"""
Module to deal with Tables used in Admin console pages

Table:

    access_action_item()                :     Selects the action item in table

    access_context_action_item()        :     Selects the action item in table right click menu

    access_link()                       :     Selects the entity from list page and navigates to
                                              given entity's details

    access_link_by_column()             :     Search by column value and access link text

    get_column_data()                   :     Get column data

    get_number_of_columns()             :     Gets number of columns present in table

    get_visible_column_names()          :     Get visible Column names

    search_for()                        :     Clears the search bar and fills given value

CVTable:

    access_action_item()                :     Selects the action item in the table action menu

    access_link()                       :     Navigates to the given entity

    search_for()                        :     Performs the search with the keyword on the table

ReactTable:

    access_action_item()                :     Selects the action item in table

    access_context_action_item()        :     Selects the action item in table right click menu

    access_link()                       :     Selects the entity from list page and navigates to
                                              given entity's details

    access_link_by_column()             :     Search by column value and access link text

    get_column_data()                   :     Get column data

    get_number_of_columns()             :     Gets number of columns present in table

    get_visible_column_names()          :     Get visible Column names

    get_all_column_names()              :     Gets all column names from table

    search_for()                        :     Clears the search bar and fills given value

    get_table_data()                    :     returns entire table data

    is_entity_present_in_column()       :     checks whether given entity is present in column or not

    get_total_rows_count()              :     gets total table rows count

    select_all_rows()                   :     selects all rows in table

    select_rows()                       :     Selectes rows based on list input

    access_toolbar_menu()               :     Access table toolbar menu to perform operation

    access_menu_from_dropdown()         :     Access table toolbar menu item from dropdown

    apply_filter_over_column()          :     Applies filter values over column in table

    clear_column_filter()               :     clears the column filter applied on table

    display_hidden_column()             :     Method to select non default column to be displayed on table

    apply_sort_over_column()            :     Applies sorting order on column

    get_grid_actions_list()             :     Gets visible grid action menu list from table

    view_by_title()                     :     Select table view from top of table title

Rfilter()       -- class which contains column filter criteria enum for react tables

Integration tests for components are added in below mentioned test cases, for any new functionality added here
add the corresponding integration method

        TC 59679        --      Test case for Rtable class(React table)

        TC 56174        --      Test case for Table class(Kendo Grid table)

"""
import enum
from time import sleep

from selenium.common.exceptions import NoSuchElementException
from selenium.webdriver.common.action_chains import ActionChains
from selenium.webdriver.common.keys import Keys

from Web.Common.page_object import (WebAction, PageService)
from Web.Common.exceptions import CVWebAutomationException


class Table:
    """Table Component used in Command Center"""

    def __init__(self, admin_console):
        self._admin_console = admin_console
        self._driver = admin_console.driver
        self._xp = "//div[contains(@class,'cv-kendo-grid')]"

    @WebAction()
    def __fetch_table_title(self):
        """Fetches the table title"""
        title = self._driver.find_element_by_xpath("//div[@class='cv-k-grid-title']/h2")
        return title.text

    @WebAction()
    def __get_column_names(self):
        """Read Column names"""
        col_xp = "//th[@role='columnheader' and not(@style='display:none')]"
        columns = self._driver.find_elements_by_xpath(self._xp + col_xp)
        return [column.text for column in columns if column.is_displayed()]

    @WebAction(delay=0)
    def __get_data_from_column_by_idx(self, col_idx):
        """Read data from column"""
        row_xp = f"//td[@role='gridcell' and not(contains(@id,':checkbox')) and not(@style='display:none')][{col_idx}]"
        return [
            column.text.strip() for column in
            self._driver.find_elements_by_xpath(self._xp + row_xp)
            if column.is_displayed()
        ]

    @WebAction()
    def __get_grid_actions_list(self):
        """Reads data from grid actions menu"""
        return self._driver.find_elements_by_xpath("//*[contains(@id,'cv-permit-actions-dropdown-menu')]/div//hr | "
                                                   "//*[contains(@id,'cv-permit-actions-dropdown-menu')]/div//a")

    @WebAction()
    def __is_search_visible(self):
        """check if search bar is available"""
        search = self._driver.find_elements_by_xpath(self._xp + "//input[@type='search']")
        return search and search[0].is_displayed()

    @WebAction()
    def __click_search(self):
        """enter search on grid"""
        search = self._driver.find_element_by_xpath(
            self._xp + "//span[contains(@class, 'k-i-search')]")
        search.click()

    @WebAction(delay=0)
    def __enter_search_term(self, value):
        """enter search on grid"""
        search = self._driver.find_element_by_xpath(self._xp + "//input[@type='search']")
        search.clear()
        search.send_keys(value)

    @WebAction(delay=1)
    def __click_actions_menu(self, entity_name, partial_selection):
        """
        Clicks Action menu
        """
        if not partial_selection:
            xp = (f"//*[text() ='{entity_name}']/ancestor::tr//"
                  f"div[contains(@class,'permittedActions')]//a")
            if not self._admin_console.check_if_entity_exists("xpath", xp):
                xp = (f"//*[text() ='{entity_name}']/ancestor::tr//"
                      f"span[contains(@class,'action-btn grid-action-icon')]")
                if not self._admin_console.check_if_entity_exists("xpath", xp):
                    xp = (f"//*[text() ='{entity_name}']/ancestor::tr//"
                          f"span[contains(@class,'grid-action-icon')]")
        else:
            xp = (f"//*[contains(text(), '{entity_name}')]/ancestor::tr//"
                  f"div[contains(@class,'permittedActions')]//a")
            if not self._admin_console.check_if_entity_exists("xpath", xp):
                xp = (f"//*[contains(text(), '{entity_name}')]/ancestor::tr//"
                      f"span[contains(@class,'action-btn grid-action-icon')]")
                if not self._admin_console.check_if_entity_exists("xpath", xp):
                    xp = (f"//*[contains(text(), '{entity_name}')]/ancestor::tr//"
                          f"span[contains(@class,'grid-action-icon')]")
        self._driver.find_element_by_xpath(xp).click()

    @WebAction(delay=1)
    def __click_action_item(self, action_item):
        """Clicks on Action item under action menu"""
        elem = self._driver.find_element_by_xpath(
            f"//ul[contains(@style,'display: block')]//a[text()='{action_item}']"
        )
        ActionChains(self._driver).move_to_element(elem).click().perform()

    @WebAction(delay=1)
    def __right_click_menu(self, entity_name):
        """
        Clicks context menu
        """
        entity = self._driver.find_element_by_xpath(
            f"//a[contains(text(),'{entity_name}')]")
        actions = ActionChains(self._driver)
        actions.move_to_element(entity).context_click().perform()

    @WebAction(delay=1)
    def __click_context_action_item(self, action_item):
        """
        Clicks action item from context click drop down
        """
        elem = self._driver.find_element_by_xpath(
            f"//div[@class='k-animation-container']//a[text()='{action_item}']"
        )
        ActionChains(self._driver).move_to_element(elem).click().perform()

    @WebAction()
    def __click_title_dropdown(self):
        """
        Clicks on title drop down on top of table
        """
        self._driver.find_element_by_xpath("//div[@class='cv-k-grid-title']//a").click()
        sleep(2)

    @WebAction()
    def __filter_by_type(self, title_to_filter):
        """
        select filter by type

        Args:
            title_to_filter   (basestring):   title to select

        """
        self._driver.find_element_by_xpath(
            f"//div[@class='k-animation-container']//a[text()='{title_to_filter}']").click()

    @WebAction(delay=0)
    def __select_row(self, name):
        """Select specified row"""
        xp = self._xp + "//*[contains(text(), '%s')]/ancestor::tr/td[contains(@id,'checkbox')]" % name
        rows = self._driver.find_elements_by_xpath(xp)
        if not rows:
            raise NoSuchElementException("Rows not found with name [%s]" % name)
        for each_row in rows:
            if not each_row.is_displayed() or 'selected' in each_row.get_attribute('class'):
                #  In case of multiple tables on same page we might face some issue
                continue
            hover = ActionChains(self._driver).move_to_element(each_row)
            hover.perform()
            sleep(1)
            each_row.click()

    @WebAction()
    def __is_tool_bar_action_visible(self, menu_id):
        """Checks whether the toolbar menu is visible or not"""
        xp = f"//div[@class='toolbar']//li[contains(@data-cv-menu-item-id, '{menu_id}')]"
        menu_items = self._driver.find_elements_by_xpath(xp)
        for menu_item in menu_items:
            if menu_item is None:
                continue
            menu_item_class = menu_item.get_attribute('class')
            if 'hidden' in menu_item_class or 'disabled' in menu_item_class:
                return True
        return False

    @WebAction()
    def __click_expand_tool_bar_menu(self):
        """Clicks on the tool bar menu's expand options button"""
        xp = "//li[@id='batch-action-menu_moreItem']"
        for element in self._driver.find_elements_by_xpath(xp):
            if element is not None and element.is_displayed():
                element.click()
                break

    @WebAction()
    def __click_tool_bar_menu(self, menu_id):
        """
        click tool bar menu in table
        Args:
            menu_id: value of attribute data-cv-menu-item-id
        """
        xp = f"//li[contains(@data-cv-menu-item-id,'{menu_id}')]"
        menu_obj = [each_element for each_element in self._driver.find_elements_by_xpath(xp) if
                    each_element.is_displayed()]
        if menu_obj:
            menu_obj[0].click()
        else:
            raise CVWebAutomationException("data-cv-menu-item-id [%s] element not found" % menu_id)

    @WebAction()
    def __expand_column_filter(self, column_name):
        """ Method to expand column filter drop down """
        col_settings_drop_down = self._driver.find_element_by_xpath(
            f"//th[@role='columnheader' and @data-title='{column_name}']"
            f"//a[@aria-label='Column Settings']")
        col_settings_drop_down.click()
        sleep(2)

    @WebAction()
    def __click_menu_item(self, menu_item):
        """Method to click on any menu item"""
        menus = self._driver.find_elements_by_xpath(
            "//div[@class='k-animation-container']//ul[@role='menubar']")
        for menu in menus:
            if menu.is_displayed():
                menu.find_element_by_xpath(f".//span[contains(text(),'{menu_item}')]").click()
                break

    @PageService()
    def __click_filter_menu_item(self):
        """ Method to click on Filter menu item """
        self.__click_menu_item("Filter")

    @PageService()
    def __click_columns_menu_item(self):
        """ Method to click on Columns menu item """
        self.__click_menu_item("Columns")

    @PageService()
    def __click_sort_item(self, ascending=True):
        """Method to click on Sort Ascending/Descending menu item"""
        menu_item = "Sort Ascending" if ascending else "Sort Descending"
        self.__click_menu_item(menu_item)

    @WebAction(delay=1)
    def __clear_integer_filter_criteria_textbox(self):
        """ Method to clear integer filter criteria textbox
        Args:
            integer_type(bool) : integer type filter
        """
        f_criteria_textbox = self._driver.find_elements_by_xpath(
            "//input[contains(@data-bind, 'value:filters[0].value')]/../input"
        )
        for textbox in f_criteria_textbox:
            if textbox.is_displayed():
                textbox.clear()
                break

    @WebAction(delay=1)
    def __clear_filter_criteria_textbox(self):
        """ Method to clear filter criteria textbox
        Args:
            integer_type(bool) : integer type filter
        """
        f_criteria_textbox = self._driver.find_elements_by_xpath(
            "//input[contains(@data-bind, 'value:filters[0].value')]"
        )
        for textbox in f_criteria_textbox:
            if textbox.is_displayed():
                textbox.clear()
                break

    @WebAction(delay=1)
    def __enter_filter_criteria(self, filter_term, integer_type=False):
        """
        Method to enter filter criteria

        Args:
            filter_term (str) : string to be filtered for a column
            integer_type(bool) : integer type filter
        """
        if integer_type:
            f_criteria = self._driver.find_elements_by_xpath(
                "//input[contains(@data-bind, 'value:filters[0].value')]/../input"
            )
        else:
            f_criteria = self._driver.find_elements_by_xpath(
                "//input[contains(@data-bind, 'value:filters[0].value')]"
            )
        for criteria in f_criteria:
            if criteria.is_displayed():
                criteria.send_keys(filter_term)
                break

    @WebAction(delay=1)
    def __enter_multi_filter_criteria(self, filter_term):
        """
        Method to enter filter criteria

        Args:
            filter_term (str) : string to be filtered for a column
        """
        f_criteria = self._driver.find_elements_by_xpath(
            "//div[contains(@class, 'k-multiselect-wrap k-floatwrap')]/input"
        )
        for criteria in f_criteria:
            if criteria.is_displayed():
                criteria.send_keys(filter_term)
                break

    @WebAction(delay=1)
    def __click_filter_button(self):
        """ Method to click on filter button """
        filter_buttons = self._driver.find_elements_by_xpath(
            "//button[@type='submit' and contains(text(),'Filter')]")
        for filter_button in filter_buttons:
            if filter_button.is_displayed():
                filter_button.click()
                break

    @WebAction()
    def __select_filter_checkbox(self, filter_term):
        """ Method to click on filter term """
        xpath = f"//li[contains(@class, 'k-item')]/label[@title='{filter_term}']"
        self._driver.find_element_by_xpath(xpath).click()

    @WebAction()
    def __click_clear_filter_button(self):
        """ Method to click on clear filter button """
        filter_buttons = self._driver.find_elements_by_xpath(
            "//button[@type='reset' and contains(text(),'Clear')]")
        for filter_button in filter_buttons:
            if filter_button.is_displayed():
                filter_button.click()
                break

    @WebAction()
    def __select_hidden_column(self, column_name):
        """ Method to click on the column to be displayed """
        self._driver.find_element_by_xpath(
            f"//li[@role='menuitemcheckbox']/span[contains(text(), '{column_name}')]"
        ).click()

    @WebAction()
    def __read_paging_info(self):
        """read paging info"""
        return self._driver.find_element_by_xpath(
            self._xp + "//span[contains(@class,'pager-info')]"
        ).text

    @WebAction()
    def __get_pagination_drop_downs(self):
        """Method returns all pagination drop down elements on page"""
        drop_downs = self._driver.find_elements_by_xpath(
            "//span[@class='k-widget k-dropdown k-header']")
        return drop_downs

    @WebAction()
    def __get_pagination_drop_down_option_elements(self, pagination_value):
        """Method to get elements of all options of given pagination value"""
        elements = self._driver.find_elements_by_xpath(
            f"//ul[@data-role='staticlist']//li[text()='{pagination_value}']")
        return elements

    @WebAction()
    def __click_expand_row(self, row_text):
        """click expand icon near rows"""
        self._driver.find_element_by_xpath(
            self._xp + f"//*[text()='{row_text}']/ancestor::tr//a[@class='k-icon k-i-expand']"
        ).click()

    @WebAction()
    def __select_all_rows(self):
        """Selects all rows"""
        xp = "//th[contains(@id,'checkbox')]"
        self._driver.find_element_by_xpath(xp).click()

    @WebAction()
    def __fetch_all_column_names(self):
        """Method to get all column names"""
        column_names = self._driver.find_elements_by_xpath("//li[@role='menuitemcheckbox']/span")
        return column_names

    @PageService()
    def get_all_column_names(self):
        """Gets all column names"""
        columns = self.__get_column_names()
        self.__expand_column_filter(columns[0])
        self.__click_columns_menu_item()
        return self.__fetch_all_column_names()

    @PageService()
    def view_by_title(self, value):
        """
        Filter by type in grid

        Args:
            value   (basestring):   title to select
        """
        self.__click_title_dropdown()
        self.__filter_by_type(value)
        self._admin_console.wait_for_completion()

    @PageService()
    def search_for(self, value):
        """
        Clears the search bar and fills in the user given value

        Args:
            value (basestring)  -- the value to be searched
        """

        self.__enter_search_term(value)
        self._admin_console.wait_for_completion()

    @PageService()
    def get_number_of_columns(self):
        """
        gets number of columns present in table
        """
        return len(self.__get_column_names())

    @PageService()
    def get_visible_column_names(self):
        """Get visible Column names"""
        return self.__get_column_names()

    @PageService()
    def get_column_data(self, column_name):
        """
        Get column data
        Args:
            column_name: Column Name
        Returns:
            list of column data
        """

        column_list = self.__get_column_names()
        if column_list:
            col_idx = column_list.index(column_name) + 1
            return self.__get_data_from_column_by_idx(col_idx)
        else:
            return []

    @PageService
    def get_grid_actions_list(self, name, group_by=False):
        """Gets visible grid actions
            Args:

                name        (str)       :   Search term to be applied on table
        """
        if self.__is_search_visible():
            self.search_for(name)
        self._admin_console.wait_for_completion()
        self.__click_actions_menu(name)
        self._admin_console.wait_for_completion()
        flatten_grid_list = []
        nested_grid_list = []
        group_list = []
        grid_actions_list = self.__get_grid_actions_list()
        grid_actions_list = [action.text for action in grid_actions_list]
        # close the action menu
        self.__click_actions_menu(name)
        if group_by:
            for action in grid_actions_list:
                if action == '':
                    nested_grid_list += [group_list]
                    group_list = []
                else:
                    group_list += [action]
            nested_grid_list += [group_list]
            return nested_grid_list
        else:
            for action in grid_actions_list:
                if action != '':
                    flatten_grid_list += [action]
            return flatten_grid_list

    @PageService()
    def access_action_item(self, entity_name, action_item, partial_selection=False):
        """
        Selects the action item in table

        Args:
            entity_name (basestring): Entity against which action item has to be selected

            action_item (basestring): action item which has to be selected

            partial_selection (bool) : flag to determine if entity name should be
            selected in case of partial match or not

        Raises:
            Exception:
                if unable to click on Action item
                or if the action item is not visible

        """
        if self.__is_search_visible():
            self.search_for(entity_name)
        self._admin_console.scroll_into_view(self._xp)
        self.__click_actions_menu(entity_name, partial_selection)
        self._admin_console.wait_for_completion()
        self.__click_action_item(action_item)
        self._admin_console.wait_for_completion()

    @PageService()
    def access_link(self, entity_name):
        """
        Access the hyperlink (eg. list page for companies, plans, users etc)

        Args:
            entity_name (basestring): Entity for which the details are to be accessed

        """
        if self.__is_search_visible():
            self.search_for(entity_name)
        self._admin_console.scroll_into_view(self._xp)
        self._admin_console.select_hyperlink(entity_name)

    @PageService()
    def access_link_by_column(self, entity_name, link_text):
        """
        search by entity_name and access by link_text

        Args:
            entity_name : name to search for in table
            link_text   : link text to click

        """
        if self.__is_search_visible():
            self.search_for(entity_name)
        self._admin_console.scroll_into_view(self._xp)
        self._admin_console.select_hyperlink(link_text)

    @PageService()
    def access_context_action_item(self, entity_name, action_item):
        """
        Selects the action item in table right click menu

        Args:
            entity_name (basestring): Entity against which action item has to be selected

            action_item (basestring): action item which has to be selected

        Raises:
            Exception:
                if unable to click on Action item
                or if the action item is not visible
        """
        if self.__is_search_visible():
            self.search_for(entity_name)
        self._admin_console.scroll_into_view(self._xp)
        self.__right_click_menu(entity_name)
        self._admin_console.wait_for_completion()
        self.__click_context_action_item(action_item)
        self._admin_console.wait_for_completion()

    @PageService()
    def access_toolbar_menu(self, menu_id):
        """
        Access tool bar menu in table
        Args:
            menu_id: value of attribute data-cv-menu-item-id in the menu
        """
        if not self.__is_tool_bar_action_visible(menu_id):
            self.__click_expand_tool_bar_menu()
        self.__click_tool_bar_menu(menu_id)
        self._admin_console.wait_for_completion()

    @PageService()
    def access_menu_from_dropdown(self, menu_id):
        """Alias for access_toolbar_menu
        Args:
            menu_id: value of attribute data-cv-menu-item-id in the menu
        """
        self.access_toolbar_menu(menu_id)

    @PageService()
    def select_rows(self, names):
        """
        Select rows which contains names
        Args:
            names                  (List)       --    string to be selected which are hyperlinks
        """
        self._admin_console.scroll_into_view(self._xp)
        for each_name in names:
            self.__select_row(each_name)

    @PageService()
    def select_all_rows(self):
        """
        Select all the rows present
        """
        self.__select_all_rows()

    def __access_filter_menu(self, column_name):
        self._admin_console.scroll_into_view(self._xp)
        self.__expand_column_filter(column_name)
        self.__click_filter_menu_item()

    @PageService()
    def apply_filter_over_integer_column(self, column_name, filter_term):
        """
        Method to apply filter on integer type column (ex: jobid in Jobs page)

        Args:
            column_name (str) : Column to be applied filter on

            filter_term (str) : value to be filtered with
        """
        self.__access_filter_menu(column_name)
        self.__clear_integer_filter_criteria_textbox()
        self.__clear_filter_criteria_textbox()
        self.__enter_filter_criteria(filter_term, integer_type=True)
        self.__click_filter_button()
        self._admin_console.wait_for_completion()

    @PageService()
    def apply_filter_over_column(self, column_name, filter_term):
        """
        Method to apply filter on given column

        Args:
            column_name (str) : Column to be applied filter on

            filter_term (str) : value to be filtered with
        """
        self.__access_filter_menu(column_name)
        self.__clear_filter_criteria_textbox()
        self.__enter_filter_criteria(filter_term)
        self.__click_filter_button()
        self._admin_console.wait_for_completion()

    @PageService()
    def apply_filter_over_column_selection(self, column_name, filter_term):
        """
        Method to apply filter on a list in given column

        Args:
            column_name (str) : Column to be applied filter on

            filter_term (str) : value to be filtered with
        """
        self.__access_filter_menu(column_name)
        self.__enter_multi_filter_criteria(filter_term)
        self.__select_filter_checkbox(filter_term)
        self.__click_filter_button()
        self.__click_filter_button()
        self._admin_console.wait_for_completion()

    @PageService()
    def apply_sort_over_column(self, column_name, ascending=True):
        """
        Method to apply a sort on the column name specified
        Args:
            column_name (str): The column to apply sort on
            ascending  (bool): Whether the sort is ascending or not
        """
        self._admin_console.scroll_into_view(self._xp)
        self.__expand_column_filter(column_name)
        self.__click_sort_item(ascending)

    @PageService()
    def display_hidden_column(self, column_name):
        """
        Method to display hidden/non-default column
        Args:
            column_name (str):  The column to be displayed
        """
        self._admin_console.scroll_into_view(self._xp)
        columns = self.__get_column_names()
        self.__expand_column_filter(columns[0])
        self.__click_columns_menu_item()
        self.__select_hidden_column(column_name)
        self._admin_console.wait_for_completion()

    @PageService()
    def clear_column_filter(self, column_name):
        """
        Method to clear filter from column

        Args:
            column_name (str) : Column name, filter to be removed from
        """
        self._admin_console.scroll_into_view(self._xp)
        self.__expand_column_filter(column_name)
        self.__click_filter_menu_item()
        self.__click_clear_filter_button()
        self._admin_console.wait_for_completion()

    @PageService()
    def get_table_data(self):
        """Dump all data in the table as json"""
        return {
            col: self.get_column_data(col)
            for col in self.__get_column_names()
        }

    @PageService()
    def is_entity_present_in_column(self, column_name, entity_name):
        """
        Check entity present
        Args:
            column_name (str) : Column Name to be searched in
            entity_name (str) : Name of entity to be checked

        """
        if self.__is_search_visible():
            self.search_for(entity_name)
        self._admin_console.scroll_into_view(self._xp)
        list_of_entities = self.get_column_data(column_name)
        if entity_name in list_of_entities:
            return True
        return False

    @PageService()
    def get_total_rows_count(self, search_keyword=None):
        """get total rows count"""
        if search_keyword:
            if self.__is_search_visible():
                self.search_for(search_keyword)
        page_txt = self.__read_paging_info()
        if page_txt:
            return page_txt.split()[-2]
        else:
            return len(self.__get_data_from_column_by_idx(1))

    @PageService()
    def set_pagination(self, pagination_value):
        """ Method to set pagination """

        drop_downs = self.__get_pagination_drop_downs()
        for dd in drop_downs:
            if dd.is_displayed():
                self._admin_console.scroll_into_view(
                    "//span[@class='k-widget k-dropdown k-header']"
                )
                ActionChains(self._driver).move_to_element(dd).click().perform()
        sleep(15)
        elements = self.__get_pagination_drop_down_option_elements(pagination_value)
        for elem in elements:
            if elem.is_displayed():
                ActionChains(self._driver).move_to_element(elem).click().perform()

    @PageService()
    def expand_row(self, row_text):
        """
        expands rows
        Args:
            row_text: text exist on row where expand has to be clicked
        """
        self.__click_expand_row(row_text)
        self._admin_console.wait_for_completion()

    @PageService()
    def hover_click_actions_sub_menu(self, entity_name, mouse_move_over_id, mouse_click_id, partial_selection=False):
        """Clicks the actions menu, hover to the element1 and clicks the element2 for an entity

                Args:
                    entity_name (str)  --  Entity name whose action menu is to be clicked

                    mouse_move_over_id (xpath) -- mouse move over element id

                    mouse_click_id (xpath)    -- mouse click element id

                    partial_selection  (bool)  --  Whether partial selection to be honored

        """
        self.__click_actions_menu(entity_name, partial_selection)
        self._admin_console.wait_for_completion()
        mouse_move_over = self._driver.find_element_by_id(mouse_move_over_id)
        mouse_click = self._driver.find_element_by_id(mouse_click_id)
        self._admin_console.mouseover_and_click(mouse_move_over, mouse_click)

class CVTable:
    """Older Table Component used in Command Center"""

    def __init__(self, admin_console):
        self.__driver = admin_console.driver
        self.__admin_console = admin_console

    @WebAction()
    def __click_entity(self, entity_name):
        """Clicks the given entity"""
        link = self.__driver.find_element_by_xpath(f"//cv-grid//a[text()='{entity_name}']")
        link.click()

    @WebAction()
    def __set_search_string(self, keyword):
        """Clears the search box and sets with the given string"""
        search_box = self.__driver.find_element_by_xpath("//cv-grid//input[@type='search']")
        search_box.click()
        search_box.clear()
        search_box.send_keys(keyword)

    @WebAction()
    def __click_actions_button(self, entity_name):
        """Clicks the action button"""
        button = self.__driver.find_element_by_xpath(
            f"//cv-grid//a[text()='{entity_name}']/../following-sibling::div//*[name() = 'svg']")
        button.click()

    @WebAction()
    def __click_action(self, entity_name, action_item):
        """Clicks the given action item"""
        option = self.__driver.find_element_by_xpath(
            f"//cv-grid//a[text()='{entity_name}']/../following-sibling::div"
            f"//a[contains(text(),'{action_item}')]")
        option.click()

    @PageService()
    def search_for(self, keyword):
        """performs the search with the given keyword on the table

        Args:
            keyword (basestring)  -- the value to be searched

        """
        self.__set_search_string(keyword)
        self.__admin_console.wait_for_completion()

    @PageService()
    def access_action_item(self, entity_name, action_item):
        """Selects the action item in the table action menu

        Args:
            entity_name (basestring): Entity against which action item has to be selected

            action_item (basestring): action item which has to be selected

        """
        self.search_for(entity_name)
        self.__click_actions_button(entity_name)
        self.__click_action(entity_name, action_item)
        self.__admin_console.wait_for_completion()

    @PageService()
    def access_link(self, entity_name):
        """Navigates to the given entity

        Args:
            entity_name (basestring): Entity which has to be accessed

        """
        self.search_for(entity_name)
        self.__click_entity(entity_name)
        self.__admin_console.wait_for_completion()

    @WebAction()
    def __get_index(self, entity_name):
        """
        Method to get index out of displayed user

        Args:
            entity_name (str) : value to get index for from table structure
        """
        index = 0
        selected = []
        element_list = self.__driver.find_elements_by_xpath(
            "//div[contains(@class,'ui-grid-render-container-body')]"
            "//div[contains(@class,'ui-grid-row ng-scope')]")
        for element in element_list:
            if self.__admin_console.is_element_present(".//div[1]/span", element):
                if entity_name == element.find_element_by_xpath(".//div[1]/span").text:
                    selected.append(entity_name)
                    break
            else:
                if entity_name == element.find_element_by_xpath(".//div[1]/div").text:
                    selected.append(entity_name)
                    break
            index += 1
        return index, selected

    @WebAction()
    def __is_checked(self, index):
        """
        Method to check if the intended checkbox is checked

        Args:
            index (int) : index of the checkbox
        """
        chkbox_xp = "//div[@class='left ui-grid-render-container-left " \
                    "ui-grid-render-container']/div[2]"
        checkboxes = self.__driver.find_elements_by_xpath(
            chkbox_xp + "//div[@role='checkbox']")
        if 'ui-grid-row-selected' in checkboxes[index].get_attribute('class'):
            return True
        return False

    @WebAction()
    def __select_checkbox(self, index):
        """ select checkbox using index given """
        xp = "//td[contains(@id,'checkbox')]"
        if not self.__admin_console.check_if_entity_exists("xpath", xp):
            xp = "//div[@class='ui-grid-cell-contents']"
        checkboxes = self.__driver.find_elements_by_xpath(xp)
        checkboxes[index].click()

    @WebAction()
    def __is_header_checked(self):
        """ Method to get checkbox state """
        header = self.__admin_console.driver.find_element_by_xpath(
            "//div[@class='left ui-grid-render-container-left "
            "ui-grid-render-container']/div[1]")
        status = False
        if 'ui-grid-all-selected' in header.find_element_by_xpath(
                ".//div[@role='checkbox']").get_attribute("class"):
            status = True
        return status

    @WebAction(delay=0)
    def __select_header_checkbox(self):
        """ Method to click on checkbox to select/deselect """
        header = self.__admin_console.driver.find_element_by_xpath(
            "//div[@class='left ui-grid-render-container-left "
            "ui-grid-render-container']/div[1]")
        header.find_element_by_xpath(".//div[@role='checkbox']").click()

    @WebAction()
    def __get_column_names(self):
        """
        Read Column Names
        Returns: List of column names

        """
        path_selector = self.__admin_console.driver.find_element_by_xpath(
            "//div[contains(@class,'ui-grid-render-container-body')]"
            "//div[contains(@class,'ui-grid-header ng-scope')]"
        )
        header_list = path_selector.text.split('\n')
        return header_list

    @WebAction()
    def __get_data_from_column_by_idx(self, col_index):
        """
        Get Column Data for given column index
        Args:
            col_index: index of Table Column

        Returns: list of column data

        """
        col_path = f"//div[contains(@class,'ui-grid-render-container-body')]\
            //div[contains(@class,'ui-grid-row ng-scope')]//div[1]/div[{col_index}]"
        return [
            column.text for column in self.__admin_console.driver.find_elements_by_xpath(col_path)
            if column.is_displayed()
        ]

    @WebAction()
    def __next_button_enabled(self):
        """ Method to check if button to next page is enabled """
        return self.__admin_console.driver.find_element_by_xpath(
            "//button[@ng-disabled='cantPageForward()']").is_enabled()

    @WebAction()
    def __click_next_button(self):
        """ Method to click next button on table"""
        self.__admin_console.driver.find_element_by_xpath(
            "//button[@ng-disabled='cantPageForward()']").click()

    @PageService()
    def get_column_data(self, column_name, data_from_all_pages=False):
        """
        Get Respective Column Data
        Args:
            column_name             (str):  Name of Column in Table
            data_from_all_pages     (bool): True if data is to be read from all
                                            pages of the table

        Returns: List of elements in required column

        """
        column_list = self.__get_column_names()
        if column_list:
            col_idx = column_list.index(column_name) + 1
            if not data_from_all_pages:
                return self.__get_data_from_column_by_idx(col_idx)
            else:
                content = self.__get_data_from_column_by_idx(col_idx)
                while True:
                    if self.__next_button_enabled():
                        self.__click_next_button()
                        self.__admin_console.wait_for_completion()
                        content.extend(self.__get_data_from_column_by_idx(col_idx))
                    else:
                        break
                return content
        else:
            return []

    @PageService()
    def select_checkbox(self, index):
        """ Public Method to select user by checking corresponding checkbox using index given """
        self.__select_checkbox(index)

    @PageService()
    def get_values_from_table(self, list_of_entities):
        """
        Method to get values from a table using the entity name
        Args:
            list_of_entities: List of entities for the given row(s)
        Returns:
            list with values from table: [<row>({<column-name>, <value>})]
            eg: [{"column1": "value1", "column2": "value2"},
            {"column1": "value3", "column2": "value4"}]
        """
        table = []
        column_values = {column: self.get_column_data(column) for column in self.__get_column_names()}
        for entity_name in list_of_entities:
            entity_name = entity_name.strip()
            index, _ = self.__get_index(entity_name)
            row = {column: value[index]for column, value in column_values.items()}
            table.append(row)
        return table

    @PageService()
    def select_values_from_table(self, list_of_entities):
        """
        Method to select values from a table

        Args:
            list_of_entities (list)  :  list of entities to be added

        Returns:
            None
        """

        entity_list = []
        selected = []
        for entity_name in list_of_entities:
            entity_list.append(entity_name.strip())

        for entity_name in entity_list:
            self.search_for(entity_name)
            index, selected_entity = self.__get_index(entity_name)
            selected.extend(selected_entity)
            if not self.__is_checked(index):
                self.__select_checkbox(index + 1)

        x_list = list(set(entity_list) - set(selected))
        if x_list:
            raise Exception(
                "There are no users with the name or the user is already added " + str(x_list))

    @PageService()
    def select_all_values_from_table(self):
        """ Method to select all values from table """
        if not self.__is_header_checked():
            self.__select_header_checkbox()

    @PageService()
    def de_select_all_values_from_table(self):
        """ Method to de-select all values from table """
        if not self.__is_header_checked():
            self.__select_header_checkbox()
            self.__select_header_checkbox()
        else:
            self.__select_header_checkbox()


class Rfilter(enum.Enum):
    """Enum for selecting criteria filter on column"""
    contains = 'Contains'
    not_contains = 'Does not contain'
    equals = 'Equals'
    not_equals = 'Does not equal'


class Rtable():
    """React Table Component used in Command Center"""

    def __init__(self, admin_console, title=None):
        self._admin_console = admin_console
        self._driver = admin_console.driver
        self._xpath = "//div[contains(@class,'flex-item')]/div[contains(@class,'grid-holder')]"
        if title:
            modified_xpath = self._xpath.replace("//", "::")
            self._xpath = f"//h1[@class='grid-title' and text()='{title}']/ancestor{modified_xpath}"

    @WebAction()
    def __select_hidden_column(self, column_name):
        """ Method to click on the column to be displayed
            Args:
                column_name     (str)       :   Name of the column
        """
        self._driver.find_element_by_xpath(
            f"//div[contains(@class,'k-column-list')]/*//span[contains(text(),'{column_name}')]"
        ).click()
        sleep(1)
        save_button = self._driver.find_element_by_xpath(
            "//div[contains(@class,'k-columnmenu-actions')]/button[contains(@class,'primary-btn') and text()='Save']")
        save_button.click()

    @WebAction()
    def __click_menu_item(self, menu_item):
        """Method to click on any menu item
            Args:
                menu_item       (str)   :   Menu item's k-icon k-i- class value
        """
        menus = self._driver.find_elements_by_xpath(
            "//div[contains(@class,'dropdown-menu show')]/div")
        for menu in menus:
            if menu.is_displayed():
                menu.find_element_by_xpath(f"*//span[contains(@class,'k-icon k-i-{menu_item}')]").click()
                sleep(2)
                break

    @WebAction()
    def __get_column_names(self):
        """Read Column names from React Table"""
        col_xp = "//th[@role='columnheader' and not(@style='display:none')]"
        columns = self._driver.find_elements_by_xpath(self._xpath + col_xp)
        return [column.text for column in columns if column.is_displayed() and column.text != '']

    @WebAction(delay=0)
    def __get_data_from_column_by_idx(self, col_idx):
        """Read data from column using column position index in React Table"""
        row_xp = f"//td[contains(@role,'gridcell') and not(@style='display:none')][{col_idx}]"
        return [
            column.text.strip() for column in
            self._driver.find_elements_by_xpath(self._xpath + row_xp)
            if column.is_displayed()
        ]

    @WebAction()
    def __is_search_visible(self):
        """check if search bar is available in React Table
            Args:
                None
            Returns:
                bool        - True if search button is visible in table
                              False if search button is not visible in table
        """
        if self.__expand_search():
            search_box = self._driver.find_element_by_xpath(
                self._xpath + "//input[contains(@class,'grid-search-input')]")
            return search_box and search_box.is_displayed()
        return False

    @WebAction()
    def __expand_search(self):
        """clicks on search button on React Table"""
        search_xpath = f"{self._xpath}//button[contains(@class,'grid-search-btn')]"
        if self._admin_console.check_if_entity_exists("xpath", search_xpath):
            search_btn = self._driver.find_element_by_xpath(search_xpath)
            search_btn.click()
            return True
        return False

    @WebAction()
    def __read_paging_info(self):
        """read paging info on React Table footer"""
        page_info_xpath = f"{self._xpath}//div[contains(@class,'k-pager-info k-label')]"
        if self._admin_console.check_if_entity_exists("xpath", page_info_xpath):
            footer_element = self._driver.find_element_by_xpath(page_info_xpath)
            return footer_element.text
        return None

    @WebAction()
    def __select_all_rows(self):
        """Selects all rows on React Table"""
        all_rows_xpath = "//th//input[contains(@class,'k-checkbox')]"
        if self._admin_console.check_if_entity_exists("xpath", all_rows_xpath):
            self._driver.find_element_by_xpath(self._xpath + all_rows_xpath).click()
        raise NoSuchElementException("SelectAll rows option not found on this table")

    @WebAction()
    def __set_search_string(self, keyword):
        """Clears the search box and sets with the given string on React Table
            Args:
                keyword         (str)       :   Keyword to be searched on table
        """
        if self.__expand_search():
            search_box = self._driver.find_element_by_xpath(
                self._xpath + "//input[contains(@class,'grid-search-input')]")
            search_box.clear()
            search_box.send_keys(keyword)

    @WebAction(delay=1)
    def __click_actions_menu(self, entity_name):
        """Clicks Action menu on React Table row
            Args:
                entity_name     (str)    :    Name of entity whose action menu has to be clicked
        """
        menu_xpath = f"//*[text() ='{entity_name}']/ancestor::tr//div[contains(@class,'action-cell')]//button"
        self._driver.find_element_by_xpath(self._xpath + menu_xpath).click()

    @WebAction(delay=1)
    def __click_action_item(self, action_item):
        """Clicks on Action item under action menu on React Table row
            Args:
                action_item         (str)   :   action name to be clicked
        """
        elem = self._driver.find_element_by_xpath(
            f"//div[contains(@class,'grid-row-actions-container')]/*//a[text() ='{action_item}']")
        elem.click()
        sleep(2)

    @WebAction(delay=0)
    def __select_row(self, name):
        """Select specified row on React Table
            Args:
                name        (str)   :   entity name which needs to be selected on table
        """
        row_xpath = f"{self._xpath}//*[contains(text(), '{name}')]/ancestor::tr/td/input[contains(@class,'k-checkbox')]"
        rows = self._driver.find_elements_by_xpath(row_xpath)
        if not rows:
            raise NoSuchElementException("Rows not found with name [%s]" % name)
        for each_row in rows:
            if not each_row.is_displayed():
                #  In case of multiple tables on same page we might face some issue
                continue
            hover = ActionChains(self._driver).move_to_element(each_row)
            hover.perform()
            sleep(1)
            each_row.click()

    @WebAction()
    def __click_more_options_menu(self):
        """clicks on 'more' option menu of react table"""
        self._admin_console.driver.find_element_by_xpath(
            f"{self._xpath}//div[contains(@class,'action-item dropdown')]//button[@id='action-grid']").click()

    @WebAction()
    def __click_tool_bar_menu(self, menu_id):
        """
        click tool bar menu in react table
        Args:
            menu_id: Name of menu attribute
        """
        menu_xpath = f"{self._xpath}//div[contains(@class,'action-item')]/button[contains(text(),'{menu_id}')]"
        if self._admin_console.check_if_entity_exists("xpath", menu_xpath):
            menu_obj = self._driver.find_element_by_xpath(menu_xpath)
            menu_obj.click()
        else:
            raise CVWebAutomationException("Action item Button [%s] element not found" % menu_id)

    @WebAction()
    def __expand_filter(self):
        """ Method to expand column filter from table"""
        col_settings_drop_down = self._driver.find_element_by_xpath(
            f"{self._xpath}//button[contains(@aria-label,'Filter')]")
        col_settings_drop_down.click()
        sleep(4)

    @WebAction()
    def __apply_filter(self):
        """Applies the filter on the table"""
        apply_element = self._driver.find_element_by_xpath(
            f"{self._xpath}//div[contains(@class,'filter-panel-button')]/button[contains(@class,'primary-btn')"
            f" and text()='Apply filters']")
        apply_element.click()
        sleep(3)

    @WebAction()
    def __apply_criteria_on_column(self, column_name, criteria):
        """applies criteria on column filter dropdown
            Args:
                column_name     (str)   :   Name of the column

                criteria        (enum)  :   Criteria to be selected for applying filter on column
        """
        dropdown_xpath = f"{self._xpath}//h4[contains(text(),'{column_name}')]/ancestor::" \
                         f"div[contains(@class,'filter-panel-row')]/*//div[contains(@id,'filter-filterConditionDropdown')]"
        dropdown = self._driver.find_element_by_xpath(dropdown_xpath)
        self._admin_console.scroll_into_view(dropdown_xpath)
        if dropdown.is_displayed():
            dropdown.click()
        sleep(3)
        criteria_xpath = f"{self._xpath}//h4[contains(text(),'{column_name}')]/ancestor::" \
                         f"div[contains(@class,'filter-panel-row')]" \
                         f"/*//div[contains(@id,'filter-filterConditionDropdown')]" \
                         f"/*//li[contains(text(),'{criteria.value}')]"
        criteria_element = self._driver.find_element_by_xpath(criteria_xpath)
        if criteria_element.is_displayed():
            criteria_element.click()
        sleep(2)

    @WebAction()
    def __apply_input_filter(self, column_name, filter_term, criteria):
        """Appllies filter term on column in Input box
        Args:
                column_name     (str)   :   Name of the column

                filter_term:    (str)   :   Search Term to be applied

                criteria        (enum)  :   Criteria to be selected for applying filter on column
        """
        self.__apply_criteria_on_column(column_name=column_name, criteria=criteria)
        input_xpath = f"{self._xpath}//h4[contains(text(),'{column_name}')]" \
                      f"/ancestor::div[contains(@class,'filter-panel-row')]" \
                      f"/*//input[contains(@id,'filter-filterInput')]"
        input_element = self._driver.find_element_by_xpath(input_xpath)
        self._admin_console.scroll_into_view(input_xpath)
        if input_element.is_displayed():
            input_element.clear()
            input_element.send_keys(filter_term)

    @WebAction()
    def __apply_dropdown_filter(self, column_name, filter_term, criteria):
        """Applies filter term on column dropdown
        Args:
                column_name     (str)   :   Name of the column

                filter_term:    (str)   :   Search Term to be applied

                criteria        (enum)  :   Criteria to be selected for applying filter on column

        """
        self.__apply_criteria_on_column(column_name=column_name, criteria=criteria)
        dropdown_xpath = f"{self._xpath}//h4[contains(text(),'{column_name}')]" \
                         f"/ancestor::div[contains(@class,'filter-panel-row')]" \
                         f"/*//div[contains(@id,'filter-filterInput-multiselect')]"
        dropdown_element = self._driver.find_element_by_xpath(dropdown_xpath)
        self._admin_console.scroll_into_view(dropdown_xpath)
        if dropdown_element.is_displayed():
            dropdown_element.click()
        sleep(2)
        list_xpath = f"{self._xpath}//h4[contains(text(),'{column_name}')]" \
                     f"/ancestor::div[contains(@class,'filter-panel-row')]" \
                     f"/*//div[contains(@id,'filter-filterInput-multiselect')]" \
                     f"/*//ul/li[contains(text(),'{filter_term}')]"
        list_element = self._driver.find_element_by_xpath(list_xpath)
        if list_element.is_displayed():
            list_element.click()
        sleep(2)

    @WebAction()
    def __expand_column_filter(self, column_name):
        """ Method to expand column filter drop down
            Args:
                 column_name        (str)   :   column name where we need to expand filter menu
        """
        col_settings_drop_down = self._driver.find_element_by_xpath(
            f"//div[contains(@class,'header-cell')]/*//a/span[contains(text(),'{column_name}')]"
            f"/ancestor::div[contains(@class,'header-cell')]/*//div/button[contains(@id,'dropdown-basic')]")
        col_settings_drop_down.click()
        sleep(2)

    @WebAction()
    def __fetch_all_column_names(self):
        """Method to get all column names"""
        column_names = self._driver.find_elements_by_xpath("//div[contains(@class,'k-column-list')]/*//span")
        names = []
        for column_element in column_names:
            names.append(column_element.text)
        # close the menu bar before returning names
        self.__expand_column_filter(names[0])
        return names

    @WebAction()
    def __get_grid_actions_list(self):
        """Reads data from grid actions menu"""
        return self._driver.find_elements_by_xpath("//div[contains(@class,'grid-row-actions-container')]//a")

    @WebAction()
    def __click_title_dropdown(self):
        """
        Clicks on title drop down on top of table
        """
        self._driver.find_element_by_xpath(f"{self._xpath}/*//div[contains(@class,'dd-header-title')]").click()
        sleep(2)

    @WebAction()
    def __filter_by_type(self, title_to_filter):
        """
        select filter by type

        Args:
            title_to_filter   (basestring):   title to select

        """
        self._driver.find_element_by_xpath(
            f"//div[contains(@class,'dd-container')]/ul/li[contains(text(),'{title_to_filter}')]").click()

    @PageService(react_frame=True)
    def get_visible_column_names(self):
        """Get visible Column names from React Table"""
        return self.__get_column_names()

    @PageService(react_frame=True)
    def get_number_of_columns(self):
        """
        gets number of columns present in React Table
        """
        return len(self.__get_column_names())

    @PageService(react_frame=True)
    def get_column_data(self, column_name):
        """
        Get column data from React Table
        Args:
            column_name: Column Name
        Returns:
            list of column data
        """

        column_list = self.__get_column_names()
        if column_list:
            col_idx = column_list.index(column_name) + 1
            return self.__get_data_from_column_by_idx(col_idx)
        return []

    @PageService(react_frame=True)
    def search_for(self, keyword):
        """performs the search with the given keyword on the React Table

        Args:
            keyword (basestring)  -- the value to be searched

        """
        self.__set_search_string(keyword)
        self._admin_console.wait_for_completion()

    @PageService(react_frame=True)
    def access_link(self, entity_name):
        """
        Access the hyperlink (eg. list page for companies, plans, users etc) in React Table

        Args:
            entity_name (basestring): Entity for which the details are to be accessed

        """
        if self.__is_search_visible():
            self.__set_search_string(entity_name)
            self._admin_console.wait_for_completion()
        self._admin_console.scroll_into_view(self._xpath)
        self._admin_console.select_hyperlink(entity_name)

    @PageService(react_frame=True)
    def access_link_by_column(self, entity_name, link_text):
        """
        search by entity_name and access by link_text on React Table

        Args:
            entity_name : name to search for in table
            link_text   : link text to click

        """
        if self.__is_search_visible():
            self.__set_search_string(entity_name)
            self._admin_console.wait_for_completion()
        self._admin_console.scroll_into_view(self._xpath)
        self._admin_console.select_hyperlink(link_text)

    @PageService()
    def get_table_data(self):
        """Dump all data in the React Table as json"""
        return {
            col: self.get_column_data(col)
            for col in self.get_visible_column_names()
        }

    @PageService(react_frame=True)
    def is_entity_present_in_column(self, column_name, entity_name):
        """
        Check entity present in React Table
        Args:
            column_name (str) : Column Name to be searched in
            entity_name (str) : Name of entity to be checked

        """
        if self.__is_search_visible():
            self.__set_search_string(entity_name)
            self._admin_console.wait_for_completion()
        self._admin_console.scroll_into_view(self._xpath)
        column_list = self.__get_column_names()
        if column_list:
            col_idx = column_list.index(column_name) + 1
            list_of_entities = self.__get_data_from_column_by_idx(col_idx)
            if entity_name in list_of_entities:
                return True
        return False

    @PageService(react_frame=True)
    def get_total_rows_count(self, search_keyword=None):
        """get total rows count from React Table
            Args:
                search_keyword      (str)   :   keyword to be searched on table
        """
        if search_keyword:
            if self.__is_search_visible():
                self.__set_search_string(search_keyword)
                self._admin_console.wait_for_completion()
        page_txt = self.__read_paging_info()
        if page_txt:
            return page_txt.split()[-2]
        return 0

    @PageService(react_frame=True)
    def select_all_rows(self):
        """
        Select all the rows present in React Table
        """
        self.__select_all_rows()

    @PageService(react_frame=True)
    def select_rows(self, names):
        """
        Select rows which contains given names in React Table
        Args:
            names                  (List)       --    entity name whose row has to be selected
        """
        self._admin_console.scroll_into_view(self._xpath)
        for each_name in names:
            self.__select_row(each_name)

    @PageService(react_frame=True)
    def access_action_item(self, entity_name, action_item):
        """
        Selects the action item in React Table

        Args:
            entity_name (basestring): Entity against which action item has to be selected

            action_item (basestring): action item which has to be selected

        Raises:
            Exception:
                if unable to click on Action item
                or if the action item is not visible

        """
        if self.__is_search_visible():
            self.__set_search_string(entity_name)
            self._admin_console.wait_for_completion()
            sleep(8)
        self._admin_console.scroll_into_view(self._xpath)
        self.__click_actions_menu(entity_name)
        self._admin_console.wait_for_completion()
        sleep(5)
        self.__click_action_item(action_item)
        self._admin_console.wait_for_completion()

    @PageService(react_frame=True)
    def access_toolbar_menu(self, menu_id):
        """
        Access tool bar menu in table
        Args:
            menu_id: Name of menu attribute
        """
        self.__click_tool_bar_menu(menu_id)
        self._admin_console.wait_for_completion()

    @PageService(react_frame=True)
    def access_menu_from_dropdown(self, menu_id):
        """Access menu item from dropdown menu
        Args:
            menu_id: name of attribute in the menu
        """
        self.__click_more_options_menu()
        sleep(5)
        menu_xpath = f"{self._xpath}//a[contains(text(),'{menu_id}') and @class='dropdown-item']"
        menu_element = self._driver.find_element_by_xpath(menu_xpath)
        menu_element.click()
        self._admin_console.wait_for_completion()

    @PageService(react_frame=True)
    def apply_filter_over_column(self, column_name, filter_term, criteria=Rfilter.contains):
        """
        Method to apply filter on given column

        Args:
            column_name (str) : Column to be applied filter on

            filter_term (str) : value to be filtered with

            criteria    (enum) : Criteria to be applied over

                **Contains**    --  Column should contain this search term [Input Box]

                **Does not contain** -- Column should not contain this search term [Input box]

                **Equals**      --  Column should match this term exactly. [Drop Down]
        """
        self.__expand_filter()
        if criteria.value in (Rfilter.contains.value, Rfilter.not_contains.value):
            self.__apply_input_filter(column_name, filter_term, criteria)
        elif criteria.value in (Rfilter.equals.value):
            self.__apply_dropdown_filter(column_name, filter_term, criteria)
        self.__apply_filter()

    @PageService()
    def apply_filter_over_integer_column(self, column_name, filter_term, criteria=Rfilter.contains):
        """
        Method to apply filter on integer type column (ex: jobid in Jobs page)

        Args:
            column_name (str) : Column to be applied filter on

            filter_term (str) : value to be filtered with

            criteria    (enum) : Criteria to be applied over

                **Contains**    --  Column should contain this search term [Input Box]

                **Does not contain -- Column should not contain this search term [Input box]

        """
        self.apply_filter_over_column(column_name, filter_term, criteria)

    @PageService()
    def apply_filter_over_column_selection(self, column_name, filter_term, criteria=Rfilter.contains):
        """
        Method to apply filter on a list in given column

        Args:
            column_name (str) : Column to be applied filter on

            filter_term (str) : value to be filtered with

            criteria    (enum) : Criteria to be applied over

                **Contains**    --  Column should contain this search term [Input Box]

                **Does not contain -- Column should not contain this search term [Input box]

        """
        self.apply_filter_over_column(column_name, filter_term, criteria)

    @PageService(react_frame=True)
    def clear_column_filter(self, column_name, criteria=Rfilter.contains):
        """
        Method to clear filter from column

        Args:
            column_name (str) : Column name, filter to be removed from

             criteria    (enum) : Criteria applied over column

                **Contains**    --  Column should contain this search term [Input Box]

                **Does not contain -- Column should not contain this search term [Input box]

                **Equals**      --  Column should match this term exactly. [Drop Down]
        """
        self.__expand_filter()
        if criteria.value in (Rfilter.contains.value, Rfilter.not_contains.value):
            input_xpath = f"{self._xpath}//h4[contains(text(),'{column_name}')]/ancestor::div[contains(@class,'filter-panel-row')]/*//input[contains(@id,'filter-filterInput')]"
            if self._admin_console.check_if_entity_exists("xpath", input_xpath):
                input_element = self._driver.find_element_by_xpath(input_xpath)
                input_element.clear()
                # Adding a trick over here to make sure backend event is triggered so that apply filter works
                input_element.send_keys("D")
                input_element.send_keys(Keys.BACK_SPACE)
                sleep(2)
                self.__apply_filter()
            else:
                raise CVWebAutomationException(
                    "Not a supported operation as column filter is not a text box to be cleared in react table")
        else:
            raise CVWebAutomationException("Not a supported operation for this filter criteria  in react tables")

    @PageService(react_frame=True)
    def display_hidden_column(self, column_name):
        """
        Method to display hidden/non-default column
        Args:
            column_name (str):  The column to be displayed
        """
        self._admin_console.scroll_into_view(self._xpath)
        columns = self.__get_column_names()
        if column_name in columns:
            return
        self.__expand_column_filter(columns[0])
        self.__click_menu_item("columns")
        self.__select_hidden_column(column_name)
        self._admin_console.wait_for_completion()

    @PageService(react_frame=True)
    def apply_sort_over_column(self, column_name, ascending=True):
        """
        Method to apply a sort on the column name specified
        Args:
            column_name (str): The column to apply sort on
            ascending  (bool): Whether the sort is ascending or not
        """
        self._admin_console.scroll_into_view(self._xpath)
        self.__expand_column_filter(column_name)
        menu_item = "sort-asc" if ascending else "sort-desc"
        self.__click_menu_item(menu_item)
        self._admin_console.wait_for_completion()

    @PageService(react_frame=True)
    def get_all_column_names(self):
        """Gets all column names"""
        columns = self.__get_column_names()
        self.__expand_column_filter(columns[0])
        self.__click_menu_item("columns")
        return self.__fetch_all_column_names()

    @PageService(react_frame=True)
    def get_grid_actions_list(self, name, group_by=False):
        """Gets visible grid actions
            Args:
                name        (str)   :   Search term which needs to be applied on table

                group_by    (bool)  :   Specifies whether to do grouping or not on action item list
        """
        if self.__is_search_visible():
            self.__set_search_string(name)
            self._admin_console.wait_for_completion()
        self.__click_actions_menu(name)
        self._admin_console.wait_for_completion()
        flatten_grid_list = []
        nested_grid_list = []
        group_list = []
        grid_actions_list = self.__get_grid_actions_list()
        grid_actions_list = [action.text for action in grid_actions_list]
        if group_by:
            for action in grid_actions_list:
                if action == '':
                    nested_grid_list += [group_list]
                    group_list = []
                else:
                    group_list += [action]
            nested_grid_list += [group_list]
            return nested_grid_list
        else:
            for action in grid_actions_list:
                if action != '':
                    flatten_grid_list += [action]
            return flatten_grid_list

    @PageService(react_frame=True)
    def expand_row(self, row_text):
        """
        expands rows
        Args:
            row_text: text exist on row where expand has to be clicked
        """
        # To do
        pass

    @PageService(react_frame=True)
    def set_pagination(self, pagination_value):
        """ Method to set pagination on React Table"""

        # To do
        pass

    @PageService(react_frame=True)
    def access_context_action_item(self, entity_name, action_item):
        """
        Selects the action item in table right click menu

        Args:
            entity_name (basestring): Entity against which action item has to be selected

            action_item (basestring): action item which has to be selected

        Raises:
            Exception:
                if unable to click on Action item
                or if the action item is not visible
        """
        # To do
        pass

    @PageService(react_frame=True)
    def view_by_title(self, value):
        """
        Filter by type in grid

        Args:
            value   (basestring):   title to select
        """
        self.__click_title_dropdown()
        self.__filter_by_type(value)
        self._admin_console.wait_for_completion()
