""" Module for panel like component that opens in the right side of the screen

ModalPanel:

    cancel()                        :       Cancel panel

    submit()                        :       Submit panel irrespective of the text in submit button

    title()                         :       Returns the title of panel

    access_tab()                    :       Access tab inside panel

    search_and_select()             :       Method to transfer select a value from the search and select input

Backup:

    submit_backup()                 :       Submits backup job in the panel


PanelInfo:

    enable_toggle()                 :       Enables the toggle bar if disabled

    disable_toggle()                :       Disables the toggle bar if enabled

    get_toggle_element()            :       Gets WebElement corresponding to the label

    is_toggle_enabled()             :       Checks if toggle is enabled

    edit_tile()                     :       click on edit icon for tile contents

    add_tile_entity()               :       click on Add for specific entity inside tile

    edit_tile_entity()              :       click on edit for specific entity inside tile

    get_details()                   :       Gets all the information contained in the panel

    check_if_hyperlink_exists_on_tile():    Checks if hyperlink on tile exists

    open_hyperlink_on_tile()        :       Opens hyperlink on tile

    save_dropdown_selection         :       Clicks on the tick mark near dropdown to save the selection

RPanelInfo:

    enable_toggle()                 :       To be implemented

    disable_toggle()                :       To be implemented

    get_toggle_element()            :       To be implemented

    is_toggle_enabled()             :       To be implemented

    edit_tile()                     :       click on edit icon for tile contents

    add_tile_entity()               :       to be implemented

    edit_tile_entity()              :       to be implemented

    get_details()                   :       Gets all the information contained in the panel

    check_if_hyperlink_exists_on_tile():    Checks if hyperlink on tile exists

    open_hyperlink_on_tile()        :       Opens hyperlink on tile

Security:

    edit_security_association       :       Add/remove security associations

    get_details                     :       Gets all the information contained in the Security panel

Dropdown:

    select_drop_down_values         :       Select values in isteven-multi-select dropdown

    deselect_drop_down_values       :       Deselect values in isteven-multi-select dropdown

RDropDown:
    
    get_values_of_drop_down         :       Returns the values in a drop down next to the
    provided label value

    get_selected_values             :       Returns the list of selected values of the drop down

    select_drop_down_values         :       select values from drop down

    deselect_drop_down_values       :       deselect values from drop down

Integration tests for components are added in below mentioned test cases, for any new functionality added here
add the corresponding integration method

        TC 59742        --      Test case for RPanelInfo class(React Panel)
"""

import time
from abc import ABC
from enum import Enum
from selenium.webdriver.common.keys import Keys
from selenium.common.exceptions import NoSuchElementException

from Web.Common.exceptions import CVWebAutomationException
from Web.Common.page_object import PageService, WebAction


class ModalPanel(ABC):
    """ Modal class """

    def __init__(self, admin_console):
        """ Initialize the base panel

        Args:
            admin_console: instance of AdminConsoleBase

        """
        self._driver = admin_console.driver
        self._admin_console = admin_console
        self.__xp = "*//div[@class='button-container']"
        self._dropdown = DropDown(admin_console)

    @WebAction(delay=2)
    def __click_submit(self):
        """ Clicks submit button of the panel doesnt matter about the text in it"""
        submit_buttons = self._driver.find_elements_by_xpath(
            "//button[contains(@class,'btn btn-primary cvBusyOnAjax')]")
        for button in submit_buttons:
            if button.is_displayed():
                button.click()
                break

    @WebAction()
    def __click_cancel(self):
        """ Clicks Cancel button of the panel"""
        self._driver.find_element_by_xpath(
            "//button[contains(@class,'btn btn-default') and contains(text(),'Cancel')]"
        ).click()

    @WebAction()
    def __click_close(self):
        """ Clicks Close button of the panel"""
        self._driver.find_element_by_xpath(
            "//button[contains(@class,'btn btn-default') and normalize-space(text())='Close']"
        ).click()

    @WebAction(delay=1)
    def _expand_accordion(self, name):
        """Clicks the heading of an accordion"""

        if self._admin_console.check_if_entity_exists('xpath',
                                                      f'//span[contains(text(), "{name}")]/ancestor::a'):
            element = self._driver.find_element_by_xpath(
                f'//span[contains(text(), "{name}")]/ancestor::a'
            )
            element.click()

        elif self._admin_console.check_if_entity_exists('xpath', f'//span[contains(text(), "{name}")]'):
            element = self._driver.find_element_by_xpath(f'//span[contains(text(), "{name}")]')
            element.click()

    @WebAction()
    def _is_visible(self, element_id):
        """Checks if the element is visible"""
        try:
            element = self._driver.find_element_by_id(element_id)
            assert element.is_displayed() is True
            return True
        except (AssertionError, NoSuchElementException):
            return False

    @WebAction()
    def __click_tab(self, tab_text):
        """
        clicks on tab inside panel
        Args:
            tab_text: localized tab text
        """
        self._driver.find_element_by_xpath(
            f'//a[@data-ng-click="selectTab(pane)" and contains(text(),"{tab_text}")]'
        ).click()

    def _expand_accordion_if_not_visible(self, element_id, accordion_name):
        """checks for visibility"""
        if self._is_visible(element_id):
            return
        self._expand_accordion(accordion_name)

    @WebAction()
    def __read_title(self):
        """Reads Modal panel title"""
        return self._driver.find_element_by_xpath("//div[@class='modal-content']//h1").text

    @PageService()
    def title(self):
        """Returns the title of panel"""
        return self.__read_title()

    @PageService()
    def submit(self, wait_for_load=True):
        """submits the panel"""
        self.__click_submit()
        if wait_for_load:
            self._admin_console.wait_for_completion()

    @PageService()
    def cancel(self):
        """Cancel the panel"""
        self.__click_cancel()
        self._admin_console.wait_for_completion()

    @PageService()
    def close(self):
        """Close the panel"""
        self.__click_close()
        self._admin_console.wait_for_completion()

    @PageService()
    def access_tab(self, tab_text):
        """
        Access tab inside panel
        Args:
            tab_text: localized tab text
        """
        self.__click_tab(tab_text)
        self._admin_console.wait_for_completion()

    @WebAction()
    def __expand_search_and_select(self, label):
        """Method to expand the user group search drop down"""
        user_group_input_xpath = f"//label[./text()='{label}']/following-sibling::div//span"
        self._driver.find_element_by_xpath(user_group_input_xpath).click()

    @WebAction()
    def __enter_search_value(self, user_group):
        """ Method to search for user group in user group input"""
        search_box = self._driver.find_element_by_xpath("//div[@id='select2-drop']//input")
        search_box.send_keys(user_group)

    @WebAction()
    def __select_value(self, value):
        """ Method to select new owner from Transfer ownership pop-up """
        search_results = self._driver.find_element_by_xpath(
            "//ul[contains(@class,'select2-results')]")
        search_results.find_element_by_xpath(f".//span[contains(text(),'({value})')]").click()

    @PageService()
    def search_and_select(self, label, select_value):
        """
        Method to transfer select a value from the search and select input

        Args:
            label        (basestring): Label text of the label next to which the input

            select_value (basestring): Value to select from the search and select input

        Raises:
            Exception:
                If there is no option to transfer owner
        """
        self.__expand_search_and_select(label)
        self.__enter_search_value(select_value)
        self._admin_console.log.info("Waiting for 10 seconds.")
        time.sleep(10)
        self.__select_value(select_value)


class Backup(ModalPanel):
    """ Class for backup panel"""

    class BackupType(Enum):
        """ type of backup to submit """
        FULL = "FULL"
        INCR = "INCREMENTAL"
        SYNTH = "SYNTHETIC_FULL"
        TRANSAC = "TRANSACTION_LOG"
        DIFF = "DIFFERENTIAL"

    @WebAction()
    def __set_backup_level(self, backup_type):
        """ Sets backup type

        Args:
            backup_type (BackupType): Type of backup should be one among the types defined in
                                      BackupType enum
        """
        self._driver.find_element_by_xpath(
            "//input[@type='radio' and @value='" + backup_type.value + "']").click()

    @WebAction()
    def __enable_notify_via_email(self):
        """ Enables notify via email checkbox """
        self._driver.find_element_by_xpath("*//span[contains(text(),'Notify via email')]").click()

    @WebAction()
    def _select_backupset_and_subclient(self, backupset_name, subclient_name):
        """Selcts the required backupset and subclient
        Args:
            backupset_name (String) : Name of backupset to be selected

            subclient_name (String) : Name of the subclient to be selected

        """
        self._driver.find_element_by_xpath(
            f'//label[text()="{backupset_name}"]/../..//label[text()="{subclient_name}"]'
            "/../preceding-sibling::span[1]//input"
        ).click()

    @PageService()
    def submit_backup(self, backup_type, backupset_name=None, subclient_name=None, notify=False,
                      incremental_with_data=False, cumulative=False):
        """ Submits backup job in the panel
        Args:
            backup_type (BackupType): Type of backup should be one among the types defined in
                                      BackupType enum
            backupset_name (str)    : backupset name
            subclient_name (str)    : subclient name
            notify (bool)           : to enable by email
            incremental_with_data(bool)    : To enable data in incremental backup
            cumulative  (bool)      : to enable cumulative backup

        Returns:
            job_id: job id from notification
        """

        if backupset_name:
            self._select_backupset_and_subclient(backupset_name, subclient_name)
            self._admin_console.wait_for_completion()
            self.submit(wait_for_load=True)

        if backup_type not in self.BackupType:
            raise CVWebAutomationException("Invalid backup type, "
                                           "format should be one among the type in BackupType")
        self.__set_backup_level(backup_type)
        if incremental_with_data:
            self._admin_console.checkbox_select('data')
        if cumulative:
            self._admin_console.checkbox_select('cumulative')
        if notify:
            self.__enable_notify_via_email()
        self.submit(wait_for_load=False)
        _jobid = self._admin_console.get_jobid_from_popup()
        self._admin_console.wait_for_completion()
        return _jobid


class PanelInfo(ModalPanel):
    """ Gets all the page details """

    def __init__(self, admin_console, title=None):
        """ Initialize the panel info object

        Args:
            admin_console : instance of AdminConsoleBase

            title (str) : Header of the panel/tile to be handled
        """
        super(PanelInfo, self).__init__(admin_console)
        if title:
            self.__tile_xp = f"//span[contains(@title,'{title}')]/ancestor::div[@class='page-details group']"
        else:
            self.__tile_xp = ""

    @WebAction(delay=0)
    def __toggle_present(self, tag):
        """ Checks whether toggle is present in tag

        Args:
            tag: panel details

        Returns (bool) : If toggle exists in the tag

        """
        tags = tag.find_elements_by_xpath('.//Toggle-Control')
        return len(tags) > 0

    @WebAction(delay=0)
    def __toggle_details(self, tag):
        """ Checks toggle status

        Args:
            tag: panel details

        Returns (str) : Whether toggle is ON or OFF

        """
        toggle_class = tag.find_element_by_xpath(
            './/Toggle-Control/div').get_attribute('class')
        toggle_value = "OFF"
        if "enabled" in toggle_class:
            toggle_value = "ON"
        return toggle_value

    @WebAction()
    def is_toggle_enabled(self, element):
        """ Checks if toggle is enabled

        Args:
            element: WebElement corresponding to the toggle option.

        Returns (bool) : True if toggle is enabled

        """
        return 'enabled' in element.get_attribute('class')

    @WebAction()
    def get_toggle_element(self, label):
        """ Gets WebElement corresponding to the label.

        Args:
            label   (basestring):   Label corresponding to the toggle option.

        Returns : toggle WebElement

        """
        return self._driver.find_element_by_xpath(
            self.__tile_xp +
            "//span[contains(text(), '" + label + "')]/ancestor::li//toggle-control"
            "/div[contains(@class,'cv-material-toggle cv-toggle')]")

    @PageService()
    def enable_toggle(self, label):
        """
        Enables the toggle bar if disabled,
        Args:
            label   (basestring):   Label corresponding to the toggle option.

        """

        element = self.get_toggle_element(label)

        if not self.is_toggle_enabled(element):
            element.click()
            self._admin_console.wait_for_completion()

    @PageService()
    def disable_toggle(self, label):
        """
        Disables the toggle bar if enabled

        Args:
            label   (basestring):   Label corresponding to the toggle option.

        """

        element = self.get_toggle_element(label)

        if self.is_toggle_enabled(element):
            element.click()
            self._admin_console.wait_for_completion()

    @WebAction()
    def __panel_details(self):
        """ Retrieves panel details

        Returns :
            (list) : all visible tags containing text or toggle

        """

        info_xp = ("//span/following-sibling::div//*[contains(@class, 'pageDetailColumn') or "
                   "contains(@class,'pageDetail-three-columns-first') or "
                   "contains(@class,'pageDetail-three-columns-second') or "
                   "contains(@ng-if,'validationInfo')]")

        info_xp = self.__tile_xp + info_xp
        tags = self._driver.find_elements_by_xpath(info_xp)
        return [each_tag for each_tag in tags if each_tag.is_displayed()]

    @WebAction(delay=0)
    def __is_link_present(self, tag=None):
        """ Check if panel contains list of elements instead of pair """
        if tag:
            if not self._admin_console.is_element_present("./following-sibling::span", tag) and \
                    not self._admin_console.is_element_present("./following-sibling::a"
                                                               "[contains(@class, 'pageDetailColumn')]", tag):
                if self._admin_console.is_element_present("./following-sibling::a", tag):
                    return True
            return False
        else:
            links = self._driver.find_elements_by_xpath(
                self.__tile_xp + "//ul[contains(@class,'list-style')]/li/a"
            )
            for link in links:
                if link.is_displayed():
                    return True
            return False

    @WebAction(delay=0)
    def __fetch_panel_list(self):
        """
        fetch all visible anchor tags from panel

        Returns:
            (list) : list of elements displayed in the panel
         """
        list_elements = self._driver.find_elements_by_xpath(
            self.__tile_xp + "//ul[contains(@class,'list-style__row group')]/li")
        values = []
        for list_element in list_elements:
            values.append(list_element.text)
        return values

    @WebAction(delay=0)
    def __fetch_panel_links(self, tag=None):
        """
        fetch all visible anchor tags from panel

        Returns:
            (list) : all visible anchor tags displayed in the panel
         """
        if tag:
            link = tag.find_element_by_xpath("./following-sibling::a")
            return link.text
        else:
            links = self._driver.find_elements_by_xpath(
                self.__tile_xp + "//ul[contains(@class,'list-style')]//a")
            values = []
            for link in links:
                if link.is_displayed():
                    values.append(link.text)
            return values

    @WebAction(delay=0)
    def __copy_label_exist(self, tag):
        """Returns True if copy label exist"""
        # sample in company page with authcode
        return tag.find_elements_by_xpath('./label/span[@uib-tooltip="Copy to clipboard"]')

    @WebAction(delay=1)
    def __expand_cv_accordion(self, name):
        """Clicks the heading of an accordion inside panel"""
        xpath = self.__tile_xp + f'//span[@class="cv-accordion-text" and text()="{name}"]/ancestor::a'

        if self._admin_console.check_if_entity_exists('xpath', xpath):
            element = self._driver.find_element_by_xpath(xpath)
            element.click()

    @PageService()
    def get_details(self):
        """ Gets all the information contained in the panel

        Returns :
            details (dict) : Details of the panel in key value pair

        """
        details = {}
        self._expand_accordion('View More')
        tags = self.__panel_details()
        if not tags:
            if self.__is_link_present():
                values = self.__fetch_panel_links()
                return values
            else:
                values = self.__fetch_panel_list()
                return values
        tag_count = 0
        key = None
        for each_tag in tags:
            if self.__copy_label_exist(each_tag):
                continue
            tag_count += 1
            if tag_count % 2 != 0 and self.__is_link_present(each_tag):     # for this if condition, method
                key = each_tag.text                                         # panel_details returns non link tags only
                value = self.__fetch_panel_links(each_tag)                  # and thus links are handled through
                details.update({key: value})                                # following sibling in __fetch_panel_links
                tag_count += 1
            elif tag_count % 2 != 0:
                key = each_tag.text
            else:
                if self.__toggle_present(each_tag):
                    value = self.__toggle_details(each_tag)
                else:
                    value = each_tag.text
                details.update({key: value})
        return details

    @WebAction()
    def add_tile_entity(self, entity_name):
        """click on Add for specific entity inside tile """
        entity_add_icon = self._driver.find_element_by_xpath(
            self.__tile_xp +
            f"//span[contains(text(),'{entity_name}')]/ancestor::li//a[contains(text(),'Add')]"
        )
        entity_add_icon.click()

    @WebAction()
    def edit_tile(self):
        """click on edit icon for tile contents """
        tile_edit_icon = self._driver.find_element_by_xpath(
            self.__tile_xp +
            "//div[contains(@class,'page-details-box-links')]//a"
        )
        tile_edit_icon.click()

    @WebAction()
    def edit_tile_entity(self, entity_name):
        """click on edit for specific entity inside tile """
        xpath = self.__tile_xp + f"//span[contains(text(),'{entity_name}')]" \
                                 f"/ancestor::li//a[contains(text(),'Edit')]"
        if not self._admin_console.check_if_entity_exists("xpath", xpath):
            xpath = self.__tile_xp + \
                    f"//span[contains(text(),'{entity_name}')]/ancestor::li//span[contains(text(),'Edit')]"
        entity_edit_icon = self._driver.find_element_by_xpath(xpath)
        entity_edit_icon.click()

    @WebAction()
    def open_hyperlink_on_tile(self, hyperlink):
        """ Opens hyperlink on tile

        Args:
            hyperlink : hyperlink to be clicked on
            """
        tile_hyperlink = self._driver.find_element_by_xpath(
            self.__tile_xp + f"//a[contains(text(),'{hyperlink}')]"
        )
        tile_hyperlink.click()

    @WebAction()
    def check_if_hyperlink_exists_on_tile(self, hyperlink):
        """ Checks if hyperlink on tile exists

        Args:
            hyperlink : hyperlink to be clicked on

        Returns:
            True if hyperlink exists
            False if hyperlink doesn't exist
            """
        tile_hyperlink_xpath = self.__tile_xp + f"//a[contains(text(),'{hyperlink}')]"
        return self._admin_console.check_if_entity_exists('xpath', tile_hyperlink_xpath)

    @WebAction()
    def __click_confirm(self):
        """Click on confirm"""
        self._admin_console.driver.find_element_by_xpath(
            self.__tile_xp + "//a[@title='Confirm']").click()

    def confirm(self):
        """Click on confirm"""
        self.__click_confirm()

    @WebAction()
    def __click_tick_mark_after_dropdown_selection(self, label):
        """Clicks on the tick mark near dropdown to save the selection
        Args:
            label  :    Name of label which was edited
        """
        xpath = self.__tile_xp + f"//span[normalize-space()='{label}']" \
            f"/..//*[@class='k-icon k-i-check']"
        if not self._admin_console.check_if_entity_exists("xpath", xpath):
            xpath = self.__tile_xp + f"//span[normalize-space()='{label}']" \
                f"/../..//*[@class='k-icon k-i-check']"
        tick_mark = self._driver.find_element_by_xpath(xpath)
        tick_mark.click()

    @PageService()
    def save_dropdown_selection(self, label):
        """Clicks on the tick mark near dropdown to save the selection
                Args:
                    label  :    Name of label which was edited
                """
        self.__click_tick_mark_after_dropdown_selection(label)
        self._admin_console.wait_for_completion()


class RPanelInfo(ModalPanel):
    """ Gets all the details for React Panel """

    def __init__(self, admin_console, title=None):
        """ Initialize the panel info object

        Args:
            admin_console : instance of AdminConsoleBase

            title (str) : Header of the panel/tile to be handled
        """
        super(RPanelInfo, self).__init__(admin_console)
        if title:
            self.__tile_xp = f"//h2[(@class='tile-title' and text()='{title}')]/ancestor::div[@class='tile']"
        else:
            self.__tile_xp = ""

    @WebAction(delay=0)
    def __toggle_present(self, tag):
        """ To be implemented """
        pass

    @WebAction(delay=0)
    def __toggle_details(self, tag):
        """ To be implemented """
        pass

    @PageService()
    def __panel_details(self):
        """ Retrieves panel details

        Returns :
            (list) : all visible tags containing text or toggle
        """
        xpath = "//div[@class='tile-row']"
        info_xp = self.__tile_xp + xpath
        tiles = self._driver.find_elements_by_xpath(info_xp)
        for elem in tiles:
            if elem.is_displayed():
                info_xp = "//div[@class='tile-row-value-display' or @class='tile-row-label']"
                info_xp = self.__tile_xp + info_xp
                tags = self._driver.find_elements_by_xpath(info_xp)
                return [each_tag for each_tag in tags]

    @PageService()
    def __is_link_present(self, tag=None):
        """ Check if tag is a link or not

        Args :
            tag: Web element to check
        """
        if tag:
            if self._admin_console.is_element_present("./following-sibling::a", tag):
                return True

        return False

    @WebAction(delay=0)
    def __fetch_panel_links(self, tag=None):
        """
        fetch all visible anchor tags from panel

        Returns:
            (list) : all visible anchor tags displayed in the panel
         """
        if tag:
            link = tag.find_element_by_xpath("./following-sibling::a")
            return link.text

        return ""

    @PageService()
    def __has_overflow_item(self, tag):
        """ Checks if the web element contains overflow list items

        Args :
            tag: Web element to check for overflow items
        """
        if tag:
            if self._admin_console.is_element_present(".//div"
                                                      "[contains(@class,'overflow-list-container')]", tag):
                return True

        return False

    @WebAction()
    def __click_overflow_dropdown(self, tag=None):
        """ click overflow dropdown """
        xpath = "//a[@class='overflow-list-item overflow-dropdown-link']"

        if self._admin_console.is_element_present(xpath, tag):
            self._driver.find_element_by_xpath(xpath).click()

    @WebAction()
    def __get_overflow_items(self, tag):
        """ Gets the data from overflow list container

        Args :
            tag: Web element to get overflow items from

        Returns :
            (list) : List of overflow items
        """
        xpath = "//span[@class='overflow-list-item']"

        self.__click_overflow_dropdown(tag)
        items = tag.find_elements_by_xpath(xpath)
        overflow_list = []
        for item in items:
            if item.is_displayed():
                if item.text[-1] == " ":
                    overflow_list.append(item.text[:-2])
                else:
                    overflow_list.append(item.text)
        return overflow_list

    @WebAction()
    def __click_tile_action(self):
        """ Click on edit icon for tile contents """
        tile_edit_icon = self._driver.find_element_by_xpath(
            self.__tile_xp +
            "//div[contains(@class,'tile-header-actions')]//a"
        )
        tile_edit_icon.click()

    @WebAction()
    def __click_hyperlink(self, hyperlink):
        """ Opens hyperlink on tile

        Args:
            hyperlink : hyperlink to be clicked on
        """
        self.__click_overflow_dropdown()
        self._admin_console.select_hyperlink(hyperlink)

    @PageService(react_frame=True)
    def edit_tile(self):
        """ Edit tile info """
        self.__click_tile_action()

    @PageService(react_frame=True)
    def add_tile_entity(self):
        """ To be implemented """
        pass

    @PageService(react_frame=True)
    def edit_tile_entity(self):
        """ To be implemented """
        pass

    @WebAction()
    def is_toggle_enabled(self, element):
        """ To be implemented """
        pass

    @WebAction()
    def get_toggle_element(self, label):
        """ To be implemented """
        pass

    @PageService()
    def enable_toggle(self, label):
        """ To be implemented """
        pass

    @PageService()
    def disable_toggle(self, label):
        """ To be implemented """
        pass

    @PageService(react_frame=True)
    def get_details(self):
        """ Gets all the information contained in the panel

        Returns :
            details (dict) : Details of the panel in key value pair

        """
        details = {}
        tags = self.__panel_details()
        tag_count = 0
        key = None
        for each_tag in tags:
            tag_count += 1
            if tag_count % 2 != 0:
                key = each_tag.text
                if key == '':
                    tag_count += 1
            else:
                if self.__has_overflow_item(each_tag):
                    value = self.__get_overflow_items(each_tag)
                else:
                    value = each_tag.text
                details.update({key: value})
        return details

    @PageService(react_frame=True)
    def open_hyperlink_on_tile(self, hyperlink):
        """ Opens the provided hyperlink

        Args:
            hyperlink: hyperlink to be clicked
        """
        self.__click_hyperlink(hyperlink)

    @PageService(react_frame=True)
    def check_if_hyperlink_exists_on_tile(self, hyperlink):
        """ Checks if hyperlink on tile exists

        Args:
            hyperlink : hyperlink to be clicked on

        Returns:
            True if hyperlink exists
            False if hyperlink doesn't exist
            """
        tile_hyperlink_xpath = self.__tile_xp + f"//a[contains(text(),'{hyperlink}')]"
        return self._admin_console.check_if_entity_exists('xpath', tile_hyperlink_xpath)


class Security(ModalPanel):
    """ Class to edit security panel """

    def __init__(self, admin_console):
        """ Initialize the panel info object

        Args:
            admin_console : instance of AdminConsoleBase
        """
        super(Security, self).__init__(admin_console)
        self.__tile_xp = f"//span[@title='Security']/ancestor::div[@class='page-details group']"

    @WebAction()
    def __remove_user(self, user, role):
        """remove operator """
        self._driver.find_element_by_xpath(
            f"//label[@title='{user}']/following-sibling::label[@title='{role}']/../span/a"
        ).click()

    @WebAction()
    def __expand_user_search_bar(self):
        """expand operator search bar """
        self._driver.find_element_by_xpath(
            f"//div[contains(@class,'select2-container add-user-type')]").click()

    @WebAction()
    def __select_user(self, user):
        """select operator user in add operator panel """
        input_boxes = self._driver.find_elements_by_xpath("//div[@class='select2-search']/input")
        for box in input_boxes:
            if box.is_displayed():
                box.send_keys(user)
        time.sleep(20)
        self._driver.find_element_by_xpath(
            f"//span[contains(text(),'({user})')]").click()

    @WebAction()
    def __click_add_user(self):
        """click on add button in add operator panel """
        self._driver.find_element_by_xpath(
            "//button[@data-ng-click='addNewUserGroup()']").click()

    @PageService()
    def edit_security_association(self, associations, add=True):
        """
        edit security association

        Args:
             associations (dict) : dictionary containing user and role pairs
                Eg. -> associations = {
                                        'User1' : [['View'], ['Alert owner']],
                                        'master': [['Master'], ['Create Role', 'Edit Role', 'Delete Role']]
                                      }
            add (boolean) : True means add association, False means remove
        """
        PanelInfo(self._admin_console, 'Security').edit_tile()
        self._admin_console.wait_for_completion()

        if not add:
            for user, roles in associations.items():
                for role in roles:
                    if len(role) > 1:
                        joined_text = ' , '.join(role)
                        role = [f'[Custom] - {joined_text} ']  # Eg: Master user can have custom multiple roles
                    self.__remove_user(user, role)

        if add:
            for user, roles in associations.items():
                for role in roles:
                    if len(role) > 1:
                        continue                         # Can only add a single role at a time
                    self.__expand_user_search_bar()
                    self.__select_user(user)
                    self._admin_console.select_value_from_dropdown("adduserId", role)
                    self.__click_add_user()
                    self._admin_console.wait_for_completion()

        self.submit()
        self._admin_console.check_error_message()

    @WebAction()
    def __get_row(self):
        """ Method to get the Security detail row"""
        return self._driver.find_elements_by_xpath("//*[@id='tileContent_Security']/div[1]/div/ul[2]/li")

    @WebAction()
    def __get_user(self, row):
        """ Method to get user on row provided as input"""
        return row.find_element_by_xpath("./span[1]").text

    @WebAction()
    def __get_role(self, row):
        """ Method to get role on row provided as input"""
        return row.find_element_by_xpath("./span[2]").text

    @PageService()
    def get_details(self):
        """ Get all details displayed in the Security tile """
        rows = self.__get_row()
        temp_dict = {}
        for row in rows:
            nested_key = self.__get_user(row)
            nested_value = self.__get_role(row)
            nested_value_list = []
            if "[Custom] - " in nested_value:
                nested_value = nested_value[11:]
                nested_value_list = nested_value.split(" , ")
            else:
                nested_value_list.append(nested_value)
            if nested_key in temp_dict:
                temp_dict[nested_key].append(nested_value_list)
            else:
                temp_dict[nested_key] = [nested_value_list]
            temp_dict[nested_key].sort()
        return temp_dict


class DropDown:
    """ Class to handle isteven-multi-select drop down related operations """

    def __init__(self, admin_console):
        self.__admin_console = admin_console
        self.__driver = admin_console.driver

    @WebAction()
    def __get_drop_down_list(self):
        """get drop down list """
        drop_down_list = self.__driver.find_elements_by_xpath("//isteven-multi-select")
        return drop_down_list

    @WebAction()
    def __get_drop_down_by_id(self, drop_down_id):
        """ Method to get drop down based on id provided as input"""
        if self.__admin_console.check_if_entity_exists("xpath", f"//isteven-multi-select[@id='{drop_down_id}']"):
            drop_down = self.__driver.find_element_by_xpath(f"//isteven-multi-select[@id='{drop_down_id}']")
        else:
            drop_down = self.__driver.find_element_by_xpath(f"//isteven-multi-select[@directive-id='{drop_down_id}']")
        return drop_down

    @WebAction()
    def __expand_drop_down(self, drop_down):
        """expand drop down """
        drop_down.click()

    @WebAction()
    def __select_none(self, drop_down):
        """click 'Select none' icon and de-select all values """
        if self.__admin_console.is_element_present(
                ".//div[contains(@class,'helperContainer')]/following-sibling::div/div[1]//span[text()='Select none']"
                "/ancestor::div[contains(@class,'multiSelectItem')]", drop_down):
            drop_down.find_element_by_xpath(
                ".//div[contains(@class,'helperContainer')]/following-sibling::div/div[1]//span[text()='Select none']"
                "/ancestor::div[contains(@class,'multiSelectItem')]").click()

    @WebAction()
    def __select_all(self, drop_down):
        """click 'Select all' icon and select all values from drop down """
        if self.__admin_console.is_element_present(
                ".//div[@class='helperContainer ng-scope']/div[1]/button[1]", drop_down):
            drop_down.find_element_by_xpath(
                ".//div[@class='helperContainer ng-scope']/div[1]/button[1]").click()

    @WebAction()
    def __search_entity(self, drop_down, entity):
        """search for an entity """
        if self.__admin_console.is_element_present(
                ".//div[@class='line-search']", drop_down):
            if '\\' in entity:
                header, sep, value = entity.partition('\\')
                drop_down.find_element_by_xpath(
                    ".//div[@class='line-search']/input").send_keys(header)
            else:
                drop_down.find_element_by_xpath(
                    ".//div[@class='line-search']/input").send_keys(entity)

    @WebAction()
    def __select_entity(self, drop_down, entity, partial_selection=False):
        """click on entity to be selected"""
        if '\\' in entity:
            header, sep, value = entity.partition('\\')
            if not drop_down.find_element_by_xpath(
                    f".//*[text()='{header}']/ancestor::div[contains(@class,'multiSelectItem ng-scope vertical')]"
                    f"/following-sibling::div[contains(@class,'multiSelectItem ng-scope vertical')]"
                    f"//*[contains(text(),'{value}')]/ancestor::label/input").get_attribute('checked'):
                drop_down.find_element_by_xpath(
                    f".//*[text()='{header}']/ancestor::div[contains(@class,'multiSelectItem ng-scope vertical')]"
                    f"/following-sibling::div[contains(@class,'multiSelectItem ng-scope vertical')]"
                    f"//*[contains(text(),'{value}')]").click()
        else:
            if partial_selection:
                if not drop_down.find_element_by_xpath(
                        f".//div[@class='checkBoxContainer']//*[contains(text(),'{entity}')]"
                        "/ancestor::label/input").get_attribute("checked"):
                    elem = drop_down.find_element_by_xpath(
                        f".//div[@class='checkBoxContainer']//*[contains(text(),'{entity}')]")
                    self.__driver.execute_script("arguments[0].scrollIntoView();", elem)
                    elem.click()
            else:
                if not drop_down.find_element_by_xpath(
                        f".//div[@class='checkBoxContainer']//*[text()='{entity}']"
                        "/ancestor::label/input").get_attribute("checked"):
                    elem = drop_down.find_element_by_xpath(
                        f".//div[@class='checkBoxContainer']//*[text()='{entity}']")
                    self.__driver.execute_script("arguments[0].scrollIntoView();", elem)
                    elem.click()

    @WebAction()
    def __deselect_entity(self, drop_down, entity):
        """click on entity to be deselected"""

        if drop_down.find_element_by_xpath(
                f".//div[@class='checkBoxContainer']//*[contains(text(),'{entity}')]"
                "/ancestor::label/input").get_attribute("checked"):
            elem = drop_down.find_element_by_xpath(
                f".//div[@class='checkBoxContainer']//*[contains(text(),'{entity}')]")
            self.__driver.execute_script("arguments[0].scrollIntoView();", elem)
            elem.click()

    @WebAction()
    def __clear_search_bar(self, drop_down):
        """clear search bar """
        if self.__admin_console.is_element_present(
                ".//div[@class='line-search']/button", drop_down):
            if drop_down.find_element_by_xpath(
                    ".//div[@class='line-search']/button").is_displayed():
                drop_down.find_element_by_xpath(
                    ".//div[@class='line-search']/button").click()

    @WebAction()
    def __collapse_drop_down(self, drop_down):
        """ Collapse drop down """

        if 'show' in drop_down.find_element_by_xpath(
                ".//div[contains(@class,'checkboxLayer')]").get_attribute('class'):
            if self.__admin_console.is_element_present(
                    ".//div[@class='line-search']/following-sibling::div", drop_down):
                drop_down.find_element_by_xpath(
                    ".//div[@class='line-search']/following-sibling::div").click()
            else:
                drop_down.click()

    @PageService()
    def __deselect_default_value(self, dropdown):
        """
        deselect the the first value which is by selected by default
        the first if statement is empty to check for the search bar
        """
        if dropdown.find_elements_by_xpath("//div[@class='checkBoxContainer']/div[1][contains(@class,'selected')]"):
            list_elements = dropdown.find_elements_by_xpath(".//div[@class='checkBoxContainer']//span")
            if list_elements[0].text != '':
                list_elements[0].click()
            else:
                list_elements[1].click()

    @PageService()
    def __check_multiselect(self, dropdown) -> bool:
        """Checks to see whether the dropdown is in multiple selection mode or single selection mode"""
        return 'single' not in [dropdown.get_attribute('data-selection-mode'),
                                dropdown.get_attribute('selection-mode')]

    @PageService()
    def select_drop_down_values(
            self, index=None, values=None,
            select_all=None, drop_down_id=None, partial_selection=False, default_unselect=True):
        """
        select values from drop down

        Args:
            index (int) : order of drop down in the sequence of display on page (starting from 0)

            values (list) : values to be selected from drop down

            select_all (bool) : boolean value to select all values from drop down

            drop_down_id (str) : id of the drop down tag 'isteven-multi-select'

            partial_selection (bool) : flag to determine if dropdown values should be
            selected in case of partial match or not

                    default: False (partial match is disabled by default)

            default_unselect  (bool)  :   flag to determine whether we have to unselect the value in
                                            dropdown which is selected by default


        """

        if index is not None:
            drop_down_list = self.__get_drop_down_list()
            drop_down = drop_down_list[index]
        elif drop_down_id is not None:
            drop_down = self.__get_drop_down_by_id(drop_down_id)
        else:
            raise Exception("Please provide either index or id of the drop down to be handled here")

        self.__expand_drop_down(drop_down)
        if select_all:
            self.__select_all(drop_down)
            self.__collapse_drop_down(drop_down)
        else:
            self.__select_none(drop_down)
            if self.__check_multiselect(drop_down) and default_unselect:
                self.__deselect_default_value(drop_down)
            for value in values:
                self.__search_entity(drop_down, value)
                self.__select_entity(drop_down, value, partial_selection)
                self.__clear_search_bar(drop_down)
            self.__collapse_drop_down(drop_down)
        self.__admin_console.wait_for_completion()

    @PageService()
    def deselect_drop_down_values(self, index, values):
        """
        deselect values from drop down

        Args:
            index (int) : order of drop down in the sequence of display on page (starting from 0)

            values (list) : values to be deselected from drop down

        """

        drop_down_list = self.__get_drop_down_list()
        self.__expand_drop_down(drop_down_list[index])

        for value in values:
            self.__search_entity(drop_down_list[index], value)
            self.__deselect_entity(drop_down_list[index], value)
            self.__clear_search_bar(drop_down_list[index])
        self.__collapse_drop_down(drop_down_list[index])

        self.__admin_console.wait_for_completion()

    def get_values_of_drop_down(self, id):
        """
        Returns the values in a drop down next to the provided label value.

        Args:
            id   (str)   --  Id of the label corresponding to the drop down.

        """
        drop_down = self.__get_drop_down_by_id(id)
        drop_down.click()
        values = drop_down.text.split('\n')
        self.__collapse_drop_down(drop_down)
        return values


class RDropDown:
    """ Class to handle React drop-down component related operations """

    def __init__(self, admin_console):
        """ Initialize the ReactDropDown object

            Args:
                admin_console : instance of AdminConsoleBase

        """
        self.__admin_console = admin_console
        self.__driver = admin_console.driver

    @WebAction()
    def __get_drop_down_list(self):
        """get drop down list present in the page"""
        drop_down_list = self.__driver.find_elements_by_xpath("//div[@class='dd-wrapper']")
        return drop_down_list

    @WebAction()
    def __get_drop_down_values_list(self, drop_down):
        """ Method to get the values of the drop down"""
        items = drop_down.find_elements_by_tag_name("li")
        values = list()
        for item in items:
            values.append(item.text)
        return values

    @WebAction()
    def __get_drop_down_by_id(self, drop_down_id):
        """ Method to get drop down based on id provided as input"""
        drop_down = self.__driver.find_element_by_xpath(f"//div[@class='dd-wrapper' and @id='{drop_down_id}']")
        return drop_down

    @WebAction()
    def __expand_drop_down(self, drop_down):
        """ Expand drop down """
        drop_down.click()

    @WebAction()
    def __collapse_drop_down(self, drop_down):
        """ Collapse drop down """
        if 'true' in drop_down.find_element_by_xpath(".//div").get_attribute('aria-expanded'):
            drop_down.click()

    @WebAction()
    def __select_all(self, drop_down):
        """ Picks all the item in the drop down from multi select drop down """
        if self.__admin_console.is_element_present('.//div/button[text()="Select all"]', drop_down):
            drop_down.find_element_by_xpath('.//div/button[text()="Select all"]').click()
        else:
            all_items = self.__get_drop_down_values_list(drop_down)
            for item in all_items:
                self.__select_entity(drop_down, item)

    @WebAction()
    def __select_none(self, drop_down):
        """ Unpicks all the item in the drop down from multi select drop down """
        if self.__admin_console.is_element_present('.//div/button[text()="Select none"]', drop_down):
            drop_down.find_element_by_xpath('.//div/button[text()="Select none"]').click()
        else:
            all_items = self.__get_drop_down_values_list(drop_down)
            for item in all_items:
                self.__deselect_entity(drop_down, item)

    @WebAction()
    def __search_entity(self, drop_down, entity):
        """ search for an entity if serach field is present"""
        if self.__admin_console.is_element_present('.//div/input[@class="dd-search-field"]', drop_down):
            drop_down.find_element_by_xpath('.//div/input[@class="dd-search-field"]').send_keys(f'{entity}')

    @WebAction()
    def __clear_search_bar(self, drop_down):
        """ clear search bar is search field is present """
        if self.__admin_console.is_element_present('.//div/input[@class="dd-search-field"]', drop_down):
            drop_down.find_element_by_xpath('.//div/input[@class="dd-search-field"]').send_keys(Keys.CONTROL + "a")
            drop_down.find_element_by_xpath('.//div/input[@class="dd-search-field"]').send_keys(Keys.DELETE)

    @WebAction()
    def __check_multiselect(self, dropdown):
        """ Check to see whether the dropdown is in multiple selection mode or single selection mode"""
        return 'single' not in dropdown.find_element_by_xpath(".//ul").get_attribute('class')

    @WebAction()
    def __deselect_entity(self, drop_down, entity):
        """ click on entity to be deselected """

        if 'true' in drop_down.find_element_by_xpath(f".//ul/li[contains(text(),'{entity}')]").get_attribute('aria-selected'):
            drop_down.find_element_by_xpath(f".//ul/li[contains(text(),'{entity}')]").click()

    @WebAction()
    def __select_entity(self, drop_down, entity, partial_selection=False):
        """ click on entity to be selected """
        if partial_selection:
            if 'false' in drop_down.find_element_by_xpath(f".//ul/li[contains(text(),'{entity}')]").get_attribute('aria-selected'):
                drop_down.find_element_by_xpath(f".//ul/li[contains(text(),'{entity}')]").click()
        else:
            if 'false' in drop_down.find_element_by_xpath(f".//ul/li[text()='{entity}']").get_attribute('aria-selected'):
                drop_down.find_element_by_xpath(f".//ul/li[text()='{entity}']").click()

    @WebAction()
    def __get_selected_values(self, drop_down):
        """ gets the list of selected values of dropdown """

        values_objects = drop_down.find_elements_by_xpath(f".//ul/li[@aria-selected='true']")
        values = []
        for value in values_objects:
            values.append(value.text)
        return values

    @PageService(react_frame=True)
    def get_values_of_drop_down(self, drop_down_id):
        """
        Returns the values in a drop down next to the provided label value.

        Args:
            drop_down_id    (str)    --  Id of the label corresponding to the drop down.

        Returns:

            values (list)   --  returns the list of elements present in the drop down

        """
        drop_down = self.__get_drop_down_by_id(drop_down_id)
        self.__expand_drop_down(drop_down)
        values = self.__get_drop_down_values_list(drop_down)
        self.__collapse_drop_down(drop_down)
        return values

    @PageService(react_frame=True)
    def get_selected_values(self, drop_down_id):
        """
        Returns the list of selected values of the drop down.

        Args:
            drop_down_id    (str)    --  Id of the label corresponding to the drop down.

        Returns:

            values (list)   --  returns the list of elements selected in the drop down

        """
        drop_down = self.__get_drop_down_by_id(drop_down_id)
        self.__expand_drop_down(drop_down)
        values = self.__get_selected_values(drop_down)
        self.__collapse_drop_down(drop_down)
        return values

    @PageService(react_frame=True)
    def select_drop_down_values(
            self, index=None, values=None,
            select_all=False, drop_down_id=None, partial_selection=False):
        """
        select values from drop down

        Args:
            index (int) : order of drop down in the sequence of display on page (starting from 0)

                default: None

            values (list) : list of values to be selected from drop down

                default: None

            select_all (bool) : boolean value to select all values from drop down

                default: False

            drop_down_id (str) : id of the drop down tag 'dd-wrapper'

                default: None

            partial_selection (bool) : flag to determine if dropdown values should be
            selected in case of partial match or not

                default: False (partial match is disabled by default)
        """

        if index is not None:
            drop_down_list = self.__get_drop_down_list()
            drop_down = drop_down_list[index]
        elif drop_down_id is not None:
            drop_down = self.__get_drop_down_by_id(drop_down_id)
        else:
            raise Exception(
                "Please provide either index or id of the drop down to be handled here")

        if not select_all and not values:
            raise Exception("Please provide either values list or set select all flag")
        self.__expand_drop_down(drop_down)
        if select_all:
            self.__select_all(drop_down)
            self.__collapse_drop_down(drop_down)
        else:
            if self.__check_multiselect(drop_down):
                self.__select_none(drop_down)
            for value in values:
                self.__search_entity(drop_down, value)
                self.__select_entity(drop_down, value, partial_selection)
                self.__clear_search_bar(drop_down)
            self.__collapse_drop_down(drop_down)
        self.__admin_console.wait_for_completion()

    @PageService(react_frame=True)
    def deselect_drop_down_values(self, values, drop_down_id=None, index=None):
        """
        deselect values from drop down

        Args:

            values       (list) : values to be deselected from drop down

            drop_down_id (str) : id of the drop down tag 'dd-wrapper'

                default: None

            index        (int) : order of drop down in the sequence of display
            on page (starting from 0)

                default: None

        """

        if index is not None:
            drop_down_list = self.__get_drop_down_list()
            drop_down = drop_down_list[index]
        elif drop_down_id is not None:
            drop_down = self.__get_drop_down_by_id(drop_down_id)
        else:
            raise Exception(
                "Please provide either index or id of the drop down to be handled here")
        self.__expand_drop_down(drop_down)
        for value in values:
            self.__search_entity(drop_down, value)
            self.__deselect_entity(drop_down, value)
            self.__clear_search_bar(drop_down)
        self.__collapse_drop_down(drop_down)
        self.__admin_console.wait_for_completion()
