""" Module for dialog panel

ModalDialog:
    click_cancel()            :       Clicks Cancel button of the panel

    click_submit()            :       Clicks submit button of the panel doesnt matter about the text in it


TransferOwnership
    transfer_ownership()      :       Method to transfer ownership to a user

"""

from Web.AdminConsole.Helper.AdminConsoleBase import AdminConsoleBase
from Web.Common.page_object import (WebAction, PageService)


class ModalDialog:
    """ Modal class """
    def __init__(self, admin_console: AdminConsoleBase):
        """ Initialize the backup object

        Args:
            admin_console: instance of AdminConsoleBase

        """
        self._driver = admin_console.driver
        self._adminconsole_base = admin_console

    @WebAction()
    def click_submit(self):
        """ Clicks submit button of the panel doesnt matter about the text in it"""
        self._driver.find_element_by_xpath(
            "//div[contains(@class, 'modal-content')]//button[contains(@class, 'btn btn-primary')]"
        ).click()

    @WebAction()
    def click_cancel(self):
        """ Clicks Cancel button of the panel"""
        self._driver.find_element_by_xpath(
            "//div[contains(@class, 'modal-content')]//button[contains(@class, 'btn btn-default')]"
        ).click()

    @WebAction()
    def type_text_and_delete(self, text_val, checkbox_id=None):
        """
        Types the text on popup and clicks Delete

        Args:
            text_val (basetring)        : Text to be entered
            checkbox_id (basestring)    : Id of the checkbox that has to be selected

        """
        if self._adminconsole_base.check_if_entity_exists(
                "xpath", "//div[@class= 'modal-content']"):
            elem = self._driver.find_element_by_xpath("//div[contains(@class, 'form-group')]//input")
            elem.send_keys(text_val)
            if self._adminconsole_base.check_if_entity_exists("xpath", f"//*[@id = '{checkbox_id}']"):
                self._adminconsole_base.checkbox_select(checkbox_id)
            self._adminconsole_base.click_button_using_text("Delete")

    @WebAction(delay=0)
    def __read_title(self):
        """Reads Modal panel title"""
        return self._driver.find_element_by_xpath("//div[@class='modal-content']//h1").text

    @PageService()
    def title(self):
        """Returns the title of panel"""
        return self.__read_title()

    @WebAction()
    def __modal_details(self):
        """ Retrieves modal details

        Returns :
            (list) : all visible tags containing text or toggle

        """
        info_xp = "//span[contains(@class, 'pageDetailColumn')]"
        tags = self._driver.find_elements_by_xpath(info_xp)
        return [each_tag for each_tag in tags if each_tag.is_displayed()]

    @PageService()
    def get_details(self):
        """ Gets all the information contained in the modal

        Returns :
            details (dict) : Details of the modal in key value pair

        """
        details = {}
        tags = self.__modal_details()
        tag_count = 0
        key = None
        value = None
        for each_tag in tags:
            tag_count += 1
            if tag_count % 2 != 0:
                key = each_tag.text
            else:
                value = each_tag.text
            details.update({key: value})
        return details


class TransferOwnership(ModalDialog):
    """ Class for Transfer Ownership dialog"""

    @WebAction()
    def __search_new_owner(self, owner_transfer):
        """ Method to search for new owner in transfer ownership/Confirm Delete pop-up """
        search_box = self._driver.find_element_by_xpath(
            "//div[contains(@class,'modal-body')]//input[@name='searchComponent']")
        search_box.send_keys(owner_transfer)

    @WebAction()
    def __select_new_owner(self, owner_transfer):
        """ Method to select new owner from Transfer ownership pop-up """
        search_results = self._driver.find_element_by_xpath(
            "//div[contains(@class,'modal-body')]//ul[contains(@class,'results-container')]")
        search_results.find_element_by_xpath(f".//h5[text()='{owner_transfer}']").click()

    @PageService()
    def transfer_ownership(self, owner_transfer):
        """
        Method to transfer ownership to a user

        Args:
            owner_transfer (basestring): User to which ownership has to be transferred

        Raises:
            Exception:
                If there is no option to transfer owner
        """

        self.__search_new_owner(owner_transfer)
        self._adminconsole_base.wait_for_completion()
        self.__select_new_owner(owner_transfer)
        self.click_submit()
        self._adminconsole_base.check_error_message()


class RModalDialog:
    """ React Modal class """

    def __init__(self, admin_console: AdminConsoleBase):
        """ Initialize the React Modal Dialog Class

        Args:
            admin_console: instance of AdminConsoleBase

        """
        self._driver = admin_console.driver
        self._adminconsole_base = admin_console

    @WebAction()
    def __click_save(self):
        """ Clicks submit button of the panel doesnt matter about the text in it"""
        self._driver.find_element_by_xpath(
            "//div[contains(@class, 'modal-content')]//button[contains(@class, 'sc-bdfBwQ cIKpxU primary-btn btn')]"
        ).click()

    @WebAction()
    def __click_cancel(self):
        """ Clicks Cancel button of the panel"""
        self._driver.find_element_by_xpath(
            "//div[contains(@class, 'modal-content')]//button[contains(@class, 'sc-bdfBwQ cIKpxU secondary-btn btn')]"
        ).click()

    @PageService(react_frame=True)
    def click_submit(self):
        """ Click submit button in a react modal dialog """
        self.__click_save()

    @PageService(react_frame=True)
    def click_cancel(self):
        """ Click cancel button in a react modal dialog """
        self.__click_cancel()
