"""
Module to deal with Callout used in Admin console pages

Callout:

        check_network_status -- Gets Network status

        fetch_job_information -- Fetches the table title

        perform_action -- Performs required action from callout

"""

from Web.AdminConsole.Components.table import Table
from Web.Common.page_object import (WebAction, PageService)


class Callout:
    """Callout Component used in Command Center"""

    def __init__(self, admin_console):
        self._admin_console = admin_console
        self._driver = admin_console.driver
        self._table = Table(self._admin_console)
        self._xp = "//div[@class='popover-body']"

    @WebAction()
    def __get_network_status_callout(self, name):
        """Access network status callout information"""
        self._table.search_for(name)
        self._admin_console.wait_for_completion()
        xp = f"//*[contains(@id,'displayName')]//a[text()='{name}']/../span[@cv-toggle='callout']"
        return self._driver.find_element_by_xpath(xp)

    @WebAction()
    def perform_action(self, action):
        """Performs required action from callout"""
        self._driver.find_element_by_xpath(self._xp + f"//span[text()='{action}']").click()

    @PageService()
    def check_network_status(self, name):
        """Gets Network status"""
        callout = self.__get_network_status_callout(name)
        status = dict()
        status['Client readiness'] = callout.get_attribute("client-readiness-status")
        if status['Client readiness'] == "Offline":
            callout_content = callout.get_attribute("data-content")
            status['Offline time'] = callout_content[callout_content.find('since'):callout_content.find('since') + 22]
            return status
        return status

    @PageService()
    def fetch_job_information(self):
        """Fetches the table title"""
        information = self._driver.find_element_by_xpath(self._xp+"//span[@class='pageDetailColumn']")
        return dict(zip(information[::2], information[1::2]))
