""" Module for browse page

Browse:
Functions:

access_folder()               :       Navigates to the file path

select_multiple_version_of_files():   Selects Multiple versions of file to restore

select_hidden_items()         :       Selects hidden items to restore

select_deleted_items_for_restore():   Selects deleted files or folders to restore

select_for_preview()          :       Select to view the file

close_preview_file()          :       Closes the preview file

select_for_restore()          :       Selects files and folders to restore

show_latest_backups()         :       Selects the latest backup

select_adv_options_submit_restore(): Restores from the selected option

show_backups_by_date_range()  :       Selects backups by daterange and picks
                                      from and to time

submit_for_restore()          :       Restores the selected files

submit_for_download()         :       Downloads the selected files

select_path_for_restore()     :      Expand the source path and
                                     selects files and folders to be restore

select_from_multiple_pages    :     Selects files/items from multiple pages in browse window

switch_to_collections_view    :     Switches to collection view

get_restore_nodelist          :     returns restore node list

ContentBrowse:

Functions:

select_path()                 :       selects folder/files

get_path()                    :       list all folders shown in browse panel

save_path()                   :       Saves the selected path

CVAdvancedTree
Functions:

select_elements()               :       Selects CV Advanced Tree Type Elements

Integration for Browse component is added in TC 58048, for any new methods added here,
add the corresponding integration method

"""
import os

from Web.AdminConsole.Helper.AdminConsoleBase import AdminConsoleBase
from Web.Common.exceptions import CVWebAutomationException
from Web.Common.page_object import PageService, WebAction


class Browse:
    """ Class for restoring selected files and folders """
    def __init__(self, admin_console: AdminConsoleBase):
        self.__driver = admin_console.driver
        self.__admin_console = admin_console
        self.__selected_files = []
        self.__admin_console._load_properties(self)
        self.__xp = "//div[@class='ui-grid-canvas']"

    @WebAction()
    def __select_dropdown(self, restore_options=False, show_backup=False, cluster_view=False):
        """ selects dropdown
        Args:
            restore_options     (bool)      selects 'restore from copy' dropdown
            show_backup         (bool)      selects 'show backup by latest/daterange' dropdown
            cluster_view        (bool)      selects 'collections/replica set view' dropdown
        """
        if restore_options:
            self.__driver.find_element_by_xpath(
                "//a[@class='uib-dropdown-toggle ng-binding dropdown-toggle'"
                " and contains(text(), 'Restore from')]").click()
        elif show_backup:
            self.__driver.find_element_by_xpath(
                "//a[@class='uib-dropdown-toggle ng-binding dropdown-toggle'"
                " and contains(text(), 'Show')]").click()
        elif cluster_view:
            self.__driver.find_element_by_xpath(
                "//a[@class='uib-dropdown-toggle ng-binding dropdown-toggle'"
                " and contains(text(), 'view')]").click()
        else:
            self.__driver.find_element_by_xpath(
                "//a[@class='uib-dropdown-toggle ng-binding dropdown-toggle'  "
                "and not(contains(text(), 'Restore from'))]").click()

    @WebAction()
    def __click_latest_backup(self):
        """ selects latest backup """
        self.__driver.find_element_by_xpath(
            "//a[contains(text(),'Show latest backup')]").click()

    @WebAction()
    def __click_daterange_backup(self):
        """  selects show backups by date range  """
        elem = "//a[contains(text(),'"+self.__admin_console.props['Show_backups_date_range']+"')]"
        self.__driver.find_element_by_xpath(elem).click()

    @WebAction()
    def __click_backups_by_specific_date(self):
        """  selects show backup by specific date"""
        elem = "//a[contains(text(),'" + \
               self.__admin_console.props['Show_backups_specific_date'] + "')]"
        self.__driver.find_element_by_xpath(elem).click()

    @WebAction()
    def __click_primary_copy_restore(self):
        """  selects restore from primary copy"""
        self.__driver.find_element_by_xpath(
            "//a[contains(text(),'Primary') and"
            " not(contains(text(),'snap')) and"
            " not(contains(text(),'Snap'))]").click()

    @WebAction()
    def __click_restore(self):
        """ clicks restore """
        xp = "//a[contains(text(),'Restore') and contains(@data-ng-class,'btn btn-primary')]"
        if self.__admin_console.check_if_entity_exists("xpath", xp):
            self.__driver.find_element_by_xpath(xp).click()
        else:
            self.__driver.find_element_by_xpath(
                "//*[@id='browseActions']//a[contains(text(),'Restore')]").click()

    @WebAction()
    def __click_download(self):
        """ clicks download """
        xp = "//a[contains(text(),'Download') and contains(@data-ng-class,'btn btn-primary')]"
        self.__driver.find_element_by_xpath(xp).click()

    @WebAction()
    def __select_all(self):
        """ selects All files and folders to restore"""
        elem = self.__driver.find_element_by_xpath(
            "//div[@class='ui-grid-cell-contents']/div[@ng-model='grid.selection.selectAll']")
        if 'selected' not in elem.get_attribute('class'):
            elem.click()

    @WebAction()
    def __browse_elements(self, second_column=False):
        """ Browses all elements in the page
        Args:
            second_column   (Boolean):True to select files/items from second column of browse page
                                      False to select files/items from first column of browse page

                            Default: False
        """
        if not second_column:
            elements = self.__driver.find_elements_by_xpath(
                "//div[@class='ui-grid-row ng-scope']//div[@role='gridcell'][1]")
        else:
            elements = self.__driver.find_elements_by_xpath(
                "//div[@class='ui-grid-row ng-scope']//div[@role='gridcell'][2]")
        return [elem.text for elem in elements if elem.is_displayed()]

    @WebAction()
    def __read_column_data(self, column_idx):
        """ reads column data """
        elements = self.__driver.find_elements_by_xpath(
            f"//div[@class='ui-grid-row ng-scope']//div[@role='gridcell'][{column_idx}]"
        )
        return [elem.text for elem in elements if elem.is_displayed()]

    @WebAction()
    def __read_column_names(self):
        """ read column names """
        elements = self.__driver.find_elements_by_xpath(
            "//div[@class='ui-grid-header-canvas']"
            "//span[@class='ui-grid-header-cell-label ng-binding']"
        )
        return [elem.text for elem in elements if elem.is_displayed()]

    @WebAction()
    def __click_elem_checkbox(self, index):
        """ Clicks element checkbox

        Args:
            index (int) : Index of the file in page

        """
        self.__driver.find_element_by_xpath(
            f"//div[@class='ui-grid-canvas']/div[{index}]//"
            f"div[@ng-click='selectButtonClick(row, $event)']").click()

    @WebAction()
    def __available_folders(self):
        """ Lists all available paths

        Returns (list) : available paths

        """
        elements = self.__driver.find_elements_by_xpath(
            "//div[@class='ui-grid-canvas']//a")
        return [elem.text for elem in elements]

    @WebAction()
    def __click_access_path(self, title):
        """ Lists and clicks the element if path exists

        Args:
            title (str) : file path (eg- ['folder1','Hackathon'] )

        """
        self.__driver.find_element_by_xpath(
            f"//div[@class='ui-grid-canvas']//a[@title='{title}' or @name='{title}' or text()='{title}']").click()

    @PageService()
    def __select_files(self, file_folders, second_column=False):
        """ Finds similar files in the panel and those which are input

        Args:
            file_folders (list):    the list of files and folders to select for restore

            second_column   (Boolean): True to select files/items from second column of browse page
                                       False to select files/items from first column of browse page

                            Default: False
        """
        if not second_column:
            files = self.__browse_elements()
        else:
            files = self.__browse_elements(second_column=True)
        index = 0
        for each_file in files:
            index += 1
            if each_file not in file_folders:
                continue
            self.__click_elem_checkbox(index)
            self.__selected_files.append(each_file)

    @WebAction()
    def __select_node_from_tree(self, root_node):
        """Clicks on the root or top most node in the browse tree

        Args:
            root_node (str):  The name of the root/top most node to click in the browse screen

        """
        self.__driver.find_element_by_xpath(
            f"//div[@class='browse-tree']//span[contains(text(), '{root_node}')]").click()

    @WebAction()
    def __unselect_all(self):
        """ selects All files and folders to restore"""
        elem = self.__driver.find_element_by_xpath(
            "//div[@class='ui-grid-cell-contents']/div[@ng-model='grid.selection.selectAll']")
        if 'selected' in elem.get_attribute('class'):
            elem.click()
        else:
            elem.click()
            elem.click()

    @WebAction()
    def __expand_source_folder(self, folder):
        """
        expands the folder
        Args:
            folder (str): expands folder
        """
        self.__driver.find_element_by_xpath(
            self.__xp + f"//a[@name='{folder}']").click()

    @PageService()
    def clear_all_selection(self):
        """Clear selection"""
        self.__unselect_all()

    @PageService()
    def get_column_data(self, column_name):
        """returns column data"""
        columns = self.__read_column_names()
        if column_name not in columns:
            raise CVWebAutomationException(f"Column [{column_name}] doesn't exist in browse page")
        col_idx = columns.index(column_name) + 1
        return self.__read_column_data(col_idx)

    @PageService()
    def access_folder(self, folder):
        """ Navigates to the file path

        Args:
            folder (str) : file path (eg- folder1)

        """
        folders = self.__available_folders()
        folders_list = [entity for entity in folders if folder.lower() == entity.lower()]
        if folders_list:
            self.__click_access_path(folders_list[0])
            self.__admin_console.wait_for_completion()
        else:
            raise CVWebAutomationException(
                f"Invalid Path, {folder} doesnt exist, available are [{folders}]"
            )

    @WebAction()
    def __is_deleted_file(self, entity):
        """ select the deleted file in the panel
        Args:
            entity (string):  file name or folder name to check whether it is
                              marked as deleted or not.
                              (eg- 'filename.txt' or 'C' )
        """
        flag = False
        result = self.__driver.find_elements_by_xpath(
            f"//span[@data-ng-show='true']/parent::div/following-sibling::div/a[@name='{entity}']")
        if result:
            flag = True
        return flag

    @WebAction()
    def __select_delected_file(self, entity):
        """ select the deleted file in the panel
               Args:
                   entity (string):    file or folder to be selected.(eg- 'filename.txt' or 'C' )
        """
        elem = f"//span[@data-ng-show='true']/following::div/a[@name='{entity}']/parent::div"
        result = self.__driver.find_element_by_xpath(elem)
        get_id = ''
        if result.is_displayed():
            get_id = result.get_attribute("id").split('-')
            get_id = get_id[0] + '-' + get_id[1]

        if get_id:
            elem = f"//*[contains(@id,'{get_id}')]/div/div/div"
            self.__driver.find_element_by_xpath(elem).click()

    @WebAction()
    def __click_dropdown(self, column_idx):
        """ Selects dropdown in restore page
        Args:
            column_idx (int): dropdown to be selected for the restore
                              Example:
                              For File system browse page:
                              column_idx:0 -- restore from default copy
                              column_idx:1 -- Showing backup as of <PIT>

                              For database browse page:
                              column_idx:0 -- Showing backup as of <PIT>
                              column_idx:1 -- restore from default copy

        """
        elements = self.__driver.find_elements_by_xpath(
            "//a[@class='uib-dropdown-toggle ng-binding dropdown-toggle']")
        elements[column_idx].click()

    @WebAction()
    def __select_dropdown_value(self, storage_copy_name, plan_name=None):
        """ Selects items from dropdown in restore page
        Args:
            storage_copy_name(str): The name of the storage copy to be selected
                                    (eg: 'Primary' or 'Secondary')
            plan_name   (str): The name of the plan (eg: 'Server plan')
        """
        if plan_name:
            self.__driver.find_element_by_xpath(
                f"//li/a[contains(@class, 'ng-binding') and text()="
                f"'Restore from {storage_copy_name} ({plan_name})']").click()
        else:
            self.__driver.find_element_by_xpath(f"//li/a[contains(@class, 'ng-binding') "
                                                f"and text()='Restore from {storage_copy_name}']").click()
        self.__admin_console.wait_for_completion()

    @WebAction()
    def __click_action_dropdown(self):
        """ Selects action dropdown in restore page """
        elem = f"//a[contains(@class,'uib-dropdown-toggle')]/span[contains(@class,'right ng-binding')]"
        self.__driver.find_element_by_xpath(elem).click()

    @WebAction()
    def __click_sub_link(self, text):
        """ Selects items from action dropdown in restore page
        Args:
            text(str): The select the specific entity from dropdown
        """
        self.__driver.find_element_by_xpath(
            f"//span[@data-ng-bind-html='subLinks.label' and text()='{text}']").click()
        self.__admin_console.wait_for_completion()

    @WebAction()
    def __get_deleted_files(self):
        """ Gets the deleted files on that page """
        files = self.__driver.find_elements_by_xpath(
            f"//span[@data-ng-show='true']/parent::div/following-sibling::div/a"
        )
        file_names = [file.text for file in files]
        return file_names

    @WebAction()
    def __click_file_preview(self, file_name):
        """ Selects the file to preview
        Args:
            file_name (str): Name of the file to be selected
                             (eg: 'file.txt')
        """
        self.__driver.find_element_by_xpath(f"//a[@name='{file_name}']").click()
        self.__admin_console.wait_for_completion()

    @WebAction()
    def __click_close_preview(self):
        """ Clicks close option of previews file"""
        self.__driver.find_element_by_xpath("//a[@class='modal__close-btn']").click()

    def __select_modified_file(self, modified_file):
        """ Selects multiple versions of a file
        Args:
            modified_file (str):  modified file versions to be selected
                                        (eg- 'C:\\Files\\Text.html'  or
                                             '/opt/files/text.txt')
        """
        if "/" in modified_file:
            delimiter = "/"
        else:
            delimiter = "\\"
        paths = modified_file.split(delimiter)
        for folder in range(0, len(paths) - 1):
            self.access_folder(paths[folder])
        self.__select_files([os.path.basename(modified_file)])

    @WebAction()
    def __click_collections_view(self):
        """ selects collections view """
        self.__driver.find_element_by_xpath(
            "//a[contains(text(),'Collection view')]").click()

    @PageService()
    def select_multiple_version_of_files(self, modified_file_path, version_nums):
        """ Selects multiple versions of a file
        Args:
            modified_file_path (str):  modified file versions to be selected
                                       (eg- 'C:\\Files\\Text.html'  or
                                            '/opt/files/text.txt')

            version_nums (list): To select the specified version
                                 (eg: ['1', '2'])
        """
        self.__select_modified_file(modified_file_path)
        self.__click_action_dropdown()
        self.__click_sub_link(self.__admin_console.props['label.viewVersions'])
        files = []
        for version_no in version_nums:
            file = os.path.basename(modified_file_path) + " (" + version_no + ")"
            files.append(file)
        self.select_for_restore(file_folders=files)

    @PageService()
    def select_hidden_items(self, hidden_items, delimiter):
        """Selects hidden items for restore
        Args:
            hidden_items (list(file_paths)):  hidden files to be selected
                                              (eg- 'C:\\Files\\Text.html'  or
                                                    '/opt/files/text.txt')
            delimiter   (str):  To know windows or unix path
        """
        self.__click_action_dropdown()
        self.__click_sub_link(self.__admin_console.props['label.showHiddenItems'])
        paths = os.path.dirname(hidden_items[0])
        if delimiter == '/':
            paths = paths.strip('/')
        paths = paths.split(delimiter)
        hidden_files = [os.path.basename(file) for file in hidden_items]
        for folder in paths:
            self.access_folder(folder)
        self.select_for_restore(file_folders=hidden_files)

    @PageService()
    def select_deleted_items_for_restore(self, content_path, delimiter):
        """ Selects deleted files and folders to restore
        Args:
            content_path (str):  deleted files path location
                                 (eg- 'C:\\Files\\Text.html'  or '/opt/files/text.txt')
            delimiter   (str):  To know windows or unix path
        """
        self.__click_action_dropdown()
        self.__click_sub_link('Show deleted items')
        paths = content_path.split(delimiter)
        flag = True
        for folder in paths:
            if self.__is_deleted_file(folder):
                self.__select_delected_file(folder)
                flag = False
                break
            self.access_folder(folder)

        if flag:
            files = self.__get_deleted_files()
            for file in files:
                self.__select_delected_file(file)

    @PageService()
    def select_for_preview(self, file_path):
        """ Selects to view the file
        Args:
            file_path (str):    the file path to select for preview
        """
        file_path = file_path.replace("\\", "/").strip("/")
        paths = os.path.dirname(file_path).split("/")
        for folder in paths:
            self.access_folder(folder)
            self.__admin_console.wait_for_completion()
        self.__click_file_preview(os.path.basename(file_path))

    @PageService()
    def close_preview_file(self):
        """ Closes the preview file"""
        self.__click_close_preview()

    @PageService()
    def select_for_restore(self, file_folders=None, all_files=False):
        """ Selects files and folders to restore

        Args:
            file_folders (list):    the list of files and folders to select for restore

            all_files   (bool):     select all the files shown for restore / download

        """
        self.__selected_files = []
        if all_files:
            self.__select_all()
            self.__admin_console.wait_for_completion()
            return

        while True:
            self.__select_files(file_folders)
            file_folders = list(set(file_folders) - set(self.__selected_files))
            if file_folders:
                #  access next page as all files are not visible in current page
                if self.__driver.find_element_by_xpath(
                        "//button[@ng-disabled='cantPageForward()']").is_enabled():
                    self.__admin_console.cv_table_click_next_button()
                else:
                    break
            else:
                break
        if file_folders:
            raise Exception("Could not find the items " + str(file_folders))

    @PageService()
    def select_path_for_restore(self, path, file_folders=None):
        """ Expand the source path and selects files and folders to be restore
        Args:
            path (str):source path to be expanded

            file_folders (list):    the list of files and folders to select for restore
        Examples:
            c:\data\f1  --> select f1
            c:  --> select c:
            data/f1  --> select f1
        """
        temp_path = path.replace('\\', '/')
        paths = temp_path.split('/')
        if '' in paths:
            paths.remove('')
        if len(paths) > 1:
            for idx in range(0, len(paths)):
                self.__expand_source_folder(paths[idx])
        if file_folders:
            self.__select_files(file_folders)
            self.__admin_console.wait_for_completion()
            return
        else:
            self.__select_all()
            self.__admin_console.wait_for_completion()
            return

    @PageService()
    def select_from_multiple_pages(self, mapping_dict, subclient_name, rds_agent=False):
        """Selects files/items from multiple pages in browse window
        Args:
            mapping_dict (dict) : The dictionary containing the folder names as keys
                                and list of files to be selected under them as value
                                Example --
                                mapping_dict={
                                'FOLDER1':['file1','file2','file3']
                                'FOLDER2':['fileA','fileB','fileC']
                                }
            subclient_name  (str):  The name of the subclient on which browse operation
                                    was performed or the name of the root folder that
                                    appears on the browse page

            rds_agent  (Boolean):  Flag to indicate if browse is performed for RDS agent

                                   True if browsing from amazon RDS instance or subclient
                                   False for any other agent

        """
        if not isinstance(mapping_dict, dict):
            raise Exception("Mapping dict is empty or invalid")
        else:
            for key, value in mapping_dict.items():
                self.access_folder(key)
                if not rds_agent:
                    self.__select_files(value)
                else:
                    self.__select_files(value, second_column=True)
                self.__select_node_from_tree(subclient_name)

    @PageService()
    def show_latest_backups(self):
        """Shows the latest backup"""
        self.__select_dropdown()
        self.__click_latest_backup()
        self.__admin_console.wait_for_completion()

    @PageService()
    def select_adv_options_submit_restore(self, storage_copy_name, plan_name=None, database=False):
        """ Restores from the selected option
            Args:
                storage_copy_name(str): The name of the storage copy
                                        (eg: 'Primary' or 'Secondary')
                plan_name(str): The name of the plan (eg: 'Server plan')

                database (Boolean): Flag to indicate if browse was performed for
                                    database agents
                                default: False
        """
        if not database:
            self.__click_dropdown(0)
        else:
            self.__click_dropdown(1)
        self.__select_dropdown_value(storage_copy_name, plan_name)
        self.__admin_console.wait_for_completion()

    @PageService()
    def show_backups_by_date_range(self, to_time, from_time=None):
        """Shows backup data by date range
            Args:
                from_time   :   Time from when to backup
                    format: %d-%B-%Y-%I-%M-%p
                            (dd-Month-yyyy-hour(12 hour)-minutes-session(AM/PM))
                            (01-January-2000-11-59-PM)

                to_time   :   Time till when to backup
                    format: %d-%B-%Y-%I-%M-%p
                            (dd-Month-yyyy-hour(12 hour)-minutes-session(AM/PM))
                            (01-January-2000-11-59-PM)
        """
        self.__select_dropdown(show_backup=True)
        if from_time:
            self.__click_daterange_backup()
        else:
            self.__click_backups_by_specific_date()
        self.__admin_console.wait_for_completion()
        if from_time:
            calender = {}
            calender['date'], calender['month'], calender['year'], \
                calender['hours'], calender['mins'], \
                calender['session'] = from_time.split("-")
            self.__admin_console.date_picker(calender, time_id="from-picker")
        calender = {}
        calender['date'], calender['month'], calender['year'], \
            calender['hours'], calender['mins'], \
            calender['session'] = to_time.split("-")
        self.__admin_console.date_picker(calender, time_id="to-picker")
        self.__admin_console.driver.find_element_by_xpath(
            "//button[@data-ng-click='ok()']").click()
        self.__admin_console.wait_for_completion()

    @PageService()
    def restore_from_primary_copy(self):
        """ Selects restore from primary copy from drop down"""
        self.__select_dropdown(restore_options=True)
        self.__click_primary_copy_restore()
        self.__admin_console.wait_for_completion()

    @PageService()
    def submit_for_restore(self):
        """ Restores the selected files """
        self.__click_restore()
        self.__admin_console.wait_for_completion()

    @PageService()
    def submit_for_download(self):
        """ Downloads the selected files """
        self.__click_download()
        notification = self.__admin_console.get_notification()
        self.__admin_console.wait_for_completion()
        return notification

    @PageService()
    def get_restore_nodelist(self):
        """ returns restore node list"""
        hostport_list = self.get_column_data("Backup host:port")
        shard_list = self.get_column_data("Replica set")
        restorenodelist = {}
        for host_port, shard in zip(hostport_list, shard_list):
            restorenodelist[shard] = host_port.replace("::", "_")
        return restorenodelist

    @PageService()
    def switch_to_collections_view(self):
        """Switches to collection view"""
        self.__select_dropdown(cluster_view=True)
        self.__click_collections_view()
        self.__admin_console.wait_for_completion()


class ContentBrowse:
    """ Class to handle content browse in subclient create/edit panels"""

    def __init__(self, admin_console):
        self.__admin_console = admin_console
        self.__driver = admin_console.driver
        self.__xp = "//div[@class='browse-tree']"

    @WebAction()
    def __expand_folder(self, folder):
        """
        expands the folder
        Args:
            folder (str): expands folder
        """
        xpath = self.__xp + f"//span[@class='path-title ng-binding' and %s/../..//button"
        if self.__admin_console.check_if_entity_exists(entity_name='xpath',
                                                       entity_value=xpath % f"text()='{folder}']"):
            self.__driver.find_element_by_xpath(xpath % f"text()='{folder}']").click()
        elif self.__admin_console.check_if_entity_exists(entity_name='xpath',
                                                         entity_value=xpath % f"contains(text(), '({folder})')]"):
            self.__driver.find_element_by_xpath(xpath % f"contains(text(), '({folder})')]").click()

    @WebAction()
    def __select_folder(self, folder):
        """
        Select the folder or file
        Args:
            folder (str): selects folder
        """
        xpath = self.__xp + f"//span[@class='path-title ng-binding' and %s"
        if self.__admin_console.check_if_entity_exists(entity_name='xpath',
                                                       entity_value=xpath % f"text()='{folder}']"):
            self.__driver.find_element_by_xpath(xpath % f"text()='{folder}']").click()
        elif self.__admin_console.check_if_entity_exists(entity_name='xpath',
                                                         entity_value=xpath % f"contains(text(), '({folder})')]"):
            self.__driver.find_element_by_xpath(xpath % f"contains(text(), '({folder})')]").click()

    @WebAction()
    def __read_folder(self):
        """
        Select the folder or file
        """
        elems = self.__driver.find_element_by_xpath(
            self.__xp + f"//span[@class='path-title ng-binding']"
        )
        return [each_elem.text for each_elem in elems if each_elem]

    @WebAction()
    def _click_save(self):
        """ Method to click on save after selecting destination paths"""
        self.__driver.find_element_by_xpath(
            '//button[@id="machineBrowse_button_#2266"]'
        ).click()

    @PageService()
    def select_path(self, path):
        """
        Selects Path
        Args:
            path (str): selects paths
        Examples:
            c:/data/f1  --> select f1
            c:  --> select c:
            data/f1  --> select f1
        """
        temp_path = path.replace('\\', '/')
        paths = temp_path.split('/')
        if '' in paths:
            paths.remove('')
        if len(paths) > 1:
            for idx in range(0, len(paths)-1):
                self.__expand_folder(paths[idx])
                self.__admin_console.wait_for_completion()
        self.__select_folder(paths[-1])

    @PageService()
    def get_paths(self):
        """
        returns the visible paths as list
        """
        return self.__read_folder()

    @PageService()
    def save_path(self):
        """Method to click on save in restore panel after setting destination path"""
        self._click_save()
        self.__admin_console.wait_for_completion()


class CVAdvancedTree:
    """Class to handle content browse in CV Advanced Tree"""

    def __init__(self, admin_console):
        self.__admin_console = admin_console
        self.__driver = admin_console.driver

    @WebAction()
    def _expand_parent_elem(self, parent_elem):
        """
            Expand Client List
            Args:
                parent_elem (str) : Name of Parent Client to Expand
            """
        xp = f"//cv-advanced-tree//label[@title='{parent_elem}']/../preceding::span[1]"
        cv_arrow_elem = self.__driver.find_element_by_xpath(xp)
        cv_arrow_status = cv_arrow_elem.get_attribute('class').split()[1]
        if cv_arrow_status == 'collapsed':
            cv_arrow_elem.click()
        return

    @WebAction()
    def _get_parent_element(self, parent_elem):
        """
            Get Parent Tree Node
            Args:
                parent_elem (str): Name of Parent Client

            Returns: Node Element for given parent

            """
        parent_elem = self.__driver.find_element_by_xpath(
            f"//cv-advanced-tree//label[@title='{parent_elem}']/../parent::cv-advanced-node"
        )
        return parent_elem

    @WebAction()
    def _select_checkbox(self, parent_elem, checkbox_label):
        """
            Select Given Checkbox Element for mentioned parent elem
            Args:
                parent_elem : Parent Element which checkbox Belongs to
                checkbox_label (str): label for checkbox to be selected
            """
        label_elem = parent_elem.find_element_by_xpath(
            f"//label[@title='{checkbox_label}']"
        )
        checkbox_input = parent_elem.find_element_by_xpath(
            f"//label[@title='{checkbox_label}']/preceding::input[1]"
        )
        status = checkbox_input.get_attribute('class').split()[1]
        if status == 'unchecked':
            label_elem.click()

    @PageService()
    def select_elements(self, parent_elem, child_elems):
        """
            Select Mentioned Subclients
            Args:
                parent_elem (str) : parent element name
                child_elems (list) : list of children to be selected in the parent element
            """
        self._expand_parent_elem(parent_elem)
        self.__admin_console.wait_for_completion()
        parent_elem = self._get_parent_element(parent_elem)
        for child in child_elems:
            self._select_checkbox(parent_elem, child)

    @PageService()
    def select_elements_by_full_path(self, element_path, agent_type=None):

        """
        Selects the element by traversing through its entire ancestors
        Args:
            element_path    (list): list of strings to passed, with ancestors separated by '\' or '/'
            type            (str) : type of Agent to be passed. Ex: (sharepoint, filesystem, onedrive)
        """
        for element in element_path:
            element = element.replace('\\', '/')
            if agent_type == "sharepoint":
                *parents, child = element.split(',')
            else:
                *parents, child = element.split('/')
            for parent in parents:
                self._expand_parent_elem(parent)
                self.__admin_console.wait_for_completion()
            self.select_elements(parents[-1], [child])
