# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------
"""
Instances file has the functions to operate on instance page in Big data.

Instances:

    access_instance                   --     Accesses specified instance

    access_server                     --     Accesses specified server

    access_restore                    --     Accesses specified restore page

    access_backup_history             --     Access backup history of specified instance

    access_restore_history            --     Access restore history of specified instance

    add_data_center                   --     Add data center

    add_cassandra_server              --     adds cassandra server

    add_splunk_server                 --     adds splunk server

    add_mongodb_server                --      adds mongoDB server

    create_mongodb_instance           --     Creates MongoDB pseudoclient instance

    delete_instance_name              --     Deletes specified instance

    add_hadoop_server                 --     Add hadoop server

CassandraServer:

    set_name                         --     Sets name field

    select_gateway_node              --     Sets gateway node

    set_gateway_java_path            --    Sets gateway java path

    set_cql_host                     --    Sets CQL host address

    set_jmx_port                     --    Sets jmx port number

    enable_cql                       --    Enables CQL

    enable_jmx                       --    Enables JMX

    enable_ssl                       --    Enables SSL

    expand_authentication_settings   --    Expands authentication settings


SplunkServer:

    set_name                         --     Sets name field

    select_master_node               --     Selects master node

    set_url                          --     Sets URL field

    set_user_name                    --     Sets user name

    set_password                     --     Sets password

    set_confirm_password             --     Sets confirm password field

    select_plan                       --    Selects plan

HadoopServer:

    add_hadoop_parameters            --     Add hadoop parameters in the client creation form

    save                             --     Click Save

"""

from Web.AdminConsole.Components.table import Table
from Web.AdminConsole.Components.panel import DropDown, ModalPanel
import time
from Web.Common.page_object import PageService, WebAction


class Instances:
    """
    Instances class has the functions to operate on instance page in Big data.
    """

    def __init__(self, admin_console):
        self._admin_console = admin_console
        self._admin_console._load_properties(self)
        self.__table = Table(admin_console)

    @WebAction()
    def __click_add_hadoop(self):
        """Click on the Hadoop Icon in Add Cluster Panel"""
        self._admin_console.driver.find_element_by_id('HADOOP').click()

    @PageService()
    def access_instance(self, instance):
        """
        Access instance
        Args:
            instance                         (String)       --    Instance name
        """
        self.__table.access_link(instance)

    @PageService()
    def access_server(self, server_name):
        """
        Access server
        Args:
            server_name                     (String)       --     Server name
        """
        self.__table.access_link_by_column(server_name, 'Server name')

    @PageService()
    def add_cassandra_server(self):
        """Add cassandra server"""
        self.__table.access_toolbar_menu('add')
        xpath = "//div[@id='addFileServerOptions']//*[contains(text(), 'Cassandra')]"
        self._admin_console.driver.find_element_by_xpath(xpath).click()
        return CassandraServer(self._admin_console)

    @PageService()
    def add_mongodb_server(self):
        """Add mongodb server"""
        self.__table.access_toolbar_menu('add')
        xp = '//*[@id="addFileServerOptions"]//..//div[contains(text(),"MongoDB")]'
        self._admin_console.driver.find_element_by_xpath(xp).click()
        return MongodbServer(self._admin_console)

    @PageService()
    def create_mongodb_instance(self, master_node, name="testcase_pseudoclient",
                                db_user="cv_root", db_pwd="commvault!12",
                                os_name=None, plan=None, port_number=None, bin_path=None):
        """
        Creates  a mongodb instance
        Args:
            name                         (String)         --     name for Mongodb server
            master_node                  (String)         --     master_node for mongodb server
            port_number                  (String)         --     port number
            os_name                      (String)         --     os user name
            plan                         (String)         --     plan name
            bin_path                     (String)         --     binary file path
            db_user                      (String)         --    database username
            db_pwd                       (String)         --    database password
        """
        _mongodb_server = self.add_mongodb_server()
        _mongodb_server.set_name(name)
        _mongodb_server.select_master_node(master_node)
        if os_name is not None:
            _mongodb_server.set_os_username(os_name)
        if port_number is not None:
            _mongodb_server.set_port_number(port_number)
        if bin_path is not None:
            _mongodb_server.set_bin_file_path(bin_path)
        if db_user and db_pwd is not None:
            _mongodb_server.set_db_username(db_user)
            _mongodb_server.set_db_password(db_pwd)
        if plan is not None:
            _mongodb_server.select_plan(plan)
        _mongodb_server.save()
        self._admin_console.click_button("OK")

    @PageService()
    def add_hadoop_server(self):
        """Add hadoop server"""
        self.__table.access_toolbar_menu('add')
        self.__click_add_hadoop()
        return HadoopServer(self._admin_console)

    @PageService()
    def add_splunk_server(self):
        """Add splunk server"""
        self.__table.access_toolbar_menu('add')
        self.__table.access_toolbar_menu('addSplunk')
        return SplunkServer(self._admin_console)

    @PageService()
    def access_backup_history(self, instance):
        """
        Access backup history
        Args:
            instance                       (String)       --     Instance name
        """
        self.__table.access_action_item(instance, 'Backup history')

    @PageService()
    def access_restore_history(self, instance):
        """Access restore history
        Args:
            instance                       (String)       --     Instance name
        """
        self.__table.access_action_item(instance, 'Restore history')

    @PageService()
    def add_data_center(self, instance):
        """Add data center
        Args:
            instance                       (String)       --     Instance name
        """
        self.__table.access_action_item(instance, 'Add data center')

    @PageService()
    def access_restore(self, instance='default'):
        """Access restore of specified instance
        Args:
            instance                   (String)          --     Instance name
        """
        self.__table.access_action_item(instance, 'Restore')

    @PageService()
    def is_instance_exists(self, instance):
        """Check if instance exists"""
        return self.__table.is_entity_present_in_column('Name', instance)

    @PageService()
    def delete_instance_name(self, instance):
        """Delete instance
        Args:
            instance                       (String)       --     Instance name
        """
        self.__table.access_action_item(instance, 'Retire')
        confirm_form_id = "deleteTypedConfirmation"
        if not self._admin_console.check_if_entity_exists('id', confirm_form_id):
            confirm_form_id = "typedConfirmationRetire"
        self._admin_console.fill_form_by_id(confirm_form_id, "Retire")
        self._admin_console.click_button("Retire")
        self._admin_console.wait_for_completion()
        if self.is_instance_exists(instance):
            self.__table.access_action_item(instance, 'Delete')
            self._admin_console.fill_form_by_id(confirm_form_id, "Delete")
            self._admin_console.click_button("Delete")
            self._admin_console.wait_for_completion()


class CassandraServer(ModalPanel):
    """Add cassandra server"""

    def __init__(self, admin_console):
        super().__init__(admin_console)
        self.__dropdown = DropDown(admin_console)

    @PageService()
    def set_name(self, name):
        """
        Set name
        Args:
            name                           (String)         -- Set name for Cassandra server
        """
        self._admin_console.fill_form_by_id('name', name)

    @PageService()
    def select_gateway_node(self, node):
        """Select gateway node
        Args:
            node                           (String)         --     Set name for Cassandra server
        """
        self.__dropdown.select_drop_down_values(drop_down_id='cassandraClients', values=[node])

    @PageService()
    def set_gateway_java_path(self, path):
        """Set gateway java path
        Args:
            path                          (String)         --     Set gateway path
        """
        self._admin_console.fill_form_by_id('javaHome', path)

    @PageService()
    def set_cql_host(self, host_name):
        """Set sql host
        Args:
            host_name                    (String)         --      Set host name
        """
        self._admin_console.fill_form_by_id('ipAddress', host_name)

    @PageService()
    def select_plan(self, plan_name):
        """Select plan
         Args:
            plan_name                    (String)         --      select plan
        """
        self.__dropdown.select_drop_down_values(drop_down_id='plan', values=[plan_name])

    @PageService()
    def set_config_file_path(self, conf_file_path):
        """Set config file path
         Args:
            conf_file_path                (String)        --      set config file path
        """
        self._admin_console.fill_form_by_id('configFilePath', conf_file_path)

    @PageService()
    def save(self):
        """Click save"""
        self._admin_console.click_button('Save')

    @PageService()
    def set_cql_port(self, port):
        """Set CQL port number
         Args:
            port                           (String)       --     set CQL port number
        """
        self._admin_console.fill_form_by_id('gatewayCQLPort', port)

    @PageService()
    def set_jmx_port(self, port):
        """Set JMX port number
         Args:
            port                           (String)       --     set JMX port number
        """
        self._admin_console.fill_form_by_id('port', port)

    @PageService()
    def enable_cql(self, user_name, password):
        """Enable cql"""
        self._admin_console.enable_toggle(index=0)
        self._admin_console.fill_form_by_id(
            element_id='userName', value=user_name)
        self._admin_console.fill_form_by_id(
            element_id='password', value=password)
        self._admin_console.fill_form_by_id(
            element_id='confirmPassword', value=password)
        self._admin_console.click_button('Save')

    @PageService()
    def enable_jmx(self, user_name, password):
        """Enable jmx"""
        self._admin_console.enable_toggle(1)
        self._admin_console.fill_form_by_id(
            element_id='userName', value=user_name)
        self._admin_console.fill_form_by_id(
            element_id='password', value=password)
        self._admin_console.fill_form_by_id(
            element_id='confirmPassword', value=password)
        self._admin_console.click_button('Save')

    @PageService()
    def enable_ssl(self):
        """Enable ssl"""
        self._admin_console.enable_toggle(2)

    @PageService()
    def expand_authentication_settings(self):
        """Expand authentication settings"""
        self._expand_accordion('Authentication settings')


class MongodbServer(ModalPanel):
    """Add Mongodb server"""

    def __init__(self, admin_console):
        super().__init__(admin_console)
        self.__dropdown = DropDown(admin_console)

    @PageService()
    def set_name(self, name):
        """
        Set name
        Args:
            name                           (String)         -- Set name for Mongodb server
        """
        self._admin_console.fill_form_by_name('clientName', name)

    @PageService()
    def select_master_node(self, master_node):
        """Select master node
        Args:
            master_node                           (String)         --     Set master_node for mongodb server
        """
        self.__dropdown.select_drop_down_values(drop_down_id='mongoDBClients', values=[master_node])

    @PageService()
    def set_port_number(self, port_number):
        """Set port number
        Args:
            port_number                          (String)         --     Set port number
        """
        if port_number is not None:
            self._admin_console.fill_form_by_id('portNumber', port_number)

    @PageService()
    def set_os_username(self, os_name):
        """Set os user name
        Args:
            os_name                    (String)         --      Set os user name
        """
        if os_name is not None:
            self._admin_console.fill_form_by_id('osUserName', os_name)

    @PageService()
    def select_plan(self, plan_name):
        """Select plan
         Args:
            plan_name                    (String)         --      select plan
        """
        self.__dropdown.select_drop_down_values(drop_down_id="plan", values=[plan_name])

    @PageService()
    def set_bin_file_path(self, bin_file_path):
        """Set binary file path
         Args:
            bin_file_path                (String)        --      set binary file path
        """
        if bin_file_path is not None:
            self._admin_console.fill_form_by_name('binPathConfigured', bin_file_path)

    @PageService()
    def save(self):
        """Click save"""
        self._admin_console.click_button('Save')

    @PageService()
    def set_db_username(self, db_username):
        """Set db user name
         Args:
            db_username                           (String)       --     set db username
        """
        self._admin_console.fill_form_by_id('dbUsername', db_username)

    @PageService()
    def set_db_password(self, db_password):
        """Set db password
         Args:
            db_password                         (String)       --     set db password
        """
        self._admin_console.fill_form_by_id('dbPassword', db_password)

    @PageService()
    def expand_authentication_settings(self):
        """Expand authentication settings"""
        self._expand_accordion(self._admin_console.props['heading.additionalSettings'])



class SplunkServer:
    """Add Splunk server"""

    def __init__(self, admin_console):
        self._admin_console = admin_console

    @PageService()
    def set_name(self, name):
        """
        Set name
        Args:
            name                            (String)      --    Set name for splunk server
        """
        self._admin_console.fill_form_by_id('clientName', name)

    @PageService()
    def select_master_node(self, master_node):
        """Select master node
        Args:
            master_node                     (String)      --     select master node
        """
        self._admin_console.select_value_from_dropdown(
            select_id='clientNode', value=master_node)

    @PageService()
    def set_splunk_server_url(self, url):
        """Set URL
        Args:
            url                            (String)      --      set url
        """
        self._admin_console.fill_form_by_id('url', url)

    @PageService()
    def set_user_name(self, user_name):
        """
        Set user name
        Args:
            user_name                      (String)      --      set user name
        """
        self._admin_console.fill_form_by_id('userName', user_name)

    @PageService()
    def set_password(self, password):
        """
        Set password
        Args:
            password                      (String)       --      set password
        """
        self._admin_console.fill_form_by_id('password', password)

    @PageService()
    def set_confirm_password(self, password):
        """
        Set confirm password
        Args:
            password                      (String)     --        set confirm password string
        """
        self._admin_console.fill_form_by_id('confirmPassword', password)

    @PageService()
    def select_plan(self, plan):
        """
        Select plan
        Args:
            plan                           (String)     --       Select plan
        """
        self._admin_console.select_value_from_dropdown(
            select_id='plan', value=plan)


class HadoopServer(ModalPanel):
    """Add Hadoop server"""

    def __init__(self, admin_console):
        """Initialize object for HadoopServer class.

            Args:
                admin_console       (obj)   -- admin_console object
            Returns:
                object - instance of HadoopServer class

        """
        super().__init__(admin_console)
        self._admin_console = admin_console
        self.__dropdown = DropDown(admin_console)
        self.__modalpanel = ModalPanel(admin_console)

    @PageService()
    def add_hadoop_parameters(self, name, access_nodes, hdfs_user, plan_name):
        """Add hadoop parameters in the client creation form

            Args:
                name            (str)   -- name of hadoop server

                access_nodes    (list)  -- list of access nodes to select

                hdfs_user       (str)   -- hadoop user

                plan_name       (str)   -- plan name to select

        """
        self._admin_console.fill_form_by_id('name', name)
        self.__dropdown.select_drop_down_values(drop_down_id='dataAccessNodes', values=access_nodes)
        self._admin_console.fill_form_by_id('hdfsUser', hdfs_user)
        self.__dropdown.select_drop_down_values(drop_down_id="plan", values=[plan_name])

    @PageService()
    def save(self):
        """Click save"""
        self.__modalpanel.submit()
