# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------
"""
details file has the functions to operate on instance page where we can backup and restore the
instances in Big data.

Overview:

    access_backup_history              --      Accesses backup history

    access_restore                     --      Accesses restore page

    access_restore_history             --      Access restore history

    backup_now                         --      Initiates backup

    restore_all                        --      restore all clients with default options

    access_hdfs                        --      Access HDFS page

Restore:

    select_do_not_recover              --     selects do not recover

    set_staging_location               --     sets staging location

    select_out_of_place_restore        --     select out of place restore

    select_recover                     --     select recover

    set_number_of_streams              --     sets number of streams



    use_sstableloader_tool             --     selects use sstableloader tool option

    ok                                 --     click on ok
"""

from Web.AdminConsole.Components.table import Table
from Web.AdminConsole.Components.browse import Browse
from Web.AdminConsole.Components.panel import Backup, ModalPanel, DropDown, PanelInfo

from Web.Common.exceptions import CVWebAutomationException
from Web.Common.page_object import PageService, WebAction


class Overview:
    """
    Functions to operate on backup and restore
    """

    def __init__(self, admin_console):
        self._admin_console = admin_console
        self.__table = Table(admin_console)
        self.__browse = Browse(admin_console)
        self.__backup = Backup(admin_console)
        self.__panel = ModalPanel(admin_console)

    @WebAction()
    def __click_hadoop_app(self, app_index=0):
        """
        Click on the Hadoop App

            Args:
                app_index       (int)   -- 0 for HDFS, 1 for HBASE


        """
        xp = '//a[contains(@href,"bigDataBackupsetDetails")]'
        self._admin_console.driver.find_elements_by_xpath(xp)[app_index].click()
        self._admin_console.wait_for_completion()

    @PageService()
    def restore_all(self):
        """Initiate restore"""
        self.__browse.select_for_restore(all_files=True)
        self.__browse.submit_for_restore()
        return Restore(self._admin_console)

    @PageService()
    def backup_now(self, name='default', backup_level=Backup.BackupType.FULL):
        """
        Initiate backup
        Args:
            name                         (String)       --    Instance name
            backup_level                 (String)       --    Specify backup level from constant
                                                              present in OverView class
        """
        self.__table.access_action_item(
            name, self._admin_console.props['label.globalActions.backup'])
        _job_id = self.__backup.submit_backup(backup_level)
        return _job_id

    @PageService()
    def access_backup_history(self, instance):
        """
        Access backup history
        Args:
            instance                       (String)       --     Instance name
        """
        self.__table.access_action_item(instance, 'Backup history')

    @PageService()
    def access_restore_history(self, instance):
        """Access restore history
        Args:
            instance                       (String)       --     Instance name
        """
        self.__table.access_action_item(instance, 'Restore history')

    @PageService()
    def access_restore(self, name):
        """Access restore of specified instance
        Args:
            name                   (String)          --     Instance name
        """
        self.__table.access_link_by_column(name, 'Restore')

    @PageService()
    def access_configuration(self):
        """Access configuration page"""
        self._admin_console.select_configuration_tab()
        return Configuration(self._admin_console)

    @PageService()
    def access_hdfs(self):
        """Access HDFS page"""
        self.__click_hadoop_app(app_index=0)


class Configuration:
    """Operations on configuration page"""

    def __init__(self, admin_console):
        self._admin_console = admin_console
        self.__table = Table(admin_console)
    
    @PageService()
    def access_backup(self):
        """clicks backup button on the top menu bar"""
        self._admin_console.access_menu(self._admin_console.props['label.globalActions.backup'])

    @PageService()
    def access_restore(self):
        """clicks restore button on the top menu bar"""
        self._admin_console.access_menu(self._admin_console.props['action.restore'])

    @PageService()
    def is_nodes_exists(self):
        """Check if nodes exists"""
        xpath = '"//*[@title="Nodes"]/../..//*[@class="info-place-holder"]'
        if self._admin_console.is_element_present(xpath):
            return False
        return True

    @PageService()
    def discover_nodes(self):
        """Discovers nodes"""
        self._admin_console.select_hyperlink('Discover nodes')
        discover_text = self._admin_console.get_notification().lower()
        if not discover_text:
            raise CVWebAutomationException("No notification is popped up to extract text")
        if "success" not in discover_text:
            raise CVWebAutomationException("Failed to discover nodes")
        self._admin_console.driver.refresh()

    @PageService()
    def get_replica_set_list(self):
        """Fetches replica set information in case of mongoDB"""
        port_list = self.__table.get_column_data('Port number')
        shard_list = self.__table.get_column_data("Replica set")
        hostname_list = self.__table.get_column_data("Host name")
        replicaset_list = {}
        for shard, hostname, port in zip(shard_list, hostname_list, port_list):
            if shard == '':
                continue
            if shard not in replicaset_list:
                replicaset_list[shard] = [hostname + "_" + port]
            else:
                replicaset_list[shard].append(hostname + "_" + port)
        return replicaset_list

    @PageService()
    def access_overview(self):
        """Access overview page"""
        self._admin_console.select_overview_tab()
        return Overview(self._admin_console)

    @PageService()
    def edit_snapshot_engine(self, engine="native"):
        """
        Edits the snapshot engine
        Args:
            engine  (str)       --      Engine name
        """
        panel_info = PanelInfo(self._admin_console, 'Snapshot management')
        toggle_element = panel_info.get_toggle_element('Enable hardware snapshot')
        if panel_info.is_toggle_enabled(toggle_element):
            panel_info.edit_tile()
        else:
            panel_info.enable_toggle('Enable hardware snapshot')
        drop_down = DropDown(self._admin_console)
        drop_down.select_drop_down_values(0, [engine])
        self._admin_console.click_button("Save")

    @PageService()
    def edit_cassandra_node(self):
        """Edit cassandra nodes"""
        xpath = "//div[@id='tileContent_Nodes']/div[2]/a[contains(text(), 'Edit')]"
        if self._admin_console.is_element_present(xpath):
            self._admin_console.driver.find_element_by_xpath(xpath).click()
            self._admin_console.wait_for_completion()
        else:
            raise CVWebAutomationException("Edit nodes option does not show up")

    @PageService()
    def discover_cassandra_node(self):
        """Discovery cassandra nodes"""
        self._admin_console.select_hyperlink('Discover')
        num_of_column = self.__table.get_number_of_columns()
        if num_of_column != 5:
            raise CVWebAutomationException("Failed to discover cassandra nodes")
        self._admin_console.click_button('Save')

    @PageService()
    def enable_backup(self):
        """Enable backup"""
        panel = PanelInfo(self._admin_console, title='Activity control')
        panel.enable_toggle("Data backup")

    @PageService()
    def disable_backup(self):
        """Disable backup"""
        panel = PanelInfo(self._admin_console, title='Activity control')
        panel.disable_toggle("Data backup")


class Restore:
    """Restore operations"""

    def __init__(self, admin_console):
        self._admin_console = admin_console
        self.__panel = ModalPanel(admin_console)

    @PageService()
    def select_out_of_place_restore(self):
        """Select out of place restore checkbox"""
        self._admin_console.checkbox_select('outOfPlaceRestore')

    @PageService()
    def use_sstableloader_tool(self):
        """Enable SSTableLoader tool toggle"""
        # self._admin_console.toggle_enable('Use SSTableLoader tool')
        self._admin_console.enable_toggle(toggle_id='useSSTableLoader')

    @PageService()
    def set_staging_location(self, location):
        """Set staging location"""
        self._admin_console.fill_form_by_id(
            element_id='stagingLocation', value=location)

    @PageService()
    def set_number_of_streams(self, number):
        """Set number of streams"""
        self._admin_console.fill_form_by_id(
            element_id='noOfStreams', value=number)

    @PageService()
    def select_stage_free_restore(self):
        """Select run stage free restore"""
        # self._admin_console.toggle_enable('Run stage free restore')
        self._admin_console.enable_toggle(toggle_id='runStageFreeRestore')

    @PageService()
    def select_in_place_restore(self):
        """Select in place restore tab"""
        self.__panel.access_tab('In place')

    @PageService()
    def submit_restore(self):
        """submit restore job"""
        self.__panel.submit()
        self._admin_console.click_button('Yes')
