# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Comm vault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
Classes defined in this file
    Restores : class used for performing a restore operation.
    __init__: Initialize the Restores class
    _set_data_files: Choose data file
    restore_in_place: Starts a restore in place job
    restore_out_of_place: Starts a restore out of place job
    restore_to_disk: Starts a restore to disk job
"""

from Web.AdminConsole.Components.panel import DropDown
from Web.Common.page_object import PageService


class Restores:
    """Class  for performing restore operations"""
    def __init__(self, admin_console):
        """ Initialize the Restores class

        Args:
            admin_console: instance of AdminConsoleBase

        """
        self._admin_console = admin_console
        self.__drop_down = DropDown(admin_console)

    @PageService()
    def _set_data_files(self, data_file_name):
        """Choose data file
         Args:
            data_file_name                (String)        --      Choose data file name
        """
        self._admin_console.select_hyperlink(data_file_name)

    @PageService()
    def restore_in_place(self, staging_path=None, overwrite=False):
        """Starts a restore in place job
        Args:
            staging_path                (String)        --      Staging path location
            overwrite                   (bool)          --      unconditionally overwrites
                (overwrite option only available in case of collection based restores)
        """
        self._admin_console.select_hyperlink('In place')
        if staging_path is not None:
            self._admin_console.fill_form_by_id("stagingPathLogRestore", staging_path)
        if self._admin_console.check_if_entity_exists('xpath', "//*[@id = 'overwrite']"):
            if overwrite:
                self._admin_console.checkbox_select('overwrite')
            else:
                self._admin_console.checkbox_deselect('overwrite')
        self._admin_console.click_button(self._admin_console.props['action.submit'])
        self._admin_console.click_button(self._admin_console.props['label.yes'])

    @PageService()
    def restore_out_of_place(self, des_cluster, overwrite=False, des_rename=False,
                             data_files=None, staging_path=None, des_instance=None):
        """
        Module for restore Out of Place
        Args:
            des_cluster                 (String)        --      name of destination cluster
            overwrite                   (bool)          --      unconditionally overwrites
              (overwrite option only available in case of collection view based restores)
            des_rename                  (bool)          --      renames databases/collections
            (add '_restore' suffix in dest, only available for collection view based restores)
            staging_path                (String)        --      Staging path location
            des_instance                (String)        --      Instance of Client
            data_files                  (dict)          --      details dictionary
            data_file- {shard_name:{'Hostname': hostname,'Data Directory': dir,'Port Number': port}}
        """
        self.__drop_down.select_drop_down_values(0, [des_cluster])
        if des_instance is not None:
            self.__drop_down.select_drop_down_values(1, des_instance)
        if staging_path is not None:
            self._admin_console.fill_form_by_id("stagingPathLogRestore", staging_path)
        if data_files is not None:
            for file in data_files.keys():
                self._set_data_files(file)
                self.__drop_down.select_drop_down_values(-1, [data_files[file]['Hostname']])
                self._admin_console.fill_form_by_id('portNumber', data_files[file]['Port Number'])
        if self._admin_console.check_if_entity_exists('id', 'overwrite'):
            if overwrite:
                self._admin_console.checkbox_select('overwrite')
            else:
                self._admin_console.checkbox_deselect('overwrite')
        if des_rename:
            num = 0
            id = f'mongoDBRestore{num}'
            while self._admin_console.check_if_entity_exists('id', id):
                value = self._admin_console.driver.find_element_by_id(id).get_attribute('name')
                self._admin_console.fill_form_by_id(id, f'{value}_restore')
                num += 1
                id = f'mongoDBRestore{num}'
        self._admin_console.click_button(self._admin_console.props['action.submit'])
        self._admin_console.click_button(self._admin_console.props['label.yes'])

    @PageService()
    def restore_to_disk(self, des_cluster, disk_path, data_files=None):
        """
        Module for restore to disk
        Args:
            des_cluster  :  name of destination cluster (str)
            data_files   :   details dictionary of the form - { shard_name: {'Hostname': hostname,
                                                                    'Data Directory': dir,'Port Number': port} }
            disk_path                (String)        --      Disk path
        """
        self._admin_console.select_hyperlink('Restore to disk')
        self.__drop_down.select_drop_down_values(0, [des_cluster])
        self._admin_console.fill_form_by_name('restoreToDiskPath', disk_path)
        if data_files is not None:
            for file in data_files.keys():
                self._set_data_files(file)
                self.__drop_down.select_drop_down_values(-1, [data_files[file]['Hostname']])
                # self._admin_console.fill_form_by_name('dataPath', data_files[file]['Data directory'])
                self._admin_console.fill_form_by_id('portNumber', data_files[file]['Port Number'])
        self._admin_console.click_button(self._admin_console.props['action.submit'])
        self._admin_console.click_button(self._admin_console.props['label.yes'])
