# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
This module provides the function or operations related to Virtualization getting started page
Virtualization : This class provides methods for Virtualization configuration

Virtualization
==============

    add_hypervisor()        -- To add a new VMware vCenter hypervisor

    add_vm_group()          -- To add a new VM group

    run_backup()            -- To run first Backup

"""

from Web.AdminConsole.AdminConsolePages.AdminPage import AdminPage
from Web.Common.handlers import adminconsole_handler
from selenium.webdriver.common.keys import Keys


class Virtualization(AdminPage):
    """
    Class for virtualization getting started page in Admin console

    """
    @adminconsole_handler(hide_args=True)
    def add_hypervisor(
            self,
            hostname=None,
            hypervisor_name=None,
            username=None,
            password=None):
        """
        To add a new VMware vCenter hypervisor

        Args:
            hostname        (str)   -- Full hostname of the machine

            hypervisor_name (str)   -- Name of the hypervisor machine

            username        (str)   -- username to access the machine

            password        (str)   -- Passwor for accessing the machine

        """
        self.select_value_from_dropdown('vendorType', "VMware vCenter")
        self.fill_form_by_id('hostname', hostname)
        self.fill_form_by_id('serverName', hypervisor_name)
        self.fill_form_by_id('vsUserName', username)
        self.fill_form_by_id('vsPassword', password)

        # To click on the save button
        self.driver.find_element_by_xpath('//span[text()="Save"]/..').click()
        self.wait_for_completion()

        # To check if there is any error message
        self.check_error_message()

    @adminconsole_handler()
    def add_vm_group(
            self,
            group_name=None,
            virtual_machines=None):
        """
        To add a new VM group

        Args:
            group_name              (str)   -- Name for the VM group

            virtual_machines        (list)  -- List of virtual_machines to select

        """
        self.fill_form_by_id('name', group_name)

        search = self.driver.find_element_by_xpath('//input[@placeholder="Search VMs"]')
        for machine in virtual_machines:
            search.clear()
            search.send_keys(machine+Keys.ENTER)
            self.wait_for_completion()
            if self.check_if_entity_exists('xpath', f'//span[@title="{machine}"]'):
                self.driver.find_element_by_xpath(f'//span[@title="{machine}"]').click()
                self.wait_for_completion()

        self.submit_form()

        # To check for error messages
        self.check_error_message()

    @adminconsole_handler()
    def run_backup(self):
        """
        To run first Backup

        """
        self.click_button('Backup now')
