# -*- coding: utf-8 -*-
# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------
"""
This module provides the function or operations that can be performed on the
View logs page on the AdminConsole
Class:
    ViewLogs
Functions:
    __get_log_data()        -- web action which gets the data logged for the log present in the page
    get_log_data()          -- gets the data logged for the log present in the page
    __pause_or_resume_logs()-- web action which selects network output
    pause_or_resume_logs()  -- selects network output
    __word_wrap()           -- web action which word wraps the log file
    word_wrap()             -- word wraps the log file
    __get_window_handle     -- gets all the window handles present
    __switch_window         -- switches the window for a given window handle
    change_to_log_window    -- changes the window handle to the log window
    __filter                -- web action to click on filter
    __toggle_action         -- web action to toggle a given toggle name
    __is_toggle_enabled     -- check toggle status
    __enter_filter          -- enter the filter provided
    __clear_filter          -- clear the filter provided
    __clear_filters         -- clear all filters
    set_filters             -- set provided filters

"""

from Web.Common.page_object import PageService, WebAction
from Web.AdminConsole.Components.panel import ModalPanel

class ViewLogs:
    """ Class for ViewLogs page """

    def __init__(self, admin_console):
        """ Initialize the base panel
        Args:
            admin_console: instance of AdminConsoleBase
        """
        self._driver = admin_console.driver
        self._admin_console = admin_console
        self.__modal_panel = ModalPanel(admin_console)
        self.__filter_list = ['processId', 'jobId', 'threadId', 'includeString', 'excludeString']

    @WebAction()
    def __get_log_data(self):
        """web action which gets the data logged for the log present in the page """
        return self._driver.find_element_by_id('editor').text

    @PageService()
    def get_log_data(self):
        """gets the data logged for the log present in the page """
        return self.__get_log_data()

    @WebAction()
    def __pause_or_resume_logs(self):
        """web action which Pauses the log file from updating live"""
        self._driver.find_element_by_xpath("//*[contains(@id, 'actionDetail_pause_')]").click()

    @PageService()
    def pause_or_resume_logs(self):
        """Pauses the log file from updating live"""
        self.__pause_or_resume_logs()
        self._admin_console.wait_for_completion()

    @WebAction()
    def __word_wrap(self):
        """web action which word wraps the log file"""
        self._driver.find_element_by_xpath("//*[contains(@id, 'actionDetail_wordWrap_')]").click()

    @PageService()
    def word_wrap(self):
        """word wraps the log file"""
        self.__word_wrap()
        self._admin_console.wait_for_completion()

    @WebAction()
    def __get_window_handle(self):
        """gets the window handles"""
        return self._driver.window_handles

    @WebAction()
    def __switch_window(self, window):
        """switches the windows for the provided window handle
        Args:
            window: windowHandle object
        """
        self._driver.switch_to.window(window)

    @PageService()
    def change_to_log_window(self):
        """changes the window handle to the log window"""
        log_window = self.__get_window_handle()
        self.__switch_window(log_window[1])
        self._admin_console.wait_for_completion()

    @WebAction()
    def __filter(self):
        """web action which word wraps the log file"""
        self._driver.find_element_by_xpath("//*[contains(@id, 'actionDetail_filter_')]").click()

    @WebAction()
    def __toggle_action(self, toggle_name):
        """web action to toggle filter
            toggle_name (str): attribute name of the toggle
        """
        self._driver.find_element_by_xpath("//div[@toggle-name='%s']" % toggle_name).click()

    @WebAction()
    def __is_toggle_enabled(self, toggle_name):
        """web action to check if toggle is enabled or not
            toggle_name (str): attribute name of the toggle
        """
        try:
            self._driver.find_element_by_xpath("//div[@toggle-name='%s']/"
                                               "following-sibling::fieldset[@disabled='disabled']" % toggle_name)
            return False
        except Exception as e:
            return True

    @PageService()
    def enable_toggle(self, toggle_name):
        """Page service to enable toggle
            toggle_name (str): attribute name of the toggle
        """
        if not self.__is_toggle_enabled(toggle_name):
            self.__toggle_action(toggle_name)
            self._admin_console.wait_for_completion()
        if not self.__is_toggle_enabled(toggle_name):
            raise Exception('%s Toggle did not get enabled'%toggle_name)

    @WebAction()
    def __enter_filter(self, filter_name, filter_value):
        """web action to enter the filter in the filter window
        filter_name (str) : name of the filter to be set
        filter_value(str) : value of the filter to be set
        """
        self._driver.find_element_by_xpath("//div/input[@ng-model=\"params.%s\"]" % filter_name).send_keys(filter_value)

    @WebAction()
    def __clear_filter(self, filter_name):
        """web action to enter the filter in the filter window
        filter_name (str) : name of the filter to be set
        filter_value(str) : value of the filter to be set
        """
        self._driver.find_element_by_xpath("//div/input[@ng-model=\"params.%s\"]" % filter_name).clear()

    @WebAction()
    def __clear_filters(self):
        """Page service to clear all filters"""
        for name in self.__filter_list:
            self.__clear_filter(name)

    @PageService()
    def set_filters(self, filter_dict, clear_existing=False):
        """
        Page Service to set filters in view_logs page
        Args:
            filter_dict: (dict) with the key as the filter name(ng-model) and the value as the filter value
                                Eg: {'processId': 100}
                                Accepted values: 'processId', 'jobId', 'threadId', 'includeString', 'excludeString'
            clear_existing: (bool) this is to clear existing filters
        """
        self.__filter()
        self._admin_console.wait_for_completion()
        self.enable_toggle('enableFilter')
        if clear_existing:
            self.__clear_filters()
        for name, value in filter_dict.items():
            self.__enter_filter(name, value)
        self.__modal_panel.submit()
