# -*- coding: utf-8 -*-

# ---------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# ----------------------------------------------------------------------------

"""

This module provides the function or operations that can be performed on the theme customization page
in admin console

Class:
      Theme() -> AdminPage() -> AdminConsoleBase() ->object()

Functions :

    set_logo()                      ->  Method to set the custom logo

    set_color_settings()            -> Method to set the color settings

    reset_theme()                   ->  Method to reset the theme settings

    get_theme_values()              -> Method to get the theme customization values

"""

from Web.Common.page_object import (WebAction, PageService)
from Web.AdminConsole.AdminConsolePages.AdminPage import AdminPage


class Theme(AdminPage):
    """Class for theme customization page"""

    @WebAction()
    def _set_commcell_logo(self, logo_file_path):
        """
        Set the commcell logo

        Args:

            logo_file_path(str)     -- file path to the custom logo.    e.g: C:\\test\\Jb_image1.jpg

        Returns:
            None

        Raises:
            Exception:
                -- if failed to set the logo

        """
        self.log.info("Setting the commcell logo")
        logo = self.driver.find_element_by_xpath("//input[@type='file']")
        logo.send_keys(logo_file_path)
        self.click_button('Save')

    @WebAction()
    def _set_custom_color_settings(self, primary_color,
                                   header_color,
                                   header_text_color,
                                   navigation_color,
                                   link_color):
        """
        set the color settings

        Args:
            primary_color(str)          --  primary color of the commcell.      eg.#841a1a

            header_color(str)           --  header color of the commcell.       eg.#31b7a5

            header_text_color(str)      --  header text color of the commcell.  eg.#d6d8e9

            navigation_color(str)       --  navigation color of the commcell.   eg.#152a0f

            link_color(str)             --  logo color of the commcell.         eg.#d83ba7

        Returns:
                None
        Raises:
                Exception:
                    --- if failed to set the color values

        """
        self.log.info("Setting the primary color")
        self.fill_form_by_id("primaryColor", primary_color)

        self.log.info("Setting the header color")
        self.fill_form_by_id("headerColor", header_color)

        self.log.info("Setting the header text color")
        self.fill_form_by_id("headerTextColor", header_text_color)

        self.log.info("Setting the navigation color")
        self.fill_form_by_id("navIconColor", navigation_color)

        self.log.info("Setting the link color")
        self.fill_form_by_id("secondaryColor", link_color)

        self.click_button('Save')

    @WebAction()
    def _get_theme_customization_values(self):
        """to get the theme customization value"""
        theme_dictionary = {}
        theme_dictionary.update(
            {'logo_value': self.driver.find_element_by_xpath("//div[@class='full-width img-input']/span/span").text})
        theme_dictionary.update(
            {'primary_color_value': self.driver.find_element_by_id("primaryColor").get_attribute('value')})
        theme_dictionary.update(
            {'header_color_value': self.driver.find_element_by_id("headerColor").get_attribute('value')})
        theme_dictionary.update(
            {'header_text_color_value': self.driver.find_element_by_id("headerTextColor").get_attribute('value')})
        theme_dictionary.update(
            {'navigation_color_value': self.driver.find_element_by_id("navIconColor").get_attribute('value')})
        theme_dictionary.update(
            {'link_color_value': self.driver.find_element_by_id("secondaryColor").get_attribute('value')})
        return theme_dictionary

    @PageService()
    def set_logo(self, logo_file_path):
        """
        Method to set the custom logo

        Args:
            logo_file_path(str)     -- file path to the custom logo.    e.g: C:\\test\\Jb_image1.jpg

        Returns:
            None

        Raises:
            Exception:
                -- if failed to set the logo

        """
        self._set_commcell_logo(logo_file_path)
        self.wait_for_completion()
        self.check_error_message()

    @PageService()
    def set_color_settings(self, primary_color='#2f4e66',
                           header_color='#e4e7ea',
                           header_text_color='#2f4e66',
                           navigation_color='#eeeeee',
                           link_color='#4B8DCC'):
        """
        Method to set the color settings

        Args:
            primary_color(str)          --  primary color of the commcell.      eg.#241a1a

            header_color(str)           --  header color of the commcell.       eg.#41b7a5

            header_text_color(str)      --  header text color of the commcell.  eg.#c6d8e9

            navigation_color(str)       --  navigation color of the commcell.   eg.#e52a0f

            link_color(str)             --  logo color of the commcell.         eg.#a83ba7

        Returns:
            None

        Raises:
            Exception:
                -- if failed to set the color settings

        """
        self._set_custom_color_settings(primary_color, header_color, header_text_color, navigation_color, link_color)
        self.wait_for_completion()
        self.check_error_message()

    @PageService()
    def reset_theme(self):
        """
        Method to reset the theme customization values

        Returns:
            None

        Raises:
            Exception:
                -- if failed to reset the customization values

        """
        self.click_button('Reset to default')
        self.click_button('Yes')
        self.check_error_message()

    @PageService()
    def get_theme_values(self):
        """
        Get theme color and logo values
        Returns:

            Dictionary with all theme color and logo values
                Eg.-{logo_file_name : ['c:\\test\\test1.jpg'], primary_color_value : ['#a83ba7']}

        """
        theme_dictionary = self._get_theme_customization_values()
        return theme_dictionary
