# -*- coding: utf-8 -*-

# ---------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# ----------------------------------------------------------------------------

"""
This module provides the function or operations that can be performed on the Subclient Policy details page in
admin console

Class:
      SubclientPolicyDetails() -> AdminPage() -> login_page() -> AdminConsoleBase() ->object()

Functions :

edit_subclient_policy_name()            --- Method to edit the subclient policy name

edit_subclient_policy_association()     --- Method to edit the subclient policy associations

add_subclient()                         --- Method to add subclient to the subclient policy

delete_subclient()                      --- Method to delete the subclient from subclient policy details page

open_subclient()                        --- Method to open a subclient details from subclient policy

"""
from Web.AdminConsole.Components.table import Table
from Web.Common.handlers import adminconsole_handler
from Web.AdminConsole.AdminConsolePages.AdminPage import AdminPage
from Web.Common.page_object import PageService


class SubclientPolicyDetails(AdminPage):
    """Class for Subclient Policy Details page"""

    def __init__(self, driver):
        super().__init__(driver)
        self.__table = Table(self)

    @adminconsole_handler()
    def edit_subclient_policy_name(self, subclient_policy_name, new_subclient_policy_name):
        """
        Method to change the subclient policy name

        Args:
         subclient_policy_name (str): subclient policy name to be changed
         new_subclient_policy_name (str):name of the subclient policy after modification

        Returns:
            None
        Raises :
             Exception:
              -- if failed to change the subclient policy name
        """

        self.log.info("Editing the subclient policy name : %s", subclient_policy_name)
        if self.check_if_entity_exists("xpath", "//h1[@id]"):
            name = self.driver.find_element_by_xpath("//h1[@id]")
        else:
            raise Exception("Unable to edit the subclient policy name")
        name.click()
        name.clear()
        name.send_keys(new_subclient_policy_name)
        self.driver.find_element_by_xpath("//cv-tile-component[@data-loader]/div/div[3]/span").click()
        self.wait_for_completion()
        self.log.info("Subclient policy name was edited successfully")
        self.check_error_message()

    @adminconsole_handler()
    def edit_subclient_policy_association(self, new_associations):
        """
        Method to edit the subclient policy association
        Args:
            new_associations (list) : New list of entities to be associated with this subclient policy

        Returns:
            None
        Raises:
            Exception:
            -- if failed to edit the subclient policy association
        """

        self.log.info("Editing the subclient policy associations")

        self.driver.find_element_by_xpath("//a[@class='page-details-box-links ng-scope']").click()
        self.wait_for_completion()
        self.checkbox_select("dissociateBackupSet")
        self.log.info("dissociate backup set check box option was selected successfully")
        self.log.info("Setting the new associations")
        while self.check_if_entity_exists("xpath", "//span[@class='cv-tree-arrow collapsed']"):
            collapsed_elems = self.driver.find_elements_by_xpath("//span[@class='cv-tree-arrow collapsed']")
            for elem in collapsed_elems:
                elem.click()
                self.wait_for_completion()
        if new_associations['server_groups']:
            head = self.driver.find_element_by_xpath(
                "//ul[@class='tree-view-wrapper ng-scope']/li")
            for value in new_associations['server_groups']:
                head.find_element_by_xpath(
                    "//label[contains(.,'" + value + "')]").click()
            self.wait_for_completion()
        self.log.info("associations were set successfully")
        self.checkbox_select("showSelected")
        self.log.info("Show selected check box option was selected successfully")
        self.click_button("Save")
        self.log.info("Subclient policy association was edited successfully.")
        self.check_error_message()

    @adminconsole_handler()
    def add_subclient(self, subclient_name, storage_policy_name, subclient_path):
        """
        Method to add a new subclient to the subclient policy
        Args:
             subclient_name (str)        : subclient name to be added newly to the subclient policy
             storage_policy_name(str):  Storage policy name to be associated with the new subclient
             subclient_path(str)        : path to the subclient
        Returns:
            None
        Raises :
           Exception:
             -- if failed to add a new subclient policy
        """

        self.log.info("Adding a new subclient to this subclient policy")
        self.select_hyperlink("Add")
        self.fill_form_by_id("subclientName", subclient_name)
        self.select_value_from_dropdown("primaryStorage", storage_policy_name)
        self.select_hyperlink("Content")
        path = self.driver.find_element_by_xpath("//input[@placeholder='Enter custom path']")
        path.clear()
        path.send_keys(subclient_path)
        self.wait_for_completion()
        self.driver.find_element_by_xpath("//i[@title='Add'][@data-ng-click='addCustomPath(0)']").click()
        self.wait_for_completion()
        self.click_button("Save")
        self.log.info("Subclient was created successfully.")
        self.check_error_message()

    @PageService()
    def delete_subclient(self, subclient_name):
        """
        Deletes the subclient policy from the subclient policy details page

        Returns:
            None

        Raises:
            Exception:
                -- if fails to delete the subclient policy
        """
        self.log.info("Deleting the subclient from details page")
        self.__table.access_link_by_column(subclient_name, 'Delete')
        self.type_text_and_save("DELETE")
        self.check_error_message()

    @PageService()
    def open_subclient(self, subclient):
        """
        Opens the subclient with the given name  --- refer backupset page

        Args:
            subclient (str): name of the subclient

        Returns:
            None
        Raises:
            Exception:There is no subclient with the name

        """
        self.__table.access_link(subclient)
        self.check_error_message()
