# -*- coding: utf-8 -*-
# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
This module provides the function or operations that can be performed on the
Service CommCell page on the AdminConsole

Class:

    ServiceCommcell()

Functions:

    _init_()                            :   initialize the class object

    __select_service_commcell_tile()    :   selects the service commcell tile on the systems page

    register_commcell()                 :   registers a new service commcell

    delete_registered_commcell()        :   unregisters a service commcell

    refresh_registered_commcell()       :   synchronize a service commcell

"""

from Web.Common.page_object import (WebAction, PageService)
from Web.AdminConsole.Components.table import Table

class ServiceCommcell:
    """ Class for Service Commcell page in Admin Console """

    def __init__(self, admin_console):
        """
        Method to initiate ServiceComcell class

        Args:
            admin_console   (Object) :   Admin Console Class object
        """
        self.__admin_console = admin_console
        self.__table = Table(self.__admin_console)
        self.__log = admin_console.log
        self.__driver = admin_console.driver

    @PageService()
    def register_commcell(self, hostname, username, password, configure_as_IdP):
        """
        Function to add a new service commcell

        Args    :

        hostname (string)   :   hostname of the commcell to be registered

        username (string)   :   username of the commcell to be registered

        password (string)   :   password of the commcell to be registered

        configure_as_IdP(bool): True - Register service commcell for Jupiter

                                False - Register service commcell for Router
        """
        self.__admin_console.select_hyperlink('Add')
        self.__admin_console.fill_form_by_id('commcellHostName', hostname)
        self.__admin_console.fill_form_by_id('userName', username)
        self.__admin_console.fill_form_by_id('password', password)
        if configure_as_IdP:
            self.__admin_console.enable_toggle(index=0, cv_toggle=True)
        self.__admin_console.click_button('Save')

    @PageService()
    def delete_registered_commcell(self, hostname):
        """
        Function to delete a registered commcell

        Args    :

        hostname (string)   :   hostname of the commcell to be un-registered
        """
        self.__table.apply_filter_over_column('Service CommCell host name', hostname)
        self.__table.access_action_item(hostname, 'Delete')
        self.__admin_console.click_button('Yes')

    @PageService()
    def refresh_registered_commcell(self, hostname):
        """
        Function to delete a registered commcell

        Args    :

        hostname (string)   :   hostname of the commcell to be synchronized
        """
        self.__table.apply_filter_over_column('Service CommCell host name', hostname)
        self.__table.access_action_item(hostname, 'Refresh')
