# -*- coding: utf-8 -*-
# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
This module provides the function or operations that can be performed on the
Send logs page on the AdminConsole

Class:

    SendLogs

Functions:

    select_local_output()       -- selects local output
    select_network_output()     -- selects network output
    select_information()        -- select the information to be collected
    email()                     -- email notification configurations
    disable_auto_upload()       -- disables auto upload
    disable_email_notification()-- disables email notification

"""
from enum import Enum
from selenium.common.exceptions import NoSuchElementException

from Web.Common.page_object import PageService, WebAction


class SendLogs:
    """ Class for Sendlogs page """
    class Informationlist(Enum):
        """ list of options available to select under information"""
        CSDB = "csDatabase"
        LOGS = "machineLogs"
        INDEX = "indexLabel"
        OS_LOGS = "osLogs"
        OTHER_DB = "otherDatabases"


    class Advancedlist(Enum):
        """ list of options available to select under Advanced"""
        PROC_DUMP = "processDump"
        DB_LOGS = "databaseLogs"
        JOBS_RESULTS = "jobResults"
        INDEX_TRANS = "actionLogs"
        LATEST_DB = "getLatestUpdates"
        OTHER_DB = "otherDatabases"
        SCRUB_LOGFILES = "scrubLogFiles"

    def __init__(self, admin_console):
        """ Initialize the base panel

        Args:
            admin_console: instance of AdminConsoleBase

        """
        self._driver = admin_console.driver
        self._admin_console = admin_console

    @WebAction(delay=0)
    def _enter_addtnl_recipients(self, addtnl_recepient):
        """Enter additional recipients"""
        rcpt = self._driver.find_element_by_xpath("//input[@class='additionalMailRecipients']")
        for each_recipient in addtnl_recepient:
            rcpt.send_keys(each_recipient + '\t')

    @WebAction(delay=0)
    def _enter_description(self, description):
        """Enter description"""
        desc = self._driver.find_element_by_xpath("//textarea")
        desc.send_keys(description)

    @WebAction(delay=2)
    def _click_submit(self):
        """click_submit"""
        xp = "//button[@id='sendLogsModal_button_#2469']"
        self._admin_console.scroll_into_view(xp)
        self._driver.find_element_by_xpath(xp).click()

    @WebAction()
    def _enter_user(self, user_name):
        """Enter user"""
        self._admin_console.fill_form_by_id('s2id_autogen1', user_name)
        try:
            user = self._driver.find_element_by_xpath(
                "//ul[@class='select2-results']//span[@class='user-type plan-user-suggestion']"
            )
        except NoSuchElementException:
            raise NoSuchElementException(f"User {user_name} doesn't exist to select in users")
        if user:
            user.click()
            # to change focus out
            self._admin_console.driver.find_element_by_xpath("//body").click()

    @PageService()
    def disable_auto_upload(self):
        """
        Disables auto upload
        """
        self._admin_console.checkbox_deselect('uploadLabel')

    @PageService()
    def disable_email_notification(self):
        """
        Disables email notification
        """
        self._admin_console.expand_accordion('Information')
        self._admin_console.checkbox_deselect('sendEMailForSendLogs')

    @PageService()
    def select_local_output(self, local_path=None):
        """
        Selects the local output
        Args:
            local_path  (str) : local path
        """
        self._admin_console.checkbox_select('saveToFolderSelected')
        self._admin_console.fill_form_by_id('localPath', local_path)

    @PageService()
    def select_network_output(self, nw_path, nw_user, nw_pwd):
        """
        Selects the network output
        Args:
            nw_path     (str) :  N/w path
            nw_user     (str) :  N/w user name
            nw_pwd      (str) :  N/w password

        """
        self._admin_console.checkbox_select('saveToFolderSelected')
        self._admin_console.select_radio('pathChoice2')
        self._admin_console.fill_form_by_id('NetworkDir', nw_path)
        self._admin_console.fill_form_by_id('impersonateName', nw_user)
        self._admin_console.fill_form_by_id('impersonatePass', nw_pwd)

    @PageService()
    def select_information(self, information_list):
        """
        Args:
            information_list (list):  list from Informationlist enum
        """
        self._admin_console.expand_accordion('Information')
        for each_info in information_list:
            self._admin_console.checkbox_select(each_info.value)

    @PageService()
    def select_advanced(self, advanced_list):
        """
         Args:
                advanced_list (list):  list from AdvancedList enum
        """
        self._admin_console.expand_accordion('Advanced')
        for each_info in advanced_list:
            self._admin_console.checkbox_select(each_info.value)

    @PageService()
    def deselect_information(self, information_list=[]):
        """
        Args:
            information_list (list):  list from Informationlist enum
        """
        self._admin_console.expand_accordion('Information')
        for each_info in information_list:
            self._admin_console.checkbox_deselect(each_info.value)

    @PageService()
    def deselect_advanced(self, advanced_list=[]):
        """
        Args:
            information_list (list):  list from Informationlist enum
        """
        self._admin_console.expand_accordion('Advanced')
        for each_info in advanced_list:
            self._admin_console.checkbox_deselect(each_info.value)

    @PageService()
    def email(self, users, addtnl_recepient=None, subject=None, description=None):
        """
        email notification
        Args:
            users               (list) : user or user groups for notification
            addtnl_recepient    (list) : email id list
            subject             (str)  : mail subject, default subject is populated in page
            description         (str)  : problem description

        """
        self._admin_console.expand_accordion('Notification')
        for each_user in users:
            self._enter_user(each_user)

        if addtnl_recepient:
            self._enter_addtnl_recipients(addtnl_recepient)
        if subject:
            self._admin_console.fill_form_by_id('emailSubject', subject)
        if description:
            self._enter_description(description)

    @PageService()
    def submit(self):
        """
        Submits send logs jobs
        Returns:
            send logs Job id
        """
        self._click_submit()
        jobid = self._admin_console.get_jobid_from_popup()
        self._admin_console.wait_for_completion()
        return jobid
