# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------
"""
This module provides the function or operations that can be performed on the
replication page of the AdminConsole

"""


from Web.AdminConsole.AdminConsolePages.AdminPage import AdminPage
from Web.AdminConsole.SQL.configure_sql_replication_group import SQLServerReplication
from selenium.common.exceptions import NoSuchElementException
from Web.Common.page_object import (
    WebAction,
    PageService
)
from Web.AdminConsole.Components.table import Table
import time


class ReplicationGroupDetails(AdminPage):
    """All operations specific to Replication group goes here"""

    def __init__(self, driver):
        """
        Initialize  ReplicationGroupDetails
        Args:
            driver: Driver Object
        """
        super(ReplicationGroupDetails, self).__init__(driver)
        self.__table = Table(self)

    @WebAction()
    def __sync_status_href(self):
        """
        Get element for status string
        Returns:    Anchor element for status
        """
        return self.driver.find_element_by_xpath(
                    '//*[@id="cv-k-grid-td-statusString"]/a')

    @PageService()
    def manage_content(self):
        """Edits the replication group"""
        self.select_hyperlink("Manage content")
        self.wait_for_completion()
        return SQLServerReplication(self.driver)

    @PageService()
    def get_replication_job(self, tries=20):
        """
        Get Current Replication job ID
        Args:
            tries: Number of times to try for replication job to start, at intervals of 10s

        Returns: Current Replication job ID

        """
        try_num = 0
        while True:
            try:
                sync_status = self.__sync_status_href()
                href_path = sync_status.get_attribute("href")
                job_id = int(href_path.split("/")[6])
                return job_id

            except NoSuchElementException:
                if try_num >= tries:
                    self.log.info("No ongoing replication job found,"
                                  " relating to this replication group.")
                    raise Exception("No ongoing replication job found,"
                                    " relating to this replication group.")
                else:
                    time.sleep(10)
                    self.refresh_page()
                    self.wait_for_completion()
                    try_num += 1

    @PageService()
    def get_sync_status(self):
        """
        Returns Sync Status for replication group.
        """
        sync_status = self.__table.get_column_data(self.props['header.syncStatus'])
        return sync_status

    @PageService()
    def get_failover_status(self, retry=3):
        """
        Returns Failover status
        Returns: failover status
        """
        failover_status = ['']
        for try_num in range(retry):
            self.refresh_page()
            failover_status = self.__table.get_column_data(self.props['header.failoverStatus'])
            if failover_status[0] != '':
                return failover_status
        return failover_status

    @PageService()
    def planned_failover(self, source_vms):
        """
        Planned failover for the replication group.
        Returns: job id for planned failover job
        """
        self.log.info("Initiating Planned Failover")
        self.__table.select_rows(source_vms)
        self.select_hyperlink(self.props['label.plannedFailover.title'])
        self.click_button_using_text(self.props['button.yes'])
        job_id = self.get_jobid_from_popup()
        return job_id

    @PageService()
    def unplanned_failover(self, source_vms):
        """
        Unplanned failover for the replication group.
        Args:
            source_vms: list of source VMs
        Returns: job id for unplanned failover job
        """
        self.log.info("Initiating Unplanned Failover")
        self.__table.select_rows(source_vms)
        self.select_hyperlink(self.props['label.unplannedFailover.title'])
        self.click_button_using_text(self.props['button.yes'])
        job_id = self.get_jobid_from_popup()
        return job_id

    @PageService()
    def failback(self, source_vms):
        """
        Fail back for the replication group.
        Args:
            source_vms: list of source VMs
        Returns: job id for failback job
        """
        self.log.info("Initiating Failback")
        self.__table.select_rows(source_vms)
        self.select_hyperlink(self.props['action.failback'])
        self.click_button(self.props['action.failback'])
        job_id = self.get_jobid_from_popup()
        return job_id

    @PageService()
    def get_source_and_destination(self):
        """
        Get source and destination vm names
        Returns: source and destination vm names
        """
        source_vms = self.__table.get_column_data(self.props['header.source'])
        destination_vms = self.__table.get_column_data(self.props['label.vmDestinationVM'])
        return source_vms, destination_vms

    @WebAction()
    def __get_recovery_target(self):
        """
        Fetch Recovery Target
        Returns: Recovery Target Name
        """
        return self.driver.find_element_by_xpath(
            "//a[contains(@href,'#/lifeCyclePolicyDetails')]").text

    @PageService()
    def get_recovery_target(self):
        """
        Return Recovery Target
        Returns: recovery target name
        """
        return self.__get_recovery_target()


