# -*- coding: utf-8 -*-
# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
This module provides the function or operations that can be performed on the
Regions page on the AdminConsole

Class:

    Regions()

Functions:

_init_()                                :       initialize the class object

__fill_location(self, location)         :       Adds location to location field in the form

__select_location(self, location)       :       Clicks the specified location from location drop-down

__add_location(self, locations)         :       Searches for location from a given list of locations selects it

 __remove_location(self, locations)     :       Removes locations from selected locations

 add_region(self, region_name, region_locations)    :       Method to add a new region

 delete_region(self, region_name)                   :       Method to delete a region

 access_region_details(self, region_name)           :       Method to get the region details

 get_region_locations(self, region_name)            :       Method to get the list of locations under a given region

 edit_region_locations(self, region_name, region_locations)     :       Method to edit locations in given region

"""

from Web.Common.page_object import (WebAction, PageService)
from Web.AdminConsole.Components.table import Table
from Web.AdminConsole.Components.panel import DropDown


class Regions:
    """ Class for Regions page in Admin Console """

    def __init__(self, admin_console):
        """
        Method to initiate Regions class

        Args:
            admin_console   (Object) :   Admin Console Class object
        """
        self.__admin_console = admin_console
        self.__table = Table(self.__admin_console)
        self.__log = admin_console.log
        self.__driver = admin_console.driver
        self.__drop_down = DropDown(self.__admin_console)

    @WebAction()
    def __fill_location(self, location):
        """
        Adds location to location field in the form
        Args:
            location (string):      location name to be added
        """
        self.__driver.find_element_by_xpath('//input[@name="location"]').send_keys(location)

    @WebAction()
    def __select_location(self, location):
        """
        Clicks the specified location from location drop-down
        Args:
            location (string):      location name to be clicked
        """
        location_xpath = f'//li[contains(text(), "{location}")]'
        if self.__admin_console.check_if_entity_exists('xpath', location_xpath):
            self.__driver.find_element_by_xpath(location_xpath).click()
            return True
        else:
            return False

    @PageService()
    def __add_location(self, locations):
        """
        Searches for location from a given list of locations selects it

        Args:
            locations (list):   list of locations
        """
        for location in locations:
            self.__fill_location(location)
            self.__admin_console.wait_for_completion()
            if self.__admin_console.check_if_entity_exists('xpath', '//ul[@class="ng-scope"]'):
                if self.__select_location(location):
                    self.__admin_console.wait_for_completion()
                    self.__log.info('Location {} added'.format(location))
                else:
                    exp = "No such location found"
                    raise Exception(exp)
            else:
                exp = 'Cannot search given location'
                raise Exception(exp)

    @WebAction()
    def __click_remove_location(self, location):
        """Click the remove button for locations"""
        xpath = '//span[contains(text(),"{}") and @class="tag-text ng-binding"]/following-sibling::div' \
            .format(location)
        self.__driver.find_element_by_xpath(xpath).click()

    @PageService()
    def __remove_location(self, locations):
        """
        Removes locations from selected locations
        Args:
            locations (list):   list of locations
        """
        for location in locations:
            self.__click_remove_location(location)


    @PageService()
    def add_region(self, region_name, region_locations):
        """
        Method to add a new region
        Args:
            region_name(string) :   test region name
            region_locations (list):   list of locations
        """
        self.__admin_console.select_hyperlink('Add region')
        self.__admin_console.fill_form_by_name('regionName', region_name)
        self.__add_location(region_locations)
        self.__admin_console.click_button_using_text('Save')
        self.__admin_console.check_error_message()

    @PageService()
    def delete_region(self, region_name):
        """
        Method to delete a region
        Args:
            region_name(string) :   test region name
        """
        self.__table.access_context_action_item(region_name, 'Delete region')
        self.__admin_console.click_button('Yes')

    @PageService()
    def access_region_details(self, region_name):
        """
        Method to get the region details
        Args:
            region_name(string) :   test region name
        """
        self.__table.access_link(region_name)

    @PageService()
    def get_region_locations(self, region_name):
        """
        Method to get the list of locations under a given region
        Args:
            region_name(string) :   test region name
        """
        self.__table.search_for(region_name)
        region_locations = self.__table.get_column_data('Location')
        return region_locations

    @PageService()
    def edit_region_locations(self, region_name, region_locations):
        """
        Method to edit locations in given region
        Args:
            region_name(string) :   test region name
            region_locations (dict):   dictionary of locations
        """
        self.__table.access_link(region_name)
        self.__admin_console.select_hyperlink('Add locations')
        add_locations = region_locations['Add']
        remove_locations = region_locations['Remove']
        self.__add_location(add_locations)
        self.__remove_location(remove_locations)
        self.__admin_console.click_button('Save')
        self.__admin_console.check_error_message()
