
# -*- coding: utf-8 -*-s

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------
"""
This module provides the function or operations that can be performed on the
Maintenance page on the AdminConsole

MaintenanceTiles:

    access_tile()                       --  Method to access tile displayed on maintenance page

    access_tile_settings()              --  Method to access tile settings

    run_job_for_tile()                  --  Method to run job for given tile

    run_job_for_tile_entity()           --  Method to run job for some entity inside tile

Maintenance:

    access_http_proxy()                  --  loads http proxy tile from maintainence page

    access_dr_backup()                   --  loads DR backup tile from maintainence page

    access_internet_options()            --  loads download software tile from maintainence page

    access_download_software()           --  loads Internet options tile from maintainence page

    access_install_update_schedules()    --  loads Install update schedule tile from
                                            maintainence page

    edit_download_schedule()            --   Edits the system created download schedule.

    add_schedule()                      --   Adds new download schedule

    add_install_schedule()              --   Creates a new install schedule on the given
                                             clients/client group.

    run_install_updates()               --   runs install updates job for the given clients/client groups

    run_copy_software()                 --  Runs copy software job with configured settings

    run_download_software()             --   Runs  download software job with configured settings.

    disable_http_proxy()                --   Disables http proxy

    set_internet_options()              --   Configures internet gateway from maintenence page.

    enable_http_proxy()                 --   set up http proxy on commcell.

"""
from Web.AdminConsole.AdminConsolePages.AdminPage import AdminPage
from Web.AdminConsole.AdminConsolePages.Companies import Companies
from Web.AdminConsole.Components.panel import DropDown, PanelInfo
from Web.AdminConsole.Components.browse import CVAdvancedTree
from Web.AdminConsole.Helper import adminconsoleconstants as acdefines
from Web.Common.page_object import (
    WebAction,
    PageService
)


class MaintenanceTiles:
    """ Maintenance Tile class """

    def __init__(self, admin_console):
        """ Initialize the backup object

        Args:
            admin_console: instance of AdminConsoleBase

        """
        self._driver = admin_console.driver
        self._admin_console_base = admin_console
        self.drop_down = DropDown(admin_console)

    @WebAction()
    def access_tile(self, tile_header):
        """ Method to access tile displayed on maintenance page """
        tile = self._driver.find_element_by_xpath(
            f"//cv-tile-component[@data-title='{tile_header}']")
        tile.click()

    @WebAction()
    def access_tile_settings(self, tile_header):
        """ Method to access tile settings """
        tile_setting_icon = self._driver.find_element_by_xpath(
            f"//cv-tile-component[@data-title='{tile_header}']"
            f"//div[contains(@class,'page-details-box-links')]//div[contains(text(),'Edit')]")
        tile_setting_icon.click()

    @WebAction()
    def run_job_for_tile(self, tile_header):
        """ Method to run job for given tile """
        tile_run_icon = self._driver.find_element_by_xpath(
            f"//cv-tile-component[@data-title='{tile_header}']"
            f"//div[contains(@class,'page-details-box-links')]//div[contains(text(),'Run')]")
        tile_run_icon.click()

    @WebAction()
    def schedule_job_for_tile(self, tile_header):
        """ Method to run job for given tile """
        tile_run_icon = self._driver.find_element_by_xpath(
            f"//cv-tile-component[@data-title='{tile_header}']"
            f"//div[contains(@class,'page-details-box-links')]//div[contains(text(),'{self._admin_console_base.props['label.schedule']}')]")
        tile_run_icon.click()

    @WebAction()
    def run_job_for_tile_entity(self, tile_header, entity_label):
        """ Method to run job for some entity inside tile """
        entity_run_icon = self._driver.find_element_by_xpath(
            f"//cv-tile-component[@data-title='{tile_header}']"
            f"//a[contains(text(),'{entity_label}')]/ancestor::li//div[contains(text(),'Run')]"
        )
        entity_run_icon.click()


class _HttpProxy:
    """Class for http proxy panel use Maintenance object access_http_proxy to get this object"""

    def __init__(self, admin_console):
        """
        Args:
            admin_console: instance of AdminConsoleBase
        """
        self._driver = admin_console.driver
        self._admin_console = admin_console
        self.http_panel = PanelInfo(self._admin_console, 'HTTP proxy')

    @PageService()
    def enable_http_proxy(
            self,
            proxy_server,
            proxy_port
    ):
        """
        Enables http proxy

        Args:
            proxy_server    (str)  :  Host name or IP address of the proxy server.

            proxy_port      (int)  :  port no on which proxy server listens to.

        """
        key = self._admin_console.props['label.proxyServer']
        proxy_toggle = self.http_panel.get_toggle_element(key)
        if self.http_panel.is_toggle_enabled(proxy_toggle):
            self.http_panel.edit_tile_entity(key)
            self._admin_console.wait_for_completion()
        else:
            self.http_panel.enable_toggle(key)
            self._admin_console.wait_for_completion()
        self._admin_console.fill_form_by_id("proxyServer", proxy_server)
        self._admin_console.fill_form_by_id("proxyPort", proxy_port)
        self._admin_console.submit_form()

    @PageService()
    def disable_http_proxy(self):
        """ Disables http proxy"""
        self._admin_console.toggle_disable(
            self._admin_console.props['label.proxyServer']
        )

    @PageService()
    def enable_authentication(self, proxy_user_name, proxy_password):
        """
        Enables http proxy authentication
        Args:
            proxy_user_name (str): Proxy user name
            proxy_password  (str): Proxy password
        """
        key = self._admin_console.props['label.availableAuthentication']
        user_authentication_toggle = self.http_panel.get_toggle_element(key)
        if self.http_panel.is_toggle_enabled(user_authentication_toggle):
            self.http_panel.edit_tile_entity(key)
            self._admin_console.wait_for_completion()
        else:
            self.http_panel.enable_toggle(key)
            self._admin_console.wait_for_completion()
        self._admin_console.fill_form_by_id("userName", proxy_user_name)
        self._admin_console.fill_form_by_id("password", proxy_password)
        self._admin_console.fill_form_by_id("confirmPassword", proxy_password)
        self._admin_console.submit_form()

    @PageService()
    def disable_authentication(self):
        """ Disables http proxy authentication"""
        self._admin_console.toggle_disable(
            self._admin_console.props['label.availableAuthentication']
        )


class Maintenance:
    """
    This class provides the function or operations that can be performed on the
    maintenance page on the AdminConsole
    """

    def __init__(self, admin_console):
        """
        Method to initiate Maintenance class
        Args:
            admin_console: instance of AdminConsoleBase
        """
        self._driver = admin_console.driver
        self._admin_console = admin_console
        self._admin_console._load_properties(self)
        self.admin_page = AdminPage(self._driver)
        self.__maintenance_tile = MaintenanceTiles(self._admin_console)
        self.company = Companies(self._admin_console)
        self._cv_advanced_tree = CVAdvancedTree(admin_console)

    @WebAction()
    def _add_one_time_schedule(
            self,
            schedule_options=None
    ):
        """
        To add or edit one time schedule using the given options
        Usage:
            * sample dict for schedule options

                schedule_options:
                {
                    'year': '2017',
                    'month': 'december',
                    'date': '31',
                    'hours': '09',
                    'mins': '19',
                    'session': 'AM'
                }
        """
        self.__maintenance_tile.drop_down.select_drop_down_values(0, [self._admin_console.props['option.oneTime']])
        if schedule_options:
            self._admin_console.date_picker(schedule_options)

    @WebAction()
    def _add_automatic_schedule(
            self,
            schedule_options=None
    ):
        """
        To add or edit automatic schedule using the given options

        Args:
            schedule_options (dict)     -- automatic schedule options to be selected

        Usage:
            * sample dict for schedule options

                schedule_options:
                {
                    'frequency': 'Automatic'
                    'min_job_interval_hrs': '24',
                    'min_job_interval_mins': '30',
                    'max_job_interval_hrs': '72',
                    'max_job_interval_mins': '45',
                    'min_sync_interval_hrs': '1',
                    'min_sync_interval_mins': '30',
                    'ignore_operation_window': True,
                    'only_wired': True,
                    'min_bandwidth': True,
                    'bandwidth': '128',
                    'use_specific_network': True,
                    'specific_network_ip_address': '0.0.0.0',
                    'specific_network': '24',
                    'start_on_ac': True,
                    'stop_task': True,
                    'prevent_sleep': True,
                    'cpu_utilization_below': True,
                    'cpu_below_threshold': '10',
                    'start_only_files_bkp': True
                }

        **Note** For Automatic there is NO default value

        """
        self.__maintenance_tile.drop_down.select_drop_down_values(0, [self._admin_console.props['option.automatic']])
        # To fill the Min job interval
        if schedule_options.get('min_job_interval_hrs') and schedule_options.get(
                'min_job_interval_mins'):
            self._admin_console.fill_form_by_id("minBackupIntervalHours",
                                                schedule_options['min_job_interval_hrs'])
            self._admin_console.fill_form_by_id("minBackupIntervalMinutes",
                                                schedule_options['min_job_interval_mins'])

        # To fill the Max job interval
        if schedule_options.get('max_job_interval_hrs') and schedule_options.get(
                'max_job_interval_mins'):
            self._admin_console.fill_form_by_id("maxBackupIntervalHours",
                                                schedule_options['max_job_interval_hrs'])
            self._admin_console.fill_form_by_id("maxBackupIntervalMinutes",
                                                schedule_options['max_job_interval_mins'])

        if schedule_options.get('min_sync_interval_hrs') and \
                schedule_options.get('min_sync_interval_mins'):
            self._admin_console.fill_form_by_id("minSyncIntervalHours",
                                                schedule_options['min_sync_interval_hrs'])
            self._admin_console.fill_form_by_id("minSyncIntervalMinutes",
                                                schedule_options['min_sync_interval_mins'])

            if schedule_options.get('ignore_operation_window'):
                self._admin_console.checkbox_select("ignoreAtMaxInterval")

        # To fill the Network Management
        # To select the wired network checkbox
        if schedule_options.get('only_wired'):
            self._admin_console.checkbox_select("onlyWiredWork")

        # To select the min network bandwidth checkbox
        if schedule_options.get('min_bandwidth'):
            self._admin_console.checkbox_select("minBandwidth")
            if schedule_options.get('bandwidth'):
                self._admin_console.fill_form_by_id("bandwidth", schedule_options['bandwidth'])
            else:
                raise Exception('Bandwidth argument missing for Automatic schedule')

        # To select the specific network checkbox
        if schedule_options.get('use_specific_network'):
            self._admin_console.checkbox_select("specificNetwork")
            if schedule_options.get('specific_network_ip_address') and \
                    schedule_options.get('specific_network'):
                self._admin_console.fill_form_by_name("specificNetworkIpAddress",
                                                      schedule_options['specific_network_ip_address'])
                self._driver.find_element_by_xpath(
                    "//input[@id='specificNetwork'and @type='number']").send_keys(
                        schedule_options['specific_network'])
            else:
                raise Exception("Specific network arguments missing in automatic schedule")

        # Power Management
        # To select A/C power checkbox
        if schedule_options.get('start_on_ac'):
            self._admin_console.checkbox_select("startOnAC")

        # To select the stop task checkbox
        if schedule_options.get('stop_task'):
            self._admin_console.checkbox_select("StopTask")

        # To select the prevent sleep checkbox
        if schedule_options.get('prevent_sleep'):
            self._admin_console.checkbox_select("preventSleep")

        # Resource Utilization
        if schedule_options.get('cpu_utilization_below'):
            self._admin_console.checkbox_select("cpuBelowThresholdEnabled")
            if schedule_options.get('cpu_below_threshold'):
                self._admin_console.fill_form_by_id("cpuBelowThreshold", schedule_options['cpu_below_threshold'])
            else:
                raise Exception('CPU threshold missing in automatic schedule')

        # File Management
        if schedule_options.get('start_only_files_bkp'):
            self._admin_console.checkbox_select("startOnlyFileBackUp")

    @WebAction()
    def _add_daily_schedule(
            self,
            schedule_options=None
    ):
        """
        To add or edit daily schedule using the given options

        Args:
            schedule_options (dict)     -- daily schedule options to be selected
        Usage:
                    {
                        'frequency': 'Daily',
                        'hours': '09',
                        'mins': '15',
                        'session': 'AM',
                        'repeatMonth': '1',
                        'exceptions': True,
                        'day_exceptions': True,
                        'week_exceptions': True,
                        'exception_days': ['monday','friday'],
                        'exception_weeks': ['First', 'Last'],
                        'repeat': True,
                        'repeat_hrs': '8',
                        'repeat_mins': '25',
                        'until_hrs': '11',
                        'until_mins': '59',
                        'until_sess': 'PM'
                    }
        """
        self.__maintenance_tile.drop_down.select_drop_down_values(0, [self._admin_console.props['option.daily']])
        if schedule_options.get('repeatDay'):
            self._admin_console.fill_form_by_id("dayFrequency", schedule_options.get('repeatDay'))

    @WebAction()
    def _add_weekly_schedule(
            self,
            schedule_options=None
    ):
        """
        To add or edit weekly schedule using the given options

        Args:
            schedule_options (dict)     -- weekly schedule options to be selected
        Usage:
                    {
                        'frequency': 'Weekly',
                        'hours': '09',
                        'mins': '15',
                        'session': 'AM',
                        'days': ['Monday', 'Friday', 'Sunday'],
                        'repeatDay': '1',
                        'exceptions': True,
                        'day_exceptions': True,
                        'week_exceptions': True,
                        'exception_days': ['monday','friday'],
                        'exception_weeks': ['First', 'Last'],
                        'repeat': True,
                        'repeat_hrs': '8',
                        'repeat_mins': '25',
                        'until_hrs': '11',
                        'until_mins': '59',
                        'until_sess': 'PM'
                    }


        """

        self.__maintenance_tile.drop_down.select_drop_down_values(0, [self._admin_console.props['option.weekly']])
        if schedule_options.get('days'):
            self.__maintenance_tile.drop_down.select_drop_down_values(1, schedule_options['days'])
        if schedule_options.get('repeatWeek'):
            self._admin_console.fill_form_by_id("weekFrequency", schedule_options.get('repeatWeek'))

    @WebAction()
    def _add_monthly_schedule(
            self,
            schedule_options=None
    ):
        """
        To add or edit monthly schedule using the given options

        Args:
            schedule_options (dict)     -- monthly schedule options to be selected
        Usage:
             schedule_options =         {
                        'frequency': 'Monthly',
                        'hours': '09',
                        'mins': '15',
                        'session': 'AM',
                        'day_of_month': '25',
                        'custom_week': 'Second',
                        'custom_day': 'Weekend Day',
                        'repeatMonth': '1',
                        'exceptions': True,
                        'day_exceptions': True,
                        'week_exceptions': True,
                        'exception_days': ['Monday','Friday'],
                        'exception_weeks': ['First', 'Last'],
                        'repeat': True,
                        'repeat_hrs': '8',
                        'repeat_mins': '25',
                        'until_hrs': '11',
                        'until_mins': '59',
                        'until_sess': 'PM'
                    }

        """
        self.__maintenance_tile.drop_down.select_drop_down_values(0, [self._admin_console.props['option.monthly']])

        if schedule_options.get('day_of_month'):
            self._admin_console.fill_form_by_id("dayOfMonth", schedule_options['day_of_month'])
        elif schedule_options.get('custom_week') and schedule_options.get('custom_day'):
            self._admin_console.select_radio("monthly_relative")
            self.__maintenance_tile.drop_down.select_drop_down_values(1, schedule_options['custom_week'])

            self.__maintenance_tile.drop_down.select_drop_down_values(2, schedule_options['custom_day'])
        else:
            raise Exception('Arguments missing for Monthly schedule')

        if schedule_options.get('repeatMonth'):
            self._admin_console.fill_form_by_id("monthRelativeFrequency", schedule_options.get('repeatMonth'))

    @WebAction()
    def _add_continuous_schedule(
            self,
            schedule_options=None
    ):
        """
        To add or edit continuous schedule using the given options

        Usage:

                schedule_options:
                {
                    'continuous_interval': '30'
                }

        **Note** There are NO default values for continuous schedule
        """
        self.__maintenance_tile.drop_down.select_drop_down_values(0, self._admin_console.props['option.continous'])
        if schedule_options.get('continuous_interval'):
            self._admin_console.fill_form_by_id("intervalBetweenTwoJobs", schedule_options['continuous_interval'])
        else:
            raise Exception('Job interval missing for continuous schedule')

    @WebAction()
    def _schedule_repeat(self, schedule_options):
        """
        To edit schedule repeat using the given options
        Usage:
            * sample dict for schedule options

                schedule_options:
                {
                    'exceptions': True,
                    'day_exceptions': False
                    'week_exceptions': True,
                    'exception_days': ['monday','friday'],
                    'exception_weeks': ['First', 'Last'],
                    'repeat': True,
                    'repeat_hrs': '8',
                    'repeat_mins': '25',
                    'until_hrs': '11',
                    'until_mins': '59',
                    'until_sess': 'PM',
                }
        """
        if schedule_options.get('repeat'):
            self._admin_console.checkbox_select("repeat")

        # To fill the repeat hours and mins
        if schedule_options.get('repeat_hrs') and schedule_options.get('repeat_mins'):
            self._admin_console.fill_form_by_id("repeatHours", schedule_options['repeat_hrs'])
            self._admin_console.fill_form_by_id("repeatMinutes", schedule_options['repeat_mins'])

        # To fill the time to stop the schedule
        if schedule_options.get('until_hrs') and schedule_options.get('until_mins'):
            self._driver.find_element_by_xpath(
                "//*[@id='repeatTime']/table/tbody/tr[2]/td[1]/input").clear()
            self._driver.find_element_by_xpath(
                "//*[@id='repeatTime']/table/tbody/tr[2]/td[1]/input").send_keys(
                    schedule_options['until_hrs'])
            self._driver.find_element_by_xpath(
                "//*[@id='repeatTime']/table/tbody/tr[2]/td[3]/input").clear()
            self._driver.find_element_by_xpath(
                "//*[@id='repeatTime']/table/tbody/tr[2]/td[3]/input"
            ).send_keys(schedule_options['until_mins'])

            # To change session AM or PM in repeat
            if schedule_options.get('until_sess'):
                sess = self._driver.find_element_by_xpath(
                    "//*[@id='repeatTime']/table/tbody/tr[2]/td[6]/button")
                if not schedule_options.get('until_sess') == sess.text:
                    sess.click()

        # To fill the Exceptions in Repeat schedule
        if schedule_options.get('exceptions'):
            self._admin_console.select_hyperlink("Exceptions")

            if schedule_options.get('day_exceptions'):
                self._admin_console.select_radio("dayInMonth")
                if schedule_options.get('exception_days'):
                    self.__maintenance_tile.drop_down.select_drop_down_values(0,
                                                                              schedule_options['exception_days'])
                if self._admin_console.check_if_entity_exists('xpath', "//form/div[2]/div/button["
                                                              "contains(text(),'Add')]"):
                    self._driver.find_element_by_xpath("//form/div[2]/div/button["
                                                       "contains(text(),'Add')]"
                                                       ).click()
                    self._admin_console.wait_for_completion()

            if schedule_options.get('week_exceptions'):
                self._admin_console.select_radio("weekInMonth")

                if schedule_options.get('exception_weeks'):
                    self.__maintenance_tile.drop_down.select_drop_down_values(0,
                                                                              schedule_options['exception_weeks'])
                if schedule_options.get('exception_days'):
                    self.__maintenance_tile.drop_down.select_drop_down_values(1,
                                                                              schedule_options['exception_days'])

                if self._admin_console.check_if_entity_exists('xpath', "//form/div[2]/div/button["
                                                              "contains(text(),'Add')]"):
                    self._driver.find_element_by_xpath("//form/div[2]/div/button["
                                                       "contains(text(),'Add')]"
                                                       ).click()
                    self._admin_console.wait_for_completion()

            self._admin_console.submit_form()

    @PageService()
    def access_http_proxy(self):
        """Loads http proxy tile and returns the HttpProxy class object"""
        self.__maintenance_tile.access_tile('HTTP proxy')
        self._admin_console.wait_for_completion()
        return _HttpProxy(self._admin_console)

    @PageService()
    def access_dr_backup(self):
        """ loads DR backup tile from maintenance page """
        self.__maintenance_tile.access_tile('DR backup (Daily)')
        self._admin_console.wait_for_completion()

    @PageService()
    def access_internet_options(self):
        """ loads Internet Options tile from maintenance page """
        self.__maintenance_tile.access_tile('Internet options')
        self._admin_console.wait_for_completion()

    @PageService()
    def access_download_software(self):
        """ loads Download software tile from maintenance page """
        self.__maintenance_tile.access_tile(self._admin_console.props['label.manageSoftware'])
        self._admin_console.wait_for_completion()

    @PageService()
    def access_install_update_schedules(self):
        """ loads Install update schedule tile from maintenance page """
        self.__maintenance_tile.access_tile('Install update schedules')
        self._admin_console.wait_for_completion()

    @PageService()
    def access_web_domains(self):
        """Access Web domains"""
        self.__maintenance_tile.access_tile('Web Domains')
        self._admin_console.wait_for_completion()

    @PageService()
    def set_internet_options(self, internet_option, gateway_client=None):
        """
        Configures internet gateway from maintenence page.

        Args:
            internet_option (Enum) -- InternetOptions defined in adminconsoleconstants
                                        (Ex:InternetOptions.USE_CLIENT)

            gateway_client  (str) -- name of the client , if InternetOptions.USE_CLIENT
            option is selected
        """
        self.__maintenance_tile.access_tile_settings("Internet options")
        self._admin_console.wait_for_completion()
        self._admin_console.select_radio(internet_option.value)
        if acdefines.InternetOptions.USE_CLIENT.value == internet_option.value:
            if isinstance(gateway_client, str):
                self._admin_console.select_value_from_dropdown("gatewayClient", gateway_client)
            else:
                raise Exception("Invalid Input parameter :gateway_client")
        self._admin_console.submit_form()

    @property
    def current_service_pack(self):
        """

        Returns: Current service pack property from donwload tile.

        """
        return self._admin_console.label_getvalue("Current service pack")

    @property
    def internet_gateway(self):
        """

        Returns: Internet gateway property from donwload tile.

        """
        return self._admin_console.label_getvalue("Internet gateway")

    @PageService()
    def _copy_software(
            self,
            media_path,
            auth=False,
            username=None,
            password=None):
        """
        Fills download software page with configured settings.
        Args:
            media_path (str)  -- path to copy media from
            auth (bool) -- set to True if authentication is required. Default is False
            username (str) -- username if authentication is required
            password (str) -- password if authentication is required
        Returns:
            None
        Raises:
            Exception if inputs are invalid
        """
        self._admin_console.select_radio("downloadUsingCopy")
        self._admin_console.fill_form_by_id("updateCachePath", media_path)
        if auth:
            if not username and not password:
                raise Exception("Auth is True but no credentials provided")
            self.company.enable_toggle("label.impersonateUser")
            self._admin_console.fill_form_by_id("uname", username)
            self._admin_console.fill_form_by_id("pass", password)
            self._admin_console.fill_form_by_id("confirm", password)

    @PageService()
    def _download_software(self, download_option=None, sp_version=None, os_to_download=None):
        """
        Fills download software page with configured settings.

        Args:
            download_option (str) -- download option to be chosen

            sp_version  (str)     -- sp version to download (EX: 'SP12')

            os_to_download  (list) --  List of os to be downloaded

        Returns:
            None

        Raises:
            Exception

                if inputs are invalid

        """
        self._admin_console.select_radio("downloadUsingInternet")
        if download_option:
            # To select the download option
            self._admin_console.select_radio(download_option)

        if acdefines.DownloadOptions.GIVEN_SP_AND_HF.value == download_option:
            if isinstance(sp_version, str):
                self.__maintenance_tile.drop_down.select_drop_down_values(0, [sp_version.upper()])
            else:
                raise Exception("Invalid Input parameter :sp_version")

        if os_to_download:
            win_os = []
            unix_os = []

            for os in os_to_download:
                if 'Windows' in os:
                    win_os.append(os)
                else:
                    unix_os.append(os)

            if win_os:
                if acdefines.DownloadOSID.WINDOWS_64.value not in win_os:
                    self.__maintenance_tile.drop_down.deselect_drop_down_values(
                        1, [acdefines.DownloadOSID.WINDOWS_64.value])
                if acdefines.DownloadOSID.WINDOWS_32.value in win_os:
                    self.__maintenance_tile.drop_down.select_drop_down_values(
                        1, [acdefines.DownloadOSID.WINDOWS_32.value])

            if unix_os:
                self._admin_console.checkbox_select("unixCheck")
                self.__maintenance_tile.drop_down.select_drop_down_values(2, unix_os)
                if acdefines.DownloadOSID.UNIX_LINUX64.value not in unix_os:
                    self.__maintenance_tile.drop_down.deselect_drop_down_values(
                        2, [acdefines.DownloadOSID.UNIX_LINUX64.value])

    @PageService()
    def run_copy_software(self, media_path, auth=False, username=None, password=None):
        """
        Runs copy software job with configured settings
        Args:
            media_path (str)  -- path to copy media from
            auth (bool) -- set to True if authentication is required. Default is False
            username (str) -- username if authentication is required
            password (str) -- password if authentication is required
        Returns:
            str -- jobid of the copy software job
        Raises:
            Exception if job fails, fails to start or inputs are invalid
        """
        self.access_download_software()
        self.__maintenance_tile.run_job_for_tile(self._admin_console.props['label.manageSoftware'])
        self._copy_software(media_path=media_path,
                            auth=auth,
                            username=username,
                            password=password)
        self._admin_console.submit_form()
        return self._admin_console.get_jobid_from_popup()

    @PageService()
    def add_install_schedule(self, schedule_name, schedule_options, clients=None, client_groups=None):
        """
        Creates a new install schedule for the given client/client groups

        Args:
            schedule_name   (str)   -- name of the schedule to be created.

            schedule_options(dict) -- schedule options to create a schedule

            clients         (list) -- list of clients

            client_groups   (list) -- list of client group names.

        """
        self.access_install_update_schedules()
        self._driver.find_element_by_id('internetProxy_button_#6092').click()
        self._admin_console.fill_form_by_id("name", schedule_name)

        if (clients is None) and (client_groups is None):
            raise Exception("Both client and client group is not provided")
        self._driver.find_element_by_xpath('//a[@class="accordion-toggle"]').click()
        if clients:
            self._cv_advanced_tree.select_elements(self._admin_console.props['label.nav.activeServers'], clients)
        if client_groups:
            self._cv_advanced_tree.select_elements(self._admin_console.props['label.nav.serverGroups'], client_groups)

        frequency = schedule_options.get('frequency')
        if not frequency:
            raise Exception('Missing argument: Frequency for schedule')

        self.add_schedule(schedule_options=schedule_options)
        self._admin_console.submit_form()

    @PageService()
    def run_install_updates(self, clients=None, client_groups=None):
        """
        run install updates job for the given client/client groups

        Args:
            clients         (list) -- list of clients

            client_groups   (list) -- list of client group names.

        """
        self._admin_console.refresh_page()
        self.access_install_update_schedules()
        self._driver.find_element_by_id('internetProxy_button_#0958').click()

        if (clients is None) and (client_groups is None):
            raise Exception("Both client and client group is not provided")
        self._driver.find_element_by_xpath(f"//input[@id='searchTree']").click()
        if clients:
            self._cv_advanced_tree.select_elements(self._admin_console.props['label.nav.activeServers'], clients)
        if client_groups:
            self._cv_advanced_tree.select_elements(self._admin_console.props['label.nav.serverGroups'],
                                                   client_groups)
        self._admin_console.submit_form()
        return self._admin_console.get_jobid_from_popup()

    @PageService()
    def run_download_software(self, download_option=None, sp_version=None, os_to_download=None):
        """
        Runs download software job with configured settings.

        Args:
            download_option (str) -- download option to be chosen

            sp_version  (str)   -- sp version to download
                                    (Example: 'SP12')

            os_to_download  (list) --  List of os to be downloaded

        Returns:
            (str)       --  Job id of the download job triggered

        Raises:
            Exception

                if inputs are invalid

                if job fails

                if job failed to start

        Usage:

            * Enum DownloadOptions defined in adminconsoleconstants can be used for providing
              input to download_option

                >>> DownloadOptions.LATEST_HF_FOR_INSTALLED_SP.value

            * Enum DownloadOSID defined in adminconsoleconstants can be used for providing input
              to os_to_download

                >>> [DownloadOSID.WINDOWS_32.value, acd.DownloadOSID.UNIX_AIX32.value]

            * If no arguments are given, by default Latest hotfixes for the installed service pack
              is chosen as the
              download_option and WindowsX64 is chosen as the os_to_download

        """
        self.access_download_software()
        self.__maintenance_tile.run_job_for_tile(self._admin_console.props['label.manageSoftware'])
        self._download_software(download_option=download_option,
                                sp_version=sp_version,
                                os_to_download=os_to_download)
        self._admin_console.submit_form(wait=False)
        return self._admin_console.get_jobid_from_popup()

    @PageService()
    def add_schedule(
            self,
            schedule_name=None,
            schedule_options=None
    ):
        """
        To add or edit a schedule using the given options

        Args:
            schedule_name    (str)   -- name of the schedule to be created

            schedule_options (dict)  -- schedule options to create or edit a schedule
        """
        if schedule_name:
            self._admin_console.fill_form_by_id("name", schedule_name)
        frequency = schedule_options.get('frequency')
        if not frequency:
            raise Exception('Missing argument: Frequeny for schedule')
        else:
            frequency = frequency.lower()
        try:
            getattr(self, '_add_' + frequency + '_schedule')(schedule_options)

        except Exception as exp:
            raise Exception(str(exp))

        if frequency in ['Daily', 'Weekly', 'Monthly']:
            # To fill the Time for schedule
            if schedule_options.get('hours') and schedule_options.get('mins'):
                self._driver.find_element_by_xpath(
                    "//table/tbody/tr[2]/td[1]/input").clear()
                self._driver.find_element_by_xpath(
                    "//table/tbody/tr[2]/td[1]/input").send_keys(schedule_options['hours'])
                self._driver.find_element_by_xpath(
                    "//table/tbody/tr[2]/td[3]/input").clear()
                self._driver.find_element_by_xpath(
                    "//table/tbody/tr[2]/td[3]/input").send_keys(schedule_options['mins'])

                # To change session to AM or PM
                if schedule_options.get('session'):
                    sess = self._driver.find_element_by_xpath(
                        "//table/tbody/tr[2]/td[6]/button")
                    if not sess.text == schedule_options['session']:
                        sess.click()

        self._schedule_repeat(schedule_options)
        self._admin_console.submit_form()

    @PageService()
    def edit_download_schedule(
            self,
            download_option=None,
            sp_version=None,
            os_to_download=None,
            schedule_name=None,
            schedule_options=None):
        """
        Edits the download software schedule with configured settings.

        Args:
            download_option (str)   -- download option to be chosen

            sp_version  (str)       -- sp version to download
                                        (Example: 'SP12')

            os_to_download  (list)  --  List of os to be downloaded

            schedule_name   (str)   -- name for the schedule

            schedule_options (dict) -- schedule options to be selected

        Usage:

            * Enum DownloadOptions defined in adminconsoleconstants can be used for providing
              input to download_option

                >>> DownloadOptions.LATEST_HF_FOR_INSTALLED_SP.value

            * Enum DownloadOSID defined in adminconsoleconstants can be used for providing
              input to os_to_download

                >>> [DownloadOSID.WINDOWS_32.value, acd.DownloadOSID.UNIX_AIX32.value]

            * If no arguments are given, by default Latest hotfixes for the installed service
              pack is chosen as the download_option and WindowsX64 is chosen as the os_to_download

        """

        self.access_download_software()
        self.__maintenance_tile.schedule_job_for_tile(self._admin_console.props['label.manageSoftware'])
        self._download_software(download_option=download_option,
                                sp_version=sp_version,
                                os_to_download=os_to_download)

        if schedule_options:
            if self._admin_console.check_if_entity_exists(
                    'xpath',
                    "//toggle-control[@id='disableSchedule']"
                    "/div[@class='cv-material-toggle cv-toggle enabled']"):
                self._driver.find_element_by_xpath("//a[@data-ng-click='editSchedule()']").click()
                self._admin_console.wait_for_completion()
            elif self._admin_console.check_if_entity_exists(
                    'xpath',
                    "//toggle-control[@title='Enable/Disable Schedule settings']"):
                self._driver.find_element_by_xpath(
                    "//toggle-control[@title='Enable/Disable Schedule settings']").click()
                self._admin_console.wait_for_completion()

            self.add_schedule(schedule_name, schedule_options)
        self._admin_console.submit_form()