# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
This module provides the function or operations related to key management servers in AdminConsole
KeyManagementServers : This class provides methods for key management server related operations

KeyManagementServers:

    select_kms()        -- To select a key management server

    delete_kms()        -- To delete a key management server

    add_aws_kmsp()      -- To add a new AWS KMSP

    add_kmip()          -- To add a new KMIP

"""
from Web.AdminConsole.Components.table import Table
from Web.Common.handlers import adminconsole_handler
from Web.AdminConsole.AdminConsolePages.AdminPage import AdminPage
from Web.Common.page_object import PageService


class KeyManagementServers(AdminPage):
    """
    This class provides the function or operations that can be
    performed on key management servers Page of the Admin Console
    """

    def __init__(self, driver):
        super().__init__(driver)
        self.__table = Table(self)

    @PageService()
    def select_kms(self, server_name):
        """
        To select a key management server

        Args:
        server_name     (str)       -- Name of the key management server to be selected

        """
        self.__table.access_link(server_name)

    @adminconsole_handler()
    def delete_kms(self, server_name):
        """
        To delete a key management server

        Args:
            server_name     (str)       -- Name of the key management server to be deleted

        """
        if self.check_if_entity_exists('id', 'search-field'):
            self.__table.access_action_item(server_name, 'Delete')
            self.driver.find_element_by_xpath("//button[text()='Yes'][@ng-click='yes()']").click()
            self.check_error_message()

    @PageService()
    def add_aws_kmsp(
            self,
            name=None,
            region=None,
            access_key=None,
            secret_access_key=None):
        """
        To add a new AWS KMSP

        Args:
            name        (str)   -- Name for the KMS

            region      (str)   -- Region to be selected for the KMS

            access_key  (str)   -- Access key for the KMS

            secret_access_key   (str)   -- Secret access key for the KMS

        """
        self.select_hyperlink('Add')
        self.select_hyperlink('AWS KMSP')
        self.fill_form_by_id('name', name)
        self.cv_single_select('Region', region)
        self.fill_form_by_id('accessKey', access_key)
        self.fill_form_by_id('secretAccess', secret_access_key)
        self.submit_form()

    @PageService()
    def add_kmip(
            self,
            name=None,
            key_length=None,
            server=None,
            port=None,
            passphrase=None,
            certificate=None,
            certificate_key=None,
            ca_certificate=None):
        """
        To add a new KMIP

        Args:
            name        (str)   -- Name of the KMIP to be added

            key_length  (str)   -- Keylength for encryption

            server      (str)   -- Server for the KMIP

            port        (str)   -- Port for the KMIP

            passphrase  (str)   -- Passphrase for the KMIP

            certificate (str)   -- Certificate path for the KMIP

            certificate_key (str)   -- Certificate key path for the KMIP

            ca_certificate  (str)   -- CA Certificate path for the KMIP

        """
        self.select_hyperlink('Add')
        self.select_hyperlink('KMIP')
        self.fill_form_by_id('name', name)
        self.select_value_from_dropdown('keyLength', key_length)
        self.fill_form_by_id('server', server)
        self.fill_form_by_id('port', port)
        self.fill_form_by_id('passphrase', passphrase)
        self.fill_form_by_id('certificate', certificate)
        self.fill_form_by_id('certificateKey', certificate_key)
        self.fill_form_by_id('caCertificate', ca_certificate)
        self.submit_form()
        self.check_error_message()
