#!/usr/bin/env python

"""
This module provides the function or operations that can be performed on the
Companies page on the AdminConsole

Class:

    IdentityServers() -> AdminConsoleBase() -> object()

Functions:

_init_()                -- initialize the class object identity serveres page
click_add_saml_app()    -- clicks the add saml app link
click_saml_tab()        -- selects the SAML tab in add domain dialog box
set_application_name()   -- fills the application name field with the given application name
set_smtp()               -- fills the SMTP field with the given value
upload_idp_metadata()   -- uploads the idp metadata file
set_webconsole_url()    -- fills the webconsole_url field with the specified URL
set_company_for_app() -- fills the company field with given value
upload_jks_file()       -- uploads the keystore file
set_alias_name()     -- fills the alias name field with the given value
set_keystore_password() --sets the keystore password field with given value
set_key_password()    --sets the key password field with given value
save_saml_app()        --clicks the save button in add saml app
select_identity_server()    --clicks on the given saml app link
add_saml_app()       -- adds saml app


Class:
    Domains

Functions:

__init__()                      -- initialize the class object identity serveres page
__click_ad_tab()                -- selects the AD tab in add domain dialog box
__enable_proxy_client()         -- Checks the enable proxy client checkbox
__disable_proxy_client()        -- Un-checks the enable proxy client checkbox
__return_elements()             -- Returns elements for the given xpath
__get_text_from_element_obj()   -- Returns text from the give element object
get_domains_list()              -- Returns list of domains
add_domain()                    -- adds a new domain to the admin console
delete_domain()                 -- delete the domain with the specified name
get_domains_list()              -- returns list of all domains displayed on the page
"""

import time
from Web.AdminConsole.Components.table import Table
from Web.AdminConsole.Components.panel import ModalPanel, DropDown
from Web.AdminConsole.Components.dialog import TransferOwnership
from Web.Common.page_object import WebAction, PageService


class IdentityServers:
    """
    This class provides the operations performed on Identity servers Page of adminconsole
    """

    def __init__(self, admin_console):
        self.__driver = admin_console.driver
        self.__admin_console = admin_console
        self.__navigator = admin_console.navigator
        self.log = self.__admin_console.log
        self.__table = Table(self.__admin_console)
        self.__drop_down = DropDown(admin_console)

    @WebAction()
    def click_add_saml_app(self):
        """
        clicks the add saml app link
        Args    :    None
        Returns :    None
        """
        span_elem = self.__driver.find_element_by_xpath("//span[@class='k-link k-menu-link']")
        span_elem.find_element_by_xpath("//a[contains(text(),\
            '" + self.__admin_console.props['label.add'] + "')]").click()

    @WebAction()
    def click_saml_tab(self):
        """
        selects the SAML tab in add domain dialog box
        """
        self.__driver.find_element_by_xpath("//div[@ng-click='authenticationChoice=2']").click()

    @WebAction()
    def set_application_name(self, app_name):
        """
        fills the application name field with the given application name
        Args:
        app_name    (str):name of the SAML app to edit
        """
        self.__admin_console.fill_form_by_id("name2", app_name)

    @WebAction()
    def upload_idp_metadata(self, idp_metadata_path):
        """
        uploads the idp metadata file
        Args:
        idp_metadata_path    (str):path of idp metadata file
        """

        self.__driver.find_element_by_id("IDPMetadata")\
            .send_keys(idp_metadata_path)
        time.sleep(2)

    @WebAction()
    def set_smtp(self, smtp_addr):
        """
        fills the SMTP field with the specified value
        Args:
        smtp_addr    (str):smtp address of the redirect rule
        """
        self.__admin_console.fill_form_by_id("SMTPAddressForDummyDomain", smtp_addr)

    @WebAction()
    def set_webconsole_url(self, webconsole_url):
        """
        fills the webconsole url field with the specified URL
        Args:
        webconsole_url    (str):url of webconsole to set
        """
        self.__admin_console.fill_form_by_id("webConsoleUrl", webconsole_url)

    @WebAction()
    def set_company_for_app(self, company_name):
        """
        fills the company name field with given value
        Args:
        company_name    (str):name of the company to set on app
        """
        self.__drop_down.select_drop_down_values(2, [company_name])

    @WebAction()
    def upload_jks_file(self, jks_file_path):
        """
        uploads the keystore file
        Args:
        jks_file_path    (str):path of the jks file to upload
        """
        self.__driver.find_element_by_id("SPMetadata").send_keys(jks_file_path)

    @WebAction()
    def set_alias_name(self, alias_name):
        """
        fills the alias name field with the given value
        Args:
        alias_name    (str):sets the given alias name for keystore file
        """
        self.__admin_console.fill_form_by_id("aliasName", alias_name)

    @WebAction()
    def set_keystore_password(self, keystore_password):
        """
        fills the keystore password field with the given value
        Args:
        keystore_password    (str):sets the given keystore_password
                                 for keystore file
        """
        self.__admin_console.fill_form_by_id("keyStorePassword", keystore_password)

    @WebAction()
    def set_key_password(self, key_password):
        """
        Fills the key password field with the given value
        Args:
        alias_name    (str):sets the given key password
                             for keystore file
        """
        self.__admin_console.fill_form_by_id("keyPassword", key_password)

    @PageService()
    def save_saml_app(self):
        """
        clicks the save button in add saml app
        """
        self.__admin_console.submit_form(True, 'addAppForm')

    @WebAction()
    def select_identity_server(self, app_name, app_type ="SAML"):
        """
        selects the given app with search box
        Args:
        app_name    (str):app name to select
        type        (str):AD/SAML
        """
        self.__table.access_link(app_name)

    @PageService()
    def add_saml_app(self, app_name, idp_metadata_path, smtp_address,
                     webconsole_url, enable_company,
                     company_name, jks_file_path=None, alias_name=None,
                     keystore_password=None, key_password=None, auto_generate_key=True):
        """
        Adds the saml app with the specified inputs in the arguments.
        Args:
        app_name            (str)    : name of SAML app
        idp_metadata_path    (str)    :IDP metadata file path
        webconsole_url        (str)   :webconsole url to edit
        enable_company        (bool)    :True if created for company
        company_name            (str)    :name of company to be added
        jks_file_path        (str)    :keystore file path
        key_password        (str)     : key password for the
                                         .jks file
        keystore_password    (str)    :keystore password for
                                        the .jks file
        alias_name            (str)    :alias name for
                                        the .jks file
        auto_generate_key    (bool)    :True if key is auto generated
        """
        self.click_add_saml_app()
        self.__admin_console.wait_for_completion()
        self.click_saml_tab()
        self.set_application_name(app_name)
        self.upload_idp_metadata(idp_metadata_path)
        self.set_smtp(smtp_address)
        self.set_webconsole_url(webconsole_url)
        if enable_company:
            self.set_company_for_app(company_name)
        if not auto_generate_key:
            self.__admin_console.disable_toggle(0)
            self.upload_jks_file(jks_file_path)
            self.set_alias_name(alias_name)
            self.set_keystore_password(keystore_password)
            self.set_key_password(key_password)
        else:
            self.__admin_console.enable_toggle(0)
        self.save_saml_app()
        self.__admin_console.check_error_message()
        return True


class Domains:

    def __init__(self, admin_console):

        self.__driver = admin_console.driver
        self.__admin_console = admin_console
        self.__table = Table(self.__admin_console)
        self.__modal_panel = ModalPanel(self.__admin_console)
        self.__transfer_ownership = TransferOwnership(self.__admin_console)
        self.__drop_down = DropDown(self.__admin_console)

    @WebAction()
    def __click_ad_tab(self):
        """
        selects the AD tab in add domain dialog box
        """
        self.__driver.find_element_by_xpath(
            "//div[@ng-click='authenticationChoice=1']").click()

    @WebAction()
    def enable_proxy_client(self):
        """Checks the proxy client checkbox"""
        hidden_checkbox_xpath = "//input[@id='accessADProxy']"
        if not self.__driver.find_element_by_xpath(hidden_checkbox_xpath).is_selected():
            self.__driver.find_element_by_xpath(f"{hidden_checkbox_xpath}/../label").click()

    @WebAction()
    def disable_proxy_client(self):
        """Unchecks the proxy client checkbox"""
        hidden_checkbox_xpath = "//input[@id='accessADProxy']"
        if self.__driver.find_element_by_xpath(hidden_checkbox_xpath).is_selected():
            self.__driver.find_element_by_xpath(f"{hidden_checkbox_xpath}/../label").click()

    @WebAction()
    def return_elements(self, xpath):
        """Returns elements"""

        return self.__driver.find_elements_by_xpath(xpath)

    @WebAction()
    def get_text_from_element_obj(self, elem, xpath):
        """Returns text from element object"""
        return elem.find_element_by_xpath(xpath).text

    @WebAction()
    def get_domains_list(self):
        """Method to get domains list"""
        domain_names = []
        elems = self.return_elements(
            "//div[@class='k-grid-content k-auto-scrollable']//tbody/tr")

        for elem in elems:
            if self.get_text_from_element_obj(elem, "./td[2]/span").text == 'AD':
                domain_name = self.get_text_from_element_obj(elem, "./td[1]/span/a").text
                domain_names.append(domain_name)
        return domain_names

    @PageService()
    def add_domain(self,
                   domain_name,
                   netbios_name,
                   username,
                   password,
                   proxy_client=None,
                   proxy_client_value=None,
                   company_name=None,
                   user_group=None,
                   local_group=None,
                   quota_enabled=False,
                   quota_limit=100):
        """
        Adds a new domain to the commcell

        Args:
            domain_name (basestring) : the name of the domain

            netbios_name (basestring): the netbios name of the domain

            username (basestring)    : the username of the domain

            password (basestring)    : the password of the domain

            proxy_client(boolean)    : enable/disable Access AD via client

            proxy_client_value(basestring) : client to access AD server from

            company_name (basestring): company name to be associated with domain

            user_group (basestring): user group be associated with domain

            local_group (list): list of local user groups to be associated with domain

            quota_enabled (boolean) : enable/disable quota enabled checkbox

            quota_limit (int)       : value of quota limit

        Returns:
            None

        """
        self.__admin_console.select_hyperlink("Add")
        self.__click_ad_tab()
        self.__drop_down.select_drop_down_values(0, ["Active directory"])
        self.__admin_console.fill_form_by_id("netBiosName", netbios_name)
        self.__admin_console.fill_form_by_id("domainName", domain_name)
        self.__admin_console.fill_form_by_id("uname", username)
        self.__admin_console.fill_form_by_id("pass", password)

        if proxy_client:
            self.enable_proxy_client()
            if proxy_client_value:
                self.__drop_down.select_drop_down_values(1, [proxy_client_value])
        else:
            self.disable_proxy_client()

        if company_name is not None:
            self.__admin_console.select_value_from_dropdown("createForCompanies", company_name)
        self.__admin_console.submit_form()
        self.__admin_console.check_error_message()

        if user_group:
            self.add_user_group(user_group, local_group, quota_enabled, quota_limit)
        else:
            self.__admin_console.click_button(self.__admin_console.props['label.cancel'])

    @PageService()
    def add_user_group(self, user_group, local_group=None, quota_enabled=False, quota_limit=100):
        """
        Method to add user group to the domain

        Args:
            user_group  (string)    : the user group to be added to the domain

            local_group (list)      : list of local groups to associate with the domain

            quota_enabled (boolean) : enable/disable quota enabled checkbox

            quota_limit (int)       : value of quota limit
        """

        self.__modal_panel.search_and_select("Group name", user_group)
        if local_group:
            self.__drop_down.select_drop_down_values(1, local_group)

        if quota_enabled:
            self.__admin_console.select_checkbox("quotaEnabled")
            if quota_limit:
                self.__admin_console.fill_form_by_id("quotaLimit", quota_limit)
        else:
            self.__admin_console.checkbox_deselect("quotaEnabled")

        self.__modal_panel.submit()

    @PageService()
    def delete_domain(self, domain_name, skip_ownership_transfer=True, transfer_owner=None):
        """
        Deletes the domain with the given name

        Args:
            domain_name (string)            : the name of the domain whose information has to
                                                   modified

            skip_ownership_transfer (bool)  : whether ownerhip has to be transferred or not

            transfer_owner (string)         : Name of owner if ownership of domain has to be
                                                  transferred

        Returns:
            None
        """

        if self.__admin_console.check_if_entity_exists("link", domain_name):

            self.__table.access_action_item(domain_name, "Delete")
            if not skip_ownership_transfer:
                self.__transfer_ownership.transfer_ownership(transfer_owner)
            else:
                self.__admin_console.click_button("Delete")
            self.__admin_console.check_error_message()
        else:
            raise Exception("Domain does not exist")

    @PageService()
    def domains_list(self):

        """Returns list of domains visible on the page"""

        self.__admin_console.navigate_to_identity_servers()
        domains_names = self.get_domains_list()
        return domains_names
