#!/usr/bin/env python

"""
This module provides the function or operations that can be performed on the
IdentityServerDetails Page on the AdminConsole

Class:

    IdentityServerDetails() -> AdminPage() -> login_page()
    login_page()-> AdminConsoleBase() -> object()

Functions:

get_app_name()              --Gets the SAML app name
is_app_enabled()            --Returns True/False if app is enabled /disabled
get_auto_create_user_flag()--Returns flag set for auto create user
get_default_user_group     --Returns the default user group set
get_company_name()         --Returns the company for which app is created
edit_idp_details()        --Selects the IDP edit option from app details page
download_sp_metadata()        --Downloads the SP metadata file from SAML app details page
modify_enable_app()       --changes the enable option in app
modify_auto_create_user()   -- Edits the auto user create option
modify_user_group()       -- Edits the user group option
copy_sp_entity_id()       --returns sp initiated link in app
copy_sso_url()            --returns sso url link in app
fetch_associations()      --fetches the associations of SAML app
delete_saml_app() --deletes SAML app
add_redirect_rule()       --adds redirect rule to the SAML app
edit_redirect_rule()      --edits redirect rule on the SAML app
delete_redirect_rule()    --deletes redirect rule on SAML app
fetch_redirect_rule()   --fetches redirect rule of SAML app
add_attribute_mappings()  --adds attribute mappings to the SAML app
edit_attribute_mapping() --edits attribute mappings on the SAML app
delete_attribute_mapping()--deletes attribute mappings on the SAML app
fetch_attribute_mappings()-- fetches attribute mappings on the SAML app
"""

import time
import os
from Web.AdminConsole.AdminConsolePages.identity_servers import IdentityServers
from Web.AdminConsole.Components.table import Table
from Web.AdminConsole.Components.panel import DropDown, PanelInfo
from Web.Common.page_object import PageService, WebAction
from Web.Common.exceptions import CVWebAutomationException


class IdentityServerDetails:
    """
    Class for the Identity server Details page
    """

    def __init__(self, admin_console):
        """
        Init method to create objects of classes used in the file.

        Args:
            admin_console  (Object) : Admin console class object
        """
        self.__driver = admin_console.driver
        self.__admin_console = admin_console
        self.__navigator = admin_console.navigator
        self.log = self.__admin_console.log
        self.__admin_console._load_properties(self)
        self.identity_servers_obj = IdentityServers(self.__admin_console)
        self.__table = Table(self.__admin_console)
        self.__drop_down = DropDown(self.__admin_console)

    @WebAction()
    def get_app_name(self):
        """
        Gets the SAML app name
        Args:    None
        Returns    (str):    app name
        """
        app_name = self.__driver.find_element_by_xpath(
            "//h1[@class='page-title ng-binding']").text
        return app_name

    @WebAction()
    def is_app_enabled(self):
        """
        Returns True/False if app is enabled/disabled
        Args:    None
        Returns (bool):    True/False
        """
        toggle_class = self.__driver.find_element_by_xpath(
            "//toggle-control[@class='tc.activityInfo.isEnabled \
                    ng-isolate-scope']/div").get_attribute('class')
        if toggle_class == "cv-material-toggle cv-toggle enabled":
            return True
        if toggle_class == "cv-material-toggle cv-toggle":
            return False
        raise CVWebAutomationException("SAML app in unknown state")

    @WebAction()
    def get_auto_create_user_flag(self):
        """
        Returns flag set for auto create user
        Args:    None
        Returns    (bool):    True/False
        """
        toggle_class = self.__driver.find_element_by_xpath(
            "//toggle-control[@class=\
            'tc.activityInfo.createUserAutomatically ng-isolate-scope']\
            /div").get_attribute('class')
        if toggle_class == "cv-material-toggle cv-toggle enabled":
            return True
        if toggle_class == "cv-material-toggle cv-toggle":
            return False
        raise CVWebAutomationException("SAML app in unknown state")

    @WebAction()
    def get_default_user_group(self):
        """
        Gets the default user group set in app
        Args:    None
        Returns    (str):    user group name
        """
        user_group_elem = self.__driver.find_element_by_xpath("//span\
                        [@class='pageDetailColumn' and \
                        contains(text(),'" + self.__admin_console.props['label.userGroup'] + "')]")
        default_user_group = user_group_elem.find_element_by_xpath(
            ".//../../span[2]").text
        default_user_group = default_user_group.split('\n')[0]
        return default_user_group

    @WebAction()
    def get_company_name(self):
        """
        Returns the company name for which the app is created
        Args:    None
        Returns    (str):    company name
        """
        company_name = None
        if self.__admin_console.check_if_entity_exists(
                "xpath", "//span[contains(text(),\
                        '" + self.__admin_console.props['label.createdForCompany'] + "')]"):
            company_elem = self.__driver.find_element_by_xpath(
                "//span[contains(text(),'" + self.__admin_console.props['label.createdForCompany'] + "')]")
            company_name = company_elem.find_element_by_xpath(
                ".//ancestor::li/span[2]").get_attribute('title')
        return company_name

    @PageService()
    def download_sp_metadata(self, app_name, download_dir):
        """
        Downloads the SP metadata and returns the location
        Args:
        app_name    (str): SAML app name for which
                            metadata has to be downloaded
        download_dir    (str):location where the file has to be downloaded
        Returns    (str):    SP metadata file downloaded location
        """
        filename = "SPMetadata_" + app_name + ".xml"
        downloaded_file_path = download_dir + "\\" + filename
        if os.path.exists(downloaded_file_path):
            os.remove(downloaded_file_path)
        self.__admin_console.select_hyperlink(self.__admin_console.props['action.downloadSpMetadata'])
        return downloaded_file_path

    @WebAction()
    def edit_saml_app_details(self):
        """
            Clicks Edit option in SAML app details page
        """
        self.__driver.find_element_by_xpath(
            "//a[@data-ng-click=\
                'tc.showAddThirdPartyAppDialog(tc.thirdPartyApp)']").click()

    @WebAction()
    def __expand_keystore_section(self):
        """ Method to expand keystore section in SAML App panel """
        self.__driver.find_element_by_xpath(
            "//legend[@class='cursor-pointer']").click()

    @PageService()
    def edit_idp_details(self, idp_metadata_path=None, web_console_url=None,
                         jks_file_path=None, key_password=None, keystore_password=None,
                         alias_name=None):
        """
        Edits the SAML app IDP details and keystore configurations

        Args:
        idp_metadata_path    (str)    :IDP metadata file path
        webconsole_url        (str)   :webconsole url to edit
        jks_file_path        (str)    :keystore file path
        key_password        (str)     :key password for the
                                         .jks file
        keystore_password    (str)    :keystore password for
                                        the .jks file
        alias_name            (str)    :alias name for
                                        the .jks file
        Returns:     None
        Raises:
            Exception:
                -- if editing app details failed
        """
        self.edit_saml_app_details()
        if idp_metadata_path:
            self.identity_servers_obj.upload_idp_metadata(idp_metadata_path)
        if web_console_url:
            self.identity_servers_obj.set_webconsole_url(web_console_url)
        if all([jks_file_path, alias_name, key_password, keystore_password]):
            self.__expand_keystore_section()
            self.__admin_console.wait_for_completion()
            self.identity_servers_obj.upload_jks_file(jks_file_path)
            self.identity_servers_obj.set_alias_name(alias_name)
            self.identity_servers_obj.set_key_password(key_password)
            self.identity_servers_obj.set_keystore_password(keystore_password)
        else:
            raise CVWebAutomationException("invalid inputs for jks file")
        self.__admin_console.submit_form()
        self.__admin_console.check_error_message()

    @PageService()
    def modify_enable_app(self, enable_app=True):
        """
        Changes the enable option in app
        Args
        enable_app    (bool):True/False
        Returns    None
        """
        if enable_app:
            self.__admin_console.enable_toggle(0)
        else:
            self.__admin_console.disable_toggle(0)

    @PageService()
    def modify_auto_create_user(self, enable_auto_create_user=True):
        """
        Edits the auto user create option
        Args
        enable_auto_create_user    (bool):True/False
        Returns    None
        """
        if enable_auto_create_user:
            self.__admin_console.enable_toggle(1)
        else:
            self.__admin_console.disable_toggle(1)
        self.__admin_console.wait_for_completion()

    @PageService()
    def add_default_usergroup(self, user_group_name):
        """
        Add's the default user group
        Args:
            user_group_name    (str):user group name
        Returns:
            None
        """
        PanelInfo(self.__admin_console, 'General').add_tile_entity(self.__admin_console.props['label.userGroup'])
        self.__admin_console.wait_for_completion()
        self._select_usergroup_from_dropdown(user_group_name)
        self.__admin_console.submit_form()
        self.__admin_console.check_error_message()

    @WebAction()
    def _select_usergroup_from_dropdown(self, user_group):
        """
        selects the user group from drop down list

        user_group (str)    :   user group name to select
        """
        self.__driver.find_element_by_xpath('//*[@id="defaultUserGroup"]/span/button').click()
        self.__driver.find_element_by_xpath('//*[@id="defaultUserGroup"]'
                                            '/span/div/div[1]/div/div[1]/input').send_keys(user_group)
        time.sleep(2)
        self.__driver.find_element_by_xpath('//*[@id="defaultUserGroup"]/span/div/div[2]').click()

    @PageService()
    def modify_user_group(self, user_group_name):
        """
        Edits the default user group
        Args:
            user_group_name    (str):new user group name
        Returns:
            None
        """
        PanelInfo(self.__admin_console, 'General').edit_tile_entity(self.__admin_console.props['label.userGroup'])
        self.__admin_console.wait_for_completion()
        self._select_usergroup_from_dropdown(user_group_name)
        self.__admin_console.submit_form()
        self.__admin_console.check_error_message()

    @WebAction()
    def copy_sp_entity_id(self):
        """
        Returns the sp entity id
        Args   : None
        Returns    (str):SP entity id of the SAML app
        """
        sp_entity_elem = self.__driver.find_element_by_xpath(
            "//span[contains(text(),'" + self.__admin_console.props['label.spEntityId'] + "')]")
        sp_entity_id = sp_entity_elem.find_element_by_xpath(
            ".//ancestor::li/span/span/input").get_attribute('title')
        return sp_entity_id

    @WebAction()
    def copy_sso_url(self):
        """
        Args       : None
        Returns    (str):Single sign on URL of the SAML app
        """
        sso_elem = self.__driver.find_element_by_xpath(
            "//span[contains(text(),'" + self.__admin_console.props['label.singleSignonUrl'] + "')]")
        sso_url = sso_elem.find_element_by_xpath(
            ".//ancestor::li/span/span/input").get_attribute('title')
        return sso_url

    @WebAction()
    def fetch_associations(self):
        """
        Returns the list of associations on SAML app
        Args   : None
        Returns    (dict):values as list of associations
                    {Associations:list of associated entities}
        """
        associated_entity = []
        ul_elements = self.__driver.find_elements_by_xpath("//ul[@class='list-style__\
                                            row group associated-users ng-scope']")
        for ul_elem in ul_elements:
            associated_entity.append(ul_elem.find_element_by_xpath(".//li/span").text)
        return associated_entity

    @PageService()
    def delete_saml_app(self):
        """
            Deletes the SAML app
            Args   : None
            Returns :   None

        """
        self.__admin_console.select_hyperlink(self.__admin_console.props['action.delete'])
        self.__admin_console.click_button('Yes')

    @WebAction()
    def add_redirect_rule(self, rule_dict):
        """
        Adds the given list of values in redirect rule
        Args   (dict):redirect rule to add
                        {"domain":"smtp_address1,smtp_address2"}
                        if domain is commcell then,
                        {"Commcell":"smtp_address1,smtp_address2"}
        Returns    :None

        """
        for domain_name, smtp_addr in rule_dict.items():
            self.__admin_console.select_hyperlink(self.__admin_console.props['pageHeader.addIdentityRule'])
            if domain_name != 'Commcell':
                self.__driver.find_element_by_xpath(
                    "//div[@class='cv-editable-select']/div/div/input").clear()
                self.__driver.find_element_by_xpath(
                    "//div[@class='cv-editable-select']/div/div/input").send_keys(domain_name)
            smtp_address_list = smtp_addr.split(",")
            for smtp_val in smtp_address_list:
                self.__admin_console.fill_form_by_id("newSMTP", smtp_val)
                self.__driver.find_element_by_xpath(
                    "//button[@class='btn btn-primary']").click()
                self.__admin_console.wait_for_completion()
            self.__admin_console.submit_form()
            self.__admin_console.check_error_message()

    @WebAction()
    def edit_redirect_rule(self, rule_dict):
        """
        Edits the given list of smtp values in redirect rule
        Args   (dict):redirect rule to edit
                    {"domain":"smtp_addr1,smtp_addr2"}
        Returns    :None
        """
        displayed_rule = self.fetch_redirect_rule()
        if rule_dict:
            if displayed_rule == rule_dict:
                self.log.info("Displayed redirect rule is same as given")
            elif not displayed_rule.keys():
                self.add_redirect_rule(rule_dict)
            else:
                for domain_name, smtp_address in rule_dict.items():
                    div_elem = self.__driver.find_element_by_xpath(
                        "//div[@class='ui-grid-row ng-scope']")
                    domain_elem = div_elem.find_element_by_xpath(
                        ".//div/div/span[@class='crop ng-scope']/span")
                    displayed_domain = domain_elem.text
                    if displayed_domain == '[ Commcell ]':
                        displayed_domain = 'Commcell'
                    if displayed_domain == domain_name:
                        svg_elem = div_elem.find_element_by_xpath(".//div[3]/div/a/span")
                        svg_elem.click()
                        div_elem.find_element_by_xpath(
                            ".//div[3]//ul[@class='dropdown-menu']/li[1]/a").click()
                        self.__admin_console.wait_for_completion()
                        delete_links = self.__driver.find_elements_by_xpath(
                            "//div[@class='btn-group ng-scope dropdown']/span/a[2]")
                        for delete_link in delete_links:
                            delete_links[0].click()
                            time.sleep(2)
                        smtp_addrress_list = smtp_address.split(",")
                        for smtp_val in smtp_addrress_list:
                            self.__admin_console.fill_form_by_id("newSMTP", smtp_val)
                            self.__driver.find_element_by_xpath(
                                "//button[@class='btn btn-primary']").click()
                        self.__admin_console.submit_form()
                        self.__admin_console.check_error_message()
                    else:
                        self.delete_redirect_rule()
                        self.add_redirect_rule(rule_dict)
        else:
            self.delete_redirect_rule()

    @WebAction()
    def delete_redirect_rule(self):
        """
        Deletes the redirect rule
        Args   :None
        Returns    :None
        """
        displayed_rule = self.fetch_redirect_rule()
        for key, value in displayed_rule.items():
            div_elem = self.__driver.find_element_by_xpath(
                "//div[@class='ui-grid-row ng-scope']")
            svg_elem = div_elem.find_element_by_xpath(".//div[3]/div/a/span")
            svg_elem.click()
            div_elem.find_element_by_xpath(
                ".//div[3]//ul[@class='dropdown-menu']/li[2]/a").click()
            self.__admin_console.wait_for_completion()
            self.__admin_console.click_button('Yes')
            self.__admin_console.check_error_message()

    @WebAction()
    def fetch_redirect_rule(self):
        """
        Fetches the list of values in redirect rule
        Args   :None
        Returns    (dict):redirect rule set on the SAML app
        """
        redirect_rule = {}
        div_elements = self.__driver.find_elements_by_xpath(
            "//div[@class='ui-grid-row ng-scope']")
        for div_elem in div_elements:
            count = 0
            smtp_address = " "
            domain_elem = div_elem.find_element_by_xpath(
                ".//div/div/span[@class='crop ng-scope']/span")
            domain_name = domain_elem.text
            if domain_name == '[ Commcell ]':
                domain_name = 'Commcell'
            label_elements = div_elem.find_elements_by_xpath(
                ".//div[2]/cv-fancy-tooltip/span[1]/span/label")
            for label_elem in label_elements:
                smtp = label_elem.text
                if 0 < count <= len(label_elements):
                    smtp_address = smtp_address + "," + smtp
                else:
                    smtp_address = smtp
                count += 1
            redirect_rule[domain_name] = smtp_address
        return redirect_rule

    @WebAction()
    def __enter_attribute_value(self, value):
        """
        Method to enter attribute value for attribute mapping.

        Args:
            value (basestring)      -- the value to be filled in the element
        """
        element = self.__driver.find_element_by_name("saml-attribute-input")
        element.clear()
        element.send_keys(value)

    @WebAction()
    def add_attribute_mappings(self, mapping_dict, from_edit=False):
        """
        Adds the given set of values in attribute mapping
        Args
        mapping_dict    (dict): attribute mappings to add.
                                {"saml_attr1":"user_attr1",
                                 "saml_attr2":"user_attr2"}
        from_edit    (bool):True when called from edit attribute
                            mappings
        Returns    :None
        """
        if mapping_dict:
            if not from_edit:
                try:
                    attribute_elem = self.__driver.find_element_by_xpath(
                        "//a[@data-ng-click='tc.editAttributeMappings()']")
                    attribute_elem.click()
                    time.sleep(8)
                except BaseException:
                    raise CVWebAutomationException("There is no option to add attribute mapping")
            for saml_attr, user_attr in mapping_dict.items():
                self.__admin_console.select_hyperlink(self.__admin_console.props['label.addMappings'])
                self.__enter_attribute_value(saml_attr)
                self.__admin_console.wait_for_completion()

                self.__driver.find_element_by_xpath("//button[@id='dropdown-toggle']").click()
                drop_down_elems = self.__driver.find_elements_by_xpath(
                    "//div[@class='cv-dropdown-item ng-binding ng-scope']")
                found = False
                for drop_down_elem in drop_down_elems:
                    if drop_down_elem.text == user_attr:
                        drop_down_elem.click()
                        found = True
                        break
                if not found:
                    self.__driver.find_element_by_xpath(
                        "//input[@data-ng-blur='validateInputField()']").clear()
                    self.__driver.find_element_by_xpath(
                        "//input[@data-ng-blur='validateInputField()']").send_keys(user_attr)
                self.__driver.find_element_by_xpath(
                    "//i[@class='glyphicon glyphicon-ok']").click()
            if not from_edit:
                self.__admin_console.submit_form()
                self.__admin_console.check_error_message()
        else:
            raise CVWebAutomationException("Empty values in add attribute mapping dictionary")

    @WebAction()
    def edit_attribute_mappings(self, mapping_dict):
        """
        Edits the given set of values in attribute mapping
        Args
        mapping_dict    (dict): attribute mappings to edit.
                                {"saml_attr1":"user_attr1",
                                 "saml_attr2":"user_attr2"}
        Returns    :None
        """
        displayed_attributes = self.fetch_attribute_mappings()
        if displayed_attributes == mapping_dict:
            self.log.info("No need to edit the mappings")
        else:
            attribute_elem = self.__driver.find_element_by_xpath(
                "//a[@data-ng-click='tc.editAttributeMappings()']")
            attribute_elem.click()
            self.__admin_console.wait_for_completion()
            matching_dict1, to_be_del_mappings = self.__comparedicts(
                displayed_attributes, mapping_dict)
            to_be_edited_mappings, to_be_added_mappings = self.__comparedicts(
                mapping_dict, displayed_attributes)
            if to_be_del_mappings.items():
                self.delete_attribute_mapping(to_be_del_mappings, from_edit=True)
            if to_be_added_mappings.items():
                self.add_attribute_mappings(to_be_added_mappings, from_edit=True)
            if to_be_edited_mappings.items():
                for saml_attr, user_attr in to_be_edited_mappings.items():
                    div_elements = self.__driver.find_elements_by_xpath(
                        "//div[@class='ui-grid-row ng-scope']")
                    for div_elem in div_elements:
                        saml_text = div_elem.find_element_by_xpath(
                            "./div/div[1]/span/label").text
                        if saml_text == saml_attr:
                            div_elem.find_element_by_xpath(".//div[3]/span[1]/\
                                i[@class='glyphicon glyphicon-pencil']").click()
                            match_found = True
                            self.__admin_console.fill_form_by_id('saml-attribute-input', saml_attr)
                            self.__driver.find_element_by_xpath(
                                "//button[@id='dropdown-toggle']").click()
                            drop_down_elements = self.__driver.find_elements_by_xpath(
                                "//div[@class='cv-dropdown-item ng-binding ng-scope']")
                            found = False
                            for drop_down_elem in drop_down_elements:
                                if drop_down_elem.text == user_attr:
                                    drop_down_elem.click()
                                    found = True
                                    break
                            if not found:
                                self.__driver.find_element_by_xpath(
                                    "//div[@class='input-group']/input").clear()
                                self.__driver.find_element_by_xpath("//div\
                                    [@class='input-group']/input").send_keys(user_attr)
                            self.__driver.find_element_by_xpath(
                                "//i[@class='glyphicon glyphicon-ok']").click()
                            if match_found:
                                break
        self.__admin_console.submit_form()
        self.__admin_console.check_error_message()

    @WebAction()
    def delete_attribute_mapping(self, mapping_dict, from_edit=False):
        """
        Deletes the given set of values in attribute mapping
        Args
        mapping_dict    (dict): attribute mappings to delete.
                                {"saml_attr1":"user_attr1",
                                 "saml_attr2":"user_attr2"}
        from_edit    (bool):True when called from edit attribute
                            mappings
        Returns    :None
        """
        if not from_edit:
            try:
                attribute_elem = self.__driver.find_element_by_xpath(
                    "//a[@data-ng-click='tc.editAttributeMappings()']")
                attribute_elem.click()
                self.__admin_console.wait_for_completion()

            except BaseException:
                raise CVWebAutomationException("There is no option to delete attribute mapping")

        for saml_attr, user_attr in mapping_dict.items():
            div_elements = self.__driver.find_elements_by_xpath(
                "//div[@class='ui-grid-row ng-scope']")
            match_found = False
            for div_elem in div_elements:
                if ((div_elem.find_element_by_xpath("./div/div[1]/span/label").text
                     == saml_attr)
                        and
                        (div_elem.find_element_by_xpath("./div/div[2]/span/label").text
                         == user_attr)):
                    match_found = True
                    div_elem.find_element_by_xpath(".//div[3]/\
                        span[2]/i[@class='glyphicon glyphicon-trash']").click()
                    time.sleep(2)
                    break
            if not match_found:
                raise CVWebAutomationException("Attributes given do not match with any")
        if not from_edit:
            self.__admin_console.submit_form()
            self.__admin_console.check_error_message()

    @WebAction()
    def fetch_attribute_mappings(self):
        """
        Fetches the set of values in attribute mapping
        Args   :None
        Returns    (dict):dict of attribute mappings set on SAML app
        """
        attribute_mappings = {}
        li_elements = self.__driver.find_elements_by_xpath("//li[@data-ng-repeat=\
            'mappings in tc.thirdPartyApp.attributeMappings']")
        for li_elem in li_elements:
            saml_attr = li_elem.find_element_by_xpath("./span[1]").text
            user_attr = li_elem.find_element_by_xpath("./span[2]").text
            attribute_mappings[saml_attr] = user_attr

        return attribute_mappings

    @PageService()
    def saml_app_info(self):
        """
        Displays all the information about the SAML app
        returns
        SAML_info    (dict)  -- all info about the SAML app
        """
        saml_app_name = self.get_app_name()
        auto_user_create_flag = self.get_auto_create_user_flag()
        default_user_group = self.get_default_user_group()
        company_name = self.get_company_name()
        sp_entity_id = self.copy_sp_entity_id()
        sso_url = self.copy_sso_url()
        associations = self.fetch_associations()
        attribute_mappings = self.fetch_attribute_mappings()
        redirect_rules = self.fetch_redirect_rule()
        saml_app_info = {"AppName": saml_app_name,
                         "Auto user create flag": auto_user_create_flag,
                         "Default user group": default_user_group,
                         "Company name": company_name,
                         "SP entity ID": sp_entity_id,
                         "SSO URL": sso_url,
                         "Associations": associations,
                         "Redirect rule": redirect_rules,
                         "Attribute mapping": attribute_mappings}
        return saml_app_info

    def __comparedicts(self, dict1, dict2):
        """
        Compares the 2 given dicts and returns common and uniques keys
        :param dict1:
        :param dict2:
        returns: matching_dictionary    (dict)
                unique_dictionary        (dict)
        """
        matching_dict = {}
        unique_dict = {}
        for key, value in dict1.items():
            if key in dict2:
                matching_dict.update({key: value})
            else:
                unique_dict.update({key: value})
        return matching_dict, unique_dict
