# -*- coding: utf-8 -*-s

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
This module provides functions or operations that can be performed on the Domains Details page

__return_elements_from_element_obj()    -- Returns elements with xpath from element object
modify_domain()                         -- edits the properties of a domain
extract_domain_info()                   -- returns details of the domain displayed on
                                            the Domain Details page
"""

from Web.Common.page_object import PageService, WebAction
from Web.AdminConsole.AdminConsolePages.identity_servers import Domains


class DomainDetails:
    """
        This class provides the function or operations that can be performed
        on the Domains page on the AdminConsole
    """

    def __init__(self, adminpage_obj):

        self.__driver = adminpage_obj.driver
        self.__adminpage_obj = adminpage_obj
        self.__domains_obj = Domains(self.__adminpage_obj)

    @WebAction()
    def __return_elements_from_element_obj(self, elem, xpath):
        """Returns elements with xpath from element object"""

        return elem.find_elements_by_xpath(xpath)

    @PageService()
    def modify_domain(self, domain_dict):
        """
        Modifies the domain information

        Args:

            domain_dict (dict)       : Dict of values to be modified.
                                       Accepted keys - username, password, domainName.


        Returns:
            None
        """

        self.__adminpage_obj.select_hyperlink("Edit")

        for key, value in domain_dict.items():

            if key == 'proxy_client_enabled':
                if value:
                    self.__domains_obj.enable_proxy_client()
                    if domain_dict.get('proxy_client_value'):
                        self.__adminpage_obj.cv_single_select('Client name',
                                                              domain_dict['proxy_client_value'])
                else:
                    self.__domains_obj.disable_proxy_client()

            elif key == 'proxy_client_value':
                pass

            else:
                self.__adminpage_obj.fill_form_by_id(key, value)

        self.__adminpage_obj.submit_form()
        self.__adminpage_obj.check_error_message()

    @PageService()
    def extract_domain_info(self):
        """Returns dict of domain details of selected domain on the Domains Details page"""

        details_page_elems = self.__domains_obj.return_elements(
                                                            "//*[@id='tileContent_General']/ul/li")
        domain_info = {}

        for elem in details_page_elems:
            span = self.__return_elements_from_element_obj(elem,
                                                    ".//span[contains(@class,'pageDetailColumn')]")
            key = span[0].text
            value = span[1].text
            domain_info[key] = value
        return domain_info
