# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

""" Operations on admin console dashboard page

get_dash_pane_titles           -- Gets all the pane's titles present in dashboard page

access_details_page            -- Access details page of specified pane

get_page_title                 -- Gets title of the current page
"""

from enum import Enum
from Web.Common.page_object import (WebAction, PageService)


class OverviewDashboard(Enum):
    """Panes used in Overview Dashboard page"""
    pane_environment = "Environment"
    pane_needs_attentions = "Needs attention"
    pane_sla = "SLA"
    pane_jobs_in_last_24_hours = "Jobs in the last 24 hours"
    pane_health = "Health"
    pane_disk_space = "Disk space"
    pane_storage_usage = "Storage"
    pane_storage_data_retention = "Storage - data retention"
    pane_current_capacity_usage = "Current capacity"
    pane_last_week_backup_job_summary = "Last week backup job summary"
    pane_top_5_largest_servers = "Top 5 largest servers"

    # Entities present in different panes
    entity_file_servers = "FILE SERVERS"
    entity_servers = "SERVERS"
    entity_vms = "VMs"
    entity_laptops = "LAPTOPS"
    entity_users = "USERS"
    entity_infrastructures = "INFRASTRUCTURE MACHINES"
    entity_jobs = "JOBS"
    entity_running = "RUNNING"
    entity_success = "SUCCESS"
    entity_failed = "FAILED"
    entity_events = "EVENTS"
    entity_disk_library = "DISK LIBRARY"
    entity_space_savings = "SPACE SAVINGS"


class ApplianceDashboard(Enum):
    """Panes used in Overview Dashboard page"""
    pane_environment = "Environment"
    pane_needs_attentions = "Needs attention"
    pane_system = "System"
    pane_hardware = "Hardware Component"
    pane_disk_space = "Disk space"
    pane_sla = "SLA"
    pane_jobs_in_last_24_hours = "Jobs in the last 24 hours"
    pane_current_capacity_usage = "Top 5 largest servers"

    # Entities present in different panes
    entity_appliances = "APPLIANCES"
    entity_servers = "SERVERS"
    entity_vms = "VMs"
    entity_critical_alerts = "CRITICAL ALERTS"
    entity_infrastructures = "INFRASTRUCTURES"
    entity_jobs = "JOBS"
    entity_running = "RUNNING"
    entity_success = "SUCCESS"
    entity_failed = "FAILED"
    entity_critical_events = "CRITICAL EVENTS"


class VirtualizationDashboard(Enum):
    """Panes used in Overview Dashboard page"""
    pane_hypervisors = "Hypervisors"
    pane_vms = "VMs"
    pane_sla = "CommCell SLA"
    pane_jobs_in_last_24_hours = "Jobs in the last 24 hours"
    pane_last_week_backup_job_summary = "Last week backup job summary"
    pane_largest_hypervisors = "Largest hypervisors"
    # Entities present in different panes
    entity_protected = "PROTECTED"
    entity_not_protected = "NOT PROTECTED"
    entity_backed_up_with_error = "BACKED UP WITH ERROR"
    entity_running = "RUNNING"
    entity_success = "SUCCESS"
    entity_failed = "FAILED"
    entity_events = "EVENTS"


class OrchestrationDashboard(Enum):
    """Panes used in Orchestration Dashboard page"""
    header_databases = "Databases"
    header_file_servers = "File servers"
    header_applications = "Applications"
    header_vms = "VMs"
    # Panes present under different headers
    pane_overview = "Overview"
    pane_last_month_stats = "Last month stats"
    pane_replication_groups = "Replication groups"
    # Entities present in different panes
    entity_servers = "SERVERS"
    entity_clones = "CLONES"
    entity_cloud_migration = "CLOUD MIGRATIONS"
    entity_failover_runs = "FAILOVER RUNS"
    entity_file_servers = "FILE SERVERS"
    entity_live_mounts = "LIVE MOUNTS"
    entity_vms = "VMs"


class Dashboard:

    def __init__(self, admin_console):
        """
        Args:
            admin_console (AdminConsole): adminconsole base object
        """
        self.__admin_console = admin_console
        self.driver = admin_console.driver

    """Operations on dashboard page"""
    @WebAction()
    def _read_dash_pane_titles(self, header=None):
        """Get panels titles present in dashboard page
        Args:
            header  (String) :   Header name for different dashboard sections
        Returns:
            List with all pane titles
        """
        if header:
            xpath = f"//span[contains(.,'{header}') and contains(@data-ng-bind,"\
                    f"'orchestrationPaneCtrl.paneData."\
                    f"title.text')]/../../..//*[contains(@class, 'dash-pane-header-title')]"
        else:
            xpath = "//*[contains(@class, 'dash-pane-header-title')]"
        return [each_panel.text for each_panel in self.driver.find_elements_by_xpath(xpath)]

    @WebAction()
    def _read_headers(self):
        """Get header names"""
        xpath = "//*[contains(@data-ng-bind, 'orchestrationPaneCtrl.paneData.title.text')]"
        return [each_header.text for each_header in self.driver.find_elements_by_xpath(xpath)]

    @WebAction()
    def _get_page_title(self):
        """Get page title"""
        title_obj = self.driver.find_elements_by_xpath("//*[contains(@class, 'page-title')]")
        if not title_obj:
            title_obj = self.driver.find_elements_by_xpath("//h2")
        if not title_obj:
            self.__admin_console.switch_to_react_frame()
            title_obj = self.driver.find_elements_by_class_name("grid-title")

        return title_obj[0].text

    @WebAction()
    def _access_details_page(self, pane_name=None, entity_name=None, header_name=None):
        """
        Click on details page specified pane
        """
        if header_name:
            if entity_name:
                xpath = f"//span[contains(.,'{header_name}') and contains(@data-ng-bind," \
                        f"'orchestrationPaneCtrl.paneData.title.text')]/../../.." \
                        f"//cv-reports-pane-header/span/*[text() = '{pane_name}']/../../../.."\
                        f"//span[text() = '{entity_name}']"
            elif pane_name:
                xpath = f"//span[contains(.,'{header_name}') and contains(@data-ng-bind," \
                        f"'orchestrationPaneCtrl.paneData.title.text')]/../../.." \
                        f"//cv-reports-pane-header/span/*[text() ='{pane_name}']"
            else:
                xpath = f"//span[contains(text(),'{header_name}') and contains(@data-ng-bind,"\
                        f"'orchestrationPaneCtrl.paneData.title.text')]"
        else:
            if entity_name:
                xpath = "//cv-reports-pane-header/span/*[text() = '{0}']/../../.." \
                        "//span[text() = '{1}']"\
                        .format(pane_name, entity_name)
            else:
                xpath = "//cv-reports-pane-header/span/*[text() = '{0}']".format(pane_name)
        self.driver.find_element_by_xpath(xpath).click()

    @WebAction()
    def _access_dashboard(self, dashboard_type):
        """
        Click on dashboard header and select given dashboard type
        Args:
            dashboard_type  (String) :  name of dashboard to be accessed
        """
        self.driver.find_element_by_xpath(
            f"//button[contains(@class,'btn btn-link dropdown-toggle')]").click()
        self.driver.find_element_by_xpath(
            f"//a[contains(text(),'{dashboard_type}')]").click()
        self.__admin_console.wait_for_completion()

    @PageService()
    def get_dash_pane_titles(self, dashboard_type=None):
        """Get dash pane tiles present in dashboard page"""
        return self._read_dash_pane_titles(dashboard_type)

    @PageService()
    def access_details_page(self, pane_name=None, entity_name=None, header_name=None):
        """
        Access details page of specified pane
        Args:
            pane_name        (String) : Name of the pane on which details page to be accessed
            entity_name      (String) : Name of the entity within pane
            header_name      (String) : Name of the header containing the pane
        """
        self._access_details_page(pane_name, entity_name, header_name)
        self.__admin_console.wait_for_completion()

    @PageService()
    def get_page_title(self):
        """Get page title"""
        return self._get_page_title()

    @PageService()
    def navigate_to_given_dashboard(self, dashboard_type):
        """
        Navigate to respective dashboard page as given
        Args:
            dashboard_type  (String) :  name of dashboard to be accessed
        """
        self._access_dashboard(dashboard_type)

    @PageService()
    def get_header_and_dash_pane_titles(self):
        """
        Get headers and dash pane tiles present in dashboard page
        Returns:
            Dictionary with all header and pane titles
                Eg.-{header1 : [panes], header2 : [panes]}
        """
        orchestration_dict = {}
        headers = self._read_headers()
        for header in headers:
            panes = self._read_dash_pane_titles(header)
            orchestration_dict['header'] = panes
        return orchestration_dict
