# -*- coding: utf-8 -*-s

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------
"""
This module provides the function or operations that can be performed on the
CredentialManager page on the AdminConsole

Class:
    CredentialManager()

Functions:
__expand_security_panel()                   -- opens the security panel to make the User/Group
                                                drop down accessible
__select_option_from_dropdown()             -- select option from drop down
__confirm_yes()                             -- clicks the yes button on the confirm delete pop up
__click_action_button()                     -- clicks the action button and selects
                                                the given action
__get_users_list()                          -- gets the list of credential names displayed
                                                on the credential manager page
__no_credentials_present()                  -- Checks if no credentials are present on
                                                the Credential Manager page
__click_element_by_xpath()                  -- Click element with the given xpath
__get_element_value_by_id()                 -- Get value of element with the given element id
__get_element_text_by_xpath()               -- Get text of element with the given xpath
__get_account_type()                        -- Get account type of credential as displayed on
                                                edit credential panel
__get_owner()                               -- Get owner of credential as displayed on
                                                edit credential panel
__get_user_group_drop_down_elems()          -- Gets elements objects from the User/Group drop down
click_ok_subclient_backup_pane()            -- Clicks OK button on subclient backup pane
get_job_url_from_notification()             -- Gets the job url from the growl notification
confirm_backupset_delete()                  -- Confirm backupset delete by typing "DELETE" in the
                                                confirmation dialog box
check_if_credential_is_available()          -- Checks if the credential is available to the account

add_credential()                            -- adds a new credential on the credential manager page
edit_credential()                           -- edits the credential
action_remove_credential()                  -- Deletes the credential
extract_edit_pane_displayed_details()       -- Displays details on the edit pane
"""
import time

from selenium.webdriver.support.ui import WebDriverWait
from selenium.webdriver.support import expected_conditions as ec
from selenium.webdriver.common.by import By

from Web.AdminConsole.Components.table import Table
from Web.Common.page_object import WebAction, PageService


class UserOrGroupSelection:
    """ Class to handle drop down related operations """

    def __init__(self, admin_console):
        self.__admin_console = admin_console
        self.__driver = admin_console.driver

    @WebAction()
    def __get_drop_down_list(self):
        """ Method to get drop down list """
        drop_down_list = self.__driver.find_elements_by_xpath("//isteven-multi-select")
        return drop_down_list

    @WebAction()
    def __expand_drop_down(self, drop_down):
        """ Method to expand drop down """
        drop_down.click()

    @WebAction()
    def __select_none(self, drop_down):
        """ Method to click 'Select none' icon and de-select all values """
        if self.__admin_console.is_element_present(
                ".//div[@class='helperContainer ng-scope']/div[1]/button[2]", drop_down):
            drop_down.find_element_by_xpath(
                ".//div[@class='helperContainer ng-scope']/div[1]/button[2]").click()

    @WebAction()
    def __select_all(self, drop_down):
        """ Method to click 'Select all' icon and select all values from drop down """
        if self.__admin_console.is_element_present(
                ".//div[@class='helperContainer ng-scope']/div[1]/button[1]", drop_down):
            drop_down.find_element_by_xpath(
                ".//div[@class='helperContainer ng-scope']/div[1]/button[1]").click()

    @WebAction()
    def __search_entity(self, drop_down, entity):
        """ Method to search for an entity """
        if self.__admin_console.is_element_present(
                ".//div[@class='line-search']", drop_down):
            drop_down.find_element_by_xpath(
                ".//div[@class='line-search']/input").send_keys(entity)

    @WebAction()
    def __select_entity(self, drop_down, entity):
        """ Method to click on entity to be selected"""
        if not drop_down.find_element_by_xpath(
                f".//div[@class='checkBoxContainer']//*[text()='{entity}']"
                "/ancestor::label/input").get_attribute("checked"):
            elem = drop_down.find_element_by_xpath(
                f".//div[@class='checkBoxContainer']//*[./text()='{entity}']")
            self.__driver.execute_script("arguments[0].scrollIntoView();", elem)
            elem.click()

    @WebAction()
    def __clear_search_bar(self, drop_down):
        """ Method to clear search bar """
        if self.__admin_console.is_element_present(
                ".//div[@class='line-search']/button", drop_down):
            if drop_down.find_element_by_xpath(
                    ".//div[@class='line-search']/button").is_displayed():
                drop_down.find_element_by_xpath(
                    ".//div[@class='line-search']/button").click()

    @WebAction()
    def __collapse_drop_down(self, drop_down):
        """ Method to collapse drop down """
        if 'show' in drop_down.find_element_by_xpath(
                ".//div[contains(@class,'checkboxLayer')]").get_attribute('class'):
            if self.__admin_console.is_element_present(
                    ".//div[@class='line-search']/following-sibling::div", drop_down):
                drop_down.find_element_by_xpath(
                    ".//div[@class='line-search']/following-sibling::div").click()
            else:
                drop_down.click()

    @WebAction()
    def __fill_values(self, entity, entity_id):
        """ Inputs the entity value """
        if entity_id == 1:
            elem = self.__driver.find_element_by_xpath(
                f"//div[@search-options='searchConfigOwner']//following-sibling::input[@name='searchComponent']")
        else:
            elem = self.__driver.find_element_by_xpath(
                f"//div[@search-options='searchConfig']//following-sibling::input[@name='searchComponent']")
        elem.clear()
        elem.click()
        elem.send_keys(entity)
        time.sleep(2)

    @WebAction()
    def __select_value(self, entity, entity_id):
        """ Selects the entity """
        if entity_id == 1:
            xp = f"//div[@search-options='searchConfigOwner']//h5[text()='{entity}']"
            if self.__admin_console.check_if_entity_exists("xpath", xp):
                self.__driver.find_element_by_xpath(xp).click()
        elif entity_id == 2:
            xp = f"//div[@search-options='searchConfig']//h5[text()='{entity}']"
            if self.__admin_console.check_if_entity_exists("xpath", xp):
                self.__driver.find_element_by_xpath(xp).click()
        time.sleep(2)
        xp = "//span[text()='No results found']"
        if self.__admin_console.check_if_entity_exists("xpath", xp):
            self.__driver.find_element_by_xpath(
                f"//div[@search-options='searchConfig']//following-sibling::input[@name='searchComponent']").clear()
        else:
            elem = self.__driver.find_element_by_xpath(
                "//div[@search-options='searchConfig']/ul[@ng-show='ctrl.isResultPanelVisible']")
            if not elem.get_attribute('class').__contains__('ng-hide'):
                self.__driver.find_element_by_xpath(
                    f"//div[@search-options='searchConfig']//following-sibling::input[@name='searchComponent']").click()

    @WebAction()
    def fill_security_entites(self, entity_id, entities):
        """ Method to select entity value """
        if not self.__driver.find_element_by_xpath("//div[@role='tablist']//a").get_attribute('aria-expanded'):
            self.__driver.find_element_by_xpath("//div[@role='tablist']//a").click()
        for entity in entities:
            self.__fill_values(entity, entity_id)
            self.__select_value(entity, entity_id)

    @PageService()
    def select_drop_down_values(self, index, values, select_all=None):
        """
        Method to select values from drop down

        Args:
            index (int) : order of drop down in the sequence of display on page (starting from 0)

            values (list) : values to be selected from drop down

            select_all (bool) : boolean value to select all values from drop down
        """

        drop_down_list = self.__get_drop_down_list()
        self.__expand_drop_down(drop_down_list[index])
        if select_all:
            self.__select_all(drop_down_list[index])
            self.__collapse_drop_down(drop_down_list[index])
        else:
            self.__select_none(drop_down_list[index])
            for value in values:
                self.__search_entity(drop_down_list[index], value)
                self.__select_entity(drop_down_list[index], value)
                self.__clear_search_bar(drop_down_list[index])
            self.__collapse_drop_down(drop_down_list[index])


class CredentialManager:
    """
    This class provides the function or operations that can be performed on the CredentialManager
     page on the AdminConsole
    """

    def __init__(self, admin_console):

        self.__driver = admin_console.driver
        self.__admin_console = admin_console
        self.__navigator = self.__admin_console.navigator
        self.__table = Table(admin_console)
        self.__user_or_group_selection = UserOrGroupSelection(admin_console)

    @WebAction()
    def __expand_security_panel(self):
        """
        Expand security panel to select more user groups for the credential
        Present on Add and Edit panels
        """
        self.__driver.find_element_by_xpath("//span[contains(text(),'Security')]").click()

    @WebAction()
    def __select_option_from_dropdown(self, owner):
        """
        Method to select options from the Account type/Owner/User group dropdown
        """
        owner_xpath = f"//div[contains(@title, '{owner}')]"
        self.__driver.find_element_by_xpath(owner_xpath).click()

    @WebAction()
    def __return_credential_name_and_type(self, elem):
        """Get text from an element object"""

        credential_name = elem.find_element_by_xpath("./div/div[1]/div").text
        credential_type = elem.find_element_by_xpath("./div/div[2]/div").text
        return credential_name, credential_type

    @PageService()
    def __get_credentials_list(self):
        """
        Method to get credentials list
        """
        credential_list = []
        elems = self.__driver.find_elements_by_xpath("//div[@class='ui-grid-canvas']/div")
        no_data_xpath = "//*[@id='wrapper']//h3"

        if self.__admin_console.check_if_entity_exists("xpath", no_data_xpath):
            self.__admin_console.log.info(self.__get_element_text_by_xpath(no_data_xpath))
            self.__admin_console.log.info("No credentials are present")

        else:
            for elem in elems:
                cred_dict = {}
                credential_name_type = self.__return_credential_name_and_type(elem)
                credential_name = credential_name_type[0]
                credential_type = credential_name_type[1]
                cred_dict[credential_name] = credential_type
                credential_list.append(cred_dict)

        return credential_list

    @WebAction()
    def __click_element_by_xpath(self, xpath):
        """Clicks on element with the given xpath"""
        self.__driver.find_element_by_xpath(xpath).click()

    @WebAction()
    def __get_element_value_by_id(self, element_id):
        """Get element value by id"""
        return self.__driver.find_element_by_id(element_id).get_attribute("value")

    @WebAction()
    def __get_element_text_by_xpath(self, element_xpath):
        """Get element text by xpath"""
        return self.__driver.find_element_by_xpath(element_xpath).text

    @WebAction()
    def __get_account_type(self):
        """Get account type selected in account dropdown"""
        account_type_xpath = "//*[@id='recordTypeList']/div"
        return self.__driver.find_element_by_xpath(account_type_xpath).text

    @WebAction()
    def __get_owner(self):
        """Get selected owner from owner dropdown"""
        xp = "//div[@search-options='searchConfigOwner']//following-sibling::input[@name='searchComponent']"
        elem = self.__driver.find_element_by_xpath(xp)
        return elem.get_attribute("value")

    @WebAction()
    def __get_user_group_drop_down_elems(self):
        """Get selected user/groups from user/group drop down"""
        user_or_group_drop_down_xpath = "//span[contains(@class,'multi-select-search-selected')]"
        return self.__driver.find_elements_by_xpath(user_or_group_drop_down_xpath)

    @WebAction()
    def click_ok_subclient_backup_pane(self):
        """Click OK  button on Run Backup panel"""
        ok_button_xpath = "//button[text()='OK'][2]"
        self.__driver.find_element_by_xpath(ok_button_xpath).click()

    @WebAction()
    def get_job_url_from_notification(self):
        """Get URL of job from notification"""
        notification_job_hyperlink_xpath = "//*[@id='wrapper']//div[@class='growl']//a"
        WebDriverWait(self.__driver, 120).until(ec.presence_of_element_located(
            (By.XPATH, "//*[@id='wrapper']//div[@class='growl']//a")))
        return self.__driver.find_element_by_xpath(
                                            notification_job_hyperlink_xpath).get_attribute('href')

    @WebAction()
    def confirm_backupset_delete(self):
        """Confirm backup set delete by typing 'DELETE'"""
        self.__driver.find_element_by_xpath("//input[@type='text']").send_keys("DELETE")

    @PageService()
    def add_credential(self,
                       account_type,
                       credential_name,
                       username,
                       password,
                       owner,
                       description=None,
                       user_or_group=None):
        """
        Adds a new credential to the commcell

        Args:
            account_type (basestring) : the account type of the credential

            credential_name (basestring) : the name of the domain

            username (basestring)        : the username of the domain

            password (basestring)        : the password of the domain

            owner (basestring)           : owner of credential

            description (basestring)     : description of the credential

            user_or_group (list)         : list of user/groups to be associated with the credential

        """
        self.__admin_console.select_hyperlink("Add")
        self.__admin_console.cv_single_select("Account type", account_type)
        self.__admin_console.fill_form_by_id("credentialName", credential_name)
        self.__admin_console.fill_form_by_id("userName", username)
        self.__admin_console.fill_form_by_id("password", password)

        if self.__admin_console.check_if_entity_exists("id", "confirmPassword"):
            self.__admin_console.fill_form_by_id("confirmPassword", password)

        if description:
            self.__admin_console.fill_form_by_id("description", description)

        if owner:
            self.__user_or_group_selection.fill_security_entites(1, [owner])

        if user_or_group:
            self.__user_or_group_selection.fill_security_entites(2, user_or_group)

        self.__admin_console.submit_form()
        self.__admin_console.check_error_message()

    @PageService()
    def edit_credential(self,
                        credential_name,
                        new_credential_name=None,
                        username=None,
                        password=None,
                        owner=None,
                        description=None,
                        user_or_group=None):

        """
        Edits the credential with the given name

        Args:
            credential_name (string)     : the name of the credential which has to
                                                   be edited

            new_credential_name (string) : the new credential name,
                                                  pass None to leave it unchanged

            username (string)            : credential username

            password (string)            : credential password, pass None to leave
                                                  credential password unchanged

            owner (basestring)           : owner of credential

            description (basestring)     : description of the credential

            user_or_group (list)         : list of user/groups to be associated with the credential

        Returns:
            None
        """

        self.__table.access_link(credential_name)

        if new_credential_name is not None:
            self.__admin_console.fill_form_by_id("credentialName", new_credential_name)

        if username is not None:
            self.__admin_console.fill_form_by_id("userName", username)

        if password is not None:
            self.__admin_console.fill_form_by_id("password", password)

        if self.__admin_console.check_if_entity_exists("id", "confirmPassword"):
            self.__admin_console.fill_form_by_id("confirmPassword", password)

        if description is not None:
            self.__admin_console.fill_form_by_id("description", description)

        self.__expand_security_panel()

        if owner is not None:
            self.__user_or_group_selection.fill_security_entites(1, [owner])

        if user_or_group is not None:
            self.__user_or_group_selection.fill_security_entites(2, user_or_group)

        self.__admin_console.submit_form()
        self.__admin_console.check_error_message()

    @PageService()
    def action_remove_credential(self, credential_name):
        """
        Deletes the credential with the given name

        Args:
            credential_name (basestring)    : Name of the credential which has to
                                                be deleted

        Returns:
            None
        """

        self.__table.access_action_item(credential_name, "Delete")
        self.__admin_console.click_button('Yes')

    @PageService()
    def extract_edit_pane_displayed_details(self, credential_name):
        """
        Method to extract and returns details displayed on the edit pane

        Args:
            credential_name(basestring) : Name of the credential whose details are to be fetched

        Returns:
            displayed_details_dict(dict) : Dictionary of the details displayed
        """

        displayed_details_dict = {}
        self.__table.access_link(credential_name)

        displayed_details_dict["Account type"] = self.__get_account_type()
        displayed_details_dict["Credential name"] = self.__get_element_value_by_id(
                                                                                  "credentialName")
        displayed_details_dict["User name"] = self.__get_element_value_by_id("userName")
        displayed_details_dict["description"] = self.__get_element_value_by_id("description")

        self.__expand_security_panel()
        self.__admin_console.wait_for_completion()

        displayed_details_dict["Owner"] = self.__get_owner()

        user_group_elems = self.__get_user_group_drop_down_elems()
        user_or_group_value = ""

        for elem in user_group_elems:
            user_or_group_value = user_or_group_value + elem.text[:-1] + ','
        displayed_details_dict["User/Group"] = user_or_group_value[:-1]

        self.__admin_console.click_button("Cancel")
        return displayed_details_dict
