# -*- coding: utf-8 -*-
# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
This module provides the function or operations that can be performed on the
Users page on the AdminConsole

Classes:

    Users()

Functions:

        filter_all_users()      -- Method to apply 'All Users' filter
        filter_laptop_users()   -- Method to apply 'Laptop Users' filter
        add_local_user          -- Method to add local user
        add_external_user       -- Method to add external user
        delete_user()           -- delete the user with the specified username


"""

from selenium.webdriver.common.keys import Keys

from Web.Common.page_object import (WebAction, PageService)
from Web.AdminConsole.Components.table import Table
from Web.AdminConsole.Components.dialog import TransferOwnership
from Web.AdminConsole.Components.panel import DropDown


class Users:
    """
    Class for users page
    """

    def __init__(self, admin_page):
        """
        Method to initiate Users class

        Args:
            admin_page   (Object) :   Admin Page Class object
        """
        self.__admin_console = admin_page
        self.__admin_console._load_properties(self)
        self.__table = Table(admin_page)
        self.__transfer_owner = TransferOwnership(admin_page)
        self.__drop_down = DropDown(admin_page)
        self.__driver = admin_page.driver

    @WebAction()
    def __append_username(self, username):
        """
        Method to enter username for external user

        Args:
            username  (basestring):   the username of the user
        """
        user_name = self.__driver.find_element_by_id("userName")
        user_name.send_keys(Keys.END + username)

    @PageService()
    def filter_all_users(self):
        """
        Method to apply 'All Users' filter
        """
        self.__table.view_by_title("All Users")
        self.__admin_console.wait_for_completion()

    @PageService()
    def filter_laptop_users(self):
        """
        Method to apply 'Laptop Users' filter
        """
        self.__table.view_by_title("All Users")
        self.__admin_console.wait_for_completion()

    def add_local_user(self,
                       email,
                       username=None,
                       name=None,
                       groups=None,
                       system_password=False,
                       password=None,
                       invite_user=False):
        """
        Method to create new local user with the specified details

        Args:

            email               (basestring):   the email of the user

            username            (basestring):   the username of the user

            name                (basestring):   the display name of the user

            groups               (basestring/List) :  one group name or list of user
                                                    groups to attach to

            system_password     (bool):         if the system password needs to be used

            password            (basestring):   the password of the user

            invite_user         (bool):         if the users should be sent an email invite

        Raises:
            Exception:
                if failed to create local user

        """
        self.__admin_console.select_hyperlink(self.__admin_console.props['pageHeader.addUser'])
        self.__admin_console.select_radio('localUser')

        if name:
            self.__admin_console.fill_form_by_id("fullName", name)
        if username:
            self.__admin_console.fill_form_by_id("userName", username)
        self.__admin_console.fill_form_by_id("email", email)

        if not system_password:
            self.__admin_console.fill_form_by_id("password", password)
            self.__admin_console.fill_form_by_id("confirmUserPassword", password)
        else:
            self.__admin_console.checkbox_select("systemGeneratePassword")

        if groups:
            self.__drop_down.select_drop_down_values(0, groups)

        if invite_user:
            self.__admin_console.checkbox_select("inviteUser")
        else:
            self.__admin_console.checkbox_deselect("inviteUser")

        self.__admin_console.submit_form()
        self.__admin_console.check_error_message()

    def add_external_user(self,
                          external_provider,
                          username,
                          email=None,
                          groups=None):
        """
        Method to create new External/AD user with the specified details

        Args:

            external_provider   (basestring):   if external user, then the provider name

            username            (basestring):   the username of the user

            email               (basestring):   the email of the user

            groups               (list) :  one group name or list of user
                                                    groups to attach to

        Raises:
            Exception:
                if failed to create AD/External user
        """

        self.__admin_console.select_hyperlink(self.__admin_console.props['pageHeader.addUser'])
        self.__admin_console.select_radio('externalUser')

        self.__drop_down.select_drop_down_values(0, [external_provider])
        self.__append_username(username)

        if email:
            self.__admin_console.fill_form_by_id("providerEmail", email)

        if groups:
            self.__drop_down.select_drop_down_values(1, groups)

        self.__admin_console.submit_form()
        self.__admin_console.check_error_message()

    @PageService()
    def delete_user(self,
                    user_name,
                    owner_transfer=None):
        """
        Deletes the specified user

        Args:
            user_name       (basestring):   username of the user that needs to be deleted
            owner_transfer  (basestring):   a user or user group to transfer ownership to
        """
        self.__table.access_action_item(user_name, 'Delete')
        self.__transfer_owner.transfer_ownership(owner_transfer)
        self.__admin_console.log.info('User : %s is deleted successfully', user_name)
