# -*- coding: utf-8 -*-
# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
This module provides the function or operations that can be performed on the
User Groups page on the AdminConsole

Class:

    UserGroups()

Functions:

__select_user_group_type       --  Method to select user group based on type given as input

add_user_group()               --  adds a new user to the admin console

open_user_group()              --  opens the user group with the specified name

action_delete_user_group()     --  deletes the user group with the specified name

active_directory_user_groups() --  Selects AD User Groups

all_user_groups()              --  Selects All User Groups

local_user_groups()            --  Selects Local User Groups

list_user_group()              --  displays the complete info of all the user groups

action_add_users()             --  Method to add users to user group
"""
import time

from Web.AdminConsole.Components.table import Table, CVTable
from Web.AdminConsole.Components.dialog import TransferOwnership
from Web.Common.page_object import (WebAction, PageService)


class UserSelection:
    """ Class to support add user action on user group and user group details page """

    def __init__(self, admin_page):
        self.__admin_console = admin_page
        self.__driver = admin_page.driver
        self.__table = CVTable(admin_page)

    @WebAction()
    def __get_user_index(self, user):
        """ Method to get index out of displayed user """
        index = 0
        selected = []
        users_list = self.__driver.find_elements_by_xpath(
            "//div[contains(@class,'ui-grid-render-container-body')]//div[@class='ui-grid-row ng-scope']")
        for username in users_list:
            index += 1
            if user == username.find_element_by_xpath(".//div[1]/span").text:
                selected.append(user)
                break
        return index, selected

    @WebAction()
    def __select_user_checkbox(self, index):
        """ Method to select user by checking corresponding checkbox using index given """
        xp = "//div[contains(@class,'ui-grid-selection-row-header-buttons')]"
        checkbox = self.__driver.find_elements_by_xpath(xp)
        checkbox[index].click()

    @PageService()
    def add_users(self, users):
        """
        Adds users to the given group

        Args:
            users (list)  :  list of users to be added to the group

        Returns:
            None
        """

        selected = []
        for user in users:
            self.__admin_console.log.info("adding user with username : %s", user)
            self.__table.search_for(user)
            index, selected_user = self.__get_user_index(user)
            selected.extend(selected_user)
            self.__select_user_checkbox(index)

        x_list = list(set(users) - set(selected))
        self.__admin_console.submit_form()
        if x_list:
            raise Exception("There are no users with the name or the user is already added " + str(x_list))


class UserGroups:
    """ class for user groups page """

    def __init__(self, admin_page):
        """ Initialization method for User Groups Class """

        self.__admin_console = admin_page
        self.__table = Table(admin_page)
        self.__transfer_owner = TransferOwnership(admin_page)
        self.__user_selection = UserSelection(admin_page)
        self.__driver = admin_page.driver

    @PageService()
    def add_user_group(self,
                       group_name,
                       description=None,
                       quota=False,
                       quota_limit=None):
        """
        Adds a new user group with the given name

        Args:
            group_name (str)         :   Name of the user group to be added

            description (str)        :   Description of the user group

            quota (bool)             :   To enable/disable quota

            quota_limit (int)        :   Size in GB of the group quota limit
        """
        self.__admin_console.select_hyperlink(self.__admin_console.props['pageHeader.addUserGroup'])
        self.__admin_console.fill_form_by_id("userGroupName", group_name)

        if description:
            self.__admin_console.fill_form_by_id("groupDesc", description)

        if quota:
            self.__admin_console.checkbox_select("quotaEnabled")
            if quota_limit:
                self.__admin_console.fill_form_by_id("quotaLimit", quota_limit)
        else:
            self.__admin_console.checkbox_deselect("quotaEnabled")

        self.__admin_console.submit_form()
        self.__admin_console.check_error_message()

    def open_user_group(self, user_group):
        """
        opens the user group with specified name
        """
        self.__table.access_link(user_group)

    @PageService()
    def delete_user_group(self, user_group, owner_transfer):
        """
        Deletes the user group with the given name

        Args:
            user_group (str)     :   Name of the user group to be deleted

            owner_transfer (str) :   a user or user group to transfer ownership to before deleting

        Raises:
            Exception:
                if unable to delete user group
        """
        self.__table.access_action_item(user_group, 'Delete')
        self.__transfer_owner.transfer_ownership(owner_transfer)
        self.__admin_console.check_error_message()

    @PageService()
    def filter_active_directory_user_groups(self):
        """
        Selects AD User Groups

        Raises:
            Exception:
                if unable to apply filter AD on user groups
        """
        self.__table.view_by_title("Active Directory")
        self.__admin_console.wait_for_completion()

    @PageService()
    def filter_all_user_groups(self):
        """
        Selects All User Groups

        Raises:
            Exception:
                if unable to apply filter All on user groups
        """
        self.__table.view_by_title("All")
        self.__admin_console.wait_for_completion()

    @PageService()
    def filter_local_user_groups(self):
        """
        Selects Local User Groups

        Raises:
            Exception:
                if unable to apply filter local on user groups
        """
        self.__table.view_by_title("Local")
        self.__admin_console.wait_for_completion()

    @PageService()
    def add_users_to_group(self, group_name, users):
        """
        Adds users to the given group

        Args:
            group_name (str)  :  Name of the group to which the users must be added

            users (list)  :  list of users to be added to the group

        Raises:
            Exception:
                if unable to users to given user group
        """
        self.__table.access_action_item(group_name, 'Add users')
        self.__user_selection.add_users(users)
