# -*- coding: utf-8 -*-
# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
This module provides the function or operations that can be performed on the
User Group details page on the AdminConsole

Class:

 UserGroupDetails()

Functions:

edit_user_group()         -- edit the user group details
add_users()               -- Adds users to the user group
remove_users()            -- Removes users from the group
list_users()              -- Lists all the users associated with the group

"""
from Web.AdminConsole.Components.table import CVTable
from Web.Common.page_object import (WebAction, PageService)
from Web.AdminConsole.AdminConsolePages.UserGroups import UserSelection
from Web.AdminConsole.Components.panel import PanelInfo
from Web.AdminConsole.Components.panel import DropDown


class UserGroupDetails:
    """
    Class for User Group Details
    """
    def __init__(self, admin_page):
        """Initialization method for UserGroupDetails Class"""

        self.__admin_console = admin_page
        self.__user_selection = UserSelection(admin_page)
        self.__driver = admin_page.driver
        self.__panel_info = PanelInfo(admin_page)
        self.__table = CVTable(admin_page)
        self.__drop_down = DropDown(admin_page)

    @WebAction()
    def __click_remove(self, user):
        """ Method to click on remove option for given user in user group details page"""
        self.__driver.find_element_by_xpath(
            f"//a[@title='{user}']/ancestor::div[@role='row']/div[4]//a").click()
        self.__admin_console.wait_for_completion()

    @WebAction()
    def __extract_users_details(self):
        """
        Method to extract details of all users associated with the group

        Returns:
            users (list)  --  List of dictionaries with details for all users associated with the group
        """
        elements = self.__driver.find_elements_by_xpath(
            "//div[contains(@class,'ui-grid-render-container ng-isolate-scope')]//div[@class='ui-grid-canvas']/div")
        users = []
        for elem in elements:
            user = {'User name': elem.find_element_by_xpath("./div/div[3]/a").text,
                    'Full name': elem.find_element_by_xpath("./div/div[2]/div").text,
                    'Email': elem.find_element_by_xpath("./div/div[1]/div").text}
            users.append(user)
        return users

    @PageService()
    def edit_user_group(self,
                        group_name=None,
                        description=None,
                        plan=None,
                        quota=False,
                        group_enabled=True,
                        laptop_admins=False,
                        quota_limit=None):
        """
        Method to edit the user group details

        Args:
            quota_limit   (int)   :  the size in GB of the group quota

            group_name    (str)   :  the new name of the group

            plan         (list)  :  list of plans to be associated with user group

            group_enabled (bool)  :  enable / disable the group

            description   (str)   :  description of the user group

            quota         (bool)  :  enable / disable quota

            laptop_admins (bool)  :  enable / disable device activation

        Raises:
            Exception:
                if unable to edit user group
        """

        self.__admin_console.select_hyperlink("Edit")

        if group_name:
            self.__admin_console.fill_form_by_id("userGroupName", group_name)

        if description:
            self.__admin_console.fill_form_by_id("groupDesc", description)

        if quota:
            self.__admin_console.checkbox_select("quotaEnabled")
            if quota_limit:
                self.__admin_console.fill_form_by_id("quotaLimit", quota_limit)
        else:
            self.__admin_console.checkbox_deselect("quotaEnabled")

        if group_enabled:
            self.__admin_console.checkbox_select("groupEnabled")
        else:
            self.__admin_console.checkbox_deselect("groupEnabled")

        if laptop_admins:
            self.__admin_console.checkbox_select("isBlackListed")
        else:
            self.__admin_console.checkbox_deselect("isBlackListed")

        if plan:
            self.__drop_down.select_drop_down_values(0, [plan])

        self.__admin_console.submit_form()

    @PageService()
    def add_users_to_group(self, users):
        """
        Adds new users to the group

        Args:
            users (list)   :   List of users to be added

        Raises:
            Exception:
                if unable to add user
        """
        self.__admin_console.select_hyperlink("Add users")
        self.__user_selection.add_users(users)

    @PageService()
    def remove_users_from_group(self, users):
        """
        Removes users from the group

        Args:
            users (list)  :  List of users to be removed
        """
        for user in users:
            self.__table.search_for(user)
            self.__click_remove(user)
            self.__admin_console.click_button('Yes')

    @PageService()
    def get_user_group_details(self):
        """
        Fetches and displays all the details about the user group like name, description,
        enabled, plans and Username of users etc.

        Returns:
            group_details (dict) :  Dictionary of user group details

        Raises:
            Exception:
                if unable to get user group details
        """
        group_details = self.__panel_info.get_details()
        return group_details

    @PageService()
    def list_users(self):
        """Lists details of all the users associated with the group

        Returns:
            users (list)  --  List of dictionaries with details for all users associated with the group

        Raises:
            Exception:
                if unable to list users
        """
        users = self.__extract_users_details()
        return users