# -*- coding: utf-8 -*-
# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
This module provides the function or operations that can be performed on the
User Details page on the AdminConsole

Class:

    UserDetails()

Functions:

    edit_user()         -- edits the email and group properties of a user with the specified username
    get_user_details()  -- displays the complete info of all the users
    open_user_group()   -- opens the given user group which is associated with this user

"""
from Web.Common.page_object import PageService
from Web.AdminConsole.Components.panel import PanelInfo
from Web.AdminConsole.Components.panel import DropDown


class UserDetails:
    """ Class for user details page """

    def __init__(self, admin_page):
        """
        Method to initiate UserDetails class

        Args:
            admin_page  (Object)  :   AdminPage Class object
        """
        self.__driver = admin_page.driver
        self.__admin_console = admin_page
        self.__admin_console._load_properties(self)
        self.__panel_info = PanelInfo(admin_page)
        self.__drop_down = DropDown(admin_page)

    @PageService()
    def edit_user(self,
                  email,
                  full_name=None,
                  user_name=None,
                  groups=None,
                  plan=None,
                  enabled=True,
                  password=None,
                  current_user_password=None):
        """
        Modifies the user with given details

        Args:
            full_name    (basestring):   full name of the user

            user_name    (basestring):   the user name of the user

            email        (basestring):   the email ID of the user

            groups       (list)      :   list of user groups to be associated with the user

            plan         (list)      :   list of plans to be associated with the user

            enabled      (bool)      :   if the user should be enabled/ disabled

            password     (basestring):   Password to be set for the user

            current_user_password (basestring) : Password of the logged in user

        Returns:
            None
        """

        self.__admin_console.select_hyperlink("Edit")
        if full_name:
            self.__admin_console.fill_form_by_id("fullName", full_name)

        if user_name:
            self.__admin_console.fill_form_by_id("userName", user_name)

        self.__admin_console.fill_form_by_id("email", email)

        if groups:
            self.__drop_down.select_drop_down_values(0, groups)

        if password:
            self.__admin_console.fill_form_by_id("password", password)
            self.__admin_console.fill_form_by_id("confirmUserPassword", password)
            self.__admin_console.fill_form_by_id("currentUserPassword", current_user_password)

        if plan:
            self.__drop_down.select_drop_down_values(1, [plan])

        self.__admin_console.submit_form()

        if enabled:
            self.__admin_console.enable_toggle(0)
        else:
            self.__admin_console.disable_toggle(0)

        self.__admin_console.check_error_message()

    @PageService()
    def get_user_details(self):
        """
        Displays all the details about the user like name, email, user group and plans

        Returns:
            user_details   (dict):  dict of users and their info
        """
        user_details = self.__panel_info.get_details()
        return user_details

    @PageService()
    def open_user_group(self, user_group):
        """
        Opens the user group associated with user

        Args:
            user_group  (basestring):   the name of the user group to open
        """
        self.__admin_console.select_hyperlink(user_group)
