# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------
"""
This module provides the function or operations that can be performed on
a servers of the all agents on the AdminConsole

Class:

    Servers()-> AdminPage() -> AdminConsoleBase() -> object()

Functions:

    select_client()                 -- select and open a client

    server_backup()                 -- Backup the given server

    action_jobs()                   -- opens the jobs page of the client

    select_client()                 -- select and open a client

    action_job()                    -- opens the job page of the given client

    action_add_software()           -- displays support for adding software for the client

    action_update_software()        -- displays support for updating software for the client

    action_release_licence()        -- opens the release license page of the given client

    retire_server()                 -- performs retire action for the given server

    delete_server()                 -- performs delete action for the given server

    action_check_readiness()        -- performs check readiness on client

    action_send_logs()              -- opens the los page of client

    add_server_new_windows_or_unix_server()       -- To create a new Windows/UNIX Server

    add_server_existing_windows_or_unix_server()  -- To add packages in existing Windows/UNIX Server

    add_server_ibmi_server()        -- To create new IBMi Server

    add_server_openvms_server()     -- To create new OpenVMS Server

    OpeniDA()                       -- opens the iDA associated with the client

"""
from Web.AdminConsole.Components.dialog import ModalDialog
from Web.AdminConsole.Components.panel import Backup
from Web.AdminConsole.Components.table import Table
from Web.AdminConsole.AdminConsolePages.AdminPage import AdminPage
from Web.Common.page_object import WebAction, PageService
from Web.AdminConsole.Components.panel import DropDown
from Web.Common.exceptions import CVWebAutomationException


class Servers(AdminPage):

    """
    Class for Server page of adminconsole
    """

    def __init__(self, driver):
        super().__init__(driver)
        self.__table = Table(self)
        self._admin_console = AdminPage(driver)
        self._drop_down = DropDown(self._admin_console)
        self._dialog = ModalDialog(self._admin_console)
        self._admin_console._load_properties(self)

    @PageService()
    def select_client(self, client_name):
        """
        Navigates to the client page

        Args:
            client_name (str): name of the client we need to select

        Returns:
            None

        Raises:
            Exception:
                There is no server with given name

        """
        self.__table.access_link(client_name)

    @WebAction()
    def server_backup(self, server_name, agent_name, subclient_name, backup_level):
        """
        Backup the given server

        Args:
            server_name     (str): Server name to backed up

            agent_name      (str): Agent type of server

            subclient_name  (str): subclient to be backed up

            backup_level  (BackupType) : type of backup, among the type in Backup.BackupType enum

        Returns:
            None

        Raises:
            Exception:
                Correct type of backup not selected
        """
        col_no = self.table_column_no('Backup')
        self.__table.search_for(server_name)
        if self.check_if_entity_exists("link", server_name):
            self.driver.find_element_by_xpath("//a[text()='" + server_name + "']/../../../div["
                                              + str(col_no) + "]/span/a").click()
            self.wait_for_completion()
            self.check_error_message()
            elements = self.driver.find_elements_by_xpath("//label[contains(text(),'"
                                                          + agent_name + "')]/../../div/ul/li")
            for elem in elements:
                if elem.find_element_by_xpath("./div/span[3]/label[contains(text(),'"
                                              + subclient_name + "')]"):
                    elem.find_element_by_xpath("./div/span[2]//input").click()
                    break
            self.submit_form()
            backup = Backup(self)
            return backup.submit_backup(backup_level)

    @PageService()
    def add_gateway(self, client_name, host_name):
        """
        Adds gateway settings on the existing clients

        Args:
            client_name (str): he server on which gateway needs to be added

            host_name (str): the host name of the server

        Returns:
            None

        Raises:
            Exception:
                There is no option to add gateway
        """
        self.select_hyperlink("Add gateway")
        self.fill_form_by_id("clientName", client_name)
        self.fill_form_by_id("hostName", host_name)
        self.submit_form()

    @PageService()
    def action_jobs(self, client_name):
        """Displays the jobs running on the given client

        Args:
            client_name     (str) -- client name for displaying its job manager

        Returns:
            None

        Raises:
            Exception

                if client_name is invalid

                if there is no jobs option for the client

        """
        self.__table.access_action_item(client_name, 'Jobs')

    @PageService()
    def action_add_software(
            self,
            client_name=None,
            select_all_packages=False,
            packages=None,
            reboot=False):
        """selects the Add software option for the given client

        Args:
            client_name     (str)       -- client to add software on

            select_all_packages  (bool)  -- selects all the packages if set True
                                            default: False

            packages        (list)      -- list of packages to be installed

            reboot          (bool)      -- set to True if reboot required
                                            default: False

        Returns:
            (str)    --   the job id for the submitted request

        Raises:
            Exception

                if given input is invalid

                if there is no add software option for the client

        """
        self.__table.access_action_item(client_name, 'Add software')
        if select_all_packages:
            self._drop_down.select_drop_down_values(drop_down_id='agents', select_all=True)
        elif packages:
            self._drop_down.select_drop_down_values(drop_down_id='agents', values=packages)
        else:
            raise CVWebAutomationException('Packages list is not provided')
        if reboot:
            self.driver.find_element_by_xpath("//span[contains(text(), 'Reboot if required')]").click()
        self.submit_form()

        if self.check_if_entity_exists(
                "xpath",
                '//div[@data-ng-controller="pushInstallController"]//a[contains(@href,"jobs")]'):

            # To get the job id
            jobid = self.driver.find_element_by_xpath(
                '//div[@data-ng-controller="pushInstallController"]//a[contains(@href,"jobs")]'
            ).text

        else:
            raise CVWebAutomationException("Job not started, please check the logs")

        # To click OK button
        self.click_button("OK")

        # return the job id
        return jobid

    @WebAction()
    def action_update_software(self, client_name=None, reboot=False):
        """selects the update software option for the given client

        Args:
            client_name     (str) -- client to update software on

            reboot      (bool)    -- set to True if reboot required
                                        default: False

        Returns:
            (str)    --   the job id for the submitted request

        Raises:
            Exception

                if given input is invalid

                if there is no update software option for the client

        """
        # To select Update software in the action item
        self.__table.access_action_item(client_name, 'Update software')

        if reboot:
            self.driver.find_element_by_xpath("//span[contains(text(), 'Reboot if required')]").click()

        self.click_button_using_text('Yes')

        if self.check_if_entity_exists("xpath", '//div[contains(text(), "Job ID:")]'):
            job_text = self.driver.find_element_by_xpath('//div[contains(text(), "Job ID:")]').text

        else:
            raise Exception("Update Job cannot be started, please check the logs")

        jobid = job_text.split(': ')[1]

        # To click on OK button
        self.click_button("OK")

        # return the job id
        return jobid

    @PageService()
    def action_release_license(self, client_name):
        """Displays the release license for the given client

        Args:
            client_name     (str) -- client name to display the release license

        Returns:
            None

        Raises:
            Exception

                if client_name is invalid

                if there is no release license option for the client

        """
        self.__table.access_action_item(client_name, 'Release license')

    @PageService()
    def retire_server(self, server_name):
        """Performs retire action for the given server

        Args:
                server_name     (str) -- server name to retire
        """
        self.__table.access_action_item(server_name, self.props['action.commonAction.retire'])
        self.click_button_using_text(self.props['button.yes'])
        self.check_error_message()
        return self.get_jobid_from_popup()

    @PageService()
    def delete_server(self, server_name):
        """Performs delete action for the given server

        Args:
                server_name     (str) -- server name to delete
        """
        self.__table.access_action_item(server_name, self.props['action.delete'])
        self.click_button_using_text(self.props['button.yes'])
        self.check_error_message()

    @PageService()
    def action_check_readiness(self, client_name):
        """performs check readiness for the given client

        Args:
            client_name     (str) -- client to perform check readiness on

        Returns:
            None

        Raises:
            Exception

                if client_name is invalid

                if there is no uninstall software option for the client

        """
        self.__table.access_action_item(client_name, 'Check readiness')

    @PageService()
    def action_send_logs(self, client_name):
        """selects the send logs option for the given client

        Args:
            client_name     (str) -- client logs to be send

        Returns:
            None

        Raises:
            Exception

                if client_name is invalid

                if there is no send logs option for the client

        """
        self.__table.access_action_item(client_name, 'Send logs')

    @WebAction(hide_args=True)
    def add_server_new_windows_or_unix_server(
            self,
            hostname=None,
            username=None,
            password=None,
            os_type='windows',
            packages=None,
            select_all_packages=False,
            plan=None,
            unix_group=None,
            log_path=None,
            reboot=False,
            install_path=None):
        """To create a new server

        Args:

            hostname   (list)  -- list of servers to install packages on

            username    (str)   -- username of the server machine

            password    (str)   -- password of the server machine

            os_type     (str)   -- os type of the server machine
                                    default: windows

            packages    (list)  -- packages to be installed on the machine

            select_all_packages (bool) -- set to True to install all the packages
                                            default: False

            plan        (str)   -- plan to run install

            unix_group  (str)   -- unix group for UNIX machine

            log_path    (str)   -- path to store the DB2 logs

            reboot      (bool)  -- set to True to reboot if required
                                    default: False

            install_path (str)  -- Installing client on specified path ( Optional )

        Returns:
            (str)    --   the job id for the submitted request

        Raises:
            Exception

                if given inputs are not valid

                if there is no add server option
        """
        self.select_hyperlink("Add server")
        self.driver.find_element_by_xpath('//div[img/@src="/adminconsole/common/svg/FileServers.svg"]').click()
        hostname = [f"{name.strip()}\n" for name in hostname]
        for each in hostname:
            self.fill_form_by_id('hostName', each)
            self.driver.find_element_by_id('add-host-name').click()
        self.fill_form_by_id('fakeusernameremembered', username)
        self.fill_form_by_id('fakepasswordremembered', password)
        self.fill_form_by_id('confirmPasswordPushInstall', password)

        if 'windows' in os_type.lower():
            self.select_radio('osTypeWINDOWS')
            if plan:
                self._drop_down.select_drop_down_values(drop_down_id='plans', values=[plan])
        else:
            self.select_radio('osTypeUNIX')
            if unix_group:
                self.fill_form_by_id("unixGroup", unix_group)

        if select_all_packages:
            self._drop_down.select_drop_down_values(drop_down_id='agents', select_all=True)
        # To install selected packages
        elif packages:
            self._drop_down.select_drop_down_values(drop_down_id='agents', values=packages)
        else:
            raise CVWebAutomationException('Packages list is not provided')

        # To reboot client if required
        if reboot:
            self.driver.find_element_by_xpath("//span[contains(text(), 'Reboot if required')]").click()

        if log_path:
            if self.check_if_entity_exists("name", "db2LogsText"):
                self.fill_form_by_id("db2LogsText", log_path)

        if install_path and self.check_if_entity_exists("id", "fakeinstalllocationremembered"):
            self.fill_form_by_id("fakeinstalllocationremembered", install_path)

        self.submit_form()

        if self.check_if_entity_exists(
                "xpath",
                '//div[@data-ng-controller="pushInstallController"]//a[contains(@href,"jobs")]'):

            # To get the job id
            jobid = self.driver.find_element_by_xpath(
                '//div[@data-ng-controller="pushInstallController"]//a[contains(@href,"jobs")]'
            ).text

        else:
            raise Exception("Job ID not found")

        # To click OK button
        self.submit_form()

        # return the job id
        return jobid

    @PageService()
    def add_server_existing_windows_or_unix_server(self):
        """To create a new server

        Args:

        Returns:
            None
        """
        self.select_hyperlink("Add server")
        self.select_hyperlink("Add Windows/UNIX file server")

    @PageService()
    def add_server_ibmi_server(self):
        """To create a new server
        """
        self.select_hyperlink("Add Server")
        self.select_hyperlink("Add IBM i server")

    @PageService()
    def add_server_openvms_server(self):
        """To create a new server
        """
        self.select_hyperlink("Add Server")
        self.select_hyperlink("Add OpenVMS server")

    @WebAction()
    def open_agent(self, agent_name, server_name):
        """
        Directly opens the given agent of the client.

        Args:
            agent_name    (str):  name of the agent we need to select

            server_name   (str): name of the server we need to select

        Returns:
            None

        Raises:
            Exception:
                There is no server with the name

        """
        self.__table.search_for(server_name)
        if self.check_if_entity_exists("link", server_name):
            if self.check_if_entity_exists(
                    "xpath",
                    "//a[text()='" +
                    server_name +
                    "']/../../../div[2]//a[text()='" +
                    agent_name +
                    "']"):
                self.driver.find_element_by_xpath(
                    "//a[text()='" +
                    server_name +
                    "']/../../../div[2]//a[text()='" +
                    agent_name +
                    "']").click()
                self.wait_for_completion()
            else:
                raise Exception("There is no iDA {0} for client {1}".format(
                    agent_name, server_name))
        else:
            raise Exception("There is no client with the name {0}".format(server_name))

    @WebAction()
    def sap_hana_client(
            self,
            client_name,
            instance_name,
            instance_number,
            os_user,
            server,
            sql_location,
            users,
            data_storage_policy,
            log_storage_policy,
            command_storage_policy,
            store_key=None,
            db_user=None,
            db_password=None):
        """
            Adds a SAP HANA client
            :param client_name:
            :param instance_name:
            :param instance_number:
            :param os_user:
            :param server:
            :param sql_location:
            :param users:
            :param data_storage_policy:
            :param log_storage_policy:
            :param command_storage_policy:
            :param store_key:
            :param db_user:
            :param db_password:
        """
        self.log.info("Adding a SAP HANA client")
        self.driver.find_element_by_name("hostName").send_keys(client_name)
        self.driver.find_element_by_name("databaseName").send_keys(instance_name)
        self.driver.find_element_by_name("databaseNumber").send_keys(instance_number)
        self.driver.find_element_by_name("osUsername").send_keys(os_user)
        self.select_value_from_dropdown("primaryHanaServer", server)
        self.driver.find_element_by_name("hdbSqlLocation").clear()
        self.driver.find_element_by_name("hdbSqlLocation").send_keys(sql_location)
        if not store_key is None:
            self.driver.find_element_by_xpath(
                "//form/cv-tabset-component/div/div[1]/div/label[7]/div/input[1]").click()
            self.driver.find_element_by_name("hdbStorekey").send_keys(store_key)
        else:
            self.driver.find_element_by_name("dbUsername").send_keys(db_user)
            self.driver.find_element_by_name("dbPassword").send_keys(db_password)
        self.driver.find_element_by_link_text("Details").click()
        if users is not None:
            for user in users:
                self.select_value_from_dropdown("users", user)
                self.driver.find_element_by_xpath(
                    "//button[@class='btn btn-primary add-user-btn cvBusyOnAjax']").click()
        self.driver.find_element_by_link_text("Storage").click()
        self.select_value_from_dropdown("dataStoragePolicy", data_storage_policy)
        self.select_value_from_dropdown("logStoragePolicy", log_storage_policy)
        self.select_value_from_dropdown("commandStoragePolicy", command_storage_policy)

        self.driver.find_element_by_xpath("//section/form/div/button[2]").click()
        self.wait_for_completion()

    @PageService()
    def add_cluster_client(self, clustername, hostname, os_type, backup_plan, dir, nodes, agents, force_sync=None):
        """
        Adding Cluster Client
        clustername         (str)   : name of the cluster client
        hostname            (str)   : hostname of the cluster client
        os_type             (str)   : Windows / Linux
        backup_plan         (list)  : Plan to set for a client
        dir                 (str)   : Resultant directory
        Nodes               (list)  : List of cluster nodes
        Agents              (list)  : List of common agents present in all the nodes
        force_sync          (bool)  : Enable/Disable Force sync configuration on remote nodes
        :return:
        """
        self._admin_console.select_hyperlink("Add server")
        self._admin_console.access_sub_menu(self._admin_console.props['label.clusterServer'])
        self._admin_console.fill_form_by_id('clusterName', clustername)
        self._admin_console.fill_form_by_id('hostName', hostname)
        if os_type == "Windows":
            self._admin_console.select_radio('osTypeWINDOWS')
        elif os_type == 'Unix and Linux':
            self._admin_console.select_radio('osTypeUNIX')
        self._drop_down.select_drop_down_values(0, backup_plan)
        self._admin_console.fill_form_by_id('jobResultsDirectory', dir)
        self._drop_down.select_drop_down_values(1, nodes)
        self._drop_down.select_drop_down_values(2, agents)
        if force_sync:
            self.driver.enable_toggle(0)

        self._admin_console.submit_form(True, 'addClusterServerForm')
        self._admin_console.check_error_message()
        return True

    @PageService()
    def delete_cluster_client(self, client_name):
        """ Deleting cluster client """
        self.__table.access_action_item(client_name, self.props['action.delete'])
        self._dialog.type_text_and_delete('Delete')
