# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
This module provides the function or operations that can be performed on the
Roles page on the AdminConsole

Class:

    Roles() -> AdminPage() -> login_page() -> AdminConsoleBase() -> object()

Functions:

add_role()                      -- adds a new role to the admin console
edit_role()                     -- Edits a role with the given permissions
action_delete_role()            -- Adds a new role with the given
toggle_given_permissions()      -- Selects the permissions passed as argument
toggle_permission_checkbox()    -- Checks/un-checks the permission checkbox
expand_category()               -- Expands the category to display the permissions under
                                   the category
list_roles()                    -- Lists all the roles
scrape_available_permissions()  -- Method to get list of all available permissions

"""

from Web.AdminConsole.AdminConsolePages.AdminPage import AdminPage
from Web.AdminConsole.Components.table import Table
from Web.Common.page_object import WebAction, PageService


class Roles(AdminPage):
    """
    Roles page class
    """

    def __init__(self, driver):
        super().__init__(driver)
        self.__table = Table(self)

    @PageService()
    def add_role(self, role_name, permissions, enable_role=True, visible_to_all=False):
        """
        Adds a new role with the given permissions

        Args:
            role_name (str)       : the name of the role
            permissions (list)    : the list of all permissions associated with the role
                                    E.g.[
                                        "Access Policies",
                                        {"Developer Tools":["Workflow",
                                                            {"Datasource":["Add Datasource",
                                                                            "Edit Datasource"]
                                                            }]
                                        },
                                        {"Monitoring Policy":["Edit Monitoring Policy",
                                                              "Search Share"]
                                        }
                                    ]
            enable_role    (bool) : Pass True to select Enable role checkbox, False to deselect
            visible_to_all (bool) : Pass True to select Visible to all checkbox, False to deselect

        Returns:
            None
        """
        self.select_hyperlink("Add role")
        self.fill_form_by_id("name", role_name)

        self.toggle_given_permissions(permissions)

        if enable_role:
            self.checkbox_select("rolesEnabled")
        else:
            self.checkbox_deselect("rolesEnabled")

        if visible_to_all:
            self.checkbox_select("visibleToAll")
        else:
            self.checkbox_deselect("visibleToAll")

        self.submit_form()
        self.check_error_message()

    @PageService()
    def edit_role(self, new_role_name, roles_info, enable_role=False, visible_to_all=True):
        """
        Edits a role with the given permissions

        Args:
            new_role_name (str) : the new name of the role
            roles_info (dict)   : it contains the roles to be added and removed as list
                              E.g. {
                                    "Add":[{"Region Management":["Create Region",
                                                                 "Edit Region"]
                                           },
                                           "Monitoring Policy"
                                    ],
                                    "Remove":["Access Policies",
                                              {"Developer Tools":[{"Datasource":["Add Datasource"]
                                                                  }
                                                                ]
                                              }
                                    ]
                                    }
            enable_role    (bool) : Pass True to select Enable role checkbox, False to deselect
            visible_to_all (bool) : Pass True to select Visible to all checkbox, False to deselect

        Returns:
            None
        """

        self.fill_form_by_id("name", new_role_name)
        self.checkbox_deselect("showSelected")

        for action, roles in roles_info.items():
            if action == "Add":
                self.toggle_given_permissions(roles, select=True)
            if action == "Remove":
                self.toggle_given_permissions(roles, select=False)

        if enable_role:
            self.checkbox_select("rolesEnabled")
        else:
            self.checkbox_deselect("rolesEnabled")

        if visible_to_all:
            self.checkbox_select("visibleToAll")
        else:
            self.checkbox_deselect("visibleToAll")

        self.submit_form()
        self.check_error_message()

    @PageService()
    def action_delete_role(self, role_name):
        """
        Deletes the role with the given name
        Args:
            role_name (str): the name of the role

        Returns:
            None
        """
        self.__table.access_action_item(role_name, "Delete")
        self.click_button('Yes')
        self.check_error_message()

    @PageService()
    def toggle_given_permissions(self, permissions_list, select=True):
        """
        Method to expand categories and select/deselect permissions

        permissions_list (list): List of permissions to be selected/deselected
                                 E.g.[
                                        "Access Policies",
                                        {"Developer Tools":["Workflow",
                                                            {"Datasource":["Add Datasource",
                                                                            "Edit Datasource"]
                                                            }]
                                        },
                                        {"Monitoring Policy":["Edit Monitoring Policy",
                                                              "Search Share"]
                                        }
                                    ]

        select          (bool): Pass True or False to select/deselect permissions
                                 provided in the permissions_list parameter

        """
        for permission in permissions_list:

            if isinstance(permission, str):
                if 'Operations on Storage Policy' in permission:
                    permission = "Operations on Storage Policy \\  Copy"
                self.toggle_permission_checkbox(str(permission), select)

            elif isinstance(permission, dict):
                for key, value in permission.items():
                    self.expand_category(str(key))
                    self.toggle_given_permissions(value, select)

            else:
                raise Exception("Invalid permission")

    @WebAction()
    def toggle_permission_checkbox(self, permission, select=True):
        """
        Method to select/deselect a permission.
        Args:
            permission (str) : permission to be selected/deselected
            select (bool)    : pass True or False to select or deselect permission
        """

        permission_checkbox_xpath = f"//div[@title='{permission}']/span[2]/span/input"
        permission_checkbox_element = \
            self.driver.find_element_by_xpath(permission_checkbox_xpath)

        if select:
            if not permission_checkbox_element.is_selected():
                permission_checkbox_element.click()

        else:
            if permission_checkbox_element.is_selected():
                permission_checkbox_element.click()

    @WebAction()
    def expand_category(self, category):
        """
        Expand the category to access permissions in that category

        Args:
            category (str): Category to be expanded
        """
        expand_permission_xpath = f"//div[@title='{category}']/span[1]"
        category_li_xpath = f"//div[@title='{category}']/.."
        category_li_element = self.driver.find_element_by_xpath(category_li_xpath)
        category_li_class = category_li_element.get_attribute("class")

        if "ivh-treeview-node-collapsed" in category_li_class:
            self.driver.find_element_by_xpath(expand_permission_xpath).click()

    @WebAction()
    def list_roles(self):
        """
        Lists all the roles

        Args:
            None

        Returns:
            Roles List
        """
        elements = self.driver.find_elements_by_xpath("//div[@class='ui-grid-canvas']/div")
        roles_list = []
        for elem in elements:
            roles_list.append(elem.find_element_by_xpath("./div/div[1]/span/a").text)
        self.log.info(roles_list)
        return roles_list

    @WebAction()
    def scrape_available_permissions(self,
                                 permission_xpath="//ul[@class='ivh-treeview']/li/div/span[3]"):
        """Method to get list of all available permissions"""
        permission_elements = self.driver.find_elements_by_xpath(permission_xpath)
        permissions = []

        for element in permission_elements:
            sub_permission_xpath = \
                f"//li/div[@title='{element.text}']/span[3]/../div/ul/li/div/span[3]"
            element.click()
            if element.text and self.driver.find_elements_by_xpath(sub_permission_xpath):
                perm_dict = dict()
                perm_dict[element.text] = self.scrape_available_permissions(sub_permission_xpath)
                permissions.append(perm_dict)
            else:
                permissions.append(element.text)
            element.click()

        return permissions
