# -*- coding: utf-8 -*-
# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
This module provides the function or operations that can be performed on the
plans page on the AdminConsole

Class:

    Plans() -> AdminPage() -> AdminConsoleBase() -> object

Functions:

    _add_replication_details()                      --  add replication details
    _select_entities()                              --  Selects Entities Needed For a
                                                        Data Classification Plan
    _deselect_entities()                            --  Delects Entities For a given
                                                        Data Classification Plan
    _click_create_index_server()                    --  Clicks the create index server button
    create_replication_plan()                       --  creates backup and replication plan
    create_data_classification_plan()               --  Creates a Data Classification plan
    edit_data_classification_plan()                 --  Edit a given data classification
                                                         plan and add/remove the given entities
    deselect_entities_in_data_classification_plan() --  Deselects all entities
                                                         in a given data classification plan
    action_delete_plan()                            --  deletes the plan with the given name
    list_plans()                                    --  lists all the plans
    select_plan()                                   --  selects the plan
    create_server_plan()                            --  create server plan by entering necessary data
    create_laptop_plan()                            --  create laptop plan by entering necessary data
    delete_plan                                     --  deletes the plans given in the dictionary
    create_laptop_derived_plan()                    --  create laptop derived plan by entering necessary data
    create_archive_plan()                           --  Method to create Archival plan
"""

import time
from selenium.webdriver.support.ui import Select
from Web.Common.page_object import (WebAction, PageService)
from Web.AdminConsole.Components.table import Table
from Web.AdminConsole.Components.panel import ModalPanel, DropDown
from Web.AdminConsole.GovernanceAppsPages.GovernanceApps import GovernanceApps


class Plans:
    """ Class for the Plans page """

    def __init__(self, admin_console):
        """
        Method to initiate Plans class

        Args:
            admin_console   (Object) :   admin console object
        """
        self.__admin_console = admin_console
        self.__driver = self.__admin_console.driver
        self.__admin_console._load_properties(self)
        self.log = self.__admin_console.log
        self.__table = Table(self.__admin_console)
        self.__modal_panel = ModalPanel(self.__admin_console)
        self.__drop_down = DropDown(self.__admin_console)
        self.__governance_apps = GovernanceApps(self.__admin_console)

    @WebAction()
    def _check_if_entity_selected(self, entity):
        """
        Checks if the entity is already selected

            Args:
                entity (list)   :   List of entities
                    Example Values:
                        "Credit card number",
                        "Social Security number",
                        "Email"
            Returns:
                True if entity already selected
                False if entity not selected
        """
        return self.__admin_console.check_if_entity_exists(
            "xpath", "//cv-entity-multi-select//label[text() = '%s']\
            /preceding-sibling::input[@checked='checked']" % entity)

    @WebAction()
    def __enable_entity_detection(self):
        """
        Enable Entity selection checkbox while DC plan creation
        """
        self.__driver.find_element_by_xpath(
            "//label[text()='Entity detection']").click()

    @WebAction()
    def _expand_classification(self):
        """Expands the classification accordion"""
        accordion = self.__driver.find_element_by_xpath(
            f"//div[contains(@class,'cv-accordion')]/*//span[contains(text(),'{self.__admin_console.props['label.classification']}')]")
        accordion.click()

    @WebAction()
    def _select_entities(self, entities_list):
        """
        Selects Entities Needed For a Data Classification Plan

            Args:
                entities_list (list)   :   List of entities
                    Example Values:
                        "Credit card number",
                        "Social Security number",
                        "Email"
        """
        for entity in entities_list:
            if not self._check_if_entity_selected(entity):
                self.__driver.find_element_by_xpath(
                    "//cv-entity-multi-select//label[text() = '%s']" % entity).click()
                self.__admin_console.wait_for_completion()
            else:
                self.log.info("Entity: %s already selected" % entity)

    @WebAction()
    def _deselect_entities(self, entities_list):
        """
        Deselects Entities For a Given Data Classification Plan

            Args:
                entities_list (list)   :   List of entities
                    Example Values:
                        "Credit card number",
                        "Social Security number",
                        "Email"
        """
        for entity in entities_list:
            if self._check_if_entity_selected(entity):
                self.__driver.find_element_by_xpath(
                    "//cv-entity-multi-select//label[text() = '%s']" % entity).click()
                self.__admin_console.wait_for_completion()
            else:
                self.log.info("Entity: %s already deselected" % entity)

    @WebAction()
    def backup_window_selection(self, not_run_interval):
        """
        Edits the backup window time slot based on the input given

        Args:
            not_run_interval    (dict)   --  the time slots when the backup should not be run
        """
        self.__driver.find_element_by_link_text("Clear").click()
        self.__admin_console.wait_for_completion()
        keys = not_run_interval.keys()
        days = ['Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday', 'Sunday']
        for day in days:
            self.__driver.find_element_by_xpath("//a[text()='" + day + "']").click()
            self.__admin_console.wait_for_completion()
            if day in keys:
                slot_values = self.__driver.find_elements_by_xpath(f"//a[text()='{day}']/../../td")
                for slot in slot_values:
                    class_name = slot.get_attribute('class')
                    if class_name != 'label-day':
                        if int(slot.get_attribute('date-time-id')) in not_run_interval[day]:
                            if 'selected' in class_name:
                                slot.click()

    @WebAction()
    def _click_create_index_server(self):
        """Clicks the create index server button"""
        self.log.info("Clicking create index server button")
        self.__driver.find_element_by_class_name("k-i-plus-outline").click()
        self.__admin_console.wait_for_completion()

    @PageService()
    def create_data_classification_plan(
            self, plan, index_server, content_analyzer, entities_list=None,
            select_all=False, content_search=True, metadata_only=False,
            content_analysis=True, enable_ocr=False, target_app='gdpr',
            create_index_server=False, node_name=None, index_directory=None, classifier_list=None):
        """
        Creates a Data Classification plan

            Args:
                plan (str)           :   Name of the plan to be created
                index_server (str)        :   Index Server name
                content_analyzer (str)      :   Content Analyzer name
                entities_list (list)   :   List of entities
                    Values:
                        "Credit card number",
                        "Social Security number",
                        "Email"
                select_all (bool)   :   True/False to select all entities
                content_search (bool)    : True/False to enable content search
                metadata_only (bool)    : True to select Metadata.
                                          False to select both Metadata and content
                content_analysis (bool)    : True/False to enable content analysis for Case Manager
                enable_ocr (bool)    : True/False to enable
                                            content extraction from scanned docs for SDG
                target_app (str)     :    The target app for the plan to be created for.
                                        Default is gdpr
                    Values:
                        "gdpr"(default),
                        "casemanager",
                        "fso"
                create_index_server (bool)  :   True if new index server has to be created
                                                False if existing index server to use
                node_name   (str)           :   Index server node name for new index server
                index_directory (str)       :   Index directory for the new index server
                classifier_list (list)      :   list of classifier to be selected

            """
        self.log.info("Creating data classification plan with name: %s", plan)
        self.__driver.find_element_by_link_text("Create plan").click()
        self.__admin_console.wait_for_completion()
        self.__driver.find_element_by_link_text("Data classification").click()
        self.__admin_console.wait_for_completion()
        if target_app == 'casemanager':
            self.__governance_apps.select_case_manager()
        elif target_app == 'fso':
            self.__governance_apps.select_file_storage_optimization()
        else:
            self.__governance_apps.select_sensitive_data_analysis()
        self.log.info("Entering plan name")
        self.__admin_console.fill_form_by_id("planName", plan)
        if create_index_server:
            self.log.info("Creating Index Server with node: %s" % node_name)
            self._click_create_index_server()
            self.log.info("Filling in Index server name")
            self.__admin_console.fill_form_by_id("newIndexServerName", index_server)
            self.log.info("Selecting index server node name")
            self.__drop_down.select_drop_down_values(0, [node_name])
            self.log.info("Filling in Index directory")
            self.__admin_console.fill_form_by_id(node_name, index_directory)
            self.__modal_panel.submit()
            self.__admin_console.wait_for_completion()
        self.log.info("Selecting Index Server")
        self.__drop_down.select_drop_down_values(0, [index_server])
        if target_app != 'fso':
            if target_app == 'casemanager':
                self.__admin_console.button_next()
                if content_analysis:
                    self.log.info("Clicking on Enabling Entity detection")
                    self.__enable_entity_detection()
                    self.__admin_console.wait_for_completion()
            if content_analysis:
                self.log.info("Selecting content analyzer")
                self.__drop_down.select_drop_down_values(1, [content_analyzer])
                if select_all is True:
                    self._select_entities(['Select all'])
                else:
                    self._select_entities(entities_list)
                if classifier_list:
                    self._expand_classification()
                    time.sleep(2)
                    self.__drop_down.select_drop_down_values(2, classifier_list, default_unselect=False)
            if target_app == 'gdpr':
                self.__admin_console.button_next()
                if enable_ocr:
                    self.__admin_console.enable_toggle(index=0, cv_toggle=True)
        self.__admin_console.submit_form()

    @PageService()
    def edit_data_classification_plan(self, plan, entities_list):
        """
        Edit a given data classification plan and add/remove the given entities
        :param plan    (str)   --  Plan name to be edited
        :param entities_list    (list)    -- Entities list to be added or removed in the plan
        """
        self.log.info("Click Edit in plan '%s'", plan)
        self.__driver.find_element_by_xpath(
            "//*[@title='Entity detection']//a[contains(text(),'Edit')]").click()
        self.__admin_console.wait_for_completion()
        self._select_entities(entities_list)
        self.log.info("Clicking on Save after the selection of the entities")
        self.__admin_console.submit_form()

    @PageService()
    def deselect_entities_in_data_classification_plan(self, plan):
        """
        Deselects all entities in a given data classification plan
        :param plan    (str)   --  Plan name from which entities will be deselcted
        """
        self.log.info("Click Edit in plan '%s'", plan)
        self.__driver.find_element_by_xpath(
            "//*[@title='Entity detection']//a[contains(text(),'Edit')]").click()
        self.__admin_console.wait_for_completion()
        self._deselect_entities(['Select all'])
        self.__modal_panel.submit()
        self.__admin_console.wait_for_completion()

    @PageService()
    def action_delete_plan(self, plan):
        """Deletes the plan with the given name
            plan    :   a string, the name of the plan to be deleted
        """
        self.__table.access_action_item(plan, "Delete")
        self.__admin_console.click_button('Yes')
        self.__admin_console.check_error_message()

    @PageService()
    def list_plans(self):
        """Lists all the plans
        """
        plans_dict = self.__table.get_table_data()
        self.log.info("The list of all plan details is %s", str(plans_dict))
        return plans_dict

    @PageService()
    def select_plan(self, plan):
        """
        Selects the given plan
        """
        self.__table.access_link(plan)

    @WebAction()
    def __set_plan_name(self, plan_name):
        """
        Method to set plan name
        Args:
            plan_name (str) : name of the plan
        """
        self.__admin_console.fill_form_by_id("planName", plan_name)

    @WebAction()
    def __open_edit_backup_window_dialog(self):
        """ Method to open edit backup window dialog """
        edit_backup_window_link = self.__driver.find_element_by_xpath(
            "//cv-backup-window-list[@data-cv-model='globalTemplate.backupWindow']")
        edit_backup_window_link.click()

    @WebAction()
    def __open_edit_full_backup_window_dialog(self):
        """ Method to open edit backup window dialog """
        edit_full_backup_window_link = self.__driver.find_element_by_xpath(
            "//cv-backup-window-list[@data-cv-model='globalTemplate.fullBackupWindow']")
        edit_full_backup_window_link.click()

    @WebAction()
    def __set_backup_window(self, backup_day, backup_time):
        """
        Method to set back up window
        Args:
            backup_day (dict): dictionary containing values of backup days for backup window
                Eg.- dict.fromkeys(['1', '2', '3'], 1)
                        backup days for schedule
                            1 : Monday
                            2: Tuesday and so on
            backup_time (dict): dictionary containing values of backup time for backup window
                Eg.- dict.fromkeys(['2', '3', '4', '5'], 1)
                    backup time for schedule
                        2 : 12:00am to 1:00 am
                        3 : 1:00am to 2:00 am
                        and so on..
        """

        self.__admin_console.select_hyperlink(self.__admin_console.props['action.clear'])
        for key_day, val_day in backup_day.items():
            if val_day == 1:
                for key_time, val_time in backup_time.items():
                    if val_time == 1:
                        self.__driver.find_element_by_xpath(
                            f"//table[@class='time-obj']/tbody/tr[{key_day}]/td[{key_time}]").click()
        self.__driver.find_element_by_xpath("//*[@id='editPlanOperationWindowModal_button_#4242']").click()
        self.__admin_console.check_error_message()

    @WebAction()
    def __open_edit_storage_dialog(self):
        """ Method to set primary storage of a plan """
        self.__driver.find_element_by_xpath(
            "//cv-grid//a[contains(@class,'uib-dropdown-toggle')]").click()
        self.__admin_console.wait_for_completion()
        self.__driver.find_element_by_xpath(
            f"//a[contains(text(),'{self.__admin_console.props['action.edit']}')]").click()
        self.__admin_console.wait_for_completion()

    @WebAction()
    def __set_retention_period(self, ret_period):
        """
        Method to set retention period

        Args:
            ret_period (str) : Value of retention period
        """
        textbox = self.__driver.find_element_by_xpath(
            "//input[@data-ng-change='changePolicyRetention()']")
        textbox.clear()
        textbox.send_keys(ret_period)

    @WebAction()
    def __set_retention_unit(self, ret_unit):
        """
        Method to set retention unit

        Args:
            ret_unit (str) : retention unit to be set
        """
        retention_unit = Select(self.__driver.find_element_by_xpath(
            "//select[@data-ng-change='changePolicyRetention()']"))
        retention_unit.select_by_visible_text(ret_unit)

    @WebAction()
    def __set_rpo_unit(self, rpo_unit):
        """
        Method to set unit for RPO

        Args:
            rpo_unit (str): RPO Unit to be set

        """
        frequency_unit = Select(self.__driver.find_element_by_xpath(
            "//select[@data-ng-model='incrementalBackupFreq.type']"))
        frequency_unit.select_by_visible_text(rpo_unit)

    @WebAction()
    def __set_rpo_hours(self, rpo_hours):
        """
        Method to set hours for Backup Start Time for RPO

        Args:
            rpo_hours (str):    Value of hour to be set

        """
        textbox = self.__driver.find_element_by_xpath(
            "//input[@ng-change='updateHours()']")
        textbox.clear()
        textbox.send_keys(rpo_hours)

    @WebAction()
    def __set_rpo_minutes(self, rpo_minutes):
        """
        Method to set minutes for Backup Start Time for RPO

        Args:
            rpo_minutes (str):  Value of minutes to be set

        """
        textbox = self.__driver.find_element_by_xpath(
            "//input[@ng-change='updateMinutes()']")
        textbox.clear()
        textbox.send_keys(rpo_minutes)

    @WebAction()
    def __set_am_or_pm(self, am_or_pm):
        """
        Set whether AM/PM for Backup Start Time under RPO

        Args:
            am_or_pm (str): Whether AM/PM

        """
        if am_or_pm.upper() == 'AM':
            self.__driver.find_element_by_xpath(
                "//button[@ng-click='toggleMeridian()']"
            ).click()

    @WebAction()
    def __click_plan_type(self, plan_type_id):
        """
        Method to select type of plan to be created from create plan option menu

        Args:
            plan_type_id (str) : id of the plan type to be selected

        """
        self.__driver.find_element_by_xpath(
            f"//div[@class='k-animation-container']//a[@id='{plan_type_id}']").click()

    @WebAction()
    def __expand_sub_panel(self, sub_panel_text):
        """ Method to expand sub panel """
        panel_xpath = self.__driver.find_element_by_xpath(
            f"//span[contains(text(),'{sub_panel_text}')]")
        panel_xpath.click()

    @WebAction()
    def __select_file_system_tab(self, file_system):
        """ Method to select file system """
        self.__driver.find_element_by_xpath(
            "//ul[@class='nav nav-tabs nav-justified']//a[contains(text(),\
            '" + file_system + "')]").click()

    @WebAction()
    def __set_sla_hours(self, sla_hours):
        """ Method to set sla hours """
        elems = self.__driver.find_elements_by_id("logSlaHours")
        for elem in elems:
            if elem.is_displayed():
                elem.clear()
                elem.send_keys(sla_hours)

    @WebAction()
    def __set_sla_minutes(self, sla_minutes):
        """ Method to set sla minutes """
        elems = self.__driver.find_elements_by_id("logSlaMinutes")
        for elem in elems:
            if elem.is_displayed():
                elem.clear()
                elem.send_keys(sla_minutes)

    @WebAction()
    def __enable_allow_override(self):
        """ Method to set allow over ride check box state """
        if not self.__driver.find_element_by_xpath(
                "//label[contains(@for,'derivation-enabled')]/preceding-sibling::input")\
                .get_attribute('checked') == 'checked':
            self.__driver.find_element_by_xpath(
                "//label[contains(@for,'derivation-enabled')]").click()

    @WebAction()
    def __select_override_backup_content(self):
        """ Method to select the overide backup content"""
        if not self.__driver.find_element_by_xpath(
                "//label[contains(text(),'Override base settings')]")\
                .get_attribute('checked') == 'checked':
            self.__driver.find_element_by_xpath("//label[contains(text(),'Override base settings')]").click()

    @WebAction()
    def __select_override_retention(self):
        """ Method to select the overide backup content"""
        xpath = "//div[@class='bootstrap-wrapper ng-scope']/div/div/div/label[@class='override-toggle-label']"
        if not self.__driver.find_element_by_xpath(xpath).get_attribute('checked') == 'checked':
            self.__driver.find_element_by_xpath(xpath).click()

    @WebAction()
    def __select_override_storage_rpo(self, overrid_checkbox_id):
        """ Method to select the overide backup content"""

        if not self.__driver.find_element_by_xpath(
                f"//div[@id='{overrid_checkbox_id}']//label[@class='override-toggle-label']")\
                    .get_attribute('checked') == 'checked':
            self.__driver.find_element_by_xpath(
                f"//div[@id='{overrid_checkbox_id}']//label[@class='override-toggle-label']").click()

    @WebAction()
    def __set_override_option(self, override_drop_down_id, override_value):
        """ Method to set override option """
        override_drop_down = Select(self.__driver.find_element_by_xpath(
            f"//select[contains(@id,'{override_drop_down_id}')]"))
        override_drop_down.select_by_visible_text(override_value)

    @WebAction()
    def __click_browse_button(self):
        """ Method to click on browse button """
        self.__driver.find_element_by_id("generalSetup_button_#6532").click()

    @WebAction()
    def __select_browse_root(self):
        """ Method to select Browse root  """
        self.__driver.find_element_by_xpath(
            "//span[@class='path-title ng-binding']").click()

    @WebAction()
    def __click_new_folder_button(self):
        """ Method to click on New folder button """
        self.__driver.find_element_by_xpath(
            "//*[@id='machineBrowse_button_#5674']").click()

    @PageService()
    def __create_index_server(self):
        """ Method to create index server """
        self.__click_browse_button()
        self.__admin_console.wait_for_completion()
        self.__select_browse_root()
        self.__click_new_folder_button()
        self.__admin_console.wait_for_completion()
        self.__admin_console.fill_form_by_id('folder-name', "IndexServer")
        self.__modal_panel.submit()
        self.__admin_console.click_button(self.__admin_console.props['label.save'])

    @WebAction()
    def __enable_secondary_storage(self):
        """ Method to enable secondary storage for laptop plan """
        if not self.__driver.find_element_by_xpath(
                "//input[@id='enableSecondaryStorage']").is_selected():
            self.__driver.find_element_by_xpath(
                "//label[@for='enableSecondaryStorage']").click()

    @WebAction()
    def __select_radio_by_id(self, radio_button_id):
        """ Method to select radio button based on id """
        if not self.__driver.find_element_by_id(radio_button_id).is_selected():
            self.__driver.find_element_by_id(radio_button_id).click()

    @WebAction()
    def __select_checkbox_by_label(self, label):
        """ Method to select checkbox by label """
        if not self.__driver.find_element_by_xpath(
                f"//label[text()='{label}']/preceding-sibling::input").is_selected():
            self.__driver.find_element_by_xpath(f"//label[text()='{label}']").click()

    @WebAction()
    def __click_next_button(self):
        """ Method to click next button on Laptop plan creation page """
        next_buttons = self.__driver.find_elements_by_xpath(
            "//div[contains(@class,'button-container')]/button[contains(text(),'Next')]")
        for next_button in next_buttons:
            if next_button.is_displayed():
                next_button.click()
                break

    @WebAction()
    def __setup_edge_drive_features(self,
                                    allowed_features,
                                    media_agent):
        """
        Method to setup edge drive features

        Args:
            allowed_features (dictionary): dictionary containing features to be enabled for the
                             plan and corresponding attributes
                Eg. - allowed_features = {'Edge Drive': 'ON',
                                          'audit_drive_operations': True,
                                          'notification_for_shares': True,
                                          'edge_drive_quota':'200',
                                          'DLP': 'ON',
                                          'Archiving': 'ON'}

            media_agent (string)    :   media agent to be used to configure edge drive for plan
        """
        if self.__admin_console.check_if_entity_exists("id", "client"):
            self.__admin_console.select_value_from_dropdown('client', media_agent)
        else:
            self.__admin_console.select_value_from_dropdown('mediaAgent', media_agent)

        if self.__admin_console.check_if_entity_exists("id", "mountPath"):
            self.__create_index_server()

        if allowed_features['audit_drive_operations']:
            self.__admin_console.checkbox_select('auditDriveActivities')
        else:
            self.__admin_console.checkbox_deselect('auditDriveActivities')

        if allowed_features['notification_for_shares']:
            self.__admin_console.checkbox_select('enableNotificationsForShares')
        else:
            self.__admin_console.checkbox_deselect('enableNotificationsForShares')

        if allowed_features['edge_drive_quota']:
            self.__admin_console.checkbox_select('isEdgeDriveQuotaEnabled')
            self.__admin_console.fill_form_by_id("edgeDriveQuota", allowed_features['edge_drive_quota'])
        else:
            self.__admin_console.checkbox_deselect('isEdgeDriveQuotaEnabled')

    @WebAction()
    def __click_remove_content(self, tab_id):
        """
        Method to click on remove all content button

        Args:
            tab_id (str) : id of backup/exclusion content tab
        """
        remove_button = f"//div[@class='tab-pane ng-scope active']//div[@id='{tab_id}']" \
                        "//a[contains(@data-ng-click,'deleteAll')]"
        if self.__admin_console.check_if_entity_exists("xpath", remove_button):
            if self.__driver.find_element_by_xpath(remove_button).is_displayed():
                self.__driver.find_element_by_xpath(remove_button).click()
                self.__admin_console.wait_for_completion()

    @WebAction()
    def __click_add_content(self, tab_id):
        """
        Method to click on add content button

        Args:
            tab_id (str) : id of backup/exclusion content tab
        """
        add_button = f"//div[@class='tab-pane ng-scope active']//div[@id='{tab_id}']" \
                     "//a[contains(text(),'Add')]"
        self.__driver.find_element_by_xpath(add_button).click()

    @WebAction()
    def __select_content(self,
                         content,
                         content_library):
        """
        Method to select backup/exclusion content

        Args:
            content (list) : List with folder names to be selected
                            for content backup
                Eg. - content_backup:["Content Library", "Documents", "Desktop",
                                        "Office", "Music", "Pictures", "Videos"]
            content_library (list): List with content library values to be
                            selected for backup
                Eg. - content_library = ['Content Library', 'GoogleDrive', 'Image',
                                         'Office', 'OneDrive', 'System',
                                        'Temporary Files', 'Text',
                                        'Thumbnail Supported', 'Video',
                                        'iCloudDrive']
        """
        for value in content:
            if not value == 'Content Library':
                if not self.__driver.find_element_by_xpath(
                        "//label[@for='" + value + "']/../..//input").is_selected():
                    self.__driver.find_element_by_xpath(
                        "//label[@for='" + value + "']/../..//input").click()
                    if value == 'Library':
                        break
            else:
                if self.__admin_console.check_if_entity_exists(
                        "xpath", "//li[@class='ivh-treeview-node ng-scope ng-isolate-scope "
                                 "ivh-treeview-node-collapsed']"):
                    self.__driver.find_element_by_xpath(
                        "//li[@class='ivh-treeview-node ng-scope ng-isolate-scope ivh-treeview"
                        "-node-collapsed']//span[@class='glyphicon glyphicon-chevron-right']"
                    ).click()
                    for value_lib in content_library:
                        if not self.__driver.find_element_by_xpath(
                                "//label[@for='" + value + "']/../..//input").is_selected():
                            self.__driver.find_element_by_xpath(
                                "//label[@for='" + value_lib + "']/../..//input").click()

    @WebAction()
    def __click_add_custom_content(self):
        """ Method to click on add custom content link """
        self.__driver.find_element_by_xpath(
            "//a[contains(text(),'" + self.__admin_console.props['label.customContent'] + "')]").click()

    @WebAction()
    def __click_create_derived_plan_button(self):
        """ Method to click on derive plan"""
        self.__driver.find_element_by_xpath(
            "//a[@id='actionDetail_DERIVE_PLAN_ENTITY']").click()

    @WebAction()
    def __select_file_size_unit(self, file_size_unit):
        """ Method to select File Size Unit """
        self.__driver.find_element_by_xpath(
            "//select[@ng-model='fileSizeConfig.rule']/option[text()='{}']".format(
                file_size_unit)).click()

    @WebAction()
    def __select_last_modified_unit(self, last_modified_unit):
        """ Method to select Last Modified Unit """
        self.__driver.find_element_by_xpath(
            "//select[@ng-model='fileTimestampConfig.timestampRule']/option[text()='{}']".format(
                last_modified_unit)).click()

    @WebAction()
    def __select_last_accessed_unit(self, last_accessed_unit):
        """ Method to select Last Modified Unit """
        self.__driver.find_element_by_xpath(
            "//select[@ng-model='fileTimestampConfig.timestampRule']/option[text()='{}']".format(
                last_accessed_unit)).click()

    @WebAction()
    def __select_access_time_rule(self):
        """ Method to select access time rule. """
        self.__driver.find_element_by_xpath(
            "//select[@ng-model='fileTimestampConfig.accessRule']/option[text()='Last accessed']").click()

    @WebAction()
    def __select_modify_time_rule(self):
        """ Method to select modify time rule. """
        self.__driver.find_element_by_xpath(
            "//select[@ng-model='fileTimestampConfig.accessRule']/option[text()='Last modified']").click()

    @WebAction()
    def __select_archive_frequency_unit(self, archive_frequency_unit):
        """ Method to set Archive frequency unit. """
        self.__driver.find_element_by_xpath(
            "//select[@ng-model='incrementalBackupFreq.type']/option[text()='{}']".format(
                archive_frequency_unit)).click()

    @WebAction()
    def __set_office365_retention_period(self, period):
        """
        Method to set the Office 365 Plan Retention

        Args:
            period (int):   No. of days/months/years data should be retained

        """
        self.__driver.find_element_by_xpath(
            "//input[@picker-id='deletedRetentionDays']").click()
        input_element = self.__driver.find_element_by_xpath(
            "//input[@data-ng-model='ctrl.pickerNum']")
        input_element.clear()
        input_element.send_keys(str(period))

    @WebAction()
    def __click_edit_onedrive_filters(self):
        """Clicks on Edit option for OneDrive Filters for Office 365 Plan"""
        self.__driver.find_element_by_xpath(
            "//*[@title='OneDrive']//ancestor::div[@class="
            "'page-details group']//a[contains(text(),'Edit')]"
        ).click()

    @WebAction()
    def __click_file_filters_tab(self):
        """Clicks on File Filters Tab for Office 365 Plan"""
        self.__driver.find_element_by_xpath(
            "//li/a[normalize-space()='File filters']").click()

    @WebAction()
    def __add_include_folder_filter(self, include_filter):
        """
        Adds the include folder filter

        Args:
            include_filter (str):   Filter to be added

        """
        input_element = self.__driver.find_element_by_id('includefolderName')
        input_element.clear()
        input_element.send_keys(include_filter)
        self.__driver.find_element_by_xpath(
            "//h4[normalize-space()='Include folder']//ancestor::div"
            "[@id='backup-content-folders']//*[contains(@class, 'k-i-plus')]"
        ).click()

    @WebAction()
    def __add_exclude_folder_filter(self, exclude_filter):
        """
        Adds the exclude folder filter

        Args:
            exclude_filter (str):   Filter to be added

        """
        input_element = self.__driver.find_element_by_id('excludefolderName')
        input_element.clear()
        input_element.send_keys(exclude_filter)
        self.__driver.find_element_by_xpath(
            "//h4[normalize-space()='Exclude folder']//ancestor::div"
            "[@id='exclude-folders']//*[contains(@class, 'k-i-plus')]"
        ).click()

    @WebAction()
    def __add_include_file_filter(self, include_filter):
        """
        Adds the include file filter

        Args:
            include_filter (str):   Filter to be added

        """
        input_element = self.__driver.find_element_by_id('includefileName')
        input_element.clear()
        input_element.send_keys(include_filter)
        self.__driver.find_element_by_xpath(
            "//h4[normalize-space()='Include file']//ancestor::div"
            "[@id='backup-content-folders']//*[contains(@class, 'k-i-plus')]"
        ).click()

    @WebAction()
    def __add_exclude_file_filter(self, exclude_filter):
        """
        Adds the exclude file filter

        Args:
            exclude_filter (str):   Filter to be added

        """
        input_element = self.__driver.find_elements_by_id('includefolderName')[1]
        input_element.clear()
        input_element.send_keys(exclude_filter)
        self.__driver.find_element_by_xpath(
            "//h4[normalize-space()='Exclude file']//ancestor::div"
            "[@id='exclude-folders']//*[contains(@class, 'k-i-plus')]"
        ).click()

    @PageService()
    def click_create_derived_plan_button(self):
        """
        click on derive plan button

        """
        self.__click_create_derived_plan_button()
        self.__admin_console.wait_for_completion()

    @PageService()
    def backup_content_selection(self,
                                 content_backup,
                                 content_library,
                                 custom_content):
        """
        Method to select backup content for the plan

        Args:
            content_backup (list): List with folder names to be selected
                            for content backup
                Eg. - content_backup:["Content Library", "Documents", "Desktop",
                                        "Office", "Music", "Pictures", "Videos"]
            content_library (list): List with content library values to be
                            selected for backup
                Eg. - content_library = ['Content Library', 'GoogleDrive', 'Image',
                                         'Office', 'OneDrive', 'System',
                                        'Temporary Files', 'Text',
                                        'Thumbnail Supported', 'Video',
                                        'iCloudDrive']
            custom_content (str): path for custom(extra) content to be backed up

        Raises:
            Exception:
                -- if fails to select Backup content
        """

        if content_backup:
            self.__click_remove_content('backup-content-folders')
        self.__click_add_content('backup-content-folders')

        if content_backup:
            self.__select_content(content_backup,
                                  content_library)

        if custom_content:
            self.__click_add_custom_content()
            self.__admin_console.fill_form_by_id("path", custom_content)

        self.__driver.find_element_by_xpath("//*[@id='createNewFolder_button_#8189']").click()
        self.__admin_console.check_error_message()

    @PageService()
    def exclusion_content_selection(self,
                                    exclude_folder,
                                    exclude_folder_library,
                                    exclude_folder_custom_content):

        """
        Method to select content to be excluded from backup for the plan

        Args:
            exclude_folder (list): List with folder names to be excluded
                            from backup
                Eg. - exclude_folder = ['Content Library', 'Documents']
            exclude_folder_library (list): List with content library folders to be
                                excluded from backup
                Eg. - exclude_folder_library = ['DropBox', 'EdgeDrive', 'Executable',
                                                'GoogleDrive', 'Image', 'Office',
                                                'OneDrive', 'Thumbnail Supported',
                                                'Video', 'iCloudDrive']
            exclude_folder_custom_content (str): path for custom(extra) content to be
                                            excluded

        Raises:
            Exception:
                -- if fails to provide exclusion values for the plan
        """

        if exclude_folder:
            self.__click_remove_content('exclude-folders')
        self.__click_add_content('exclude-folders')

        if exclude_folder:
            self.__select_content(exclude_folder, exclude_folder_library)

        if exclude_folder_custom_content:
            self.__click_add_custom_content()
            self.__admin_console.fill_form_by_id("folderName", exclude_folder_custom_content)

        self.__driver.find_element_by_xpath("//*[@id='createNewFolder_button_#8189']").click()
        self.__admin_console.check_error_message()

    @PageService()
    def set_network_resources(self,
                              throttle_send,
                              throttle_receive):
        """
        Method to set network resources

        Args:
            throttle_send (string ot integer): Network Resource send parameter value

            throttle_receive (string or integer): Network Resource receive parameter value
        """
        if throttle_send == "No limit":
            self.__admin_console.checkbox_select('throttleSend')
        else:
            self.__admin_console.checkbox_deselect('throttleSend')
            self.__admin_console.fill_form_by_id('maxLANBandwidth', throttle_send)

        if throttle_receive == "No limit":
            self.__admin_console.checkbox_select('throttleReceive')
        else:
            self.__admin_console.checkbox_deselect('throttleReceive')
            self.__admin_console.fill_form_by_id('maxWANBandwidth', throttle_send)

    @WebAction()
    def __alert_checkbox(self, alert):
        """ Method to get alert checkbox element """
        xp = f"//*[@id='tileContent_Alerts']//*[text()='{alert}']"
        return self.__driver.find_element_by_xpath(xp)

    @WebAction()
    def __edit_alert_checkbox(self, alert):
        """ Method to edit alert checkbox element """
        xp = f"//form[@name='editAlertsForm']//*[text()='{alert}']"
        return self.__driver.find_element_by_xpath(xp)

    @PageService()
    def set_alerts(self, alerts, edit_alert=False):
        """
        Method to set alerts

        Args:
            alerts (dictionary): Dictionary with values for determining whether alerts should
                be Enabled/Disabled
                Eg. - alerts = {"Backup" : "No backup for last 4 days",
                                "Jobfail": "Restore Job failed",
                                "edge_drive_quota":"Edge drive quota alert",
                                "edge_drive_operations":"Edge drive/share operations alert"}
            edit_alert (boolean) = Whether to edit alerts or not
        """
        if alerts.get("Backup", None):
            if not edit_alert:
                elem = self.__alert_checkbox('No backup for last 4 days')
            else:
                elem = self.__edit_alert_checkbox('No backup for last 4 days')
            if alerts["Backup"] == "No backup for last 4 days":
                self.__admin_console.checkbox_select(elem.get_attribute('for'))
            else:
                self.__admin_console.checkbox_deselect(elem.get_attribute('for'))

        if alerts.get("Jobfail", None):
            if not edit_alert:
                elem = self.__alert_checkbox('Restore Job failed')
            else:
                elem = self.__edit_alert_checkbox('Restore Job failed')
            if alerts["Jobfail"] == "Restore Job failed":
                self.__admin_console.checkbox_select(elem.get_attribute('for'))
            else:
                self.__admin_console.checkbox_deselect(elem.get_attribute('for'))

        if alerts.get("edge_drive_quota", None):
            if not edit_alert:
                elem = self.__alert_checkbox('Edge drive quota alert')
            else:
                elem = self.__edit_alert_checkbox('Edge drive quota alert')
            if alerts["edge_drive_quota"] == "Edge drive quota alert":
                self.__admin_console.checkbox_select(elem.get_attribute('for'))
            else:
                self.__admin_console.checkbox_deselect(elem.get_attribute('for'))

        if alerts.get("file_system_quota", None):
            if not edit_alert:
                elem = self.__alert_checkbox('File system quota alert')
            else:
                elem = self.__edit_alert_checkbox('File system quota alert')
            if alerts["file_system_quota"] == "File system quota alert":
                self.__admin_console.checkbox_select(elem.get_attribute('for'))
            else:
                self.__admin_console.checkbox_deselect(elem.get_attribute('for'))

    @PageService()
    def set_override_options(self, allow_override, edit_override_laptop=False):
        """
        Method to set override options

        Args:
            allow_override (dictionary): dictionary containing values for Override parameters
                Eg. - allow_override = {"Storage_pool": "Override required",
                                        "RPO": "Override optional",
                                        "Folders_to_backup": "Override not allowed",
                                        "Retention": "Override not allowed"}

            edit_override_laptop (boolean) = Whether to edit override or not
        """
        if allow_override.get('Folders_to_backup', None):
            if not edit_override_laptop:
                self.__set_override_option('override-select-SubclientPolicyWin', allow_override['Folders_to_backup'])
            else:
                self.__set_override_option('override-select-SubclientContentWin', allow_override['Folders_to_backup'])

        if allow_override.get('Storage_pool', None):
            self.__set_override_option('override-select-StoragePolicy', allow_override['Storage_pool'])

        if allow_override.get('RPO', None):
            self.__set_override_option('override-select-SchedulePolicy', allow_override['RPO'])

        if allow_override.get('Retention', None):
            self.__set_override_option('override-select-SubclientRetentionWin', allow_override['Retention'])

    @PageService()
    def set_retention(self, retention):
        """
        Method to set retention values

        Args:
             retention (dict) : dictionary containing retention attributes for laptop plan
                Eg. - retention = {'deleted_item_retention': {'value': '5', 'unit': 'day(s)'},
                                   'file_version_retention': {'duration': {'value': '4',
                                                                          'unit': 'day(s)'},
                                                              'versions': '5',
                                                              'rules': {'days': '4',
                                                                        'weeks': '5',
                                                                        'months': '6'}}}
                    OR
                        retention = {'deleted_item_retention': {'value': '5', 'unit': 'day(s)'},
                                       'file_version_retention': {'duration': None,
                                                                  'versions': None,
                                                                  'rules': {'days': '4',
                                                                            'weeks': '5',
                                                                            'months': '6'}}}
        """
        if retention.get('deleted_item_retention', None):

            if retention['deleted_item_retention']['value'] == "Indefinitely":
                self.__select_radio_by_id("radio2_1")
            else:
                self.__select_radio_by_id("radio1_1")
                self.__admin_console.fill_form_by_id("radio1_1years",
                                                     retention['deleted_item_retention']['value'])
                self.__admin_console.select_value_from_dropdown("deleted_time",
                                                                retention['deleted_item_retention']['unit'])

        if retention.get('file_version_retention', None):

            if retention['file_version_retention']['duration']:
                self.__select_radio_by_id("radio1_2")
                self.__admin_console.fill_form_by_id("radio1_2years",
                                                     retention['file_version_retention']['duration']['value'])
                self.__admin_console.select_value_from_dropdown("file_time",
                                                                retention['file_version_retention']['duration']
                                                                ['unit'])

            elif retention['file_version_retention']['versions']:
                self.__select_radio_by_id("radio2_2")
                self.__admin_console.fill_form_by_id("radio2_2versions",
                                                     retention['file_version_retention']['versions'])

            elif retention['file_version_retention']['rules']:
                self.__select_radio_by_id("radio3_3")
                self.__admin_console.fill_form_by_id("radio3_2perday",
                                                     retention['file_version_retention']['rules']['days'])
                self.__admin_console.fill_form_by_id("radio3_3perweek",
                                                     retention['file_version_retention']['rules']['weeks'])
                self.__admin_console.fill_form_by_id("radio3_4permonth",
                                                     retention['file_version_retention']['rules']['months'])

    @PageService()
    def set_archiving_rules(self, archiving_rules):
        """
        Method to set archiving rules

        Args:
             archiving_rules (dictionary): dictionary containing values for Archive Feature rules
                Eg. - archiving_rules = {"do_edit": True, "start_clean": "40", "stop_clean": "90",
                                        "file_access_time": "85", "file_modify_time": None,
                                        "file_create_time": "2", "archive_file_size": None,
                                        "max_file_size": None, "archive_read_only": True,
                                        "replace_file": None, "delete_file": None}
        """
        if archiving_rules.get("start_clean", None):
            self.__admin_console.fill_form_by_id("startCleanDisk", archiving_rules["start_clean"])
        if archiving_rules.get("stop_clean", None):
            self.__admin_console.fill_form_by_id("stopCleanDisk", archiving_rules["stop_clean"])
        if archiving_rules.get("file_access_time", None):
            self.__admin_console.fill_form_by_id("fileAccessTime", archiving_rules["file_access_time"])
        if archiving_rules.get("file_modify_time", None):
            self.__admin_console.fill_form_by_id("fileModifiedTime", archiving_rules["file_modify_time"])
        if archiving_rules.get("file_create_time", None):
            self.__admin_console.fill_form_by_id("fileCreatedTime", archiving_rules["file_create_time"])
        if archiving_rules.get("archive_file_size", None):
            self.__admin_console.fill_form_by_id("archiveFileSize", archiving_rules["archive_file_size"])
        if archiving_rules.get("max_file_size", None):
            self.__admin_console.fill_form_by_id("maxFileSize", archiving_rules["max_file_size"])

        if archiving_rules.get("archive_read_only", None):
            self.__admin_console.checkbox_select('archiveReadOnlyFile')
        else:
            self.__admin_console.checkbox_deselect('archiveReadOnlyFile')

        if archiving_rules.get("replace_file", None):
            self.__admin_console.select_radio('replaceFile')
        else:
            self.__admin_console.select_radio('deleteFile')
        self.__modal_panel.submit()

    @PageService()
    def create_server_plan(
            self,
            plan_name,
            storage,
            rpo=None,
            allow_override=None,
            backup_day=None,
            backup_duration=None,
            backup_data=None,
            snapshot_options=None,
            database_options=None,
            system_state=True,
            full_backup=False,
            vss=True):
        """
        Method to create Server plan

        Args:
            plan_name (string): Name of the plan to be created

            storage (dict) : Dict containing storage attributes for admin console
                Eg. - self._storage = {'pri_storage': None,
                         'pri_ret_period':'30',
                         'sec_storage': None,
                         'sec_ret_period':'45',
                         'ret_unit':'day(s)'}

            rpo (dict): Dictionary containing values for RPO
                Eg. - rpo_dict = {'bkp_frequency': '1',
                                  'frequency_unit': 'Day(s)',
                                  'hours': '09',
                                  'minutes': '00',
                                  'am_or_pm': 'PM'}

            allow_override (dictionary): dictionary containing values for Override parameters
                Eg. - allow_override = {"Storage_pool": "Override required",
                                        "RPO": "Override optional"}

            backup_day (dictionary): dictionary containing values of backup days for backup
                window
                Eg. - backup_day = dict.fromkeys(['1', '2', '3'], 1)

            backup_duration (dictionary): dictionary containing values of excluded backup
                time for backup window
                Eg. - backup_duration = dict.fromkeys(['2', '3', '4', '5'], 1)

            backup_data (Dictionary): Dictionary with data to be selected
                for backup or excluded from backup
                Eg. - backup_data = {'file_system':["Mac", "Unix", "Windows"],
                                    'content_backup':["Content Library"],
                                    'content_library':['Content Library', 'GoogleDrive'],
                                    'custom_content':None,
                                    'exclude_folder':['Content Library', 'Documents'],
                                    'exclude_folder_library':['DropBox', 'EdgeDrive', 'Executable']
                                    'exclude_folder_custom_content':None}

            system_state (boolean): value, if "Backup system state" should be checked or not

            full_backup (boolean): value, if "Only with full backup" should be checked or not

            vss (boolean): value, if "Use VSS for system state" should be checked or not

            snapshot_options (dictionary) : dictionary containing values for snapshot parameters
                Eg. - snapshot_options = {'snap_recovery_points':'5',
                                          'Enable_backup_copy':True,
                                          'sla_hours':'5',
                                          'sla_minutes':'20'}

            database_options (dictionary) : dictionary containing values for databse parameters
                Eg. - database_options = {'sla_hours':'5',
                                          'sla_minutes':'20'}
        Returns:
            None

        Raises:
            Exception:
                -- if fails to create the plan
        """

        self.__admin_console.select_hyperlink(self.__admin_console.props['label.createProfile'])
        self.__click_plan_type('CreateServerPlan')
        self.__admin_console.wait_for_completion()
        self.__set_plan_name(plan_name)

        self.__admin_console.select_hyperlink("Add")
        self.__admin_console.fill_form_by_id("storageName", "Primary")
        self.__drop_down.select_drop_down_values(0, [storage['pri_storage']])

        if storage.get('pri_ret_period'):
            if storage['pri_ret_period'] == 'Infinite':
                self.__set_retention_unit(storage['ret_unit'])
            else:
                self.__set_retention_period(storage['pri_ret_period'])
                if storage.get('ret_unit', None):
                    self.__set_retention_unit(storage['ret_unit'])
        self.__admin_console.submit_form()

        if storage.get('sec_storage', None):
            self.__admin_console.select_hyperlink("Add copy")
            self.__admin_console.fill_form_by_id("storageName", "Secondary")
            self.__drop_down.select_drop_down_values(0, [storage['sec_storage']])

            if storage['sec_ret_period']:
                if storage['sec_ret_period'] == 'Infinite':
                    self.__set_retention_unit(storage['ret_unit'])
                else:
                    self.__set_retention_period(storage['sec_ret_period'])
                    if storage.get('ret_unit', None):
                        self.__set_retention_unit(storage['ret_unit'])
            self.__admin_console.submit_form()

        if rpo:
            if rpo.get('bkp_frequency', None):
                self.__admin_console.fill_form_by_id("rpo", rpo['bkp_frequency'])
                self.__set_rpo_unit(rpo['frequency_unit'])
            if rpo.get('hours', None):
                self.__set_rpo_hours(rpo['hours'])
                self.__set_rpo_minutes(rpo['minutes'])
                self.__set_am_or_pm(rpo['am_or_pm'])

        if backup_day and backup_duration:
            self.__open_edit_backup_window_dialog()
            self.__admin_console.wait_for_completion()
            self.__set_backup_window(backup_day, backup_duration)

            self.log.info("setting up full back up window")
            self.__open_edit_full_backup_window_dialog()
            self.__admin_console.wait_for_completion()
            self.__set_backup_window(backup_day, backup_duration)

        if backup_data:
            self.__expand_sub_panel('Folders to backup')
            self.__admin_console.wait_for_completion()

            for value in backup_data['file_system']:
                self.__select_file_system_tab(value)
                self.__admin_console.wait_for_completion()
                time.sleep(2)

                self.backup_content_selection(
                    backup_data.get('content_backup', None),
                    backup_data.get('content_library', None),
                    backup_data.get('custom_content', None))

                if backup_data['exclude_folder'] or backup_data['exclude_folder_library'] or backup_data[
                        'exclude_folder_custom_content']:
                    self.exclusion_content_selection(
                        backup_data.get('exclude_folder', None),
                        backup_data.get('exclude_folder_library', None),
                        backup_data.get('exclude_folder_custom_content', None))

                if value == "Windows":
                    if system_state:
                        self.__admin_console.checkbox_select('backupSystemState')

                        if full_backup:
                            self.__admin_console.checkbox_select('backupSystemStateforFullBkpOnly')
                        else:
                            self.__admin_console.checkbox_deselect('backupSystemStateforFullBkpOnly')

                        if vss:
                            self.__admin_console.checkbox_select('useVSSForSystemState')
                        else:
                            self.__admin_console.checkbox_deselect('useVSSForSystemState')
                    else:
                        self.__admin_console.checkbox_deselect('backupSystemState')

        if snapshot_options:
            self.__expand_sub_panel('Snapshot options')
            self.__admin_console.wait_for_completion()
            if snapshot_options.get('snap_recovery_points', None):
                if self.__admin_console.check_if_entity_exists("id", "snapRecoveryPoints"):
                    self.__admin_console.fill_form_by_id("snapRecoveryPoints",
                                                         snapshot_options['snap_recovery_points'])

            if snapshot_options.get('Enable_backup_copy', None):
                if snapshot_options['Enable_backup_copy'] == 'ON':
                    self.__admin_console.enable_toggle(index=1)
                else:
                    self.__admin_console.disable_toggle(index=1)

            if snapshot_options.get('sla_hours', None):
                self.__set_sla_hours(snapshot_options['sla_hours'])

            if snapshot_options.get('sla_minutes', None):
                self.__set_sla_minutes(snapshot_options['sla_minutes'])

        if database_options:
            self.__expand_sub_panel('Database options')
            self.__admin_console.wait_for_completion()

            if database_options.get('sla_hours', None):
                self.__set_sla_hours(database_options['sla_hours'])

            if database_options.get('sla_minutes', None):
                self.__set_sla_minutes(database_options['sla_minutes'])

        if allow_override:
            self.__expand_sub_panel(self.__admin_console.props['label.overrideRestrictions'])
            self.__admin_console.wait_for_completion()
            self.__enable_allow_override()
            self.__admin_console.wait_for_completion()
            self.set_override_options(allow_override)

        self.__admin_console.click_button(self.__admin_console.props['label.save'])
        self.__admin_console.wait_for_completion()
        self.__admin_console.check_error_message()

    @PageService()
    def create_laptop_plan(
            self,
            plan_name,
            allowed_features=None,
            media_agent=None,
            archiving_rules=None,
            backup_data=None,
            file_system_quota=None,
            storage=None,
            rpo_hours=None,
            retention=None,
            throttle_send=None,
            throttle_receive=None,
            alerts=None,
            allow_override=None):
        """
        Method to create Laptop plan

        Args:

            plan_name (string): Name of the plan to be created

            rpo_hours (string): RPO Hours value

            allowed_features (dictionary): dictionary containing features to be enabled for the
                             plan and corresponding attributes
                Eg. - allowed_features = {'Edge Drive': 'ON',
                                          'audit_drive_operations': True,
                                          'notification_for_shares': True,
                                          'edge_drive_quota':'200',
                                          'DLP': 'ON',
                                          'Archiving': 'ON'}

            media_agent (string)    :   media agent to be used to configure edge drive for plan

            archiving_rules (dictionary): dictionary containing values for Archive Feature rules
                Eg. - archiving_rules = {"do_edit": True, "start_clean": "40", "stop_clean": "90",
                                        "file_access_time": "85", "file_modify_time": None,
                                        "file_create_time": "2", "archive_file_size": None,
                                        "max_file_size": None, "archive_read_only": True,
                                        "replace_file": None, "delete_file": None}

            backup_data (dictionary) : Dictionary containing multiple lists with values for back
                up content to be selected and excluded while creating the plan
                Eg. - backup_data = {'file_system':["Mac", "Unix", "Windows"],
                                    'content_backup':["Content Library"],
                                    'content_library':['Content Library', 'GoogleDrive'],
                                    'custom_content':None,
                                    'exclude_folder':['Content Library', 'Documents'],
                                    'exclude_folder_library':['DropBox', 'EdgeDrive', 'Executable']
                                    'exclude_folder_custom_content':None
                                   }

            file_system_quota (string): Variable containing value for File system quota

            storage (dict) : Dict containing storage attributes for admin console
                Eg. - self._storage = {'pri_storage': None,
                         'pri_ret_period':'30',
                         'sec_storage': None,
                         'sec_ret_period':'45',
                         'ret_unit':'day(s)'}

            throttle_send (string ot integer): Network Resource send parameter value

            throttle_receive (string or integer): Network Resource receive parameter value

            alerts (dictionary): Dictionary with values for determining whether alerts should
                be Enabled/Disabled
                Eg. - alerts = {"Backup" : "No backup for last 4 days",
                                "Jobfail": "Restore Job failed",
                                "edge_drive_quota":"Edge drive quota alert",
                                "edge_drive_operations":"Edge drive/share operations alert"}

            allow_override (dictionary): dictionary containing values for Override parameters
                Eg. - allow_override = {"Storage_pool": "Override required",
                                        "RPO": "Override optional",
                                        "Folders_to_backup": "Override not allowed"}

            retention (dict) : dictionary containing retention attributes for laptop plan
                Eg. - retention = {'deleted_item_retention': {'value': '5', 'unit': 'day(s)'},
                                   'file_version_retention': {'duration': {'value': '4',
                                                                          'unit': 'day(s)'},
                                                              'versions': '5',
                                                              'rules': {'days': '4',
                                                                        'weeks': '5',
                                                                        'months': '6'}}}
                    OR
                        retention = {'deleted_item_retention': {'value': '5', 'unit': 'day(s)'},
                                       'file_version_retention': {'duration': None,
                                                                  'versions': None,
                                                                  'rules': {'days': '4',
                                                                            'weeks': '5',
                                                                            'months': '6'}}}

        Returns:
            None

        Raises:
            Exception:
                -- if fails to create the plan
        """

        self.__admin_console.select_hyperlink(self.__admin_console.props['label.createProfile'])
        self.__click_plan_type('CreateLaptopPlan')
        self.__admin_console.wait_for_completion()
        self.__set_plan_name(plan_name)

        if allowed_features:
            if allowed_features.get("Edge Drive", None):
                if allowed_features["Edge Drive"] == "ON":
                    self.__admin_console.checkbox_select('146')
                    self.__setup_edge_drive_features(allowed_features,
                                                     media_agent)
                else:
                    self.__admin_console.checkbox_deselect('146')

            if allowed_features.get("DLP", None):
                if allowed_features["DLP"] == "ON":
                    self.log.info("Enabling DLP for the plan")
                    self.__admin_console.checkbox_select('148')
                else:
                    self.__admin_console.checkbox_deselect('148')

            if allowed_features.get("Archiving", None):
                if allowed_features["Archiving"] == "ON":
                    self.__admin_console.checkbox_select('215')

                    if archiving_rules:
                        self.__admin_console.select_hyperlink(self.__admin_console.props['action.edit'])
                        self.__admin_console.wait_for_completion()
                        self.set_archiving_rules(archiving_rules)
                else:
                    self.__admin_console.checkbox_deselect('215')

        self.__click_next_button()
        self.__admin_console.wait_for_completion()

        if backup_data:
            for value in backup_data['file_system']:
                self.__select_file_system_tab(value)
                self.__admin_console.wait_for_completion()
                time.sleep(2)

                self.backup_content_selection(
                    backup_data.get('content_backup', None),
                    backup_data.get('content_library', None),
                    backup_data.get('custom_content', None))

                self.exclusion_content_selection(
                    backup_data.get('exclude_folder', None),
                    backup_data.get('exclude_folder_library', None),
                    backup_data.get('exclude_folder_custom_content', None))

        if file_system_quota:
            self.__admin_console.checkbox_select('isQuotaEnabled')
            self.__admin_console.fill_form_by_id("quota", file_system_quota)

        self.__click_next_button()
        self.__admin_console.wait_for_completion()

        if storage.get('pri_storage', None):
            self.__admin_console.select_value_from_dropdown('storage', storage['pri_storage'])

        if storage.get('sec_storage', None):
            self.__enable_secondary_storage()
            self.__admin_console.wait_for_completion()
            self.__admin_console.select_value_from_dropdown('secondaryStorage', storage['sec_storage'])

        if rpo_hours:
            self.__admin_console.fill_form_by_id("rpoHours", rpo_hours)

        self.__click_next_button()
        self.__admin_console.wait_for_completion()

        self.set_retention(retention)
        self.__click_next_button()

        self.set_network_resources(throttle_send, throttle_receive)
        self.set_alerts(alerts)
        if allow_override:
            self.__enable_allow_override()
            self.__admin_console.wait_for_completion()
            self.set_override_options(allow_override)

        self.__click_next_button()
        self.__admin_console.wait_for_completion()

        self.__admin_console.click_button(self.__admin_console.props['label.save'])
        self.__admin_console.check_error_message()

    @PageService()
    def delete_plan(self, plan_name):
        """
        Method to delete an existing plan

        Args:
            plan_name (str): Name of the plan to be deleted

        Raises:
            Exception:
                -- if fails to delete plan
        """
        self.__table.access_action_item(plan_name, self.__admin_console.props['action.delete'])
        self.__admin_console.click_button_using_text('Yes')
        notification = self.__admin_console.get_notification(wait_time=120)
        if not notification == 'The plan was successfully deleted':
            raise Exception(notification)


    @PageService()
    def create_laptop_derived_plan(self,
                                   derived_plan,
                                   backup_data=None,
                                   storage=None,
                                   rpo_hours=None,
                                   retention=None):
        """
        Method to create Laptop derived plan

        Args:

            derived_plan (string): Name of the derived plan to be created

            backup_data (dictionary) : Dictionary containing multiple lists with values for back
                up content to be selected and excluded while creating the plan
                Eg. - backup_data = {'file_system':["Mac", "Unix", "Windows"],
                                    'content_backup':["Content Library"],
                                    'content_library':['Content Library', 'GoogleDrive'],
                                    'custom_content':None,
                                    'exclude_folder':['Content Library', 'Documents'],
                                    'exclude_folder_library':['DropBox', 'EdgeDrive', 'Executable']
                                    'exclude_folder_custom_content':None
                                   }

            storage (dict) : Dict containing storage attributes for admin console
                Eg. - self._storage = {'pri_storage': None,
                         'pri_ret_period':'30',
                         'sec_storage': None,
                         'sec_ret_period':'45',
                         'ret_unit':'day(s)'}

            rpo_hours (string): RPO Hours value

            retention (dict) : dictionary containing retention attributes for laptop plan
                Eg. - retention = {'deleted_item_retention': {'value': '5', 'unit': 'day(s)'},
                                   'file_version_retention': {'duration': {'value': '4',
                                                                          'unit': 'day(s)'},
                                                              'versions': '5',
                                                              'rules': {'days': '4',
                                                                        'weeks': '5',
                                                                        'months': '6'}}}
                    OR
                        retention = {'deleted_item_retention': {'value': '5', 'unit': 'day(s)'},
                                       'file_version_retention': {'duration': None,
                                                                  'versions': None,
                                                                  'rules': {'days': '4',
                                                                            'weeks': '5',
                                                                            'months': '6'}}}

        Returns:
            None

        Raises:
            Exception:
                -- if fails to create the plan

        """

        self.__set_plan_name(derived_plan)
        self.__click_next_button()
        self.__admin_console.wait_for_completion()

        if backup_data:
            self.__select_override_backup_content()
            for value in backup_data['file_system']:
                self.__select_file_system_tab(value)
                self.__admin_console.wait_for_completion()
                time.sleep(2)

                self.backup_content_selection(
                    backup_data.get('content_backup', None),
                    backup_data.get('content_library', None),
                    backup_data.get('custom_content', None))

                self.exclusion_content_selection(
                    backup_data.get('exclude_folder', None),
                    backup_data.get('exclude_folder_library', None),
                    backup_data.get('exclude_folder_custom_content', None))

            self.__click_next_button()
            self.__admin_console.wait_for_completion()

        if storage:
            self.__select_override_storage_rpo('tileContent_Copies')
            self.__admin_console.select_value_from_dropdown('storage', storage['pri_storage'])

        if rpo_hours:
            self.__select_override_storage_rpo('tileContent_RPO')
            self.__admin_console.fill_form_by_id("rpoHours", rpo_hours)

        self.__click_next_button()
        self.__admin_console.wait_for_completion()

        if retention:
            self.__select_override_retention()
            self.set_retention(retention)
        self.__click_next_button()
        self.__admin_console.wait_for_completion()

        self.__click_next_button()
        self.__admin_console.wait_for_completion()

        self.__admin_console.click_button(self.__admin_console.props['label.save'])
        self.__admin_console.check_error_message()

    @PageService()
    def create_archive_plan(
            self,
            plan_name,
            storage,
            rpo=None,
            archive_day=None,
            archive_duration=None,
            archiving_rules=None,
            allow_override=None):
        """
        Method to create Archival plan

        Args:
            plan_name (string): Name of the plan to be created

            storage (dict) : Dict containing storage attributes for admin console
                Eg. - self._storage = {'pri_storage': None,
                         'pri_ret_period':'30',
                         'sec_storage': None,
                         'sec_ret_period':'45',
                         'ret_unit':'day(s)'}

            rpo (dictionary): dictionary containing RPO values
                Eg. -   rpo = {
                "archive_frequency": 2
                "archive_frequency_unit": Hours
                }

            archive_day (dictionary): dictionary containing values of backup days for backup
                window
                Eg. - archive_day = dict.fromkeys(['1', '2', '3'], 1)

            archive_duration (dictionary): dictionary containing values of excluded backup
                time for backup window
                Eg. - archive_duration = dict.fromkeys(['2', '3', '4', '5'], 1)

            archiving_rules (dictionary):   dictionary containing values of diskcleanup/archiving rules
                Eg. -   archiving_rules = {
                "last_accessed_unit":   "days",
                "last_accessed": 2,
                "last_modified_unit": "hours",
                "last_modified": 3,
                "file_size": 20,
                "file_size_unit": "KB"
                }

            allow_override (dictionary): dictionary containing values for Override parameters
                Eg. - allow_override = {"Storage_pool": "Override required",
                                        "RPO": "Override optional"}

        """
        self.__admin_console.select_hyperlink(self.__admin_console.props['label.createProfile'])
        self.__click_plan_type('CreateArchivePlan')
        self.__admin_console.wait_for_completion()
        self.__set_plan_name(plan_name)

        self.__admin_console.select_hyperlink("Add")

        self.__drop_down.select_drop_down_values(0, [storage['pri_storage']])

        if storage['pri_ret_period']:
            if storage['pri_ret_period'] is not 'Infinite':
                self.__set_retention_unit(storage['ret_unit'])
                self.__set_retention_period(storage['pri_ret_period'])
        self.__admin_console.submit_form()

        if rpo:
            if rpo['archive_frequency']:
                self.__admin_console.fill_form_by_id("rpo", rpo['archive_frequency'])
            if rpo['archive_frequency_unit']:
                self.__select_archive_frequency_unit(rpo['archive_frequency_unit'])

        if archive_day and archive_duration:
            self.__open_edit_backup_window_dialog()
            self.__admin_console.wait_for_completion()
            self.__set_backup_window(archive_day, archive_duration)

        if archiving_rules:

            if archiving_rules['last_accessed']:
                self.__select_access_time_rule()
                self.__admin_console.fill_form_by_id("accessedAgo", archiving_rules['last_accessed'])

            if archiving_rules['last_accessed_unit']:
                self.__select_last_accessed_unit(archiving_rules['last_accessed_unit'])

            if archiving_rules['last_modified']:
                self.__select_modify_time_rule()
                self.__admin_console.fill_form_by_id("accessedAgo", archiving_rules['last_modified'])

            if archiving_rules['last_modified_unit']:
                self.__select_last_modified_unit(archiving_rules['last_modified_unit'])

            if archiving_rules['file_size']:
                self.__admin_console.fill_form_by_id("size", archiving_rules['file_size'])

            if archiving_rules['file_size_unit']:
                self.__select_file_size_unit(archiving_rules['file_size_unit'])

        if allow_override:
            self.__expand_sub_panel(self.__admin_console.props['label.overrideRestrictions'])
            self.__admin_console.wait_for_completion()
            self.__enable_allow_override()
            self.__admin_console.wait_for_completion()
            self.set_override_options(allow_override)

        self.__driver.find_element_by_xpath("//*[@id='archivalPlanTemplate_button_#4019']").click()
        self.__admin_console.wait_for_completion()
        self.__admin_console.check_error_message()

    @PageService()
    def create_office365_plan(
            self,
            plan_name,
            retention=None,
            backup_archive_mailbox=False,
            set_content_search=False,
            generate_previews=False,
            set_entity_search=False,
            backup_all_versions=True,
            onedrive_filters=None):
        """
        Method to create Office365 Plan

        Args:
            plan_name (str):                Name of the plan

            retention (dict):               How long the backed up data should be retained
                                            Eg: retention = {'ret_period': '365',
                                                             'ret_unit': 'day(s)'}

            backup_archive_mailbox (bool):  Whether or not archive mailbox has to be backed up

            set_content_search (bool):      Whether or not Content Search has to be enabled

            generate_previews (bool):       Whether or not Pre-generate previews has to be enabled

            set_entity_search (bool):       Whether or not Entity Search has to be enabled

            backup_all_versions (bool):     Whether or not all versions have to be backed up

            onedrive_filters (dict):        Filters that have to be applied
                                            Eg: filters = {'include_folders': ['test_folder'],
                                                           'include_files': ['*.log', 'test_file.txt']}
                                            Eg: filters = {'exclude_folders': ['test_folder'],
                                                           'exclude_files': ['*.log', 'test_file.txt']}

        """
        self.__admin_console.select_hyperlink(self.__admin_console.props['label.createProfile'])
        self.__click_plan_type('CreateOffice365Plan')
        self.__admin_console.wait_for_completion()
        self.__set_plan_name(plan_name)

        if backup_archive_mailbox:
            self.__admin_console.enable_toggle(index=1, cv_toggle=True)

        if retention:
            if retention['ret_period'] != 'Infinite':
                self.__select_radio_by_id('retainBasedOnDeletionTime')
                self.__set_office365_retention_period(retention['ret_period'])
                if retention.get('ret_unit', None):
                    self.__admin_console.select_value_from_dropdown(
                        select_id='cvTimeRelativePicker_isteven-multi-select_#6209',
                        value=retention['ret_unit'])

        if set_content_search:
            self.__admin_console.enable_toggle(index=2, cv_toggle=True)
            if generate_previews:
                self.__admin_console.enable_toggle(index=3, cv_toggle=True)
            if set_entity_search:
                self.__admin_console.enable_toggle(index=4, cv_toggle=True)
        elif set_entity_search:
            self.__admin_console.enable_toggle(index=3, cv_toggle=True)

        self.__admin_console.click_button(self.__admin_console.props['label.save'])
        self.__admin_console.wait_for_completion()
        self.__admin_console.check_error_message()

        if not backup_all_versions:
            self.__admin_console.disable_toggle(index=5, cv_toggle=True)
            self.__admin_console.disable_toggle(index=6, cv_toggle=True)

        if onedrive_filters:
            self.__click_edit_onedrive_filters()
            self.__admin_console.wait_for_completion()

            if onedrive_filters.get('include_folders', None):
                for item in onedrive_filters['include_folders']:
                    self.__add_include_folder_filter(item)

            if onedrive_filters.get('exclude_folders', None):
                for item in onedrive_filters['exclude_folders']:
                    self.__add_exclude_folder_filter(item)

            if onedrive_filters.get('include_files', None):
                self.__click_file_filters_tab()
                for item in onedrive_filters['include_files']:
                    self.__add_include_file_filter(item)

            if onedrive_filters.get('exclude_files', None):
                self.__click_file_filters_tab()
                for item in onedrive_filters['exclude_files']:
                    self.__add_exclude_file_filter(item)

            self.__admin_console.click_button(self.__admin_console.props['label.save'])
            self.__admin_console.wait_for_completion()
            self.__admin_console.check_error_message()
