# -*- coding: utf-8 -*-
# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
This module provides the function or operations that can be performed on the
plan detail page on the AdminConsole

Class:

    PlanDetails() -> AdminPage() -> AdminConsoleBase() -> object

Functions:

plan_info()                             --  Displays all the information of the plan in the form of a dictionary
edit_plan_storage_pool()                --  Edits the storage pool associated with the plan
edit_plan_alerts()                      --  Edits the alerts associated with the plan
edit_plan_network_resources()           --  Edits the network resources associated with the plan
edit_plan_rpo()                         --  Edits the backup SLA associated with the plan
edit_plan_options()                     --  Edits the options associated with the plan
edit_plan_backup_content()              --  Edits the backup content associated with the plan
edit_plan_associated_users_and_groups() --  Edits the users and user groups
                                            associated with the plan
remove_associated_users_and_groups      --  Method to remove associated users or user groups
                                            from a plan
edit_plan_retention()                   --  Method to edit plan retention properties
edit_plan_override_restrictions()       --  Method to edit plan override settings
data_classication_plan_entities()       --  Returns list of entities selected for DC plan
edit_server_plan_storage_pool           --  Method to edit storage pool associated with the plan
access_snap_copy_type_dialog()          --  Method to access copy type edit dialog
delete_server_plan_storage_copy()       --  Method to delete storage copy in a server plan
edit_plan_rpo_start_time()              --  Method to edit RPO start time associated to the plan
edit_plan_full_backup_rpo()             --  Method to edit Full backup RPO settings

"""
import time

from selenium.webdriver.common.keys import Keys

from Web.Common.page_object import (WebAction, PageService)
from Web.AdminConsole.AdminConsolePages.Plans import Plans
from Web.AdminConsole.Components.panel import PanelInfo, ModalPanel, DropDown
from Web.AdminConsole.Components.table import Table


class PlanDetails:
    """ Class for the Plans page """

    def __init__(self, admin_console):
        """
        Method to initiate PlanDetails class

        Args:
            admin_console   (Object) :   admin console object
        """
        self.__admin_console = admin_console
        self.__driver = self.__admin_console.driver
        self.__admin_console._load_properties(self)
        self.log = self.__admin_console.log
        self.__table = Table(self.__admin_console)
        self.__modal_panel = ModalPanel(self.__admin_console)
        self.__plans = Plans(self.__admin_console)
        self.__panel_dropdown_obj = DropDown(self.__admin_console)

    @WebAction()
    def __check_if_copy_present(self, copy_name):
        """
                Method to check if a storage policy copy exists

                Args:
                    copy_name: Name of copy to be checked

                Returns:
                    boolean: True/False based on copy presence
                """
        if self.__admin_console.is_element_present(
                f'//*[@id="tileContent_Backup destinations"]//a[contains(text(), "{copy_name}")]'):
            return True
        return False
    
    @WebAction()
    def __extract_plan_info(self, plan_type):
        """
        Method to extract plan information and return a dictionary

        Args:
            plan_type : Type of plan to be fetched details of (Laptop/Server)

        Returns:
            details dict() : dictionary containing plan information displayed in UI
        """
        self.log.info("Getting all plan information")
        details = {}
        title = self.__driver.find_element_by_xpath(
            '//h1[@class="no-margin page-title editable js-page-title"]')
        details.update({"PlanName": title.text})
        elems = self.__driver.find_elements_by_xpath("//div[@class='page-details group']")

        for elem in elems:
            keyvalue_dict = {}
            keyvalue_list = []
            key = elem.find_element_by_xpath("./span").text
            list_items = ['Inheriting from base plan', 'Overriding base plan']
            for each_item in list_items:
                if key.endswith(each_item):
                    key = key.replace(each_item, "").rstrip()
            if key != '':
                if key == "Backup destinations":
                    table_rows = elem.find_elements_by_xpath("./../div//table[@role='grid']/tbody/tr")
                    if not plan_type == 'laptop_plan':
                        for table_row in table_rows:
                            nested_key = table_row.find_element_by_xpath(
                                "./td/a").text
                            nested_value_1 = table_row.find_element_by_xpath(
                                "./td[3]/span").text
                            nested_value_2 = table_row.find_element_by_xpath(
                                "./td[4]/span").text
                            keyvalue_dict.update({nested_key: [nested_value_1, nested_value_2]})
                        details.update({key: keyvalue_dict})
                    else:
                        xpath_add = "//span[contains(text(),'Backup destinations')]/ancestor::div" \
                                    "//div[contains(@class,'page-details-box-links')]/a[text()='Add']"
                        xpath_add_copy = "//span[contains(text(),'Backup destinations')]/ancestor::div" \
                                         "//div[contains(@class,'page-details-box-links')]/a[text()='Add copy']"
                        if self.__admin_console.check_if_entity_exists("xpath", xpath_add):
                            accordians = self.__driver.find_elements_by_xpath(
                                "//uib-accordion[@class='text-left']//a[descendant::label]")
                            for accordian in accordians:
                                region_name = accordian.find_element_by_xpath(
                                    ".//label").text
                                accordian.click()
                                self.__admin_console.wait_for_completion()
                                for table_row in table_rows:
                                    keyvalue_list.append(table_row.find_element_by_xpath(
                                        "./td[3]/span/a").text)
                                details.update({key: {region_name: keyvalue_list}})
                        elif self.__admin_console.check_if_entity_exists("xpath", xpath_add_copy):
                            for table_row in table_rows:
                                keyvalue_list.append(table_row.find_element_by_xpath("./td[3]/span/a").text)
                            details.update({key: keyvalue_list})
                        else:
                            for table_row in table_rows:
                                keyvalue_list.append(table_row.find_element_by_xpath("./td[3]/span/a").text)
                            details.update({key: keyvalue_list})
                elif key == "Security":
                    rows = self.__driver.find_elements_by_xpath("//*[@id='tileContent_Security']/div[1]/div/ul[2]/li")
                    temp_dict = {}
                    for row in rows:
                        nested_key = row.find_element_by_xpath("./span[1]").text
                        nested_value = row.find_element_by_xpath("./span[2]").text
                        nested_value_list = []
                        if "[Custom] - " in nested_value:
                            nested_value = nested_value[11:]
                            nested_value_list = nested_value.split(" , ")
                        else:
                            nested_value_list.append(nested_value)
                        if nested_key in temp_dict:
                            temp_dict[nested_key].append(nested_value_list)
                        else:
                            temp_dict[nested_key] = [nested_value_list]
                        temp_dict[nested_key].sort()
                    details.update({key: temp_dict})
                elif key == "RPO" or "RPO" in key:
                    list_store = elem.find_elements_by_xpath(".//ul[@class='list-style-secondary ng-scope']/li")
                    for list_value in list_store:
                        if self.__admin_console.is_element_present('.//Toggle-Control', list_value):
                            toggle_class = list_value.find_element_by_xpath(
                                './/Toggle-Control/div').get_attribute('class')
                            toggle_value = "OFF"
                            if 'enabled' in toggle_class:
                                toggle_value = "ON"
                            keyvalue_dict.update({list_value.find_element_by_xpath(".//label").text: toggle_value})
                        else:
                            keyvalue_dict.update({
                                list_value.find_element_by_xpath(
                                    "./span[1]").text: list_value.find_element_by_xpath("./span[2]").text})
                    details.update({key: keyvalue_dict})
                else:
                    values = PanelInfo(self.__admin_console, key).get_details()
                    details.update({key: values})

        self.log.info(details)
        return details

    @WebAction()
    def __dissociate_all_users_user_groups(self):
        """ Method to dissociate all users and user groups"""
        user_list = self.__driver.find_elements_by_xpath("//div[@class='users-list']/ul/li")
        for user in user_list:
            user.find_element_by_xpath(".//span[@class='delete-row']").click()

    @WebAction()
    def __dissociate_all_users(self):
        """ Method to dissociate all users """
        users = self.__driver.find_elements_by_xpath("//div[@class='users-list']//span[@class='user-type']")
        for user in users:
            user.find_element_by_xpath(".//span[@class='delete-row']").click()

    @WebAction()
    def __dissociate_all_user_groups(self):
        """ Method to dissociate all user groups """
        users = self.__driver.find_elements_by_xpath("//div[@class='users-list']//span[@class='group-type']")
        for user in users:
            user.find_element_by_xpath(".//span[@class='delete-row']").click()

    @WebAction()
    def __dissociate_specific_users_user_groups(self, user_and_group_list):
        """ Method to dissociate specific users or user groups """
        user_list = self.__driver.find_elements_by_xpath("//div[@class='users-list']/ul/li")
        for user in user_list:
            for value in user_and_group_list['Delete_Specific_user_or_group']:
                if user.find_element_by_xpath(".//span[@class='ng-binding']").text == value:
                    user.find_element_by_xpath(".//span[@class='delete-row']").click()

    @WebAction()
    def __access_rpo_edit_dialog(self):
        """ Method to access RPO edit dialog """
        self.__driver.find_element_by_xpath(
            "//a[@data-ng-click='enableEditModeForIncrementalRPO()']").click()

    @WebAction()
    def __access_snap_points_edit_dialog(self):
        """ Method to access snap recovery points edit dialog """
        if not self.__driver.find_element_by_id("retention-"
                                                "jobs-mode-for-plan").is_selected():
            self.__driver.find_element_by_id("retention-jobs-"
                                             "mode-for-plan").click()
            self.__admin_console.click_button("Yes")
        self.__driver.find_element_by_xpath(
            "//*[@class='group retention-period-options']//..//a[@data-ng-click='ctrl.toggleEdit()']").click()

    @WebAction()
    def __access_snap_copy_type_dialog(self, type="vault"):
        """ Method to access copy type edit dialog """
        if type == "vault":
            self.__driver.find_element_by_xpath("//input[@data-ng-value='obj.value'][@value='0']").click()
        else:
            self.__driver.find_element_by_xpath("//input[@data-ng-value='obj.value'][@value='1']").click()

    @WebAction()
    def __select_snap_vendor(self, snap_engine):
        """Method to select snap vendor while adding SVM Mappings"""
        self.__driver.find_element_by_id("snapVendor").send_keys(snap_engine)

    @WebAction()
    def __click_add_svm_mappings(self):
        """Method to click on adding svm mappings during secondary snap copy creation"""
        self.__driver.find_element_by_xpath("//button[@data-ng-click='svmMappingCtrl.addMappingPair()']").click()

    @WebAction()
    def __click_delete_storage_copy(self, copy_name):
        """Method to click on delete copy in a plan"""

        self.__driver.find_element_by_xpath(f'//*[@id="tileContent_Backup destinations"]//a[contains(text(), "{copy_name}")]/ancestor::td/ancestor::tr//*[@id="cv-k-grid-td-ActionCol"]').click()
        self.__admin_console.click_button_using_text('Yes')

    @WebAction()
    def __access_retention_period_edit_dialog(self):
        """Method to access retention period edit dialog"""
        if not self.__driver.find_element_by_id("retention-non-"
                                                "jobs-mode-for-plan").is_selected():
            self.__driver.find_element_by_id("retention-non-"
                                             "jobs-mode-for-plan").click()
            self.__admin_console.click_button("Yes")
        self.__driver.find_element_by_xpath("//*[@id='tileContent_Retention']//.."
                                            "//li[@class='group retention-period-options']//../"
                                            "a[@data-ng-click='ctrl.toggleEdit()']").click()

    @WebAction()
    def __access_snap_rpo_edit_dialog(self):
        """ Method to access snap recovery points edit dialog """
        self.__driver.find_element_by_xpath("//*[@id='tileContent_Snapshot options']//.."
                                            "//span[contains(text(),'Backup copy frequency (in HH:MM)')]"
                                            "//..//a[@data-ng-click='ctrl.toggleEdit()']").click()

    @WebAction()
    def __access_log_rpo_edit_dialog(self):
        """ Method to access database log rpo edit dialog """
        xpath = "//span[contains(text(),'Log backup RPO')]/ancestor::li//a[@data-ng-click='ctrl.toggleEdit()']"
        self.__driver.find_element_by_xpath(xpath).click()

    @WebAction()
    def __fill_snap_retention_period(self, retention_period):
        """Method to set the snap retention period"""
        retentionval = self.__driver.find_element_by_xpath(
            "//*[@class='group retention-period-options']//../input[@data-ng-model='uiModel.retentionInfo.value']")
        retentionval.click()
        retentionval.clear()
        retentionval.send_keys(retention_period['retention_value'])
        self.__admin_console.select_value_from_dropdown("retentionPeriodOptionsAsRadioBtn", retention_period['period'])

    @WebAction()
    def __fill_rpo(self, backup_rpo, flag_snapshot=True, flag_db_log=False):
        """ Method to fill database log rpo edit dialog """
        if flag_snapshot:
            backup_copy_label = self.__admin_console.props['label.backupCopyRPOinHHMM']
            hour_xpath = f"//span[contains(text(),'{backup_copy_label}')]//..//input[@placeholder='HH']"
            minute_xpath = f"//span[contains(text(),'{backup_copy_label}')]//..//input[@placeholder='MM']"
            save_xpath = f"//span[contains(text(),'{backup_copy_label}')]//..//" \
                         "a[@data-ng-click='ctrl.saveEdit()']"

        elif flag_db_log:
            log_backup_label = self.__admin_console.props['label.logBackupRPO']
            hour_xpath = f"//span[contains(text(),'{log_backup_label}')]/ancestor::li//input[@placeholder='HH']"
            minute_xpath = f"//span[contains(text(),'{log_backup_label}')]/ancestor::li//input[@placeholder='MM']"
            save_xpath = f"//span[contains(text(),'{log_backup_label}')]/ancestor::li//a[@data-ng-click='ctrl.saveEdit()']"

        self.__driver.find_element_by_xpath(hour_xpath).click()
        self.__driver.find_element_by_xpath(hour_xpath).clear()
        self.__driver.find_element_by_xpath(hour_xpath).send_keys(str(backup_rpo['hours']))
        self.__driver.find_element_by_xpath(minute_xpath).click()
        self.__driver.find_element_by_xpath(minute_xpath).clear()
        self.__driver.find_element_by_xpath(minute_xpath).send_keys(str(backup_rpo['minutes']))
        self.__driver.find_element_by_xpath(save_xpath).click()

    @WebAction()
    def __search_user(self, user_or_group):
        """
        Method to search for user or group in associate users dialog

        Args:
            user_or_group (str) : user or group name to be associated
        """
        self.__driver.find_element_by_xpath(
            "//input[@class='select2-input select2-default']").send_keys(user_or_group)

    @WebAction()
    def __select_user(self, user_or_group):
        """
        Method to associate users and user groups

        Args:
            user_or_group (str) : user or group name to be associated
        """
        self.__driver.find_element_by_xpath(
            f"//ul[@class='select2-results']//span[contains(text(),'({user_or_group})')]").click()
        time.sleep(2)

    @WebAction()
    def __access_allowed_feature_settings(self, data_ng_click):
        """
        Method to click on settings hyperlink for Edge drive

        Args:
            data_ng_click (str) : data-ng-click attribute of the element
        """
        settings_link = self.__driver.find_element_by_xpath(
            f"//a[@data-ng-click='{data_ng_click}']")
        settings_link.click()

    @WebAction()
    def __click_add_storage_copy(self, region=None, snap=False):
        """ Method to click on Add 'Copy' or 'Snap Copy' to add additional storage """
        if region:
            self.__driver.find_element_by_xpath(f"//label[text()='{region}']/ancestor::div[@class='panel-group']"
                                                f"//a[contains(@data-ng-click,'addStorage')]").click()
        else:
            self.__admin_console.tile_select_hyperlink('Backup destinations', 'Add')
            if snap:
                self.__table.access_toolbar_menu('AddSnapCopy')
            else:
                self.__table.access_toolbar_menu('AddCopy')

    @WebAction()
    def __click_edit_storage_action_item(self, storage_name, region=None):
        """
        Method to click on edit storage action item for existing storage

        Args:
            storage_name (str)  : name of storage to be edited
        """
        if region:
            self.__driver.find_element_by_xpath(
                f"//label[text()='{region}']/ancestor::div[@class='panel-group']"
                f"//div[contains(@class,'plan-storage-grid')]//td[@role='gridcell']/a[text()='{storage_name}']")\
                .click()
        else:
            self.__driver.find_element_by_xpath(
                f"//div[contains(@class,'plan-storage-grid')]//td[@role='gridcell']/a[text()='{storage_name}']")\
                .click()

    @WebAction()
    def __edge_drive_settings(self, quota):
        """
        Method to set edge drive settings

        Args:
            quota (str) : quota to be set for edge drive
        """

        self.__admin_console.checkbox_select("isEdgeDriveQuotaEnabled")
        self.__admin_console.fill_form_by_id("edgeDriveQuota", quota)
        self.__modal_panel.submit()

    def __edit_plan_storage_ret_period(self, ret_value):
        """ """
        elem = self.__driver.find_element_by_xpath("//input[contains(@class,'policy-retention')]")
        elem.clear()
        elem.send_keys(ret_value)

    @WebAction()
    def __edit_snap_recovery_point(self, snap_recovery_points):
        """Edits number snap recovery points"""
        if self.__admin_console.check_if_entity_exists(
                "xpath", "//*[@id='cv-k-grid-td-StoragePolicyCopy.copyName']/a[contains(text(),'snap copy')]"):
            self.__driver.find_element_by_xpath(
                "//*[@id='cv-k-grid-td-StoragePolicyCopy.copyName']/a[contains(text(),'snap copy')]").click()
            self.__admin_console.wait_for_completion()
            self.__driver.find_element_by_id("retention-jobs-mode").click()
            self.__admin_console.fill_form_by_id("snapRecoveryPoints", snap_recovery_points)
            self.__admin_console.submit_form()
        else:
            self.__driver.find_element_by_xpath(
                "//div[@value='snapshotOptions.snapRecoveryPoints']//a[@data-ng-click='ctrl.toggleEdit()']").click()
            self.__driver.find_element_by_id('retention-jobs-mode').click()
            self.__admin_console.fill_form_by_id("snapRecoveryPoints", snap_recovery_points)
            self.__driver.find_element_by_xpath(
                "//div[contains(@value,'snapRecovery')]//a[@data-ng-click='ctrl.saveEdit()']").click()

    @PageService()
    def plan_info(self, plan_type):
        """
        Retrieves all plan information displayed from Plan Details Section

        Args:
            plan_type : Type of plan to be fetched details of (Laptop/Server)

        Returns:
            plan_details dict() : dictionary containing plan information displayed in UI
        """
        plan_details = self.__extract_plan_info(plan_type)
        return plan_details

    @PageService()
    def is_copy_present(self, copy):
        """
        Checks if provided copy is present inside plan

        Args:
            copy :(str) Name of copy

        Returns:
            boolean: True/False based on if copy is present
        """
        return self.__check_if_copy_present(copy)

    @PageService()
    def edit_plan_storage_pool(self, additional_storage, edit_storage, region=None):
        """
        Method to edit storage pool associated with the plan

        Args:
            additional_storage (dictionary): Dictionary containing values for an additional
                storage, if it is to be added and the values of it
                Eg. - additional_storage = {'Add':False, 'storage_name':'Store',
                                                    'storage_pool':'Secondary',
                                                    'ret_period':'10'}
            edit_storage (dictionary): Dictionary containing values to edit existing storage
                Eg. - edit_plan_storage = {'Edit': True,
                                          'old_storage_name': 'Primary',
                                          'new_storage_name':'New Primary',
                                          'new_ret_period':'20'}
            region (string): region name
        """
        xpath_region = "//a[contains(text(),'Configure region based storage')]"
        xpath_add = "//span[contains(text(),'Backup destinations')]/ancestor::div//a[contains(text(),'Add')]"
        if region and self.__admin_console.check_if_entity_exists("xpath", xpath_region):
            self.__driver.find_element_by_xpath(xpath_region).click()
            if additional_storage:
                self.__additional_storage(additional_storage)
            if edit_storage:
                self.__edit_storage(edit_storage)
            self.__panel_dropdown_obj.select_drop_down_values(values=[region], index=0)
            self.__admin_console.submit_form()
        elif region and region == self.__driver.find_element_by_xpath("//uib-accordion//label").text:
            if additional_storage:
                self.__additional_storage(additional_storage, region)
            if edit_storage:
                self.__edit_storage(edit_storage, region)
        elif self.__admin_console.check_if_entity_exists("xpath", xpath_add):
            if additional_storage:
                self.__additional_storage(additional_storage)
            if edit_storage:
                self.__edit_storage(edit_storage)

    @PageService()
    def __additional_storage(self, additional_storage, region=None):
        """Add additional storage"""
        self.__click_add_storage_copy(region)
        self.__admin_console.wait_for_completion()
        self.__admin_console.fill_form_by_id("storageName", additional_storage['storage_name'])
        self.__panel_dropdown_obj.select_drop_down_values(values=[additional_storage['storage_pool']],
                                                          drop_down_id="storage")
        self.__admin_console.submit_form()

    @PageService()
    def __edit_storage(self, edit_storage, region=None):
        """Edit name of storage"""
        self.__click_edit_storage_action_item(edit_storage['old_storage_name'], region)
        self.__admin_console.wait_for_completion()
        self.__admin_console.fill_form_by_id("storageName", edit_storage['new_storage_name'])
        self.__admin_console.submit_form()

    @PageService()
    def edit_server_plan_storage_pool(self, additional_storage, edit_storage, snap=False):
        """
        Method to edit storage pool associated with the plan or Add additional Storage

        Args:
            additional_storage (dictionary): Dictionary containing values for an additional
                storage, if it is to be added and the values of it
                Eg. - additional_storage = {'Add':False, 'storage_name':'Store',
                                                    'storage_pool':'Secondary',
                                                    'ret_period':'10'}
            edit_storage (dictionary): Dictionary containing values to edit existing storage
                Eg. - edit_plan_storage = {'Edit': True,
                                          'old_storage_name': 'Primary',
                                          'new_storage_name':'New Primary',
                                          'new_ret_period':'20'}
        """
        if additional_storage and snap:
            self.__click_add_storage_copy(snap=snap)
            self.__admin_console.wait_for_completion()
            self.__admin_console.fill_form_by_id("storageName", additional_storage['storage_name'])
            self.__access_snap_copy_type_dialog(additional_storage['copy_type'])
            self.__driver.find_element_by_id("sourceCopy").send_keys(additional_storage['source_copy'])
            self.__panel_dropdown_obj.select_drop_down_values(
                drop_down_id='storage',
                values=[additional_storage['storage_pool']]
            )
            if additional_storage['snap_engine'] in ["NetApp", "Amazon"]:
                self.__driver.find_element_by_xpath("//a[@id='AddSVMMapping']").click()
                self.__admin_console.wait_for_completion()
                if additional_storage['default_replica']:
                    self.__select_snap_vendor(additional_storage['snap_engine'])

                self.__panel_dropdown_obj.select_drop_down_values(
                    drop_down_id='sourceSVM', values=[additional_storage['mappings']['src_svm']])
                self.__panel_dropdown_obj.select_drop_down_values(
                    drop_down_id='targetSVM', values=[additional_storage['mappings']['dest_svm']])
                self.__click_add_svm_mappings()
                self.__admin_console.wait_for_completion()
                self.__admin_console.submit_form()

            self.__admin_console.submit_form()

        if additional_storage and not snap:
            self.__click_add_storage_copy()
            self.__admin_console.wait_for_completion()
            self.__admin_console.fill_form_by_id("storageName", additional_storage['storage_name'])
            self.__panel_dropdown_obj.select_drop_down_values(
                drop_down_id='storage',
                values=[additional_storage['storage_pool']]
            )
            self.__admin_console.submit_form()

        if edit_storage:
            self.__admin_console.select_hyperlink(edit_storage['old_storage_name'])
            self.__admin_console.wait_for_completion()

            if edit_storage['new_storage_name']:
                self.__admin_console.fill_form_by_id("storageName", edit_storage['new_storage_name'])

            if edit_storage['new_ret_period']:
                self.__edit_plan_storage_ret_period(edit_storage['new_ret_period'])
            self.__admin_console.submit_form()

    @PageService()
    def edit_plan_alerts(self, update_alerts):
        """
        Method to edit alerts associated with the plan

        Args:
            update_alerts (dict): dictionary containing alert details,
                    which alerts to enable/disable.
                Eg. - update_alerts = {"Backup" :None,
                                       "Jobfail": "Restore Job failed",
                                       "edge_drive_quota":None}
        """

        PanelInfo(self.__admin_console, 'Alerts').edit_tile()
        self.__plans.set_alerts(update_alerts, edit_alert=True)
        self.__modal_panel.submit()

    @PageService()
    def edit_plan_network_resources(self, throttle_send, throttle_receive):
        """
        Method to edit  network resources associated with the plan

        Args:
            throttle_send (str or int): Network resource value
            throttle_receive (str or int): Network resource value
        """
        PanelInfo(self.__admin_console, 'Network resources').edit_tile()
        self.__plans.set_network_resources(throttle_send, throttle_receive)
        self.__modal_panel.submit()

    @WebAction()
    def edit_plan_rpo_hours(self, rpo_hours):
        """
        Method to edit the RPO (Backup SLA) associated with the plan

        Args:
            rpo_hours (string): Value of RPO hours for the plan
        """
        self.__access_rpo_edit_dialog()
        self.__admin_console.wait_for_completion()
        self.__admin_console.fill_form_by_id("rpo", rpo_hours)
        self.__driver.find_element_by_xpath("//*[@id='tileContent_RPO']//a[contains(@class,'save-frequency')]").click()

    @WebAction()
    def edit_plan_rpo_start_time(self, start_time):
        """
        Method to edit the RPO start time associated with the plan

        Args:
             start_time (str):  Start time of the plan in format: (%I:%M %p) eg. "07:00 AM"
        """
        self.__access_rpo_edit_dialog()
        self.__admin_console.wait_for_completion()
        hours_minutes, session = start_time.split(" ")
        self.__admin_console.date_picker(time_value={'date': "",
                                                     'hours': hours_minutes.split(":")[0],
                                                     'mins': hours_minutes.split(":")[1],
                                                     'session': session}, pick_time_only=True)
        self.__driver.find_element_by_xpath("//*[@id='tileContent_RPO']//a[contains(@class,'save-frequency')]").click()

    @WebAction()
    def edit_plan_full_backup_rpo(self, start_time=None, start_days=None, enable=True):
        """Method to edit Full backup RPO settings
            Args:
                start_days  (list): Day(s) on which full backup is to be done weekly
                start_time  (str):  start time of schedule
                enable      (bool): True if full backup schedule is to be enabled
        """
        if not enable:
            self.__admin_console.disable_toggle(index=0)
        else:
            self.__admin_console.disable_toggle(index=0)
            self.__admin_console.enable_toggle(index=0)
            self.__admin_console.wait_for_completion()
            if start_days:
                self.__panel_dropdown_obj.select_drop_down_values(drop_down_id="isteven-multi-select2",
                                                                  values=start_days)
            hours_minutes, session = start_time.split(" ")
            self.__admin_console.date_picker(time_value={'date': "",
                                                         'hours': hours_minutes.split(":")[0],
                                                         'mins': hours_minutes.split(":")[1],
                                                         'session': session}, pick_time_only=True)
            self.__driver.find_element_by_xpath("//*[@id='tileContent_RPO']//a[contains(@class,'save-frequency')]").click()

    @WebAction()
    def edit_plan_options(self, file_system_quota):
        """
        Method to edit options (file system quota) associated with the plan

        Args:
            file_system_quota (string): value of upper limit of File system quota for plan

        Returns:
            None

        Raises:
            Exception:
                None
        """

        PanelInfo(self.__admin_console, 'Options').edit_tile()

        if file_system_quota:
            self.__admin_console.checkbox_select('isQuotaEnabled')
            self.__admin_console.fill_form_by_id("quota", file_system_quota)
        else:
            self.__admin_console.checkbox_deselect('isQuotaEnabled')

        self.__modal_panel.submit()

    @WebAction()
    def edit_plan_backup_content(self,
                                 file_system,
                                 content_backup,
                                 content_library,
                                 custom_content,
                                 exclude_folder,
                                 exclude_folder_library,
                                 exclude_folder_custom_content):
        """
        Method to edit backup content associated with the plan

        Args:
            file_system (list): Type of operating system (Windows/Unix/Mac)
                Eg. - file_system = ["Mac", "Unix", "Windows"]
            content_backup (list): List with folder names to be selected/unselected
                            for content backup
                Eg. - content_backup:["Content Library", "Documents", "Desktop",
                                        "Office", "Music", "Pictures", "Videos"]
            content_library (list): List with content library values to be
                            selected/unselected for backup
                Eg. - content_library = ['Content Library', 'GoogleDrive', 'Image',
                                         'Office', 'OneDrive', 'System',
                                        'Temporary Files', 'Text',
                                        'Thumbnail Supported', 'Video',
                                        'iCloudDrive']
            custom_content (str): path for custom(extra) content to be backed up
            exclude_folder (list): List with folder names to be excluded(or not)
                            from backup
            exclude_folder_library (list): List with content library folders to be
                                excluded(or not) from backup
            exclude_folder_custom_content (str): path for custom(extra) content to be
                                            excluded
        Returns:
            None

        Raises:
            Exception:
                None
        """

        PanelInfo(self.__admin_console, 'Backup content').edit_tile()

        for value in file_system:
            self.__driver.find_element_by_xpath("//ul[@class='nav nav-tabs nav-justified']"
                                                f"//a[contains(text(),'{value}')]").click()
            self.__admin_console.wait_for_completion()

            self.__plans.backup_content_selection(
                content_backup,
                content_library,
                custom_content)

            self.__plans.exclusion_content_selection(
                exclude_folder,
                exclude_folder_library,
                exclude_folder_custom_content)

        self.__modal_panel.submit()

    @PageService()
    def edit_plan_associate_users_and_groups(self, user_or_group):
        """
        Method to edit users/user groups associated with the plan

        Args:
            user_or_group (list): List of Users or user groups to be associated to the plan

                    Eg.- user_or_group = ["Media",
                                          "Tester1",
                                          "TestNew2",
                                          "View All"]
        """

        PanelInfo(self.__admin_console, 'Associated users and user groups').edit_tile()
        for value in user_or_group:
            self.__search_user(value)
            self.__admin_console.wait_for_completion()
            self.__select_user(value)

        self.__admin_console.click_button('Preview')
        if self.__admin_console.check_if_entity_exists(
                "xpath", "//button[@class='btn btn-primary cvBusyOnAjax']"):
            self.__modal_panel.submit()

        elif self.__admin_console.check_if_entity_exists(
                "xpath", "//h4[contains(text(),'Affected client list')]"):
            self.__admin_console.click_button('Yes')

    @PageService()
    def edit_plan_features(self, allowed_features, archiving_rules, quota=None, media_agent=None):
        """
        Method to edit plan features associated with the plan

        Args:
            allowed_features (dict)   :     dictionary containing features to be enabled for the
                             plan

                Eg. - allowed_features = {"Laptop": "ON", "Edge Drive": "ON", "DLP": "ON",
                                         "Archiving": "ON"}
            archiving_rules (dict)    :     dictionary containing values for Archive Feature rules

                Eg. - archiving_rules = {"do_edit": True, "start_clean": "40", "stop_clean": "90",
                                        "file_access_time": "85", "file_modify_time": None,
                                        "file_create_time": "2", "archive_file_size": None,
                                        "max_file_size": None, "archive_read_only": True,
                                        "replace_file": None, "delete_file": None}

            quota (str)             :       variable containing value for Edge Drive quota

            media_agent ()          :       media agent to be used for edge drive
        """

        if allowed_features["Edge Drive"] == "ON":
            if not self.__admin_console.check_if_entity_exists("xpath", "//a[@data-ng-click='editEdgeDrive()']"):
                PanelInfo(self.__admin_console, 'Allowed features').enable_toggle('Edge Drive')
                self.__admin_console.select_value_from_dropdown(media_agent)
                if quota:
                    self.__edge_drive_settings(quota)
            else:
                if quota:
                    self.__access_allowed_feature_settings("editEdgeDrive()")
                    self.__admin_console.wait_for_completion()
                    self.__edge_drive_settings(quota)

        if allowed_features["DLP"] == "ON":
            PanelInfo(self.__admin_console, 'Allowed features').enable_toggle('DLP')
        else:
            PanelInfo(self.__admin_console, 'Allowed features').disable_toggle('DLP')

        if allowed_features["Archiving"] == "ON":
            if not self.__admin_console.check_if_entity_exists("xpath", "//a[@data-ng-click='editArchiving()']"):
                PanelInfo(self.__admin_console, 'Allowed features').enable_toggle('Archiving')
                if archiving_rules:
                    self.__plans.set_archiving_rules(archiving_rules)
                    self.__modal_panel.submit()
                else:
                    self.__modal_panel.submit()
            else:
                if archiving_rules:
                    self.__access_allowed_feature_settings("editArchiving()")
                    self.__plans.set_archiving_rules(archiving_rules)
                    self.__modal_panel.submit()
        else:
            PanelInfo(self.__admin_console, 'Allowed features').disable_toggle('Archiving')
            self.__admin_console.checkbox_select("disable-archiving-input")
            self.__admin_console.click_button('Save')

    @PageService()
    def remove_associated_users_and_groups(self, user_user_group_de_association):
        """
        Method to remove users/user groups associated to a plan

        Args:
            user_user_group_de_association (dictionary) : Dictionary containing users or users groups to be
                                        de-associated from a plan (if All or just Users or just user groups)

                Eg. - "user_user_group_de_association" : {"DeleteAll": true,
                                                          "DeleteAllUsers": false,
                                                          "DeleteAllUserGroups": false,
                                                          "Delete_Specific_user_or_group": false}
        """
        PanelInfo(self.__admin_console, 'Associated users and user groups').edit_tile()
        if user_user_group_de_association['DeleteAll']:
            self.__dissociate_all_users()
        else:
            if user_user_group_de_association['DeleteAllUsers']:
                self.__dissociate_all_users()

            if user_user_group_de_association['DeleteAllUserGroups']:
                self.__dissociate_all_user_groups()

            if user_user_group_de_association['Delete_Specific_user_user_or_group']:
                self.__dissociate_specific_users_user_groups(
                    user_user_group_de_association['Delete_Specific_user_user_or_group'])

        self.__admin_console.click_button('Preview')

        if self.__admin_console.check_if_entity_exists(
                "xpath", "//h4[contains(text(),'Affected client list')]"):
            self.__admin_console.click_button('Yes')

        elif self.__admin_console.check_if_entity_exists(
                "xpath", "//button[@class='btn btn-primary cvBusyOnAjax']"):
            self.__modal_panel.submit()

    @PageService()
    def edit_plan_retention(self, retention):
        """
        Method to edit retention properties for a plan

        Args:
            retention (dict) : dictionary containing retention attributes for laptop plan
                Eg. - retention = {'deleted_item_retention': {'value': '5', 'unit': 'day(s)'},
                                   'file_version_retention': {'duration': {'value': '4',
                                                                          'unit': 'day(s)'},
                                                              'versions': '5',
                                                              'rules': {'days': '4',
                                                                        'weeks': '5',
                                                                        'months': '6'}}}
                    OR
                        retention = {'deleted_item_retention': {'value': '5', 'unit': 'day(s)'},
                                       'file_version_retention': {'duration': None,
                                                                  'versions': None,
                                                                  'rules': {'days': '4',
                                                                            'weeks': '5',
                                                                            'months': '6'}}}
        """
        PanelInfo(self.__admin_console, 'Retention').edit_tile()
        self.__plans.set_retention(retention)
        self.__modal_panel.submit()

    @PageService()
    def edit_plan_override_restrictions(self, allow_override, override_laptop_plan=False):
        """
        Method to edit override restrictions

        Args:
            allow_override (dictionary): dictionary containing values for Override parameters
                Eg. - allow_override = {"Storage_pool": "Override required",
                                        "RPO": "Override optional",
                                        "Folders_to_backup": "Override not allowed"}
            override_laptop_plan (boolean) = To check if it is laptop plan type
        """
        PanelInfo(self.__admin_console, 'Override restrictions').edit_tile()
        self.__plans.set_override_options(allow_override, edit_override_laptop=override_laptop_plan)
        self.__modal_panel.submit()

    @WebAction()
    def data_classication_plan_entities(self):
        """
        Returns a list of all the entities selected for a data classification plan
        """
        self.log.info("Clicking on the entities to view all")
        self.__driver.find_element_by_xpath('//*[@id="entities"]/span/button').click()
        entities = self.__driver.find_elements_by_xpath(
            '//div[@class="checkBoxContainer"]/div/div[@class="acol"]\
            //input/../span')
        entities_list = []
        for entity_id in range(1, len(entities) + 1):
            xpath_value = '//div[@class="checkBoxContainer"]/div[%d]\
            /div[@class="acol"]//input[@checked="checked"]/../span' % entity_id
            if self.__admin_console.check_if_entity_exists('xpath', xpath_value):
                entities_list.append(str(self.__driver.find_element_by_xpath(
                    xpath_value).text).strip())

        self.log.info("Entities found are: %s" %entities_list)
        return entities_list

    @PageService()
    def edit_snapshot_options(self, backup_rpo, snap_recovery_points=None, retention_period=None):
        """
        Method to edit snapshot options for server plan details

         Args:
            snap_recovery_points (string): value of number of snap recovery points
            backup_rpo (dictionary): dictionary containing backup_rpo value
                Eg. - backup_rpo = {"hours": "10", "minutes": "20"}
            retention_period (dictionery): dictionery containing retention value and period
                Eg. - retention_period = {"retention_value": "1", "period" : "Day(s)"}
        """
        if snap_recovery_points:
            self.__edit_snap_recovery_point(snap_recovery_points)
            self.__admin_console.wait_for_completion()
            self.__access_snap_rpo_edit_dialog()
            self.__fill_rpo(backup_rpo, flag_snapshot=True)
            self.__admin_console.wait_for_completion()
        else:
            self.__access_retention_period_edit_dialog()
            self.__admin_console.wait_for_completion()
            self.__fill_snap_retention_period(retention_period)

    @PageService()
    def edit_database_options(self, backup_rpo):
        """
        Method to edit database options for server plan details

         Args:
            backup_rpo (dictionary): dictionary containing backup_rpo value
                Eg. - backup_rpo = {"hours": "10", "minutes": "20"}
        """

        self.__access_log_rpo_edit_dialog()
        self.__admin_console.wait_for_completion()
        self.__fill_rpo(backup_rpo, flag_snapshot=False, flag_db_log=True)
        self.__admin_console.wait_for_completion()

    @PageService()
    def delete_server_plan_storage_copy(self, copy_name):
        """
        Method to delete storage copy in a server plan

        Args:
            copy_name (string): name of the copy
        """

        self.__click_delete_storage_copy(copy_name)
        self.__admin_console.wait_for_completion()