# -*- coding: utf-8 -*-
# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# -------------------------------------------------------------------------

"""
Main file for performing Operation window (Admin Console) related operations

Classes:

    OperationWindow()

Functions:

    add_operation_rule           :    Method to create Backup rule

    edit_operation_rule          :    Method to modify the operation rule

    delete_operation_rule        :    Method to Delete operation rule

    fetch_operation_rule_details :  Method to extract operation rule details from UI
"""

import datetime
from datetime import date

from Web.Common.page_object import (WebAction, PageService)
from Web.AdminConsole.Components.table import Table
from Web.AdminConsole.Components.panel import DropDown


class OperationWindow:
    """ Class for the Backup Window page """

    def __init__(self, admin_page):
        """
        Method to initiate Operation Window class

        Args:
            admin_page   (Object) :   Admin Page Class object
        """
        self.__admin_console = admin_page
        self.__admin_console._load_properties(self)
        self.__driver = admin_page.driver
        self.__table = Table(self.__admin_console)
        self.__drop_down = DropDown(self.__admin_console)

    @PageService()
    def __select_backup_operations(self, backup_operations):
        """
        Method to select backup operations for operation window

        Args:
            backup_operations (dict): dictionary with operations to be enabled/disabled
                                    for operation window
                Eg.- backup_operations = {'Full':1,
                                          'Incremental and Differential':1,
                                          'Synthetic Full':1}
        """

        if backup_operations.get('Full', None):
            if backup_operations['Full'] == '1':
                self.__admin_console.checkbox_select('Full')
            else:
                self.__admin_console.checkbox_deselect('Full')

        if backup_operations.get('Incremental and Differential', None):
            if backup_operations['Incremental and Differential'] == '1':
                self.__admin_console.checkbox_select('Incremental and Differential')
            else:
                self.__admin_console.checkbox_deselect('Incremental and Differential')

        if backup_operations.get('Synthetic Full', None):
            if backup_operations['Synthetic Full'] == '1':
                self.__admin_console.checkbox_select('Synthetic Full')
            else:
                self.__admin_console.checkbox_deselect('Synthetic Full')

    @WebAction()
    def __select_no_run_days(self, no_run_days):
        """
        Method to select no run days for operation window

        Args:
            no_run_days (dict): dictionary with days for which the operation
                                rule configured will not be running
                Eg.- no_run_days = {'Monday':1,
                                    'Wednesday':1,
                                    'Friday':1,
                                    'Sunday':1}
        """
        no_run_days1 = dict.fromkeys(['Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday', 'Sunday'], 0)
        no_run_days1.update(no_run_days)
        for key, value in no_run_days1.items():
            if value == 1:
                if not self.__driver.find_element_by_id(
                        "" + key + "").is_selected():
                    self.__driver.find_element_by_xpath(
                        "//label[contains(text(),'" + key + "')]").click()
            else:
                if self.__driver.find_element_by_id(
                        "" + key + "").is_selected():
                    self.__driver.find_element_by_xpath(
                        "//label[contains(text(),'" + key + "')]").click()

    @WebAction()
    def __select_no_run_time(self, no_run_time):
        """
        Method to select no run time for operation window

        Args:
            no_run_time (dict): dictionary with intervals for which the
                                operation rule not to be run
                Eg.- no_run_time = {'0':1, '1':1, '2':1, '3':1,
                                    '4':1, '14':1, '15':1, '16':1,
                                    '17':1, '18':1}
        """
        no_run_time1 = dict.fromkeys(['0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '10',
                                      '11', '12', '13', '14', '15', '16', '17', '18', '19', '20',
                                      '21', '22', '23', '24'], 0)
        no_run_time1.update(no_run_time)
        for key, value in no_run_time1.items():
            if value == 1:
                if self.__admin_console.check_if_entity_exists(
                        "xpath", "//td[@id='" + key + "' and @class='ng-scope']"):
                    self.__driver.find_element_by_xpath(
                        "//tr[@id='TIME']/td[@id='" + key + "']").click()
            else:
                if self.__admin_console.check_if_entity_exists(
                        "xpath", "//td[@id='" + key + "' and @class='ng-scope eng-selected-item']"):
                    self.__driver.find_element_by_xpath(
                        "//tr[@id='TIME']/td[@id='" + key + "']").click()

    @WebAction()
    def __select_calendar_current_date(self):
        """ Method to select current date from Date Picker """
        self.__driver.find_element_by_xpath(
            "//label[@for='startDate']/..//button").click()
        self.__admin_console.wait_for_completion()
        self.__driver.find_element_by_xpath(
            "//button[contains(@class,'btn btn-default btn-sm') "
            "and contains(@class,'active')]").click()
        self.__admin_console.wait_for_completion()

    @WebAction()
    def __select_calendar_succeeding_date(self):
        """ Method to select tomorrow's date from Date Picker """
        self.__driver.find_element_by_xpath(
            "//label[@for='endDate']/..//button").click()
        self.__admin_console.wait_for_completion()
        self.__driver.find_element_by_xpath(
            "//button[contains(@class,'btn btn-default btn-sm') and "
            "contains(@class,'active')]/../following-sibling::td/button").click()
        self.__admin_console.wait_for_completion()

    @WebAction()
    def __set_start_date(self, start_date):
        """
        Method to select start date

        Args:
            start_date (str) : start date to be selected for operation rule
        """
        tomorrow = date.today() + datetime.timedelta(days=1)
        if start_date == date.today().strftime("%m/%d/%Y"):
            self.__select_calendar_current_date()
        elif start_date == tomorrow.strftime("%m/%d/%Y"):
            self.__select_calendar_succeeding_date()
        else:
            self.__admin_console.fill_form_by_id("startDate", start_date)

    @WebAction()
    def __set_end_date(self, end_date):
        """
        Method to select start date

        Args:
            end_date (str) : end date to be selected for operation rule
        """
        tomorrow = date.today() + datetime.timedelta(days=1)
        if end_date == date.today().strftime("%m/%d/%Y"):
            self.__select_calendar_current_date()
        elif end_date == tomorrow.strftime("%m/%d/%Y"):
            self.__select_calendar_succeeding_date()
        else:
            self.__admin_console.fill_form_by_id("endDate", end_date)

    @WebAction()
    def __clear_text_field(self, element_id):
        """
        Method to clear text field

        Args:
            element_id (str) : id of the WebElement to be cleared
        """
        self.__driver.find_element_by_id(element_id).clear()

    @PageService()
    def __select_start_end_dates(self, start_date, end_date):
        """
        Method to select start sate and end date for operation window

        Args:
            start_date (str)    :   Start date of the operation rule

            end_date (str)      :   End date of the operation rule

        Returns:
            None
        """
        if start_date or end_date:
            self.__admin_console.enable_toggle(0)
            if start_date:
                self.__set_start_date(start_date)
            else:
                self.__clear_text_field("startDate")

            if end_date:
                self.__set_end_date(end_date)
            else:
                self.__clear_text_field("endDate")
        else:
            self.__admin_console.disable_toggle(0)

    @PageService()
    def add_operation_rule(
            self,
            operation_rule_name,
            start_date,
            end_date,
            no_run_days,
            no_run_time,
            backup_operations=None,
            company_name=None):
        """
        Method to create a new Backup Rule

        Args:
            operation_rule_name (str)   : Name for operation rule to be created

            backup_operations (dict)    : Backup operations to be enabled/disabled for
                                          operation window
                Eg. - backup_operations = {'Full': '1',
                                           'Incremental and Differential': '0',
                                           'Synthetic Full': '1'}

            start_date (str)            : Start date of the operation rule

            end_date (str)              : End date of the operation rule

            no_run_days (dict)          : dictionary with days for which the operation rule
                                          configured will not be running
                Eg.- no_run_days = {'Monday':1,
                                    'Wednesday':1,
                                    'Friday':1,
                                    'Sunday':1}

            no_run_time (dict)          : dictionary with intervals for which the operation rule not
                     to be run
                Eg.- no_run_time = {'0':1, '1':1, '2':1, '3':1,
                                    '4':1, '14':1, '15':1, '16':1,
                                    '17':1, '18':1}

            company_name (str)          : Name of the company to be associated with operation rule

        Returns:
            None

        Raises:
            Exception:
                if failed to create a operation rule
        """

        self.__admin_console.select_hyperlink("Add")
        self.__admin_console.fill_form_by_id("operationRuleName", operation_rule_name)

        if company_name:
            self.__drop_down.select_drop_down_values(0, company_name.split(','))

        if backup_operations:
            self.__select_backup_operations(backup_operations)
        self.__select_no_run_days(no_run_days)
        self.__select_no_run_time(no_run_time)
        self.__select_start_end_dates(start_date, end_date)

        self.__admin_console.submit_form()
        self.__admin_console.check_error_message()

    @PageService()
    def edit_operation_rule(
            self,
            operation_rule_name,
            new_operation_rule_name=None,
            company_name=None,
            backup_operations=None,
            new_start_date=None,
            new_end_date=None,
            no_run_days=None,
            no_run_time=None):
        """
        Method to modify an existing Backup Rule

        Args:

            operation_rule_name (str)    : Name of the existing operation rule

            new_operation_rule_name (str): New name for operation rule

            company_name (str)           : Name of the company to be associated with operation rule

            backup_operations (dict)     : Backup operations to be enabled/disabled for
                                          operation window
                Eg. - backup_operations = {'Full': '1',
                                           'Incremental and Differential': '0',
                                           'Synthetic Full': '1'}

            new_start_date (str)         : new start date of the operation rule

            new_end_date (str)           : new end date of the operation operation rule

            no_run_days (dict)           : dictionary with days for which the operation
                                           rule configured will not be running
                Eg.- no_run_days = {'Monday':1, 'Wednesday':1,
                                    'Friday':1, 'Sunday':1}

            no_run_time (dict)           : dictionary with intervals for which the operation rule
                                           not to be run
                Eg.- operation_time = {'0':1, '1':1, '2':1, '3':1, '4':1, '14':1, '15':1, '16':1,
                                                    '17':1, '18':1}
                    Time duration for which the back up is to be run
                    eg. 0 : 12:00 - 01:00 AM
                        1 : 01:00 - 02:00 AM etc.

        Raises:
            Exception:
                if failed to modify a operation rule

        """
        self.__table.access_link(operation_rule_name)

        if new_operation_rule_name:
            self.__admin_console.fill_form_by_id("operationRuleName", new_operation_rule_name)

        if company_name:
            self.__drop_down.select_drop_down_values(0, company_name.split(','))

        if backup_operations:
            self.__select_backup_operations(backup_operations)

        if no_run_days:
            self.__select_no_run_days(no_run_days)

        if no_run_time:
            self.__select_no_run_time(no_run_time)

        self.__select_start_end_dates(new_start_date, new_end_date)

        self.__admin_console.submit_form()
        self.__admin_console.check_error_message()

    @PageService()
    def delete_operation_rule(self, operation_rule_name):
        """
        Method to delete a Operation Rule

        Args:
            operation_rule_name (str)   :   name of the Backup rule to be deleted

        Raises:
            Exception:
                if failed to delete a operation rule
        """

        self.__table.access_context_action_item(operation_rule_name, 'Delete')
        self.__admin_console.click_button('Yes')
        self.__admin_console.check_error_message()

    @WebAction()
    def fetch_operation_rule_details(self, operation_rule_name):
        """
        Method to extract operation rule details from UI

        Args:
            operation_rule_name (str) : name of the operation rule, details to be fetched for

        Returns:
            operation_rule_details (dict) : dictionary containing operation rule values displayed in UI
                Eg. - operation_rule_details = {'Name': ['Server plan backup window'], 'Company': ['CommCell']}
        """
        self.__table.apply_filter_over_column('Name', operation_rule_name)
        operation_rule_details = self.__table.get_table_data()
        return operation_rule_details
