"""
This module provides the function or operations that can be performed on
the Navigation Preference window on the AdminConsole

Class:

    NavigationPreferences() -> AdminConsoleBase() -> object()

Functions:

    get_commcell_admin_entities         :   Returns all the commcell admin checkbox entities

    get_navigation_names                :   Returns all the names in navigation grid

    get_tenant_admin_entities           :   Returns all the tenant admin checkbox entities

    maximize_all_plus_squared_buttons   :   Expands all the navigation entities

    reset_to_default                    :   Resets all the entities to Default values

    save_changes                        :   Saves all the updated entities
"""
import random
import time

from Web.Common.page_object import WebAction, PageService
from selenium.common.exceptions import NoSuchElementException


class Navigation:
    """ Class for Navigation Preference page of AdminConsole """
    def __init__(self, admin_console):
        self.__admin_console = admin_console
        self.__driver = admin_console.driver
        self.log = self.__admin_console.log

    @WebAction()
    def __maximize_plus_square(self):
        """ Clicks the maximize all plus squared button """
        try:
            self.__driver.find_element_by_xpath("//div[contains(@class,'ui-grid-icon-plus-squared')]").click()
        except NoSuchElementException:
            self.log.info("Already expanded")

    @WebAction()
    def __entity(self, entity_id):
        """ Fetches the checkbox entity
        Args:
            entity_id: Id of the entity
        """
        return self.__driver.find_element_by_xpath(f"//div/label[contains(@for,'{entity_id}')]")

    @WebAction()
    def __nav_details(self):
        """ Fetches all the Navigation Preferences window details """
        xpath = "//div[contains(@class,'ui-grid-render-container-body')]" \
                "//div[contains(@class,'ui-grid-tree-header-row')]"
        all_rows = self.__driver.find_elements_by_xpath(xpath)
        entity_dict = {}
        for row in all_rows:
            entity_dict.update(
                {row.find_element_by_xpath(".//div[contains(@class,'ui-grid-cell-contents')]").text: [
                    row.find_element_by_xpath(".//label[contains(@for,'CommcellAdmin')]").get_attribute('for'),
                    row.find_element_by_xpath(".//label[contains(@for,'TenantAdmin')]").get_attribute('for')
                ]})
        return entity_dict

    @WebAction()
    def __navigation_names(self):
        """ Fetches all the names in navigation grid """
        return self.__driver.find_elements_by_xpath("//*[@class='ui-grid-cell-contents ng-binding ng-scope']")

    @WebAction()
    def __save(self):
        """ Clicks the Save Button in Navigation Preferences window """
        self.__driver.find_element_by_xpath("//span/a[contains(text(),'Save')]").click()

    @WebAction()
    def __reset_to_default(self):
        """ Clicks the Reset to Default Button in Navigation Preferences window """
        self.__driver.find_element_by_xpath("//*[a[contains(text(),'Reset to default')]]").click()

    @WebAction()
    def __checkbox_deselect(self, checkbox_id):
        """
        Deselects checkbox that matches the ID
        Args:
            checkbox_id  (str)  -- id of the checkbox from dev or input tag
        """
        xp = f"//*[@id = '{checkbox_id}']"
        chkbox = self.__driver.find_element_by_xpath(xp)
        if chkbox.is_selected() or chkbox.get_attribute("data-state") == 'checked' or \
                'partial-selection' not in chkbox.get_attribute('class'):
            xpath = f"//*[@id = '{checkbox_id}']/following-sibling::label"
            self.__driver.find_element_by_xpath(xpath).click()

    @PageService()
    def get_navigation_names(self):
        """ Returns all the names in navigation grid """
        name_elem = self.__navigation_names()
        return [name.__getattribute__('text') for name in name_elem]

    @PageService()
    def get_entity(self, entity_id):
        """ Returns the respective checkbox entity
        Args:
            entity_id: Id of the entity
        """
        self.__entity(entity_id)

    @PageService()
    def get_nav_details(self):
        """ Returns the Navigation Preferences window details """
        return self.__nav_details()

    @PageService()
    def maximize_all_plus_squared_buttons(self):
        """ Expands all the navigation entities """
        self.__maximize_plus_square()

    @PageService()
    def save_changes(self):
        """ Saves all the updated entities """
        self.__save()
        self.__admin_console.click_button('Yes')

    @PageService()
    def reset_to_default(self):
        """ Resets all the entities to Default values """
        self.__reset_to_default()
        self.__admin_console.click_button('Yes')

    @PageService()
    def update_navigation(self):
        """ Update Navigation for commcell admin and tenant admin for random entities """
        self.maximize_all_plus_squared_buttons()
        time.sleep(5)
        details = self.get_nav_details()
        name_elem = list(details.keys())
        for key in name_elem:
            if key == 'Jobs':
                break
            del details[key]
        # these entities have duplicate names, and navigation hierarchy can't be modified
        pop_list = ['Monitoring', 'My apps', 'Manage', 'Customization', 'Navigation',
                    'Network', 'Cloud', 'CommCell', 'Servers', 'Store']
        for i in pop_list:
            details.pop(i, None)
        random_entities = random.sample(details.items(), k=3)
        self.log.info("Random entities being validated are: ", random_entities)
        c_names = []
        t_names = []
        for entity in range(3):
            if self.__driver.find_element_by_xpath(f"//*[@id = '{random_entities[entity][1][0]}']").is_selected():
                self.__checkbox_deselect(random_entities[entity][1][0])
                c_names.append(random_entities[entity][0])
            if self.__driver.find_element_by_xpath(f"//*[@id = '{random_entities[entity][1][1]}']").is_selected():
                self.__checkbox_deselect(random_entities[entity][1][1])
                t_names.append(random_entities[entity][0])
        self.save_changes()
        return c_names, t_names
