# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
This module provides the function or operations that can be performed on the
Login page on the AdminConsole

Class:

    LoginPage() -> AdminConsoleBase() -> object()

Functions:

_init_()            -- initialize the class object
set_username()      -- fill the username field with the specified username
set_password()      -- fill the password field with the specified password
submit_login()      -- login to admin console
login()             -- attempt login to AdminConsole using the username and password provided
forgot_password()   -- attempt to recover the password for the given username
"""
from time import sleep

from selenium.webdriver.support.ui import WebDriverWait
from selenium.webdriver.support import expected_conditions as ec
from selenium.webdriver.common.by import By

from Web.Common.exceptions import (
    CVWebAutomationException
)

from Web.Common.page_object import (
    WebAction,
    PageService
)


class LoginPage:
    """
    This class provides the operations that can be perfomed on the Login Page of adminconsole
    """
    def __init__(self, admin_console):
        """
        Args:
            admin_console: adminconsole object
        """
        self.__admin_console = admin_console
        self.__driver = admin_console.driver

    @WebAction()
    def _set_username(self, username):
        """Enter username during login"""
        txt_box = self.__driver.find_element_by_xpath(
            "//input[@id='username']")
        txt_box.send_keys(username)

    @WebAction(hide_args=True)
    def _set_password(self, password):
        """Enter password during login"""
        txt_box = self.__driver.find_element_by_xpath(
            "//input[@id='password']")
        txt_box.send_keys(password)

    @WebAction()
    def _set_stay_logged_in(self):
        """Enable stay logged in during login"""
        chk_box = self.__driver.find_element_by_id("stayactivebox")
        chk_box.click()

    @WebAction(delay=0)
    def _click_continue_button(self):
        """Click Continue button"""
        continue_btn = self.__driver.find_element_by_id("continuebtn")
        continue_btn.click()

    @WebAction(log=False, delay=0)
    def is_login_page(self):
        """Check if current page is login page"""
        return self.__driver.title == "Login"

    @WebAction(delay=0)
    def _click_login(self):
        """Click login button"""
        login_btn = self.__driver.find_element_by_id("loginbtn")
        login_btn.click()

    @WebAction()
    def _validate_creds(self):
        """Read invalid creds message during login"""
        login_err = self.__driver.find_elements_by_xpath(
            "//div[@id='loginform']//div[text()='Invalid Username or Password']"
        )
        if login_err:
            err = "Unable to login, received [%s]" % login_err[0].text
            raise CVWebAutomationException(err)

    @WebAction()
    def _select_commcell(self, service_commcell=None):
        """
        if the service commcell name is provided in the input:
            Selects the given service commcell from the availble list
        if No service commcell name is provided in the input:
            Selects the first list item from the availble redirects
        if service commcell name is provided is not found in the available list:
            raise Exception

        Args:
            service_commcell (str)  :   Service commcell name where login should happen
        """
        if service_commcell:
            list_elements = self.__driver.find_elements_by_xpath(
                "//ul[contains(@class,'idpSSORedirectUrlList list-group')]/li")
            count = 0
            for list_element in list_elements:
                if list_element.text.lower() == service_commcell.lower():
                    index = list_elements.index(list_element) + 1
                    elem = self.__driver.find_element_by_xpath('//*[@id="idpSsoredirectUrlDetail"]'
                                                               '[' + str(index) + ']/span')
                    elem.click()
                    count = 1
                    break
            if count == 0:
                err = "Service commcell name provided is not found in the available redirect list"
                raise CVWebAutomationException(err)
        else:
            self.__driver.find_element_by_xpath('//*[@id="idpSsoredirectUrlDetail"][1]/span').click()

    @WebAction()
    def _check_if_redirect_list_available(self):
        """
        Checks if there are any available redirects for the user
        if Yes  : Return True
        else    : Return False
        """
        if self.__admin_console.check_if_entity_exists("id", "idpSsoredirectUrlDetail"):
            return True
        return False

    def login(self, user_name, password, stay_logged_in=True, service_commcell=None):
        """
        Login to AdminConsole by using the username and password specified

        Args:
            user_name (basestring) : username to be used to login
            password (basestring)  : password to be used to login
            stay_logged_in (bool)  : select/deselect the keep me logged in checkbox
            service_commcell (str)   :  service commcell name where user has to login
        """
        self._set_username(user_name)
        self._click_continue_button()
        redirects_available = self._check_if_redirect_list_available()
        if redirects_available:
            self._select_commcell(service_commcell=service_commcell)
            self.__admin_console.wait_for_completion()
        self._set_password(password)
        if stay_logged_in:
            self.__admin_console.checkbox_select("stayactivebox")
        else:
            self.__admin_console.checkbox_deselect("stayactivebox")
        self._click_login()
        sleep(5)
        self.__admin_console.wait_for_completion()
        if self.is_login_page():
            self._validate_creds()

    @WebAction()
    def _click_forgot_password(self):
        """clicks on forgot password"""
        self.__driver.find_element_by_xpath("//a[@href='forgotPassword.jsp']").click()

    @WebAction(log=False)
    def _is_reset_pwd_page(self):
        """Check page is forgot password page"""
        return self.__admin_console.check_if_entity_exists("xpath", "//a[@href='index.jsp']")

    @WebAction()
    def _return_to_signin(self):
        """click on return to signin"""
        self.__driver.find_element_by_xpath("//a[@href='index.jsp']").click()

    @PageService()
    def forgot_password(self, user_name):
        """
        To reset the password

        Args:
            user_name (basestring): username to be used
        """
        self._click_forgot_password()
        self.__admin_console.wait_for_completion()
        self._set_username(user_name)
        self._click_login()
        self.__admin_console.wait_for_completion()
        WebDriverWait(self.__driver, 60).until(ec.presence_of_element_located(
            (By.ID, "forgotpass-title")))
        if self._is_reset_pwd_page():
            self._return_to_signin()
            self.__admin_console.wait_for_completion()
