# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
This module provides the function or operations that can be performed on the
jobs page on the AdminConsole

Class:

    jobs() -> AdminPage() -> login_page() -> AdminConsoleBase() -> object()

action_initiate_job_based_subclient_restore()    --    Method to initiate job based
                                                        subclient based restore
"""

import time
from selenium.common.exceptions import StaleElementReferenceException, ElementClickInterceptedException

from Web.Common.page_object import (
    WebAction,
    PageService
)
from Web.AdminConsole.Components.table import Table
from Web.AdminConsole.Components.panel import PanelInfo, ModalPanel
from Web.Common.exceptions import CVWebAutomationException


class Jobs:
    """ Class for the jobs page """
    def __init__(self, admin_console):
        """
        Method to initiate Maintenance class

        Args:
            admin_console   (Object) :   admin_console object
        """
        self.__admin_console = admin_console
        self.__driver = admin_console.driver
        self.__admin_console._load_properties(self)
        self.__navigator = self.__admin_console.navigator
        self.log = self.__admin_console.log
        self.__table = Table(self.__admin_console)
        self.__panel = PanelInfo(self.__admin_console)
        self.__modal_panel = ModalPanel(self.__admin_console)

    @WebAction()
    def __get_job_status(self):
        """Read job status"""
        while True:
            try:
                return self.__driver.find_element_by_xpath(
                    "//span[contains(text(),'Status')]/following-sibling::span"
                ).text
            except StaleElementReferenceException:
                pass

    @WebAction()
    def __get_job_progress(self):
        """Read job progress"""
        if self.__admin_console.check_if_entity_exists(
                "xpath", "//span[contains(text(),'Progress')]/following-sibling::span"):
            return self.__driver.find_element_by_xpath(
                "//span[contains(text(),'Progress')]/following-sibling::span"
            ).text
        return ""

    @WebAction()
    def __expand_job_details_panel(self):
        """Clicks on View more in the Job Details Panel"""
        if self.__admin_console.check_if_entity_exists(
                "xpath", "//*[@class='job-details-view-more ng-scope']"):
            self.__driver.find_element_by_xpath(
            "//*[@class='job-details-view-more ng-scope']"
        ).click()

    @WebAction()
    def if_job_exists(self, job_id):
        """
        Method to check if a job with given job id is displayed on page

        Args:
            job_id (str) : Id of the job
        """
        job_existence = False
        if not self.__driver.find_element_by_xpath(
                "//h3[@class='no-margin' and text()='No jobs available']").is_displayed():
            self.__table.apply_filter_over_integer_column('Job Id', job_id)
            attempts = 0
            while attempts < 3:
                try:
                    if self.__admin_console.check_if_entity_exists(
                            "xpath", f"//a[text()='{job_id}']"):
                        job_existence = True
                        break
                except StaleElementReferenceException:
                    pass
                finally:
                    attempts += 1
        return job_existence

    @PageService()
    def job_completion(self, job_id, retry=2, skip_job_details=False):
        """
        Waits for job completion
        :param job_id             (str)   --  ID of the job to wait for
        :param retry              (int)   --  the number of times the job has to be resumed if it goes into pending
        :param skip_job_details   (bool)  --  if true, skips getting job details
        """
        job_id = str(job_id)
        self.__navigator.navigate_to_jobs()
        if self.if_job_exists(job_id):
            self.access_job_by_id(job_id)
        else:
            self.access_job_history()
            self.show_admin_jobs()
            if not self.if_job_exists(job_id):
                raise CVWebAutomationException("jobs is not present in Active jobs or job history")
            self.access_job_by_id(job_id)
        while True:
            if self.__admin_console.check_if_entity_exists(
                    "xpath", "//span[contains(text(), 'Status')]/following-sibling::span"):
                status = self.__get_job_status()
                if status == "Running":
                    progress = self.__get_job_progress()
                    if progress == '100%':
                        retry = 0
                        while retry < 18:
                            time.sleep(10)
                            status = self.__get_job_status()
                            if status == "Running":
                                retry += 1
                                continue
                            else:
                                break
                        else:
                            raise CVWebAutomationException("Job Status is not getting updated correctly. "
                                                           "job progress says 100 % but status still shows as "
                                                           "running even 3 minutes of wait")
                    else:
                        self.log.info("Waiting for Job %s to complete.", str(job_id))
                        time.sleep(10)
                        continue
                if status in ["Completed", "Failed", "Completed w/ one or more errors",
                              "Pending", "Killed", "Failed to Start", "Committed"]:
                    if status in ["Completed", "Committed", "Completed w/ one or more errors", "Completed w/ one or more warnings"]:
                        break
                    elif status in ["Failed", "Killed", "Failed to Start"]:
                        break
                    else:
                        if retry > 0:
                            self.resume_job(job_id)
                            retry = retry - 1
                            continue
                        else:
                            self.kill_job(job_id)
                            time.sleep(30)
                            continue

        details = None
        if not skip_job_details:
            details = self.job_details()
        if self.__admin_console.check_if_entity_exists("xpath", "//button[@data-ng-click='close()']"):
            self.__modal_panel.close()
            self.__admin_console.wait_for_completion()
        return details

    @PageService()
    def access_active_jobs(self):
        """
        Shows all the active jobs in the jobs page
        """
        self.__admin_console.access_tab("Active jobs")

    @PageService()
    def access_job_history(self):
        """
        Shows all the active jobs in the jobs page
        """
        self.__admin_console.access_tab("Job history")

    @PageService()
    def view_jobs_of_last_24_hours(self):
        """
        Shows all the jobs that were run in the last 24 hrs
        """
        self.access_job_history()
        self.__table.view_by_title("Last 24 hours")

    @PageService()
    def view_jobs_of_last_3_months(self):
        """
        Shows all the jobs that were run in the last 3 months
        """
        self.access_job_history()
        self.__table.view_by_title("Last 3 months")

    @PageService()
    def get_job_ids(self):
        """
        Gets the jobs ids in the jobs page
        Returns:
            job_ids - list of job_ids present in the job page
        """
        job_ids = self.__table.get_column_data('Job Id')
        return job_ids

    @PageService()
    def access_job_by_id(self, job_id):
        """
        Access the job with given ID
        Args:
            job_id - id of the job to be selected

        Return:
            True    --  if the job opened successfully
        """
        self.__table.access_link(job_id)

    @PageService()
    def access_send_log_job_by_id(self, job_id):
        """

        Args:
            job_id:

        Returns:

        """
        self.__table.access_link(job_id)
        self.__admin_console.access_page_action_menu("More actions")
        self.__admin_console.access_page_action_menu("Send logs")

    @PageService()
    def job_details(self):
        """Get all the details of the job"""

        """
        Ex: {
                'Status': 'Completed',
                'Data Written': '4.44 MB',
                'Job started by': 'Admin',
                'Backup type': 'Incremental',
                'Elapsed time': '2 min 8 sec',
                'iDataAgent': 'Virtual Server',
                'Software compression': 'Storage Policy',
                'Start time': 'Nov 18, 2015 00:23:57 AM',
                'Collection': 'NewCol',
                'Server': 'ADMINCONSOLECB',
                'Data transfered on network': '28.07 KB',
                'Instance': 'Hyper-V',
                'End time': 'Nov 18, 2015 00:26:14 AM',
                'Throughput': '3.67 MB/hr',
                'Last update time': 'Nov 18, 2015 00:26:14 AM',
                'Progress': '100%',
                'Type': 'Backup',
                'Source client computer': 'ADMINCONSOLECB',
                'Job started from': 'Interactive',
                'Size': '110.55 KB'
            }
        """

        self.__expand_job_details_panel()
        while True:
            try:
                details = self.__panel.get_details()
                return details

            except Exception as exp:
                if "stale element reference" in str(exp):
                    continue
                else:
                    raise Exception(str(exp))

    @PageService()
    def kill_job(self, job_id):
        """
        Kills the job
        Args:
            job_id: job id to kill
        """
        self.__table.access_context_action_item(job_id, "Kill")

    @PageService()
    def view_job_details(self, job_id):
        """
        access job details
        Args:
            job_id: job id to access
        """
        self.__table.access_context_action_item(
            job_id,
            self.__admin_console.props['label.viewJobDetails']
        )

    @PageService()
    def suspend_job(self, job_id):
        """
        Suspends the job
        Args:
            job_id: job id to suspend
        """
        self.__table.access_context_action_item(job_id, "Suspend")

    @PageService()
    def resume_job(self, job_id):
        """
        Resumes the job
        Args:
            job_id: job id to resume
        """
        self.__table.access_context_action_item(job_id, "Resume")

    @PageService()
    def initiate_job_based_subclient_restore(self, job_id):
        """
        Initiate subclient restore

        Args:
            job_id   (string)   : job id which restore is to be initiated
        """
        self.access_job_by_id(job_id)
        self.__admin_console.select_hyperlink("Restore")

    @PageService()
    def get_job_id_by_subclient(self, subclient):
        """
        Get job id for subclient
        Args:
            subclient               (str):     subclient name
        Returns:                    (int):     job id of sublcient
        """
        self.__table.apply_filter_over_column('Subclient', subclient)
        job_id = self.__table.get_column_data('Job Id')[0]
        if not job_id:
            raise CVWebAutomationException("Job id not found for [%s] subclient" % subclient)
        return int(job_id)

    @PageService()
    def get_job_id_by_operation(self, operation):
        """
        Get job id for server
        Args:
            operation               (str):     operation name
        Returns:                    (int):     job id of server
        """
        self.__table.clear_column_filter('Operation')
        self.__admin_console.wait_for_completion()
        self.__table.apply_filter_over_column_selection('Operation', operation)
        job_id = self.__table.get_column_data('Job Id')[0]
        if not job_id:
            raise CVWebAutomationException("Job id not found for [%s] operation" % operation)
        return int(job_id)

    @PageService()
    def show_admin_jobs(self):
        """
        Click on 'Show admin jobs'
        """
        self.__admin_console.select_hyperlink(self.__admin_console.props['label.showAdminJobs'])
        self.__admin_console.wait_for_completion()