# -*- coding: utf-8 -*-s

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------
"""
This module provides the function or operations that can be performed on the
Domains page on the AdminConsole

Class:

    Domains() -> AdminPage() -> login_page() -> AdminConsoleBase() -> object()

Functions:

add_domain()           -- adds a new domain to the admin console

modify_domain()          -- edits the properties of a domain

delete_domain()        -- delete the domain with the specified name

domain_info()          -- displays the complete info of all the domains


"""
from Web.AdminConsole.Components.table import Table
from Web.Common.handlers import adminconsole_handler
from Web.AdminConsole.AdminConsolePages.AdminPage import AdminPage
from Web.Common.page_object import PageService


class Domains(AdminPage):
    """
    This class provides the function or operations that can be performed on the Domains page on the
    AdminConsole
    """

    def __init__(self, driver):
        super().__init__(driver)
        self.__table = Table(self)

    @PageService()
    def add_domain(self, domain_name, netbios_name, username, password, sso=False):
        """
        Adds a new domain to the commcell

        Args:
            domain_name (basestring) : the name of the domain

            netbios_name (basestring): the netbios name of the domain

            username (basestring)    : the username of the domain

            password (basestring)    : the password of the domain

            sso (bool)               : by default sso is false

        Returns:
            None

        """
        self.select_hyperlink("Add domain")
        self.fill_form_by_id("netBiosName", netbios_name)
        self.fill_form_by_id("domainName", domain_name)
        self.fill_form_by_id("username", username)
        self.fill_form_by_id("password", password)
        self.fill_form_by_id("confirmPassword", password)
        if sso:
            self.checkbox_select("enableSSO")
        self.submit_form()
        self.check_error_message()

    @PageService()
    def modify_domain(self, domain_name, username, password, sso=False):
        """
        Modifies the domain information

        Args:
            domain_name (basestring): the name of the domain whose information has to modified

            username (basestring)   : the username of the domain

            password (basestring)   : the password for the domain user

            sso (bool)              :  by default the sso is False

        Returns:
            None
        """
        self.__table.access_link(domain_name)
        self.fill_form_by_id("username", username)
        self.fill_form_by_id("password", password)
        self.fill_form_by_id("confirmPassword", password)
        if sso:
            self.checkbox_select("enableSSO")
        else:
            self.checkbox_deselect("enableSSO")
        self.submit_form()
        self.check_error_message()

    @PageService()
    def delete_domain(self, domain_name, skip_ownership_transfer="Yes", transfer_owner=None):
        """
        Deletes the domain with the given name

        Args:
            domain_name (basestring)            : the name of the domain whose information has to
                                                   modified

            skip_ownership_transfer(basestring) : whether ownerhip has to be transferred or not

            transfer_owner(basestring)          : Name of owner if ownership of domain has to be
                                                  transferred

        Returns:
            None
        """
        table = Table(self)
        table.access_link_by_column(domain_name, 'Delete')

        if skip_ownership_transfer != "Yes":
            self.checkbox_deselect("skipOwnershipTransfer")
            self.click_yes_button()
            self.transfer_ownership(transfer_owner)
        else:
            self.checkbox_select("skipOwnershipTransfer")
            self.click_yes_button()
        self.check_error_message()

    @adminconsole_handler()
    def domain_info(self):
        """
        Lists the information of all the domains in the commcell

        Returns:
            List of all domains
        """

        elems = self.driver.find_elements_by_xpath("//div[@class='ui-grid-canvas']/div")
        domain_list = []
        for elem in elems:
            domain_info = {'Netbios': elem.find_element_by_xpath("./div/div[1]/span/a/span").text,
                           'DomainName': elem.find_element_by_xpath("./div/div[2]/div").text}
            domain_list.append(domain_info)
        self.log.info("The list of all domains is "+str(domain_list))
        return domain_list
