# -*- coding: utf-8 -*-
# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
This module provides the function or operations that can be performed on the
CompanyDetails Page on the AdminConsole

Class:

    CompanyDetails() -> Companies() -> AdminPage() -> login_page() ->
    AdminConsoleBase() -> object()

Functions:

edit_general_settings()               -- Edits general settings tile of the company

edit_contact()                        -- Edits the contact information of the company

edit_domains()                        -- Edits the domain information of the company

edit_sender_email()                   -- Edits the email information of the company

edit_company_plan()                   -- Edits the plans associated with the company

deactivate_and_delete_company_from_details() --Deactivates and deletes the company

company_info()                        -- Extracts and returns all information about the company

extract_company_info()                -- Extracts and returns contained information from the company

de_associate_plans_from_company()     -- Method to de-associate all plans associated to the company

edit_company_file_exceptions()        -- Edits file exceptions tile of the company

edit_company_operators()              -- Method to edit Operators for the company

edit_external_authentication()        -- Method to create domain association for company

"""
import time
from selenium.webdriver.common.keys import Keys
from selenium.webdriver.support.ui import Select

from Web.Common.page_object import (WebAction, PageService)

from Web.AdminConsole.Components.table import Table
from Web.AdminConsole.Components.table import CVTable
from Web.AdminConsole.Components.panel import PanelInfo
from Web.AdminConsole.Components.panel import DropDown


class CompanyDetails:
    """
    This class provides the operations that can be performed on CompanyDetails Page of
    admin console
    """
    def __init__(self, admin_page):
        """
        Method to initiate Companies class

        Args:
            admin_page   (Object) :   Admin Page Class object
        """
        self.__admin_console = admin_page
        self.__table = Table(self.__admin_console)
        self.__cv_table = CVTable(self.__admin_console)
        self.__driver = admin_page.driver
        self.__drop_down = DropDown(self.__admin_console)
        self.__panel_info = None

    @WebAction()
    def __select_infrastructure_type(self, infra_type):
        """ Method to select value from drop down
        Args:
            infra_type (str): Infrastructure type
                Eg. infra_type (str) : 'Rented'

        Returns:
            None

        Raises:
            Exception:
                if fails to edit infra structure type
        """
        select_infra = Select(self.__admin_console.driver.find_element_by_xpath(
            "//select[@data-ng-model='subscriptionData.selectedInfraStructuretype']"))
        select_infra.select_by_visible_text(infra_type)

    @WebAction()
    def __select_save_or_cancel_icon(self, data_ng_click_attribute):
        """ Method to select save icon """
        self.__driver.find_element_by_xpath(
            f"//a[@data-ng-click='{data_ng_click_attribute}']/i").click()

    @WebAction()
    def __enter_company_alias(self, company_alias):
        """ Method to enter company alias """
        comp_alias_textbox = self.__driver.find_element_by_xpath(
            "//input[@data-ng-model='subscriptionData.companyAlias']")
        comp_alias_textbox.clear()
        comp_alias_textbox.send_keys(company_alias)

    @WebAction()
    def __enter_associated_smtp(self, smtp):
        """ Method to edit associated smtp for a company """
        self.__admin_console.fill_form_by_id("smtp", smtp)

    @PageService()
    def __remove_contacts(self):
        """ Method to remove/dissociate contacts from a company """

        self.__cv_table.de_select_all_values_from_table()

    @PageService()
    def __add_contacts(self, contact_names):
        """ Method to add contacts to a company """
        self.__cv_table.select_values_from_table(contact_names)

    @WebAction()
    def __click_remove_link(self):
        """ MEthod to click on remove link on edit sites dialog """
        self.__driver.find_element_by_xpath(
            "//a[@data-ng-click='removeDomain(domain)']").click()

    @WebAction()
    def __remove_secondary_sites(self):
        """ Method to remove secondary sites from a company """
        old_secondary_site_entries = self.__admin_console.driver.find_elements_by_xpath(
            "//input[@data-ng-model='domain.value']")
        for old in old_secondary_site_entries:
            old.clear()
            self.__click_remove_link()

    @WebAction()
    def __add_secondary_sites(self, secondary_sites):
        """ Method to add secondary sites to a company"""

        for site in secondary_sites:
            self.__admin_console.select_hyperlink('Add an additional site')
        index = len(secondary_sites)

        for elem in range(1, index + 1):
            self.__admin_console.driver.find_element_by_xpath(
                "//ul[" + str(elem) + "]//input"
            ).send_keys(secondary_sites[elem - 1])

    @PageService()
    def __remove_plan_associations(self):
        """ Method to remove plan associations for a company """
        self.__cv_table.de_select_all_values_from_table()

    @PageService()
    def __add_plan_associations(self, plans):
        """ Method to add plan associations """
        self.__cv_table.select_values_from_table(plans)

    @WebAction()
    def __select_default_plan(self,
                              server_default_plan,
                              laptop_default_plan):
        """ Method to select default plan from edit plan panel """

        if server_default_plan or laptop_default_plan:
            if self.__admin_console.check_if_entity_exists(
                    "xpath", "//div[@class='addServerModal panel ng-isolate-scope panel-default']"):
                self.__admin_console.driver.find_element_by_xpath(
                    "//div[@class='addServerModal panel ng-isolate-scope panel-default']/div").click()
                self.__admin_console.wait_for_completion()

                if server_default_plan:
                    server_default_plan = [server_default_plan]
                    self.__drop_down.select_drop_down_values(0, server_default_plan)

                if laptop_default_plan:
                    laptop_default_plan = [laptop_default_plan]
                    self.__drop_down.select_drop_down_values(1, laptop_default_plan)

    @WebAction()
    def __extract_company_info(self):
        """
        Extracts all the information about the company

        Args:
            None

        Returns:
            company_info (dict) : info about all the companies

        Raises:
            Exception:
                -- if fails to return company info
        """
        company_info = {}
        title = self.__admin_console.driver.find_element_by_xpath(
            '//h1[@class="no-margin page-title editable js-page-title"]')
        company_info.update({"CompanyName": title.text})
        elements = self.__admin_console.driver.find_elements_by_xpath(
            "//div[@class='page-details group']")
        self.__admin_console.driver.find_element_by_xpath(
            "//cv-fancy-tooltip[@id='supportedSolutionsToolTip']"
            "//span[@data-ng-if='remainingObjects.length']/a").click()
        for elem in elements:
            values_list = []
            key = elem.find_element_by_xpath("./span").text
            if key == 'Plans':
                div_elements = elem.find_elements_by_xpath("./div[@class='tile-accordion ng-scope']")
                for div_elem in div_elements:
                    plan_rows = div_elem.find_elements_by_xpath(
                        "//div[@class='ui-grid-row ng-scope']")
                    for plan_row in plan_rows:
                        val = plan_row.find_element_by_xpath(".//a").text
                        values_list.append(val)
                company_info.update({key: values_list})
            else:
                values = PanelInfo(self.__admin_console, key).get_details()
                if key == 'General':
                    displayed_solutions = values['Supported solutions'].split(',')
                    supported_solutions = []
                    for value in displayed_solutions:
                        if '\n' in value:
                            value = value.split('\n')
                            for val in value:
                                if "+" in val:
                                    value.remove(val)
                            supported_solutions.extend(value)
                        else:
                            supported_solutions.append(value.strip())
                    values['Supported solutions'] = supported_solutions
                company_info.update({key: values})
        return company_info

    @WebAction()
    def __next_line(self, element_id):
        """
        Method to enter file exceptions

        Args:
            element_id (str) : id of the textbox to enter values in
        """
        self.__admin_console.driver.find_element_by_id(element_id).send_keys(Keys.RETURN)

    @WebAction()
    def __append_file_exception(self, element_id, exception):
        """
        Method to enter file exceptions

        Args:
            element_id (str) : id of the textbox to enter values in

            exception (str) : exception to be added to the company
        """
        self.__admin_console.driver.find_element_by_id(element_id).send_keys(exception)

    @WebAction()
    def __remove_operator(self, user, role):
        """ Method to remove operator """
        self.__admin_console.driver.find_element_by_xpath(
            f"//label[@title='{user}']/following-sibling::label[@title='{role}']/../span/a").click()

    @WebAction()
    def __expand_operator_search_bar(self):
        """ Method to expand operator search bar """
        self.__admin_console.driver.find_element_by_xpath(
            f"//div[contains(@class,'select2-container add-user-type')]").click()

    @WebAction()
    def __select_operator_user(self, user):
        """ Method to select operator user in add operator panel """
        self.__admin_console.driver.find_element_by_xpath(
            "//div[@class='select2-search']/input").send_keys(user)
        time.sleep(20)
        self.__admin_console.driver.find_element_by_xpath(
            f"//span[contains(text(),'({user})')]").click()

    @WebAction()
    def __click_add_operator(self):
        """ Method to click on add button in add operator panel """
        self.__admin_console.driver.find_element_by_xpath(
            "//button[@data-ng-click='addNewUserGroup()']").click()

    @WebAction()
    def __select_ad_panel(self):
        """ Method to select AD section in 'Add domain' panel """
        self.__admin_console.driver.find_element_by_xpath(
            "//div[@ng-click='authenticationChoice=1']").click()

    @WebAction()
    def __access_ad_details(self):
        """ Method to click on AD name and open AD details page """
        self.__admin_console.driver.find_element_by_xpath(
            "//a[@data-ng-click='authenticationAction(externalAuthentication)").click()

    @PageService()
    def edit_general_settings(self, general_settings):
        """
        Method to edit general setting associated o a company

        Args:
            general_settings (dict): dictionary containing values for general settings
                Eg. - general_settings = {'smtp':'comm.com', 'authcode': 'ON',
                                          'shared_laptop_usage': 'ON', 'UPN': 'ON',
                                          '2-factor': 'ON', 'data_encryption': ON,
                                          'infra_type': 'Rented'}

        Returns:
            None

        Raises:
            Exception:
                if fails to edit general settings
        """
        self.__admin_console.log.info("Editing general settings for the company")
        self.__panel_info = PanelInfo(self.__admin_console, 'General')

        if general_settings.get('company_alias', None):
            self.__panel_info.edit_tile_entity('Company alias')
            self.__enter_company_alias(general_settings['company_alias'])
            self.__select_save_or_cancel_icon('fSaveCompanyAlias()')
            self.__admin_console.wait_for_completion()
            self.__admin_console.click_button('Yes')

        if general_settings.get('smtp', None):
            self.__panel_info.edit_tile_entity('Associated SMTP')
            self.__enter_associated_smtp(general_settings['smtp'])
            self.__admin_console.submit_form()

        if general_settings.get('authcode', None):
            if general_settings['authcode'] == "ON":
                self.__panel_info.enable_toggle('Requires authcode for installation')
            else:
                self.__panel_info.disable_toggle('Requires authcode for installation')
                self.__admin_console.click_button('Yes')

        if general_settings.get('UPN', None):
            if general_settings['UPN'] == "ON":
                self.__panel_info.enable_toggle('Use UPN instead of e-mail')
            else:
                self.__panel_info.disable_toggle('Use UPN instead of e-mail')

        if general_settings.get('2-factor', None):
            if general_settings['2-factor'] == "ON":
                self.__panel_info.enable_toggle('Enable two factor authentication')
            else:
                self.__panel_info.disable_toggle('Enable two factor authentication')

        if general_settings.get('reseller_mode', None):
            if general_settings['reseller_mode']:
                self.__panel_info.enable_toggle('Enable reseller mode')

        if general_settings.get('data_encryption', None):
            if general_settings['data_encryption']:
                self.__panel_info.enable_toggle('Allow owners to enable data encryption')
            else:
                self.__panel_info.disable_toggle('Allow owners to enable data encryption')

        if general_settings.get('auto_discover_applications', None):
            if general_settings['auto_discover_applications']:
                self.__panel_info.enable_toggle('Auto discover applications')
            else:
                self.__panel_info.disable_toggle('Auto discover applications')

        if general_settings.get('infra_type', None):
            if general_settings['infra_type']:
                self.__panel_info.edit_tile_entity('Infrastructure type')
                self.__select_infrastructure_type(general_settings['infra_type'])
                self.__select_save_or_cancel_icon('fSaveInfraStructureType()')

        if general_settings.get('supported_solutions', None):
            if general_settings['supported_solutions']:
                self.__panel_info.edit_tile_entity('Supported solutions')
                self.__drop_down.select_drop_down_values(0, general_settings['supported_solutions'])

        self.__admin_console.check_error_message()

    @PageService()
    def edit_contacts(self, contact_names):
        """
        Edits the contact information of the company

        Args:
            contact_names (list): String containing users to be associated/de-associated
                                  to the company
                        Eg.- contact_names = 'User1,User2,User3'

        Returns:
            None

        Raises:
            Exception:
                -- if fails to set edit contact for the company
        """

        PanelInfo(self.__admin_console, 'Contacts').edit_tile()
        self.__admin_console.wait_for_completion()
        self.__remove_contacts()
        self.__add_contacts(contact_names)
        self.__admin_console.submit_form()
        self.__admin_console.check_error_message()

    @PageService()
    def edit_sites(self,
                   primary_site,
                   secondary_sites):
        """
        Edits the domain information of the company

        Args:
            primary_site(str) -- the name of the primary site to be added
            secondary_sites (list) -- list of all secondary sites to be added

        Returns:
            None

        Raises:
            Exception:
                -- if fails to set edit sites for the company
        """

        PanelInfo(self.__admin_console, 'Sites').edit_tile()
        self.__admin_console.wait_for_completion()

        if primary_site:
            self.__admin_console.fill_form_by_id("primaryDomainName", primary_site)

        if secondary_sites:
            self.__remove_secondary_sites()
            self.__add_secondary_sites(secondary_sites)

        self.__admin_console.submit_form()
        self.__admin_console.check_error_message()

    @PageService()
    def edit_sender_email(self,
                          sender_name,
                          sender_email):
        """
        Edits the email information of the company

        Args:
            sender_name      (str)   --  the name of the sender of the email
            sender_email     (str)   --  the email of the sender

        Returns:
            None

        Raises:
            Exception:
                -- if fails to set edit domains for the company
        """
        PanelInfo(self.__admin_console, 'Email settings').edit_tile()
        self.__admin_console.wait_for_completion()

        self.__admin_console.fill_form_by_id("senderName", sender_name)
        self.__admin_console.fill_form_by_id("senderEmail", sender_email)

        self.__admin_console.submit_form()
        self.__admin_console.check_error_message()

    @PageService()
    def edit_company_plans(self,
                           plans,
                           server_default_plan=None,
                           laptop_default_plan=None):
        """
        Edits the plans associated with the company

        Args:
            plans (list)                    -- list of plans to be associated to the company

            server_default_plan (str)       --  name of the default server plan to be set

            laptop_default_plan (str)       --  name of the default laptop plan to be set

        Returns:
            None

        Raises:
            Exception:
                -- if fails to set edit domains for the company
        """
        PanelInfo(self.__admin_console, 'Plans').edit_tile()
        self.__admin_console.wait_for_completion()

        self.__remove_plan_associations()
        self.__add_plan_associations(plans)
        self.__select_default_plan(server_default_plan,
                                   laptop_default_plan)

        self.__admin_console.submit_form()
        self.__admin_console.click_button('Yes')
        self.__admin_console.check_error_message()

    @PageService()
    def company_info(self):
        """
        collates and returns all the information about the company

        Returns:
            displayed_val(dict) : displayed values of the company in question

        Raises:
            Exception:
                -- if fails to return displayed_val
        """
        displayed_val = self.__extract_company_info()
        self.__admin_console.log.info(displayed_val)
        return displayed_val

    @PageService()
    def de_associate_plans_from_company(self):
        """de-associates plans associated to the company

        Args:
            None

        Returns:
            None

        Raises:
            Exception:
                -- if fails to de-associate plans associated to the company
        """

        PanelInfo(self.__admin_console, 'Plans').edit_tile()
        self.__admin_console.wait_for_completion()
        self.__remove_plan_associations()
        self.__admin_console.submit_form()
        self.__admin_console.click_button('Yes')
        self.__admin_console.check_error_message()

    @PageService()
    def deactivate_and_delete_company_from_details(self,
                                                   company_disable_login=None,
                                                   company_disable_backup=None,
                                                   company_disable_restore=None):
        """
        Deletes the company

        Args:
            company_disable_login (bool) : if login to be disabled while
                                           de-activating the company

            company_disable_backup (bool) : if backup to be disabled while
                                            de-activating the company

            company_disable_restore (bool) : if restore to be disabled while
                                             de-activating the company

        Returns:
            None

        Raises:
            Exception:
                -- if fails to delete the company
        """
        self.__admin_console.select_hyperlink('Deactivate')
        if company_disable_login:
            self.__admin_console.checkbox_deselect('Disable login ')
        elif company_disable_backup:
            self.__admin_console.checkbox_deselect('Disable backup ')
        elif company_disable_restore:
            self.__admin_console.checkbox_deselect('Disable restore')
        self.__admin_console.submit_form()
        self.__admin_console.select_hyperlink('Delete company')
        self.__admin_console.click_button('Yes')

    @PageService()
    def edit_company_file_exceptions(self,
                                     file_exceptions):
        """
        Edits file exceptions for a company

        Args:
            file_exceptions (dict of lists) : List of paths to be excluded for the company
               Eg. -  file_exceptions = {"windows_path":["C:\\Test"],
                                         "unix_path": ["/root/file1"]}

        Returns:
            None

        Raises:
            Exception:
                -- if fails to set edit file exceptions for the company
        """

        PanelInfo(self.__admin_console, 'File exceptions').edit_tile()
        self.__admin_console.wait_for_completion()

        if file_exceptions.get('windows_path', None):
            self.__admin_console.driver.find_element_by_id('windowsFiles').clear()
            for win_path in file_exceptions['windows_path']:
                self.__append_file_exception("windowsFiles", win_path)
                self.__next_line("windowsFiles")

        if file_exceptions.get('unix_path', None):
            self.__admin_console.driver.find_element_by_id('unixFiles').clear()
            for unix_path in file_exceptions['unix_path']:
                self.__append_file_exception("unixFiles", unix_path)
                self.__next_line("unixFiles")

        self.__admin_console.submit_form()
        self.__admin_console.check_error_message()

    @PageService()
    def edit_company_operators(self, operators):
        """
        Edits operators for a company

        Args:
            operators (dict) : dictionary containing operators to be assigned to the company
               Eg. -  operators = {'add': {'TestUser': 'Tenant Operator'},
                                    'remove': {'admin': 'Tenant Operator'}}

        Returns:
            None

        Raises:
            Exception:
                -- if fails to set edit operators for the company
        """
        PanelInfo(self.__admin_console, 'Operators').edit_tile()
        self.__admin_console.wait_for_completion()

        if operators['remove']:
            for key, value in operators['remove'].items():
                self.__remove_operator(key, value)

        if operators['add']:
            for key, value in operators['add'].items():
                self.__expand_operator_search_bar()
                self.__select_operator_user(key)
                time.sleep(2)
                self.__admin_console.select_value_from_dropdown("adduserId", value)
                self.__click_add_operator()
                self.__admin_console.wait_for_completion()

        self.__admin_console.submit_form()
        self.__admin_console.check_error_message()

    @PageService()
    def edit_external_authentication(self,
                                     dir_type=None,
                                     netbios_name=None,
                                     username=None,
                                     password=None,
                                     domain_name=None,
                                     proxy_client=None):
        """
        Edits external authentication for the company

        Args:
            dir_type     (str) : Type of AD to be created

            netbios_name (str) : 'Netbios name/OSX server name/Host' value
                                 based on dir_type

            username     (str) : Username for AD registration

            password     (str) : Password for AD registration

            domain_name  (str) : domain name for AD to be created

            proxy_client (str) : Name of the client serving as proxy between
                                 Host machine and AD

        Returns:
            None

        Raises:
            Exception:
                -- if fails to set edit external authentication for the company
        """
        if self.__admin_console.check_if_entity_exists(
                "xpath", "//a[@data-ng-click='authenticationAction(externalAuthentication)' and text()='Edit']"):
            PanelInfo(self.__admin_console, 'External authentication').edit_tile()
            self.__admin_console.wait_for_completion()
            self.__select_ad_panel()
        else:
            self.__access_ad_details()
            self.__admin_console.wait_for_completion()
            PanelInfo(self.__admin_console, 'General').edit_tile()
            self.__admin_console.wait_for_completion()

        if dir_type:
            dir_type = [dir_type]
            self.__drop_down.select_drop_down_values(0, dir_type)
        if netbios_name:
            self.__admin_console.fill_form_by_id('netBiosName', netbios_name)
        if domain_name:
            self.__admin_console.fill_form_by_id('domainName', domain_name)
        if username or password:
            self.__admin_console.fill_form_by_id('uname', username)
            self.__admin_console.fill_form_by_id('pass', password)
        if proxy_client:
            proxy_client = [proxy_client]
            self.__admin_console.checkbox_select('accessADProxy')
            self.__drop_down.select_drop_down_values(1, proxy_client)
        else:
            self.__admin_console.checkbox_deselect('accessADProxy')
        self.__admin_console.submit_form()
        self.__admin_console.check_error_message()
