# -*- coding: utf-8 -*-
# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
This module provides the function or operations that can be performed on the
Companies page on the AdminConsole

Class:

    Companies() -> AdminPage() -> AdminConsoleBase() -> object()

Functions:

_init_()                        :     initialize the class object

__set_company_name()            :     fill the company name field with the specified
                                      company name

__set_email()                   :     fill the email field with the specified email

__set_contact_name()            :     fill the contact name field with the specified
                                      contact name

__set_plans()                   :     fill the plans field with the specified plans

__set_company_alias()           :     fill the company alias field with the specified
                                      company alias

__set_smtp()                    :     fill the smtp field with the specified smtp alias

__enable_custom_domain()        :     Check/Uncheck the *Enable custom domain* checkbox

__set_primary_domain()          :     fill the primary domain field with the specified
                                      primary domain

__click_add_company()           :     clicks the link add company

__submit_company()              :     clicks the save button in company

add_company()                   :     adds the new company clicking save

deactivate_and_delete_company() :     deletes the selected company
"""

from Web.Common.page_object import (WebAction, PageService)
from Web.AdminConsole.Components.table import Table
from Web.AdminConsole.Components.panel import DropDown


class Companies:
    """ Class for Companies page in Admin Console """

    def __init__(self, admin_page):
        """
        Method to initiate Companies class

        Args:
            admin_page   (Object) :   Admin Page Class object
        """
        self.__admin_console = admin_page
        self.__table = Table(self.__admin_console)
        self.__driver = admin_page.driver
        self.__drop_down = DropDown(self.__admin_console)

    @WebAction()
    def __set_company_name(self, company_name):
        """
        Method to set name of the company

        Args:
            company_name (string): Name of the company to be created

        Returns:
            None

        Raises:
            Exception:
                -- if fails to set company name
        """
        self.__admin_console.fill_form_by_id("name", company_name)

    @WebAction()
    def __set_email(self, email):
        """
        Method to set email for the company

        Args:
            email (string): email for the company to be created

        Returns:
            None

        Raises:
            Exception:
                -- if fails to set email for company
        """
        self.__admin_console.fill_form_by_id("email", email)

    @WebAction()
    def __set_contact_name(self, contact_name):
        """
        Method to set contact name for the company

        Args:
            contact_name (string): contact name for the company to be created

        Returns:
            None

        Raises:
            Exception:
                -- if fails to set contact name for company
        """
        self.__admin_console.fill_form_by_id("fullName", contact_name)

    @WebAction()
    def __set_company_alias(self, company_alias):
        """
        Method to set Company Alias for the company

        Args:
            company_alias (string): company_alias to be set for the
                                    company to be created

        Returns:
            None

        Raises:
            Exception:
                -- if fails to set company_alias for the company
        """
        self.__admin_console.fill_form_by_id("companyAlias", company_alias)

    @WebAction()
    def __set_smtp(self, smtp_alias):
        """
        Method to set smtp alias for the company

        Args:
            smtp_alias (string): smtp_alias to be set for the company
                                to be created

        Returns:
            None

        Raises:
            Exception:
                -- if fails to set smtp_alias for the company
        """
        self.__admin_console.fill_form_by_id("associatedSMTP", smtp_alias)

    @WebAction()
    def is_toggle_enabled(self, element):
        """ Checks if toggle is enabled

        Args:
            element: WebElement corresponding to the toggle option.

        Returns (bool) : True if toggle is enabled

        """
        if 'expanded' in element.get_attribute('class'):
            return False
        return True

    @WebAction()
    def get_toggle_element(self, toggle_label):
        """ Gets WebElement containing attribute toggle-label.

        Args:
            toggle_label   (basestring):   toggle-label attribute of the toggle option

        Returns : toggle WebElement

        """
        return self.__driver.find_element_by_xpath(
            f"//div[@toggle-label='{toggle_label}']/..")

    @WebAction()
    def enable_toggle(self, toggle_label):
        """
        Enables the toggle bar if disabled,
        Args:
            toggle_label   (basestring):   toggle-label attribute of the toggle option

        """
        element = self.get_toggle_element(toggle_label)

        if not self.is_toggle_enabled(element):
            element.find_element_by_xpath(".//span").click()
            self.__admin_console.wait_for_completion()

    @WebAction()
    def disable_toggle(self, toggle_label):
        """
        Disables the toggle bar if enabled

        Args:
            toggle_label   (basestring):   toggle-label attribute of the toggle option

        """
        element = self.get_toggle_element(toggle_label)

        if self.is_toggle_enabled(element):
            element.find_element_by_xpath(".//span").click()
            self.__admin_console.wait_for_completion()

    @PageService()
    def add_company(self,
                    company_name,
                    email,
                    contact_name,
                    plans,
                    company_alias,
                    smtp_alias,
                    mail_template,
                    primary_domain):
        """ Method to add new company as per inputs provided as arguments

            Args:
                company_name (string) : Name of the cmpany to be created

                email (string)        : email of Tenant admin for the company to be
                                        created

                contact_name (string) : Contact name for the company

                plans (list)          : plans to be associated to the company

                company_alias (string): company alias for the company

                smtp_alias (string)   : smtp alias for the company

                mail_template (str)   : mail template to be used to send intimation after
                                        company creation

                primary_domain (string): primary_domain to be set for the company to be
                                        created

            Returns:
                None

            Raises:
                Exception:
                    -- if fails to set primary_domain for the company
        """
        self.__admin_console.select_hyperlink("Add company")
        self.__set_company_name(company_name)
        self.__set_email(email)
        self.__set_contact_name(contact_name)

        if company_alias:
            self.__set_company_alias(company_alias)

        if plans:
            self.__drop_down.select_drop_down_values(0, plans)

        if smtp_alias:
            self.__set_smtp(smtp_alias)

        if mail_template:
            mail_template = [mail_template]
            self.enable_toggle('label.sendOnboardingEmail')
            self.__admin_console.wait_for_completion()
            self.__drop_down.select_drop_down_values(1, mail_template)
        else:
            self.disable_toggle('label.sendOnboardingEmail')
            self.__admin_console.wait_for_completion()

        if primary_domain:
            self.enable_toggle('label.enableCustomDomain')
            self.__admin_console.wait_for_completion()
            self.__admin_console.fill_form_by_id("domainName", primary_domain)
        else:
            self.disable_toggle('label.enableCustomDomain')
            self.__admin_console.wait_for_completion()

        self.__admin_console.submit_form()
        self.__admin_console.check_error_message()

    @PageService()
    def deactivate_and_delete_company(self,
                                      company_name,
                                      company_disable_login=None,
                                      company_disable_backup=None,
                                      company_disable_restore=None):
        """
        Method to delete a company

        Args:
            company_name (str) : Name of the company to be deleted

            company_disable_login (bool) : if login to be disabled while de-activating
                                           the company

            company_disable_backup (bool) : if backup to be disabled while de-activating
                                            the company

            company_disable_restore (bool) : if restore to be disabled while de-activating
                                             the company

        Returns:
            None

        Raises:
            Exception:
                -- if fails delete company with given name
        """
        self.__table.access_action_item(company_name, 'Deactivate')
        if company_disable_login:
            self.__admin_console.checkbox_deselect('Disable login ')
        elif company_disable_backup:
            self.__admin_console.checkbox_deselect('Disable backup ')
        elif company_disable_restore:
            self.__admin_console.checkbox_deselect('Disable restore')
        self.__admin_console.submit_form()
        self.__table.access_action_item(company_name, 'Delete')
        self.__admin_console.click_button('Yes')
        self.__admin_console.check_error_message()
