# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
This module provides the function or operations that can be performed on the
commcell page of the File System iDA on the AdminConsole

Class:

    Commcell() -> AdminPage() -> login_page() -> AdminConsoleBase() -> object()

Functions:

Toggle Buttons:
    enable_all_job_activity             - enables all activity
    disable_all_job_activity            - disables all activity
    enable_backup                       - enables job activity
    disable_backup                      - disables job activity
    enable_restore                      - enables restores
    disable_restore                     - disables restores
    enable_schedule                     - enables schedules
    disable_schedule                    - disables schedules
    enable_authcode_for_installation    - Enables requirement of authCode for installation
    disable_authcode_for_installation   - Disables requirement of authCode for installation
    enable_upn                          - Enables usage of UPN instead of e-mail
    disable_upn                         - Disables usage of UPN instead of e-mail
    enable_tfa                          - Enables two factor authentication
    disable_tfa                         - Disables two factor authentication
    enable_data_privacy                 - Enables allowance of users to enable data privacy
    disable_data_privacy                - Disables allowance of users to enable data privacy

Page hyperlinks:
    access_sendlogs                     - click on access send logs button

All details:
    extract_all_displayed_details       - Retrieves all information displayed in Commcell page
    _extract_all_visible_panel_title    - Method to extract all visible tiles title

Tile Editing:
    edit_email_settings                 - Edit/Set/Update email settings
    edit_default_plans                  - Edit/Set/Update Default plan
    edit_password_encryption            - Edit/Set/Update password encryption

"""
from Web.Common.page_object import PageService, WebAction
from Web.AdminConsole.Components.dialog import ModalDialog
from Web.AdminConsole.Components.panel import DropDown, PanelInfo, Security
from AutomationUtils import logger


class Commcell:
    """ Class for Commcell page of admin console"""

    def __init__(self, admin_console):
        self._admin_console = admin_console
        self._driver = admin_console.driver
        self.__activity_panel_ac = PanelInfo(self._admin_console, 'Activity control')
        self.__activity_panel_es = PanelInfo(self._admin_console, 'Email settings')
        self.__activity_panel_ge = PanelInfo(self._admin_console, 'General')
        self.__activity_panel_pr = PanelInfo(self._admin_console, 'Privacy')
        self.__activity_panel_pk = PanelInfo(self._admin_console, 'Passkey')
        self.__dialog = ModalDialog(self._admin_console)
        self.__drop_down = DropDown(self._admin_console)
        self.log = logger.get_log()
        self._admin_console._load_properties(self)

    @WebAction()
    def click_test_email(self):
        """click on test email of email setting page"""
        self._driver.find_element_by_id("configureEmail_button_#3607").click()
        self._admin_console.wait_for_completion()

    @PageService()
    def enable_all_job_activity(self):
        """Enables all job activity"""
        self.__activity_panel_ac.enable_toggle(self._admin_console.props['All_Job_Activity'])

    @PageService()
    def disable_all_job_activity(self):
        """disables all job activity"""
        self.__activity_panel_ac.disable_toggle(self._admin_console.props['All_Job_Activity'])

    @PageService()
    def enable_backup(self):
        """Enables Backup job activity"""
        self.__activity_panel_ac.enable_toggle(self._admin_console.props['Data_Backup'])

    @PageService()
    def disable_backup(self):
        """Disables Backup job activity"""
        self.__activity_panel_ac.disable_toggle(self._admin_console.props['Data_Backup'])

    @PageService()
    def enable_restore(self):
        """Enables Restore job activity"""
        self.__activity_panel_ac.enable_toggle(self._admin_console.props['Data_Restore'])

    @PageService()
    def disable_restore(self):
        """Disables Backup job activity"""
        self.__activity_panel_ac.disable_toggle(self._admin_console.props['Data_Restore'])

    @PageService()
    def enable_schedule(self):
        """Enables Schedule activity"""
        self.__activity_panel_ac.enable_toggle(self._admin_console.props['Scheduler'])

    @PageService()
    def disable_schedule(self):
        """Disables Schedule activity"""
        self.__activity_panel_ac.disable_toggle(self._admin_console.props['Scheduler'])

    @PageService()
    def access_sendlogs(self):
        """access send logs menu"""
        self._admin_console.select_hyperlink(self._admin_console.props['label.submitSendLogs'])

    @PageService()
    def enable_authcode_for_installation(self):
        """Enables requirement of authCode for installation"""
        self.__activity_panel_ge.enable_toggle('Requires authcode for installation')
        self._admin_console.wait_for_completion()

    @PageService()
    def disable_authcode_for_installation(self):
        """Disables requirement of authCode for installation"""
        self.__activity_panel_ge.disable_toggle('Requires authcode for installation')
        self.__dialog.click_submit()
        self._admin_console.wait_for_completion()

    @PageService()
    def enable_upn(self):
        """Enables usage of UPN instead of e-mail"""
        self.__activity_panel_ge.enable_toggle('Use UPN instead of e-mail')

    @PageService()
    def disable_upn(self):
        """Disables usage of UPN instead of e-mail"""
        self.__activity_panel_ge.disable_toggle('Use UPN instead of e-mail')

    @PageService()
    def enable_tfa(self):
        """Enables two factor authentication"""
        self.__activity_panel_ge.enable_toggle('Enable two factor authentication')
        # Default is ALL
        self.__dialog.click_submit()
        self._admin_console.wait_for_completion()

    @PageService()
    def disable_tfa(self):
        """Disables two factor authentication"""
        self.__activity_panel_ge.disable_toggle('Enable two factor authentication')

    @PageService()
    def enable_data_privacy(self):
        """Enables allowance of users to enable data privacy"""
        self.__activity_panel_pr.enable_toggle('Allow users to enable data privacy')

    @PageService()
    def disable_data_privacy(self):
        """Disables allowance of users to enable data privacy"""
        self.__activity_panel_pr.disable_toggle('Allow users to enable data privacy')

    @PageService()
    def enable_passkey_for_restore(self, passkey):
        """Enables requirement of passkey for restores"""
        self.__activity_panel_pk.enable_toggle('Require passkey for restores')
        self._admin_console.fill_form_by_id("passkey", passkey)
        self._admin_console.fill_form_by_id("confirmPasskey", passkey)
        self.__dialog.click_submit()
        self._admin_console.click_button('Yes')
        self._admin_console.wait_for_completion()

    @PageService()
    def disable_passkey_for_restore(self, passkey):
        """Disables requirement of passkey for restores"""
        self.__activity_panel_pk.disable_toggle('Require passkey for restores')
        self._admin_console.fill_form_by_id("passkey", passkey)
        self.__dialog.click_submit()
        self._admin_console.wait_for_completion()

    @PageService()
    def edit_passkey(self, new_passkey, old_passkey):
        """Edit the passkey"""
        PanelInfo(self._admin_console, 'Passkey').edit_tile_entity('Require passkey for restores')
        self._admin_console.fill_form_by_id("existingPasskey", old_passkey)
        self._admin_console.fill_form_by_id("newPasskey", new_passkey)
        self._admin_console.fill_form_by_id("confirmPasskey", new_passkey)
        self.__dialog.click_submit()
        self._admin_console.click_button('Yes')
        self._admin_console.wait_for_completion()

    @PageService()
    def enable_users_can_enable_passkey(self):
        """Enables users to enable passkey"""
        self.__activity_panel_pk.enable_toggle('Allow users to enable passkey')

    @PageService()
    def disable_users_can_enable_passkey(self):
        """Disables users to enable passkey"""
        self.__activity_panel_pk.disable_toggle('Allow users to enable passkey')

    @PageService()
    def enable_authorize_for_restore(self, passkey):
        """Enables authorize for restore"""
        self.__activity_panel_pk.enable_toggle('Authorize for restore')
        self._admin_console.fill_form_by_id("existingPasskey", passkey)
        self.__dialog.click_submit()
        self._admin_console.wait_for_completion()

    @PageService()
    def disable_authorize_for_restore(self):
        """Disables authorize for restore"""
        self.__activity_panel_pk.disable_toggle('Authorize for restore')

    @PageService()
    def extract_all_displayed_details(self):
        """
            Retrieves all information displayed in Commcell page
        Returns:
            all_details dict() : dictionary containing commcell information displayed in UI
        """
        tile_titles = self._extract_all_visible_panel_title()
        details = {}
        for title in tile_titles:
            if title == "Security":
                values = Security(self._admin_console).get_details()
                details.update({title: values})
            else:
                values = PanelInfo(self._admin_console, title).get_details()
                details.update({title: values})
        return details

    @WebAction()
    def _extract_all_visible_panel_title(self):
        """
            Method to extract all visible tiles title
        """
        all_tiles = self._driver.find_elements_by_xpath("//div[@class='page-details group']/span/span[1]")
        return [tile.text for tile in all_tiles]

    @PageService()
    def edit_email_settings(self, email_settings, test_scenario="positive"):
        """
            Edit/Set/Update email settings
        Args:
            email_settings (dict):
                {
                    'SMTP server': String,
                    'SMTP port': integer between 1 and 6100 (both included),
                    'Sender email': String,
                    'Sender name': String,
                    'Encryption algorithm': String,
                    'Use authentication': boolean,
                    'User name': String,
                    'Password': String (Don't include this key if you want to keep same password)
                }
            test_scenario  (str):
                specify test scenario is positive or negative scenarios
                    Expected values: positive or negative
        """
        PanelInfo(self._admin_console, 'Email settings').edit_tile()
        self._admin_console.wait_for_completion()
        if email_settings.get('SMTP server'):
            self._admin_console.fill_form_by_id('smtpServer', email_settings.get('SMTP server'))
        if email_settings.get('SMTP port'):
            self._admin_console.fill_form_by_id('smtpPort', email_settings.get('SMTP port'))
        if email_settings.get('Sender email'):
            self._admin_console.fill_form_by_id('senderEmail', email_settings.get('Sender email'))
        if email_settings.get('Sender name'):
            self._admin_console.fill_form_by_id('senderName', email_settings.get('Sender name'))
        if email_settings.get('Encryption algorithm'):
            self._admin_console.select_value_from_dropdown('encryption', email_settings.get('Encryption algorithm'))
        if email_settings.get('Use authentication'):
            self._admin_console.enable_toggle(toggle_id="use-authentication")
            self._admin_console.fill_form_by_id('username', email_settings.get('User name'))
            if email_settings.get('Password'):
                self._admin_console.fill_form_by_id('password', email_settings.get('Password'))
                self._admin_console.fill_form_by_id('confirmPassword', email_settings.get('Password'))
        else:
            self._admin_console.disable_toggle(toggle_id="use-authentication")

        if test_scenario.lower() == "positive":
            self._admin_console.submit_form()
            self._admin_console.check_error_message()
        else:
            expected_error_message = "Failed to send email to SMTP Mail server"
            self.click_test_email()
            # check if error is raised for invalid setting (negative scenario)
            error_string = self._admin_console.get_error_message()
            if expected_error_message not in error_string:
                raise Exception("no error found for negative scenario")
            self.log.info("successfully got the error message %s for negative scenario",
                          error_string)
            # cancel settings
            self._admin_console.cancel_form()

    @PageService()
    def edit_password_encryption(self, password_encryption):
        """
            Edit/Set/Update password encryption
        Args:
            password_encryption (dict):
                {
                    'Key Management Server': String
                }
        """
        PanelInfo(self._admin_console, 'Password encryption').edit_tile()
        self._admin_console.wait_for_completion()
        self.__drop_down.select_drop_down_values(drop_down_id=
                                                 'configurePasswordEncryptionConfig_isteven-multi-select_#6908',
                                                 values=[password_encryption.get('Key Management Server')])
        self._admin_console.submit_form()
        self._admin_console.check_error_message()

    @PageService()
    def edit_default_plans(self, default_plan):
        """
            Helper function to Edit/Set/Update Default plan
        Args:
            default_plan (dict):
                {
                    'Server plan': String (Give 'None' if you want no default plans)
                }
        """
        PanelInfo(self._admin_console, 'Default plans').edit_tile()
        self._admin_console.wait_for_completion()
        for key, value in default_plan.items():
            if key == 'Server plan':
                if value == 'None':
                    self._admin_console.select_value_from_dropdown('serverDefault', 'Select server default plan')
                else:
                    self._admin_console.select_value_from_dropdown('serverDefault', value)
            elif key == 'Laptop plan':
                if value == 'None':
                    self._admin_console.select_value_from_dropdown('laptopDefault', 'Select laptop default plan')
                else:
                    self._admin_console.select_value_from_dropdown('laptopDefault', value)
            elif key == 'Fs plan':
                if value == 'None':
                    self._admin_console.select_value_from_dropdown('fsDefault', 'Select fs default plan')
                else:
                    self._admin_console.select_value_from_dropdown('fsDefault', value)
            elif key == 'Db plan':
                if value == 'None':
                    self._admin_console.select_value_from_dropdown('dbDefault', 'Select db default plan')
                else:
                    self._admin_console.select_value_from_dropdown('dbDefault', value)
        self._admin_console.submit_form()
        self._admin_console.check_error_message()
