# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
This module provides the function or operations that can be performed on the Arrays Page on the AdminConsole.
Below are the class defined along with the functions.


    Arrays() :  -- To add snap arrays along with editing snap configurations

                    add_arrays()            -- Creates an array if a snap plan already exists

                    action_delete_array()   -- Deletes the array with the given name

                    action_list_snaps()     --  Lists the snaps on the array with the given name

                    array_list()            -- Lists all the arrays

                    delete_snaps()          -- Deletes the snap with given jobid for an array

                    delete_all_snapshots()  -- Delete all snapshots for an array


    General() -- Includes all the General information required to add array like : Array Vendor, Array Name,Control Host,
                 Username and Password.

                    add_general()   -- To add generic details to add the array


    ArrayAccessNodes() -- This class includes array controllers to be selected for pruning purpose

                    add_array_accessnodes()  -- To add array controllers for the array

    Snap_config()   -- This class includes adding/editing snap configuration of the array

                    add_snapconfig()  -- To add snap configuration for the array


    Engine()  -- Includes the method to add array for each engine.

                    add_engine()  -- To add the Snap engine



"""

import re

from Web.AdminConsole.Components.table import Table
from Web.Common.page_object import PageService
from Web.AdminConsole.Components.panel import ModalPanel, DropDown
from Web.AdminConsole.Components.dialog import ModalDialog
from AutomationUtils import logger


class Arrays():
    """ Class for the Arrays page """

    def __init__(self, admin_console):
        """Method to initiate Arrays Class
                Args:
                        admin_console   (Object) :   Admin Console Class object"""

        self.__driver = admin_console.driver
        self.__admin_console = admin_console
        self.__table = Table(self.__admin_console)
        self.__modal_panel = ModalPanel(self.__admin_console)
        self.__drop_down = DropDown(self.__admin_console)
        self.__navigator = admin_console.navigator
        self.dialog = ModalDialog(self.__admin_console)
        self.__admin_console._load_properties(self)


    @PageService()
    def action_delete_array(self, array_name):
        """
        Deletes the array with the given name
       Args :
            array_name   (str)   --  the name of the array to be deleted
        """
        self.__navigator.navigate_to_arrays()
        self.__table.access_action_item(array_name, "Delete")
        self.__admin_console.click_button_using_text('Yes')
        self.__admin_console.get_notification(wait_time=60)

    @PageService()
    def action_list_snaps(self, array_name):
        """
        Lists the snaps on the array with the given name

        Args :
            array_name   (str)   --  the name of the array whose snaps are to listed
        """
        self.__admin_console.refresh_page()
        self.__table.access_action_item(array_name, self.__admin_console.props['action.listSnaps'])

    @PageService()
    def delete_snaps(self, job_id):
        """
        Deletes the snap with the given jobids in the List for an array
        Args :

        job_ids   (string)   --  list of job id

        """
        self.__navigator.close_popup()
        self.__table.access_action_item(job_id, self.__admin_console.props['action.delete'])
        self.dialog.click_submit()
        del_jobid = self.__admin_console.get_jobid_from_popup()
        return del_jobid

    def array_list(self):
        """
        Lists all the arrays.
        :return array_info  (dict)  --  dict of all arrays, contains name, username and snap vendor.
        """
        self.log.info("Listing all the arrays")
        array_info = []
        array_table = self.__driver.find_element_by_id('snapArrayTable')
        rows = array_table.find_elements_by_xpath(".//tr")[1:]
        for row in rows:
            name = row.find_element_by_xpath(".//td[1]").text
            user_name = row.find_element_by_xpath(".//td[2]").text
            snap_vendor = row.find_element_by_xpath(".//td[3]").text
            array_info.append({'name': name, 'user_name':user_name, 'snap_vendor': snap_vendor})
        self.log.info(arrays_info)
        return array_info

    def delete_all_snapshots(self):
        """
        Delete all  the snaps on the array
        """
        self.log.info("Deleting snaps for the given array")
        self.driver.find_element_by_xpath(
            "//div[@ng-model = 'grid.selection.selectAll']").click()
        if self.check_if_entity_exists("link", "Delete"):
            self.select_hyperlink("Delete")
            self.click_button_using_text('Yes')
            job_text = self.get_notification()
            if not job_text:
                raise Exception("Could not delete snaps")
            job_text = re.findall(r'\d+', job_text)[0]
            return job_text
        else:
            raise Exception("There is no option to delete snaps for the array")


class _General():
    """
    Class for adding general details of Arrays
    """

    def __init__(self, admin_console):
        """
        Method to initate General Class
        """
        self._admin_console = admin_console
        self._table = Table(self._admin_console)
        self._drop_down = DropDown(self._admin_console)
        self._navigator = admin_console.navigator

    @PageService()
    def add_general(self,
                    array_vendor,
                    array_name,
                    username,
                    password,
                    control_host=None,
                    Credentials=False):
        """

        Args:
            snap_vendor:   (str)            -- Name of the snap vendor
            array_name:    (str)            -- Name of the array
            control_host:  (str)            -- Name of the control host
            user_name:     (str)            -- username of the array
            password:      (str)            -- password of the array
            Credentials :  (str)            -- saved Credentials of the array


        """
        self._table.access_toolbar_menu('AddSnapArray')
        self._admin_console.select_value_from_dropdown("vendorInput", array_vendor)
        self._admin_console.fill_form_by_id("arrayNameInput", array_name)
        if self._admin_console.check_if_entity_exists("id", "arrayHostInput"):
            self._admin_console.fill_form_by_id("arrayHostInput", control_host)
        if Credentials is False:
            self._admin_console.disable_toggle(index=0)
        self._admin_console.fill_form_by_id("userName", username)
        self._admin_console.fill_form_by_id("password", password)

class _ArrayAccessNodes():
    """
    Class for adding Array access nodes for the array
    """

    def __init__(self, admin_console):
        """
        Method to initate General Class
        """
        self._admin_console = admin_console
        self._table = Table(self._admin_console)
        self._drop_down = DropDown(self._admin_console)

    @PageService()
    def add_array_accessnodes(self, controllers):
        """
        Args:
            controllers:  (list)        -- list the array controllers
        """
        if controllers:
            self._drop_down.select_drop_down_values(values=[controllers], drop_down_id='ArrayControllers')
            self._admin_console.wait_for_completion()

class _Snap_config():

    def __init__(self, admin_console,csdb):

        """
        Args:
            admin_console:  (obj)        -- browser object

            csdb: (obj)                  -- database object
        """

        self.csdb = csdb
        self._admin_console = admin_console
        self.log = logger.get_log()
        self.get_master_config_id = """SELECT config.id FROM SMMasterConfigs AS config
                                                    INNER JOIN SMVendor AS vendor ON config.VendorId = vendor.id		
                                                    WHERE config.Name LIKE '{a}' AND vendor.Name LIKE '{b}'"""
        self.get_type_input = """SELECT Type from SMMasterConfigs WHERE id = {a}"""
        self.snap_config = """ SELECT Name from SMMasterConfigs WHERE id = {a}"""

    @PageService()
    def add_snapconfig(self, snap_configs, array_vendor):
        """

        Args:
            snap_configs:  (str)        -- Snap configs to be updated
            array_vendor:  (str)        -- array vendor name

        """

        import json
        if snap_configs is not None and type(snap_configs) is not dict:
            snap_configs = json.loads(snap_configs)
        if snap_configs is not None:
            config_data = {}
            self.log.info(f"snap config {snap_configs}, array_vendor {array_vendor}")
            for config, value in snap_configs.items():
                self.csdb.execute(self.get_master_config_id.format(
                    **{'a': config, 'b': array_vendor}))
                master_config_id = self.csdb.fetch_one_row()[0]
                self.log.info(f"Config ID: {master_config_id}, value: {value}")
                config_data[master_config_id] = value

                self.csdb.execute(self.snap_config.format(**{'a': master_config_id}))
                toggle_config = self.csdb.fetch_one_row()[0]
                self.csdb.execute(self.get_type_input.format(**{'a': master_config_id}))
                type_1 = self.csdb.fetch_one_row()[0]

                if str(type_1) == '1':
                    if self._admin_console.check_if_entity_exists("xpath",
                                                                f" //*[contains(text(),'{toggle_config}')]/following::toggle-control/div[contains(@class,'cv-material-toggle cv-toggle')]"):
                        self._admin_console.enable_toggle(toggle_id=master_config_id)

                if type_1 != '1':
                    if self._admin_console.check_if_entity_exists("name", "snapConfigInput"):
                        self._admin_console.fill_form_by_id(master_config_id, value)
                        self._admin_console.wait_for_completion()

        else:
            config_data = None



class Engine(_General, _ArrayAccessNodes , _Snap_config):

    def __init__(self, admin_console , csdb):

        super().__init__(admin_console)

        from AutomationUtils.database_helper import get_csdb

        self._csdb = get_csdb()
        self._admin_console = admin_console
        self.log = logger.get_log()
        self.csdb = csdb
        self.get_master_config_id = """SELECT config.id FROM SMMasterConfigs AS config
                                                                   INNER JOIN SMVendor AS vendor ON config.VendorId = vendor.id		
                                                                   WHERE config.Name LIKE '{a}' AND vendor.Name LIKE '{b}'"""
        self.get_type_input = """SELECT Type from SMMasterConfigs WHERE id = {a}"""
        self.snap_config = """ SELECT Name from SMMasterConfigs WHERE id = {a}"""

    @PageService()
    def add_engine(self, array_vendor,
                   array_name,
                   username,
                   password,
                   control_host,
                   controllers,
                   snap_config):
       """

       Args:
           array_vendor: select the array vendor
           array_name:   (str)   --     name of the array
           username:     (str)   --     username of the array
           password:     (str)   --     password of the array
           control_host: (int)   --     control host of the array
           controllers:  (str)   --     name of the array controller
           snap_config:  (str)   --     Edit the snap configuration of the array


       """
       self.add_general(array_vendor, array_name, username, password, control_host)
       self._admin_console.button_next()
       if controllers:
           self.add_array_accessnodes(controllers)
           self._admin_console.click_button('Next')
           self._admin_console.wait_for_completion()
       if snap_config:
           self.add_snapconfig(snap_config, array_vendor)
       self._admin_console.click_button("Save")
       self._admin_console.wait_for_completion()

