# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
This module provides the function or operations that can be performed on the
Arrays Details Page on the Alert page of AdminConsole

Class:

    ArrayDetails() -> Arrays() -> AdminPage() -> login_page() -> AdminConsoleBase() -> object()

Functions:

array_info()                    --  Displays all the information about the array
delete_array()                  --  Deletes the array
edit_snap_configuration()       --  Editing the snap configuration of the array

"""

from Web.AdminConsole.Components.panel import PanelInfo ,ModalPanel
from Web.Common.page_object import PageService
from Web.AdminConsole.Components.table import Table
from Web.AdminConsole.AdminConsolePages.Arrays import _Snap_config
from AutomationUtils.database_helper import get_csdb

class ArrayDetails():
    """
    this class provides the function or operations that can be performed on the Arrays Details Page
    """
    def __init__(self, admin_console):

        """Method to initiate Array Details Class
                Args:
                        admin_console   (Object) :   Admin Console Class object"""

        self.__driver = admin_console.driver
        self.__panel_info = None
        self.__admin_console = admin_console
        self.__modal_panel = ModalPanel(self.__admin_console)
        self.__navigator = admin_console.navigator
        self.__table = Table(self.__admin_console)
        self.__csdb = get_csdb()
        self.snap_config_obj = _Snap_config(self.__admin_console, self.__csdb)

    @PageService()
    def array_info(self):
        """
        Displays all the information about the array
        :return array_info  (dict)  --  info about the array
        """
        panel_details = PanelInfo(self)
        return panel_details.get_details()

    @PageService()
    def delete_array(self):
        """
        Deletes the array
        """

        self.__admin_console.click_button_using_text('Delete')
        self.__admin_console.wait_for_completion()
        self.__admin_console.click_button_using_text('Yes')

    @PageService()
    def edit_snap_configuration(self,
                                array_vendor,
                                array_name,
                                snap_config=None):
        """
                Editing the snap configuration of the array
                Args:

                    snap_vendor                  (str)   --  the name of the vendor
                    array_name                   (str)   --  the name of the array to be added
                    snap_config                  (str)   --  snap configurations to be edited
        """
        self.__navigator.navigate_to_arrays()
        self.__table.access_link(array_name)
        PanelInfo(self.__admin_console, 'Snap configurations').edit_tile()
        self.__admin_console.wait_for_completion()
        self.snap_config_obj.add_snapconfig(snap_config, array_vendor)
        self.__admin_console.click_button("Save")
        self.__admin_console.wait_for_completion()

    @PageService()
    def edit_general(self,
                     array_name , region):
        """

        Args:
            array_name: name of the array
            region: region

        """

        self.__navigator.navigate_to_arrays()
        self.__table.access_link(array_name)
        PanelInfo(self.__admin_console, 'General').edit_tile()
        self.__admin_console.wait_for_completion()
        self.__admin_console.cvselect_from_dropdown("Region", region)
        self.__admin_console.click_button("Save")
        self.__admin_console.wait_for_completion()

