# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
This module provides the function or operations that can be performed on the
Alerts page on the AdminConsole

Class:

    Alerts() -> AdminPage() -> login_page() -> AdminConsoleBase() -> object()

Functions:

select_triggered_alerts()      --  Method to select and open the triggered alerts tab

select_alert_definitions()     --  Method to select and open the alert definitions tab

delete_triggered_alert()       --  Method to delete single triggered alert

delete_all_triggered_alerts()  --  Method to delete all triggered alerts

dump_alerts_info()             --  prints the alert info for a scpecific machine and alert

all_triggered_alerts_info()    --  prints all the alerts triggered for all the clients

select_by_alert_type()         --  Displays only the alerts of the given type

"""
import time

from Web.AdminConsole.Components.table import Table
from Web.Common.page_object import (
    WebAction,
    PageService
)
from Web.Common.exceptions import CVWebAutomationException


class Alerts:
    """
    This class provides the function or operations that can be performed on the Alerts page
    """

    def __init__(self, admin_console):
        self.__admin_console = admin_console
        self.__table = Table(admin_console)
        self.log = admin_console.log
        self.driver = admin_console.driver

    @WebAction()
    def __select_alerts_navigation_link(self, linkText):
        """
        Selects the triggered alerts link

        Args:

            linkText(string) : the link to be selected triggeredAlerts/ alertDefinitions
        Raises:
            Exception:
                -- if fails to navigate to respective alerts page
        """
        if self.__admin_console.check_if_entity_exists("xpath", "//a[@href='/adminconsole/#/"+linkText+"']"):
            self.driver.find_element_by_xpath(
                "//a[@href='/adminconsole/#/"+linkText+"']").click()
            self.__admin_console.wait_for_completion()
        else:
            raise CVWebAutomationException("There is no link with text "+str(linkText))

    @PageService()
    def select_triggered_alerts(self):
        """
        Method to navigate to triggered alerts page
        """
        self.__select_alerts_navigation_link('triggeredAlerts')

    @PageService()
    def select_alert_definitions(self):
        """
        Method to navigate to alerts definition page
        """
        self.__select_alerts_navigation_link('alertsDefinitions')

    @WebAction()
    def __select_ok_from_prompt(self):
        """Clicks the confirm button for alert prompts"""
        self.driver.find_element_by_xpath("//button[@class='btn btn-primary ng-binding']").click()

    @PageService()
    def delete_current_triggered_alert(self, alert_name, client_name):
        """
        Method to delete a triggered alert

        Args:
           alert_name   (string) : Name of the alert to be deleted

           client_name  (string) : Name of client for which triggered alert to be deleted

        Returns:
            None

        Raises:
            Exception:
                -- if fails to delete the triggered alert
        """
        if not self.__admin_console.check_if_entity_exists(
                    "xpath", "//div[@class='cv-k-grid-empty-content page-details-list grid-empty']"):
            self.__table.apply_filter_over_column('Client name', client_name)
            self.__table.access_context_action_item(alert_name, 'Delete')
            self.__admin_console.wait_for_completion()
            self.__select_ok_from_prompt()
            time.sleep(1)
            notification_text = self.__admin_console.get_notification()
            if len(notification_text) == 0:
                exp = "Alert deletion message not displayed"
                raise CVWebAutomationException(exp)
        else:
            self.log.info("No Triggered alerts displayed on the page")

    @PageService()
    def delete_all_triggered_alerts(self):
        """
        Method to delete all triggered alerts

        Args:
           None

        Returns:
            None

        Raises:
            Exception:
                -- if fails to delete the triggered alerts
        """
        if not self.__admin_console.check_if_entity_exists(
                    "xpath", "//div[@class='cv-k-grid-empty-content page-details-list grid-empty']"):
            self.__table.select_all_rows()
            self.__admin_console.select_hyperlink('Delete')
            self.__admin_console.wait_for_completion()
            self.__select_ok_from_prompt()
            time.sleep(1)
            notification_text = self.__admin_console.get_notification()
            if len(notification_text) == 0:
                exp = "Alert deletion message not displayed"
                raise CVWebAutomationException(exp)
        else:
            self.log.info("No Triggered alerts displayed on the page")
