# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
This module provides the function or operations that can be performed on the
alert definitions tab on the Alerts page on the AdminConsole

Class:

    AlertDefinitions() -> Alerts() -> AdminPage() -> login_page() -> AdminConsoleBase() -> object

Functions:

create_alert_definition()        --   Method to Add a new alert definition

delete_alert_definition()        --   Method to delete an alert definition

enable_alert_definition()        --   Method to enable an alert definition

disable_alert_definition()       --   Method to disable an alert definition

select_alert()                   --   Method to open an alert definition

"""
import time
from Web.AdminConsole.Components.table import Table
from Web.Common.page_object import PageService, WebAction
from Web.Common.exceptions import CVWebAutomationException



class AlertDefinitions:
    """
    This module provides the function or operations that can be performed on the
    alert definitions tab on the Alerts page on the AdminConsole
    """

    def __init__(self, admin_console):
        self.__admin_console = admin_console
        self.__table = Table(admin_console)
        self.log = admin_console.log
        self.driver = admin_console.driver

    @WebAction()
    def __get_collapsed_server_elements(self):
        """ Return collapsed server elements"""
        collapsed_elements = self.driver.find_elements_by_xpath(
                "//button[@class='collapsed']")
        return collapsed_elements

    @WebAction()
    def __get_serverlist_entities(self):
        """ Return server list entity list"""
        server_list_entities = self.driver.find_element_by_xpath(
                                "//div[@class='browse-tree alerts-browse']")
        return server_list_entities

    @WebAction()
    def __select_server_entity(self, value, entities):
        """ Selects the given server entity"""
        entities.find_element_by_xpath(
            "//span[./text()='" + str(value) + "']").click()

    @WebAction()
    def __fill_to_input(self, value):
        """Fills the to input field"""
        self.driver.find_element_by_xpath(
            "//div[@id='s2id_toUserList']/ul/li/input").send_keys(value)

    @WebAction()
    def __fill_cc_input(self, value):
        """Fills the cc input field"""
        self.driver.find_element_by_xpath(
            "//div[@id='s2id_ccUserList']/ul/li/input").send_keys(value)

    @WebAction()
    def __fill_bcc_input(self, value):
        """Fills the bcc input field"""
        self.driver.find_element_by_xpath(
            "//div[@id='s2id_bccUserList']/ul/li/input").send_keys(value)

    @WebAction()
    def __select_user_entity(self, value):
        """Selects the user entity"""
        self.driver.find_element_by_xpath("//span[@class='user-type plan-user-suggestion' and contains\
                                            (text(),' (" + str(value) + ")" + "')]").click()

    @WebAction()
    def __select_group_entity(self, value):
        """Selects the group entity"""
        self.driver.find_element_by_xpath("//span[@class='group-type plan-user-suggestion' and contains\
                                            (text(),' (" + str(value) + ")" + "')]").click()

    @WebAction()
    def __select_email_entity(self, value):
        """Selects the email entity"""
        self.driver.find_element_by_xpath("//span[@class='email-type plan-user-suggestion' and contains\
                                            (text(),'" + str(value) + "')]").click()

    @WebAction()
    def __select_email_dropdown(self):
        """Select the email dropdown"""
        self.driver.find_element_by_xpath("//div[@id='select2-drop-mask']").click()

    @WebAction()
    def __get_error_text(self):
        """Returns the error text message"""
        text = self.driver.find_element_by_xpath("//span[@class='server-message error']").text
        return text

    @WebAction()
    def __click_error_close_button(self):
        """Clicks the error message close button"""
        self.driver.find_element_by_xpath("//button[@class='btn btn-default cvBusyOnAjax']").click()

    @WebAction()
    def __button_next(self, step_number):
        """
            Clicks the Next button on addAlertDefinitionForm template
        Args:
            step_number: The step at which we are in creating alert definition. (Step starts at 1)
        """
        self.driver.find_element_by_xpath(f"(//button[contains(.,'Next')])[{step_number}]").click()
        self.__admin_console.wait_for_completion()

    @PageService()
    def create_alert_definition(self,
                                alert_name,
                                alert_type,
                                to_recipients,
                                cc_recipients,
                                bcc_recipients,
                                value_of_x,
                                alert_entities,
                                alert_target,
                                ind_notification=None,
                                alert_locale=None,
                                mail_subject=None):
        """
        Function to create Alert Definition

        Args:
            alert_name      (str)       : Name of alert to be created

            alert_type      (str)       : Type of alert to be created

            to_recipients  (dict)       : user, user group or email to be added to alert
                                            recipients list
                Eg. to_recipients = {"Group":["master"],
                                     "Email":["jrana@commvault.com"],
                                     "User":["admin"]}

            bcc_recipients  (dict)      : user, user group or email to be added to alert
                                            recipients list
                Eg. to_recipients = {"Group":["master"],
                                     "Email":["jrana@commvault.com"],
                                     "User":["admin"]}

            cc_recipients    (dict)     : user, user group or email to be added to alert
                                            recipients list
                Eg. to_recipients = {"Group":["master"],
                                     "Email":["jrana@commvault.com"],
                                     "User":["admin"]}

            value_of_x        (integer) : value for days/time/percent based on alert selected

            alert_entities    (list)    : server groups or clients for which alert is to be
                                            defined
                Eg. alert_entities = ["Server groups", "Clients"]

            alert_target      (dict)    : dict containing values if alert targets be
                                            enabled/disabled
                Eg. alert_target = {'Email':True, 'Event viewer':True,
                                    'Console':True, 'SNMP':True}

            ind_notification  (str)    :  individual notification to be enabled/disabled
                Eg. - ind_notification = 'ON'

            alert_locale      (str)    :   alert locale to be selected
                Eg. - 'English'

            mail_subject      (str)    :   subject line for the mailer to be updated
                Eg. - 'Test Alert'

        Returns:
            None

        Raises:
            Exception:
                -- if fails to create alert definition
        """

        to_list = []
        cc_list = []
        bcc_list = []

        for group in to_recipients['Group']:
            to_list.append(group)
        for user in to_recipients['User']:
            to_list.append(user)
        for email in to_recipients['Email']:
            to_list.append(email)
        for group in cc_recipients['Group']:
            cc_list.append(group)
        for user in cc_recipients['User']:
            cc_list.append(user)
        for email in cc_recipients['Email']:
            cc_list.append(email)
        for group in bcc_recipients['Group']:
            bcc_list.append(group)
        for user in bcc_recipients['User']:
            bcc_list.append(user)
        for email in bcc_recipients['Email']:
            bcc_list.append(email)

        self.__admin_console.access_menu("Add alert definition")
        self.__admin_console.wait_for_completion()

        self.__admin_console.fill_form_by_id("alertName", alert_name)

        self.__admin_console.cv_single_select('Alert type', alert_type)

        if self.__admin_console.check_if_entity_exists("id", "alertParam"):
            self.__admin_console.fill_form_by_id("alertParam", value_of_x)

        if ind_notification == 'ON':
            self.__admin_console.enable_toggle(toggle_id='alertNotifIndividual')
        else:
            self.__admin_console.disable_toggle(toggle_id='alertNotifIndividual')

        self.__button_next(1)

        if self.__admin_console.check_if_entity_exists(
                "xpath", "//button[@class='collapsed']"):
            collapsed_elems = self.__get_collapsed_server_elements()
            for elem in collapsed_elems:
                elem.click()
            entities = self.__get_serverlist_entities()
        for value in alert_entities['server_group_list']:
            self.__select_server_entity(value, entities)
        for value in alert_entities['server_list']:
            self.__select_server_entity(value, entities)
        self.__admin_console.wait_for_completion()

        self.__button_next(2)

        if alert_target:
            if alert_target['Email']:
                self.__admin_console.checkbox_select('alertNotifTypeEmail')

                if to_list:
                    self.log.info("Adding recipients in TO LIST")
                    for value in to_list:
                        self.__fill_to_input(value)
                        self.__admin_console.wait_for_completion()
                        if self.__admin_console.check_if_entity_exists(
                                "xpath", "//span[@class='group-type plan-user-suggestion' and \
                            contains(text(),' ("+str(value)+")"+"')]"):
                            self.__select_group_entity(value)
                        elif self.__admin_console.check_if_entity_exists(
                                "xpath", "//span[@class='user-type plan-user-suggestion' and \
                            contains(text(),' ("+str(value)+")"+"')]"):
                            self.__select_user_entity(value)
                        elif self.__admin_console.check_if_entity_exists(
                                "xpath", "//span[@class='email-type plan-user-suggestion' and \
                            contains(text(),'"+str(value)+"')]"):
                            self.__select_email_entity(value)
                        self.__admin_console.wait_for_completion()

                if cc_list:
                    self.log.info("Adding recipients in CC LIST")
                    for value in cc_list:
                        self.__fill_cc_input(value)
                        self.__admin_console.wait_for_completion()
                        if self.__admin_console.check_if_entity_exists(
                                "xpath", "//span[@class='group-type plan-user-suggestion' and \
                            contains(text(),' ("+str(value)+")"+"')]"):
                            self.__select_group_entity(value)
                        elif self.__admin_console.check_if_entity_exists(
                                "xpath", "//span[@class='user-type plan-user-suggestion' and \
                            contains(text(),' ("+str(value)+")"+"')]"):
                            self.__select_user_entity(value)
                        elif self.__admin_console.check_if_entity_exists(
                                "xpath", "//span[@class='email-type plan-user-suggestion' and \
                            contains(text(),'"+str(value)+"')]"):
                            if self.__admin_console.check_if_entity_exists(
                                    "xpath", "//div[@id='select2-drop-mask']"):
                                self.__select_email_dropdown()
                            self.__select_email_entity(value)
                        self.__admin_console.wait_for_completion()

                if bcc_list:
                    self.log.info("Adding recipients in BCC LIST")
                    for value in bcc_list:
                        self.__fill_bcc_input(value)
                        self.__admin_console.wait_for_completion()
                        if self.__admin_console.check_if_entity_exists(
                                "xpath", "//span[@class='group-type plan-user-suggestion' and \
                            contains(text(),' ("+str(value)+")"+"')]"):
                            self.__select_group_entity(value)
                        elif self.__admin_console.check_if_entity_exists(
                                "xpath", "//span[@class='user-type plan-user-suggestion' and \
                            contains(text(),' ("+str(value)+")"+"')]"):
                            self.__select_user_entity(value)
                        elif self.__admin_console.check_if_entity_exists(
                                "xpath", "//span[@class='email-type plan-user-suggestion' and \
                            contains(text(),'"+str(value)+"')]"):
                            if self.__admin_console.check_if_entity_exists(
                                    "xpath", "//div[@id='select2-drop-mask']"):
                                self.__select_email_dropdown()
                            self.__select_email_entity(value)
                        self.__admin_console.wait_for_completion()
            else:
                self.__admin_console.checkbox_deselect('alertNotifTypeEmail')

            if alert_target['Event viewer']:
                self.__admin_console.checkbox_select('alertNotifTypeEV')
            else:
                self.__admin_console.checkbox_deselect('alertNotifTypeEV')

            if alert_target['Console']:
                self.__admin_console.checkbox_select('alertNotifTypeCA')
                if not alert_target['Email']:
                    if to_list:
                        self.log.info("Adding recipients in TO LIST")
                        for value in to_list:
                            self.__fill_to_input(value)
                            self.__admin_console.wait_for_completion()
                            if self.__admin_console.check_if_entity_exists(
                                    "xpath", "//span[@class='group-type plan-user-suggestion' and \
                                contains(text(),' ("+str(value)+")"+"')]"):
                                self.__select_group_entity(value)
                            elif self.__admin_console.check_if_entity_exists(
                                    "xpath", "//span[@class='user-type plan-user-suggestion' and \
                                contains(text(),' ("+str(value)+")"+"')]"):
                                self.__select_user_entity(value)
                            elif self.__admin_console.check_if_entity_exists(
                                    "xpath", "//span[@class='email-type plan-user-suggestion' and \
                                contains(text(),'"+str(value)+"')]"):
                                if self.__admin_console.check_if_entity_exists(
                                        "xpath", "//div[@id='select2-drop-mask']"):
                                    self.__select_email_dropdown()
                                self.__select_email_entity(value)
                            self.__admin_console.wait_for_completion()
            else:
                self.__admin_console.checkbox_deselect('alertNotifTypeCA')

            if alert_target['SNMP']:
                self.__admin_console.checkbox_select('alertNotifTypeSNMP')
            else:
                self.__admin_console.checkbox_deselect('alertNotifTypeSNMP')

        self.__button_next(3)

        if alert_locale:
            self.__admin_console.select_value_from_dropdown("alertLocale", alert_locale)
        if mail_subject:
            self.__admin_console.fill_form_by_id("alertNotifEmailSub", mail_subject)

        self.__admin_console.submit_form()
        self.__admin_console.wait_for_completion()

        if self.__admin_console.check_if_entity_exists(
                "xpath", "//span[@class='server-message error']"):
            exp = self.__get_error_text()
            if "doesn't have [Create Alert] permission" in exp:
                self.__click_error_close_button()
            else:
                self.__click_error_close_button()
                raise CVWebAutomationException(exp)
        self.__admin_console.check_error_message()

    @PageService()
    def delete_alert_definition(self, alert_name):
        """
        Method to delete alert definition

        Args:
           alert_name (string): Name of the alert to be deleted

        Returns:
            None

        Raises:
            Exception:
                -- if fails to delete the alert
        """
        self.__table.access_context_action_item(alert_name, 'Delete')
        self.__admin_console.click_button_using_text('Yes')
        time.sleep(1)
        notif_text = self.__admin_console.get_notification()
        if len(notif_text) == 0:
            exp = "Delete Alert confirmation message is not displayed"
            raise CVWebAutomationException(exp)

    @WebAction()
    def __select_alert_checkbox(self, alert_name):
        """
        Select alert_checkbox
        Args:
            alert_name (string) : name of the alert to be selected

            Raise:
                Exception if element is not present.
        """
        if self.__admin_console.check_if_entity_exists("xpath", "//a[contains(text(),'" + alert_name + "')]"):
            status_check_box = self.driver.find_element_by_xpath('//input[@class="status-checkbox"]')
            status_check_box_container = self.driver.find_element_by_xpath('//a[@class="k-grid-status"]')
            if not status_check_box.is_selected():
                status_check_box_container.click()
        else:
            exp = "Alert {} not present".format(alert_name)
            raise CVWebAutomationException(exp)

    @WebAction()
    def __deselect_alert_checkbox(self, alert_name):
        """
        De-Select alert_checkbox
        Args:
            alert_name (string) : name of the alert to be de-selected

            Raise:
                Exception if element is not present.
        """
        if self.__admin_console.check_if_entity_exists("xpath", "//a[contains(text(),'" + alert_name + "')]"):
            status_check_box = self.driver.find_element_by_xpath('//input[@class="status-checkbox"]')
            status_check_box_container = self.driver.find_element_by_xpath('//a[@class="k-grid-status"]')
            if status_check_box.is_selected():
                status_check_box_container.click()
        else:
            exp = "Alert {} not present".format(alert_name)
            raise CVWebAutomationException(exp)

    @PageService()
    def enable_alert_definition(self, alert_name):
        """Enables the alert definition

          Args:
                alert_name (string) : name of the alert to be disabled
        """
        self.__table.search_for(alert_name)
        self.__select_alert_checkbox(alert_name)

    @PageService()
    def disable_alert_definition(self, alert_name):
        """
        Disables a alert definition

        Args:
            alert_name (string) : name of the alert to be disabled
        """
        self.__table.search_for(alert_name)
        self.__deselect_alert_checkbox(alert_name)

    @PageService()
    def select_alert(self, alert_name):
        """
        Method to open alert details

        Args:
            alert_name (string) : Name of the alert to be opened

        Returns:
            None

        Raises:
            Exception:
                -- if fails to open alert details
        """
        self.__table.access_link(alert_name)
