# -*- coding: utf-8 -*-
# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
This module provides the function or operations that can be performed on the
Admin page on the AdminConsole

Class:

    AdminPage() -> AdminConsoleBase() -> object()

Functions:

navigate_to_hypervisors()               -- select and navigate to the *hypervisors* page
navigate_to_servers()                   -- select and navigate to the *servers* page
navigate_to_server_groups()             -- select and navigate to the *server groups* page
navigate_to_alerts()                    -- select and navigate to the *alerts* page
navigate_to_virtual_machines()          -- select and navigate to the *VMs* page
navigate_to_vm_groups()                 -- select and navigate to the *VMs groups* page
navigate_to_jobs()                      -- select and navigate to the *jobs* page
navigate_to_events()                    -- select and navigate to the *events* page
navigate_to_storage_policies()          -- select and navigate to the *storage policies* page
navigate_to_storage_targets()           -- select and navigate to the *storage targets* page
navigate_to_storage_pools()             -- select and navigate to the *storage pools* page
navigate_to_arrays()                    -- select and navigate to the *array* page
navigate_to_media_agents()              -- select and navigate to the *media agents* page
navigate_to_network_stores()            -- select and navigate to the *network stores* page
navigate_to_companies()                 -- select and navigate to the *companies* page
navigate_to_users()                     -- select and navigate to the *users* page
navigate_to_user_groups()               -- select and navigate to the *user groups* page
navigate_to_roles()                     -- select and navigate to the *roles* page
navigate_to_identity_servers()          -- select and navigate to the *identity servers* page
navigate_to_global_exceptions()         -- select and navigate to the *global exceptions* page
navigate_to_plugins()                   -- select and navigate to the *plugins* page
navigate_to_license()                   -- select and navigate to the *license* page
navigate_to_theme()                     -- select and navigate to the *theme* page
navigate_to_email_templates()           -- select and navigate to the *email template* page
navigate_to_navigation()                -- select and navigate to the *navigation* page
navigate_to_operation_window()          -- select and navigate to the *backup window* page
navigate_to_metrics_reporting()         -- select and navigate to the *metrics reporting* page
navigate_to_snmp()                      -- select and navigate to the *snmp* page
navigate_to_access_control()            -- select and navigate to the *access control* page
navigate_to_data()                      -- select and navigate to the *data* page
navigate_to_maintenance()               -- select and navigate to the *maintenance* page
navigate_to_certificate_administration()-- select and navigate to the *certificate
                                            administration* page
navigate_to_credential_manager()        -- select and navigate to the *credential manager* page
navigate_to_schedule_policies()         -- select and navigate to the *schedule policies* page
navigate_to_subclient_policies()        -- select and navigate to the *subclient policies* page
navigate_to_plan()                      -- select and navigate to the *plan* page
navigate_to_home()                      -- select and navigate to the *home* page
navigate_to_getting_started()           -- select and navigate to the *getting started* page
navigate_to_dashboard()                 -- select and navigate to the *dashboard* page
navigate_to_commcell()                  -- select and navigate to the *commcell* page
navigate_to_windows_servers()           -- select and navigate to the *windows servers* page
navigate_to_unix_servers()              -- select and navigate to the *unix servers* page
navigate_to_NAS_servers()               -- select and navigate to the *nas servers* page
navigate_to_devices()                   -- select and navigate to the *devices* page
navigate_to_db_instances()              -- select and navigate to the *db instances* page
navigate_to_archiving()                 -- select and navigate to the *archiving* page
navigate_to_office365()                 -- select and naviagte to the *office365* page
navigate_to_exchange()                  -- select and navigate to the *exchange* page
navigate_to_sharepoint()                -- select and navigate to the *sharepoint* page
navigate_to_cloud_apps()                -- select and navigate to the *cloud apps* page
navigate_to_oracle_ebs()                -- select and navigate to the *oracle ebs* page
navigate_to_replication_targets()       -- select and navigate to the *replication target* page
navigate_to_replication_groups()        -- select and navigate to the *replication groups* page
navigate_to_replication_monitor()       -- select and navigate to the *replication monitor* page
navigate_to_reports()                   -- select and navigate to the *reports* page
navigate_to_recovery_groups()           -- select and navigate to the *recovery groups* page
navigate_to_life_cycle_policies()       -- select and navigate to the *lifecycle policies* page
navigate_to_governance_apps()           -- select and navigate to the *Governance apps* page
navigate_to_workflows()                 -- select and navigate to the *Workflows* page
navigate_to_k8s_clusters()              -- select and navigate to the kubernetes "Clusters" page
navigate_to_k8s_appgroup()              -- select and navigate to the kubernetes "Application groups" page
navigate_to_k8s_applications()          -- select and navigate to the kubernetes "Application" page
logout()                                -- sign the current user out of the AdminConsole

"""
import time

from selenium.webdriver.support.ui import WebDriverWait
from selenium.webdriver.support import expected_conditions as ec
from selenium.webdriver.common.by import By
from selenium.webdriver.common.keys import Keys
from selenium.common.exceptions import NoSuchElementException

from Web.AdminConsole.Helper.AdminConsoleBase import AdminConsoleBase
from Web.Common.page_object import (
    WebAction,
    PageService
)
from AutomationUtils import config

_CONSTANTS = config.get_config()


class AdminPage(AdminConsoleBase):
    """
    This module provides the function or operations that can be performed on the
    Admin page on the AdminConsole
    """

    def __init__(self, driver):
        """
        Initializes the properties of the class for the selected locale

        Args:
            driver  (object):   instance of the browser class

        """
        super(AdminPage, self).__init__(driver)
        self._load_properties(self)

    @WebAction()
    def __search_navigation(self, nav):
        """
        Searches for a given nav bar element by the href
        Args:
            nav (basestring): string to search
        """
        global_search = self.driver.find_element_by_xpath("//input[@id='nav-search-field']")
        global_search.clear()
        global_search.send_keys(nav)

    @WebAction()
    def __access_by_id(self, id):
        """
        Access by id
        Args:
            id(basestring): id to acess
        """
        self.driver.find_element_by_xpath(
            f"//nav[@class='nav side-nav navigation']//a[@id='{id}']/span").click()

    @WebAction()
    def __generic_access_by_id(self, generic_id):
        """
        Access by id
        Args:
            id(basestring): id to acess
        """
        elem = self.driver.find_elements_by_xpath(f'//*[@id="{generic_id}"]')
        if elem is None:
            return False
        self.driver.find_elements_by_xpath(f'//*[@id="{generic_id}"]').click()
        return True

    @WebAction()
    def __access_by_link(self, href):
        """
        Access by link
        Args:
            href: link text to acess
        """
        self.driver.find_element_by_xpath(f"//a[@href='#/{href}']/span").click()

    def search_nav_by_id(self, nav, id):
        """
        Searches for a given nav bar element by the id
        Args:
            nav (basestring): string to search
            id (basestring) : id of element to navigate
        """
        self.__search_navigation(nav)
        self.wait_for_completion()
        self.__access_by_id(id)
        self.wait_for_completion()

    def __search_navs(self, nav, href):
        """
        Searches for a given nav bar element by the href
        Args:
            nav: string to search
            href: link text to acess
        """
        self.__search_navigation(nav)
        self.wait_for_completion()
        self.__access_by_link(href)
        self.wait_for_completion()

    @WebAction()
    def _click_home_icon(self):
        """ Method to click on home icon """
        home_icon = self.driver.find_element_by_xpath("//div[@class='logo-bar-default']")
        home_icon.click()

    @WebAction()
    def _click_user_settings_drop_down(self):
        """ Method to expand user settings drop down """
        user_settings_drop_down = self.driver.find_element_by_xpath(
            "//div[@data-ng-if='showUserSettingsDropdown']")
        user_settings_drop_down.click()
        self.wait_for_completion()

    @WebAction()
    def _click_logout(self):
        """ Method to click on logout option in user settings drop down"""
        logout = self.driver.find_element_by_xpath("//a[@id='user-header-logout']")
        logout.click()
        self.wait_for_completion()

    @WebAction()
    def _navigate_back_to_login_page(self):
        """ Method to navigate back to login page from post logout screen """
        back_link = self.driver.find_element_by_xpath("//div[@class='links']/a[1]")
        back_link.click()
        self.wait_for_completion()

    @WebAction()
    def __expand_company_selection_drop_down(self):
        """ Method to expand company selection drop down """
        self.driver.find_element_by_xpath(
            "//span[contains(text(),'Select a company')]").click()
        time.sleep(2)

    @WebAction()
    def __select_company_from_drop_down(self, company_name):
        """ Method to select company from drop down"""
        self.driver.find_element_by_xpath(
            f"//a[contains(text(),'{company_name}')]").click()

    @WebAction()
    def __entity_not_visible(self):
        """

        :return:
        """
        try:
            self.driver.find_element_by_xpath("//span[contains(text(),'No items found')]")
            return True
        except NoSuchElementException:
            self.log.info("Entity is visible")
            return False

    @PageService()
    def check_if_element_exists(self, nav):
        """ Method to check if element is visible in navigation bar

        Args:
            nav (basestring): string to search
        """
        self.__search_navigation(nav)
        self.wait_for_completion()
        if self.__entity_not_visible():
            return False
        return True

    @PageService()
    def check_if_id_exists(self, nav_id):
        """ Method to check if id is present in the source code

        Args:
            nav_id (basestring): string to search
        """
        try:
            self.driver.find_element_by_xpath(f'//*[@id="{nav_id}"]')
            return True
        except NoSuchElementException:
            self.log.info(nav_id+" is not present")
            return False

    @PageService()
    def navigate_to_virtualization(self):
        """
        Navigates to Virtualization page
        """
        self.search_nav_by_id(self.props['label.vsa'], 'navigationItem_vsa')

    @PageService()
    def navigate_to_hypervisors(self):
        """
        Navigates to hypervisors page
        """
        self.navigate_to_virtualization()
        self.access_tab(self.props['label.nav.hypervisors'])

    @PageService()
    def navigate_to_server_groups(self):
        """
        Navigates to server groups page
        """
        self.search_nav_by_id(self.props['label.nav.serverGroups'], "navigationItem_serverGroups")

    @PageService()
    def navigate_to_service_commcell(self):
        """Function to navigate to service commcell page"""
        self.navigate_to_systems()
        self.access_tile('tileMenuSelection_serviceCommcells')

    @PageService()
    def navigate_to_servers(self):
        """
        Navigates to active servers page
        """
        self.search_nav_by_id(self.props['label.nav.clients'], "navigationItem_clientGroupDetails")

    @PageService()
    def navigate_to_file_servers(self):
        """
        Navigates to file servers page
        """
        self.search_nav_by_id(self.props['label.nav.serversDashboard'], "navigationItem_fsServersList")

    @PageService()
    def navigate_to_alerts(self):
        """
        Navigates to alerts page
        """
        self.search_nav_by_id(self.props['label.nav.alerts'], "navigationItem_triggeredAlerts")

    @PageService()
    def navigate_to_virtual_machines(self):
        """
        Navigates to VMs page
        """
        self.navigate_to_virtualization()
        self.access_tab(self.props['label.nav.vms'])

    @PageService()
    def navigate_to_vm_groups(self):
        """
        Navigates to VM groups page
        """
        self.navigate_to_virtualization()
        self.access_tab(self.props['label.nav.vmGroups'])

    @PageService()
    def navigate_to_jobs(self):
        """
        Navigates to jobs page
        """
        self.search_nav_by_id(self.props['label.nav.jobs'], 'navigationItem_activeJobs')

    @PageService()
    def navigate_to_events(self):
        """
        Navigates to events page
        """
        self.search_nav_by_id(self.props['label.nav.events'], "navigationItem_events")

    @PageService()
    def navigate_to_storage_pools(self):
        """
        Navigates to storage pools page
        """
        self.search_nav_by_id(self.props['label.nav.storagePool'], "navigationItem_mstoragePool")

    @PageService()
    def navigate_to_arrays(self):
        """
        Navigates to Arrays page
        """
        self.navigate_to_infrastructure()
        self.access_tile('tileMenuSelection_snapArray')

    @PageService()
    def navigate_to_media_agents(self):
        """
        Navigates to Media Agents page
        """
        self.navigate_to_infrastructure()
        self.access_tile('tileMenuSelection_mediaAgent')

    @PageService()
    def navigate_to_network_stores(self):
        """
        Navigates to Media Agents page
        """
        self.search_nav_by_id(
            self.props['label.nav.networkStore'], 'navigationItem_networkStore'
        )

    @PageService()
    def navigate_to_infrastructure(self):
        """
        Navigates to Infrastructure
        """
        self.search_nav_by_id(self.props['label.nav.infrastructure'], "navigationItem_mInfrastructure")

    @PageService()
    def navigate_to_companies(self):
        """
        Navigates to Companies page
        """
        self.search_nav_by_id(self.props['label.nav.subscriptions'], "navigationItem_subscriptions")

    @PageService()
    def navigate_to_users(self):
        """
        Navigates to users page
        """
        self.search_nav_by_id(self.props['label.nav.users'], 'navigationItem_musers')
        self.access_tile('tileMenuSelection_users')

    @PageService()
    def navigate_to_user_groups(self):
        """
        Navigates to user groups page
        """
        self.search_nav_by_id(self.props['label.nav.userGroups'], "navigationItem_musers")
        self.access_tile('tileMenuSelection_userGroups')

    @PageService()
    def navigate_to_roles(self):
        """
        Navigates to roles page
        """
        self.search_nav_by_id(self.props['label.nav.roles'], "navigationItem_musers")
        self.access_tile('tileMenuSelection_roles')

    @PageService()
    def navigate_to_identity_servers(self):
        """
        Navigates to Identity Servers page
        """
        self.search_nav_by_id(self.props['label.nav.identityServers'], "navigationItem_musers")
        self.access_tile('tileMenuSelection_identityServers')

    @PageService()
    def navigate_to_credential_manager(self):
        """
        Navigates to Credential Manager page
        """
        self.search_nav_by_id(self.props['label.nav.credentialManager'], 'navigationItem_musers')
        self.access_tile("tileMenuSelection_credentialManager")

    @PageService()
    def navigate_to_key_management_servers(self):
        """
        Navigates to the Key management servers
        """
        self.search_nav_by_id(self.props['label.nav.keyManagement'], 'navigationItem_musers')
        self.access_tile("tileMenuSelection_keyManagement")

    @PageService()
    def navigate_to_global_exceptions(self):
        """
        Navigates to Global exceptions page
        """
        self.navigate_to_systems()
        self.access_tile("tileMenuSelection_globalExceptions")

    @PageService()
    def navigate_to_plugins(self):
        """
        Navigates to Plugins page
        """
        self.navigate_to_systems()
        self.access_tile("tileMenuSelection_plugin")

    @PageService()
    def navigate_to_operation_window(self):
        """
        Navigates to Operation Window page
        """
        self.navigate_to_systems()
        self.access_tile("tileMenuSelection_operationWindow")

    @PageService()
    def navigate_to_metrics_reporting(self):
        """
        Navigates to Metrics Reporting page
        """
        self.navigate_to_systems()
        self.access_tile('tileMenuSelection_metricsReporting')

    @PageService()
    def navigate_to_snmp(self):
        """
        Navigates to snmp page
        """
        self.navigate_to_systems()
        self.access_tile('tileMenuSelection_snmp')

    @PageService()
    def navigate_to_license(self):
        """
        Navigates to License page
        """
        self.search_nav_by_id(self.props['label.nav.license'], "navigationItem_license")

    @PageService()
    def navigate_to_theme(self):
        """
        Navigates to Theme page
        """
        self.navigate_to_customization()
        self.access_tile("tileMenuSelection_customization")

    @PageService()
    def navigate_to_email_templates(self):
        """
        Navigates to Email Templates page
        """
        self.navigate_to_customization()
        self.access_tile("tileMenuSelection_emailTemplates")

    @PageService()
    def navigate_to_navigation(self):
        """
        Navigates to navigation page
        """
        self.navigate_to_customization()
        self.access_tile("tileMenuSelection_navigationPreferences")

    @PageService()
    def navigate_to_customization(self):
        """
        Navigates to customization
        """
        self.search_nav_by_id(
            self.props['label.nav.masterCustomization'], "navigationItem_masterCustomization"
        )

    @PageService
    def navigate_to_operations(self):
        """
        Navigates to Operations
        """
        self.search_nav_by_id(self.props['label.nav.operations'], "navigationItem_operation")

    @PageService()
    def navigate_to_access_control(self):
        """
        Navigates to Access control page
        """
        self.navigate_to_systems()
        self.access_tile("tileMenuSelection_accessControl")

    @PageService()
    def navigate_to_data(self):
        """
        Navigates to Data page
        """
        self.__search_navs(self.props['label.nav.data'], "dataOptions")

    @PageService()
    def navigate_to_systems(self):
        """
        Navigates to Systems
        """
        self.search_nav_by_id(self.props['label.nav.system'], "navigationItem_settings")

    @PageService()
    def navigate_to_regions(self):
        """
        Navigates to Regions
        """
        self.search_nav_by_id(self.props['label.nav.regions'], "navigationItem_regions")

    @PageService()
    def navigate_to_maintenance(self):
        """
        Navigates to Maintenance page
        """
        self.navigate_to_systems()
        self.access_tile("tileMenuSelection_maintenance")

    @PageService()
    def navigate_to_certificate_administration(self):
        """
        Navigates to Certificate Administration page
        """
        self.__search_navs(self.props['label.nav.certificate'], "certificate")

    @PageService()
    def navigate_to_schedule_policies(self):
        """
        Navigates to Schedule policies page
        """
        self.__search_navs(self.props['label.nav.schedulePolicies'], "schedulePolicies")

    @PageService()
    def navigate_to_subclient_policies(self):
        """
        Navigates to subclient policies page
        """
        self.__search_navs(self.props['label.nav.subclientPolicies'], "subclientPolicies")

    @PageService()
    def navigate_to_plan(self):
        """
        Navigates to Plan page
        """
        self.search_nav_by_id(self.props['label.nav.profile'], "navigationItem_profile")

    @PageService()
    def navigate_to_home(self):
        """
        Navigates to the main page(servers)
        """
        self._click_home_icon()

    @PageService()
    def navigate_to_getting_started(self):
        """
        Navigates to the Getting Started page
        """
        self.search_nav_by_id(
            self.props['label.nav.gettingStarted'], "navigationItem_gettingStarted"
        )

    @PageService()
    def navigate_to_dashboard(self):
        """
        Navigates to the dashboard page
        """
        self.search_nav_by_id(self.props['label.nav.dashboard'], "navigationItem_dashboard")

    @PageService()
    def navigate_to_commcell(self):
        """
        Navigates to the commcell page
        """
        self.search_nav_by_id(self.props['label.nav.commCell'], "navigationItem_commCell")

    @PageService()
    def navigate_to_nas_servers(self):
        """
        Navigates to the NAS File servers Page
        """
        self.__search_navs(self.props['label.nav.NASFS'], "nasServers/")

    @PageService()
    def navigate_to_devices(self):
        """
        Navigates to the Devices page
        """
        self.__search_navs(self.props['label.nav.devices'], "devices")

    @PageService()
    def navigate_to_databases(self):
        """
        Navigate to databases
        """
        self.search_nav_by_id(self.props['label.dbs'], "navigationItem_dbs-new")

    @PageService()
    def navigate_to_db_instances(self):
        """
        Navigates to the DB instances page
        """
        self.navigate_to_databases()
        self.access_tab('Instances')

    @PageService()
    def navigate_to_archiving(self):
        """
        Navigates to the Archiving page
        """
        self.search_nav_by_id(self.props['label.nav.archiving'], "navigationItem_archiveFileServers")

    @PageService()
    def navigate_to_office365(self):
        """
        Navigates to the Office 365 Page
        """
        self.search_nav_by_id(self.props['label.nav.office365'], 'navigationItem_office365V2')

    @PageService()
    def navigate_to_exchange(self):
        """
        Navigates to the exchange Apps page
        """
        self.__search_navs(self.props['label.exchange'], "exchange")

    @PageService()
    def navigate_to_sharepoint(self):
        """
        Navigates to the sharepoint Apps page
        """
        self.__search_navs(self.props['label.sharepoint'], "sharepoint/")

    @PageService()
    def navigate_to_cloud_apps(self):
        """
        Navigates to the cloud apps page
        """
        self.search_nav_by_id(self.props['label.capps'], "navigationItem_cappsClients")

    @PageService()
    def navigate_to_oracle_ebs(self):
        """
        Navigates to the Oracle EBS page
        """
        self.__search_navs(self.props['label.ebs'], "ebsApps")

    @PageService()
    def navigate_to_replication_targets(self):
        """
        Navigates to the Replication Targets page
        """
        self.__search_navs(self.props['label.nav.lifeCyclePolicies'], "lifeCyclePolicies")

    @PageService()
    def navigate_to_replication_groups(self):
        """
        Navigates to the Replication Groups page
        """
        self.__search_navs(self.props['label.nav.replication'], "replicationGroups")

    @PageService()
    def navigate_to_replication_monitor(self):
        """
        Navigates to the Replication Monitor page
        """
        self.search_nav_by_id(self.props['label.nav.replication.monitor'], "navigationItem_vsaReplicationMonitor")

    @PageService()
    def navigate_to_continuous_replication(self):
        """
        Navigates to the continuous tab of the Replication monitor page
        """
        self.search_nav_by_id(self.props['label.nav.replication.monitor'], "navigationItem_vsaReplicationMonitor")
        self.access_tab("Continuous")

    @PageService()
    def navigate_to_reports(self):
        """
        Navigates to the reports page
        """
        self.search_nav_by_id(self.props['label.nav.reports'], "navigationItem_reports")

    @PageService()
    def navigate_to_big_data(self):
        """
        Navigates to the big data page
        """
        self.__search_navs(self.props['label.nav.bigDataApps'], "bigDataApps")

    @PageService()
    def navigate_to_recovery_groups(self):
        """
        Navigates to the recovery Groups page
        """
        self.__search_navs(self.props['label.nav.replication'], "nav/replicationParent")
        self.access_tab('Recovery groups')

    @PageService()
    def navigate_to_life_cycle_policies(self):
        """
        Navigates to the Life cycle policies page
        """
        self.__search_navs(self.props['label.nav.lifeCyclePolicies'], "lifeCyclePolicies")

    @PageService()
    def navigate_to_governance_apps(self):
        """
        Navigates to the Governance apps page
        """
        self.__search_navs(self.props['label.activate'], "activate")

    @PageService()
    def navigate_to_object_storage(self):
        """
        Navigates to object storage page
        """
        self.search_nav_by_id(self.props['label.nav.cloudStorage'], 'navigationItem_cloudStorageApp')

    @PageService()
    def navigate_to_storage_policies(self):
        """
        Navigates to storage policies page
        """
        self.__search_navs(self.props['label.nav.storage='], "storagePolicies")

    @PageService()
    def navigate_to_storage_targets(self):
        """
        Navigates to storage targets page
        """
        self.__search_navs(self.props['label.nav.storage'], "nav/mstoragePool")
        self.access_tile('Storage targets')

    @PageService()
    def navigate_to_disk_storage(self):
        """
        Navigates to disk storage page
        """
        self.search_nav_by_id(self.props['label.nav.disk'], "navigationItem_storagePoolDisk")

    @PageService()
    def navigate_to_tape_storage(self):
        """
        Navigates to tape storage page
        """
        self.search_nav_by_id(self.props['label.nav.tape'], "navigationItem_storageTape")

    @PageService()
    def navigate_to_cloud_storage(self):
        """
        Navigates to cloud storage page
        """
        self.search_nav_by_id(self.props['label.nav.cloud'], "navigationItem_storagePoolCloud")

    @PageService()
    def navigate_to_workflows(self):
        """
        Navigates to workflows page
        """
        self.search_nav_by_id(self.props['label.nav.webconsole.forms'], "navigationItem_formsNav")

    @PageService()
    def navigate_to_kubernetes(self):
         """
         Navigates to kubernetes page
         """
         self.search_nav_by_id(self.props['label.nav.kubernetes'], 'navigationItem_kubernetes')

    @PageService()
    def navigate_to_k8s_clusters(self):
        """
        Navigates to the Kubernetes clusters page
        """
        self.navigate_to_kubernetes()
        self.access_tab(self.props['label.nav.clusters'])

    @PageService()
    def navigate_to_k8s_appgroup(self):
        """
        Navigates to the Kubernetes application group page
        """
        self.navigate_to_k8s_clusters()
        self.access_tab(self.props['label.nav.applicationGroups'])

    @PageService()
    def navigate_to_k8s_applications(self):
        """
        Navigates to the Kubernetes application page
        """
        self.navigate_to_k8s_clusters()
        self.access_tab(self.props['label.nav.applications'])

    @PageService()
    def logout(self, url=None):
        """        Logs out from the current user

        Args:
                url      (str)   --  URL to be loaded

        """
        if not _CONSTANTS.SECURITY_TEST:  # skip logout while doing security test
            self._click_user_settings_drop_down()
            self._click_logout()
            if url:
                self.navigate(url)
            else:
                self._navigate_back_to_login_page()

    @PageService()
    def switch_company_as_operator(self, company_name):
        """
        To Select the company

        Args:
            company_name: name of the company that has to be selected

        Returns:
        """
        time.sleep(5)
        self.__expand_company_selection_drop_down()
        self.__select_company_from_drop_down(company_name)
        self.wait_for_completion()

    @WebAction()
    def __wait_for_universal_search_options(self, option_text):
        """
        Wait till options appear after typing into the universal search

        Args:
            option_text (str)   : text expected in the options
        """
        cmd_complete_xp = f"//ul[@class='dropdown-menu']/li//strong[./text()='{option_text}']"
        wait = WebDriverWait(self.driver, 30)
        wait.until(ec.presence_of_element_located((By.XPATH, cmd_complete_xp)))

    @WebAction()
    def __type_into_universal_search(self, search_string):
        """
        Args:
            search_string   (str):  key or string to typed in to universal search

        """

        if search_string in ["TAB", "UP", "DOWN"]:
            self.driver.find_element_by_id("globalSearchInput").send_keys(
                                                                    eval(f"Keys.{search_string}"))
        else:
            self.driver.find_element_by_id("globalSearchInput").send_keys(search_string)

    @WebAction()
    def __highlight_given_option(self, result_xpath):
        """Method to cycle through options till given option is highlighted"""

        while "active" not in self.driver.find_element_by_xpath(result_xpath).get_attribute(
                'class'):
            self.__type_into_universal_search("UP")
            self.wait_for_completion()

    @PageService()
    def __select_search_result(self, result, result_header=None):
        """
        Args:
            result            (str): Name of entity to be selected
            result_header     (str): Entity type to be selected

        """

        if result_header:
            result_xpath = f"//span[contains(text(),'{result_header}')]/../following-sibling::a/" \
                           f"div/strong[contains(text(),'{result}')]"
            self.scroll_into_view(result_xpath)
        else:
            result_xpath = f"//div[@class='custom-popup-wrapper ng-isolate-scope']" \
                           f"//strong[text()='{result}']/ancestor::li"
            self.scroll_into_view(result_xpath)
            self.__type_into_universal_search("DOWN")
            self.__highlight_given_option(result_xpath)

        self.__type_into_universal_search("TAB")

    @PageService()
    def add_entity_from_search_bar(self, entity_type):
        """Adds entity using /Add command on universal search bar
        Args:
            entity_type (str)   -- Type of entity to be added e.g. File server, Company, Plan etc.
        """

        self.fill_form_by_id("globalSearchInput", "/Add")
        self.__wait_for_universal_search_options("/Add")
        self.__type_into_universal_search("TAB")

        self.__type_into_universal_search(entity_type)
        self.__wait_for_universal_search_options(entity_type)
        self.__type_into_universal_search("TAB")

    @PageService()
    def manage_entity_from_search_bar(self, entity_type, entity_name, action):
        """Manage entity from search bar by searching entity and selecting action
           from the search bar

        Args:
            entity_type     (str)   -- Type of the entity
            entity_name     (str)   --  Name of the entity to manage
            action          (str)   -- Action you want to perform on the entity

        """

        self.fill_form_by_id("globalSearchInput", entity_name)
        self.__select_search_result(result=entity_name,
                                    result_header=entity_type)
        self.__type_into_universal_search(action)
        self.__wait_for_universal_search_options(action)
        self.__select_search_result(action)
