# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------
"""
Main file that does all operations on AliCloud

Classes:

AliHelper - Ali Cloud class that provides elastic compute service to automation

AliVm - AliThis is the main file for all  Ali Cloud VM operations

AliHelper:

build_request_string()	    --	Builds the request string to be used with the API requests

execute_action()		    --	Computes RPC URL based on Ali Cloud standard.

get_regions()			    --	Returns all the regions available.

get_all_vms_in_hypervisor()	--	gets all vms from all regions in ali cloud.

compute_free_resources()    --  computes the region, network and security groups based on proxy


AliVM:

delete_vm()			--	Delete the VM.

power_on()			--	power on the VM.

power_off()         --  power off the VM.

update_vm_info()    --  updates vm information to current state.

get_vm_basic_prop() --  gets the basic properties of the VM like guest os, IP, power state and GUID


"""

from datetime import datetime
import uuid
from urllib.parse import quote, urlencode
import hmac
import base64
import hashlib
import json
import requests
from VirtualServer.VSAUtils.HypervisorHelper import Hypervisor
from VirtualServer.VSAUtils.VMHelper import HypervisorVM


class AliCloudHelper(Hypervisor):
    """
    Ali Cloud class that provides elastic compute service to automation
    """

    # pylint: disable=too-many-arguments
    # pylint: disable=too-many-instance-attributes
    # This requirement is coming from base class.

    def __init__(self,
                 server_host_name,
                 host_machine,
                 user_name,
                 password,
                 instance_type,
                 commcell):
        """

        Args:

            server_host_name  (str) -- Url of Ali Cloud.defaults to public url

            host_machine (str)  -- the client of CS where we will execute all the powershell cmd

            user_name (str)  -- Access key of the user

            password (str)  -- Secret key corresponding to the access key.

            instance_type (str)  -- Will be defaulted to ALI , when object is created for
                                        hypervisor class.

            commcell (Object) -- commcell object created using python SDK.

        """
        # pylint: disable=too-many-instance-attributes
        # pylint: disable=unused-argument
        # We inherit all this from base class , it might be required for ali cloud ,but will be
        # required for other hypervisors. Cant do much.
        super(AliCloudHelper, self).__init__(server_host_name, host_machine,
                                             user_name, password, instance_type, commcell)
        self.ali_url = "https://ecs.aliyuncs.com/?"
        self.access_key = user_name
        self.secret_key = password
        self.secret_key += "&"
        self.ali_common_param = {
            "AccessKeyId": self.access_key,
            "Format": "JSON",
            "SignatureMethod": "HMAC-SHA1",
            "SignatureVersion": "1.0",
            "Version": "2014-05-26"
        }
        self.region_ids = []
        self.vm_list = []
        self.vm_region_map = {}
        self.vm_details = {}

    @staticmethod
    def build_request_string(table):
        """
        Builds a request string from the request params

        Args:
            table   (dict): the params to be sent with the request

        Returns:
            res (basestring):   encoded string of all the request params

        """
        items = sorted(iter(table.items()), key=lambda d: d[0])

        res = urlencode(items)
        res = res.replace('+', '%20').replace('*', '%2A').replace('%7E', '~')

        return res

    def execute_action(self, action, additional_param=None):
        """
        Computes RPC URL based on Ali Cloud standard.
        Args:

            action (str) -- Name of the Action to be performed.

            additional_param  (dict)   -- Key value pair of additional inputs to the action
            specified.

        Returns:

            Response from the RPC call if status is 200.

        Raises:

            When RPC call failed. Dumps response in the exception message.


        """

        # pylint: disable=too-many-locals
        # url encoding and signature calculation involves multiple steps and we need locals.
        retry = 0
        while retry < 3:

            if additional_param is None:
                additional_param = {}

            now = datetime.utcnow()
            utc = now.isoformat(timespec='seconds') + 'Z'
            api = self.ali_url

            import copy
            # Have dict per execute call
            # Hence multiple request can be made simultaneously
            params = copy.deepcopy(self.ali_common_param)
            params['Action'] = action
            params['SignatureNonce'] = str(uuid.uuid1())
            params['Timestamp'] = utc
            params.update(additional_param)

            encoded_params = self.build_request_string(params)
            string_to_sign = f"GET&%2F&{quote(encoded_params)}"

            # Generate signature
            digest = hmac.new(
                bytes(self.secret_key, 'UTF-8'),
                bytes(string_to_sign, 'UTF-8'),
                hashlib.sha1
            ).digest()
            sign = base64.standard_b64encode(digest)
            signature = str(sign, "UTF-8")
            params['Signature'] = signature

            url = api + self.build_request_string(params)

            response = requests.get(url)
            if response.status_code != 200:
                retry += 1
                continue
            elif response.status_code == 200 and response.content:
                return json.loads(response.content)

        raise Exception("Ali Cloud REST call failed, Response [{}] and message [{}] ".format(
            str(response),
            str(response.content)))

    def get_regions(self):
        """
        Returns all the regions available.

        Returns:
            region_list   (list)    -- List of regions in ali cloud
        """
        regions = self.execute_action("DescribeRegions")
        for region in regions["Regions"]["Region"]:
            self.region_ids.append(region["RegionId"])

    def get_all_vms_in_hypervisor(self, region_id=None):
        """

        gets all vms from all regions in ali cloud.

        Args:
            region_id   (basestring / list):   regions whose instances are to be obtained

        Returns:
            list of all instances in the regions

        """
        if region_id:
            if isinstance(region_id, str):
                regions = [region_id]
            elif isinstance(region_id, list):
                regions = region_id
        else:
            self.get_regions()
            regions = self.region_ids
        # self.region_ids = ['ap-southeast-1', 'ap-southeast-2']
        regions.remove('cn-wulanchabu')
        for region in regions:
            stop = True
            pageNumber = 1
            while stop:
                get_vms = {
                    "RegionId": region,
                    "PageNumber": pageNumber
                }
                instances = self.execute_action("DescribeInstances", get_vms)
                for instance in instances["Instances"]["Instance"]:
                    instance_name = instance["InstanceName"]
                    self.vm_region_map[instance_name] = region
                    self.vm_details[instance_name] = instance
                if instances['PageNumber']*instances['PageSize'] >= instances['TotalCount']:
                    stop = False
                pageNumber = pageNumber + 1

        return self.vm_region_map.keys()

    def compute_free_resources(self, proxy):
        """
        Compute the free resources of the endpoint depending on proxy and source instance

        Args:

            proxy   (basestring):   name of the proxy whose region has to be found out

        Returns:

            zone    (basestring):   the zone to restore the VMs to

            vpc     (basestring):   the vpc the restored VMs should be associated with

            security_groups (basestring):   the security group to be associated to

        Raises:
            Exception:
                if there is an error in computing the resources of the endpoint.

        """
        try:
            if proxy not in self.vm_region_map.keys():
                self.get_regions()
                self.get_all_vms_in_hypervisor()
            region = self.vm_region_map[proxy]

            get_network = {'RegionId': region, 'PageSize': 50}
            proxy_details = self.vm_details[proxy]
            networks = self.execute_action("DescribeNetworkInterfaces", get_network)
            for network in networks['NetworkInterfaceSets']['NetworkInterfaceSet']:
                if network['InstanceId'] == proxy_details['InstanceId']:
                    security_groups = network['SecurityGroupIds']['SecurityGroupId']
                    vpc = network['VSwitchId']
                    zone = network['ZoneId']
                    break

            return zone, vpc, security_groups
        except Exception as err:
            self.log.exception("An exception %s occurred in computing free resources"
                               " for restore", err)
            raise Exception(err)


class AliCloudVM(HypervisorVM):
    """
    This is the main file for all  Ali Cloud VM operations
    """

    # pylint: disable=too-many-instance-attributes
    # VM property mandates many attributes.

    def __init__(self, hv_obj, vm_name):
        """

        Args:
            hv_obj   (Object)    -- hypervisor object

            vm_name   (str)    -- name of the vm

        """
        super(AliCloudVM, self).__init__(hv_obj, vm_name)
        self.ali_cloud = hv_obj
        self.vm_name = vm_name
        self.no_of_cpu = ""
        self.memory = ""
        self.disk_count = 0
        self.region_id = self.ali_cloud.vm_region_map.get(vm_name, None)
        self.instance_id = self.ali_cloud.vm_details.get(vm_name, {}).get("InstanceId", None)
        self.vm_info = None
        self.guid = self.instance_id
        self.guest_os = None
        self.ip = None
        self.power_state = ""
        self.power_status_on = "Running"
        self.power_status_off = "Stopped"
        self.instance_type = None
        self._get_vm_info()
        self._basic_props_initialized = False
        self.update_vm_info()

    def delete_vm(self):
        """
        delete the VM.

        return:
                True - when delete is successful

                False - when delete failed
        """
        delete_instance = {
            "RegionId": self.region_id,
            "InstanceId": self.instance_id
        }
        self.ali_cloud.execute_action("DeleteInstance", delete_instance)
        return True

    def power_on(self):
        """
        power on the VM.

        return:
                True - when power on is successful

        Exception:
                When power on failed

        """
        start_instance = {
            "RegionId": self.region_id,
            "InstanceId": self.instance_id
        }
        self.ali_cloud.execute_action("StartInstance", start_instance)
        return True

    def power_off(self):
        """
        power off the VM.

        return:
                True - when power off is successful

        Exception:
                When power off failed

        """
        stop_instance = {
            "InstanceId": self.instance_id,
            "ForceStop": "true"
        }
        self.ali_cloud.execute_action("StopInstance", stop_instance)
        return True

    def update_vm_info(self, prop='Basic', os_info=False, force_update=False):
        """
        Updates vm properties to the current state.

        Args:
            prop            (basestring):   the type of properties to collect from VM

                possible values:
                    Basic - Collect just the basic properties like GUID, Power state, IP and OS
                    All -   Collect all the possible properties of the VM

            os_info         (bool):         If os information needs to be collected

            force_update    (bool):         If the properties have to be refreshed again

        Raises:
            Exception:
                if there is an error while updating the VM properties

        """
        if self.vm_info:
            if not self._basic_props_initialized or force_update:
                self.get_vm_basic_prop()
                self._basic_props_initialized = True

            if os_info or prop == 'All':
                self.get_vm_basic_prop()
                self.vm_guest_os = self.guest_os
                self.instance_type = self.vm_info['InstanceType']
                self.no_of_cpu = self.vm_info['Cpu']
                self.memory = self.vm_info['Memory']
                self.security_groups = self.vm_info['SecurityGroupIds']['SecurityGroupId']
                if self.vm_info['VpcAttributes']:
                    self.network = self.vm_info['VpcAttributes']['VpcId'] + "\\" + self.vm_info[
                        'VpcAttributes']['VSwitchId']
                self.get_drive_list()
            elif prop == 'Basic':
                self.get_vm_basic_prop()
        else:
            raise Exception("The VM properties has not been obtained yet")

    def _get_vm_info(self):
        """
        Get all VM Info related to the given VM.

        Raises:
            Exception:
                if there is an error while getting the VM information

        """
        try:
            self.region_id = None
            self.instance_id = None
            self.log.info("VM information :: Getting all information of VM [%s]", self.vm_name)
            if not self.region_id:
                self.ali_cloud.get_all_vms_in_hypervisor()
                self.region_id = self.ali_cloud.vm_region_map[self.vm_name]
                self.instance_id = self.ali_cloud.vm_details[self.vm_name]["InstanceId"]
            else:
                if not self.instance_id:
                    self.ali_cloud.get_all_vms_in_hypervisor(self.region_id)
                    self.instance_id = self.ali_cloud.vm_details[self.vm_name]["InstanceId"]
            vm_info = {
                "RegionId": self.region_id,
                "InstanceId": self.instance_id
            }
            self.log.info(vm_info)
            self.ali_cloud.ali_url = f"https://ecs.{self.region_id}.aliyuncs.com/?"
            self.vm_info = self.ali_cloud.execute_action("DescribeInstanceAttribute", vm_info)
            self.log.info(self.vm_info)

        except Exception as err:
            self.log.exception("Exception in get_vm_info")
            raise Exception(err)

    def get_vm_basic_prop(self):
        """
        Gets the basic properties of the VM like GUID, PowerState, GuestOS and IP

        Raises:
            Exception:
                if there is an exception with getting the basic properties

        """
        try:
            if not self.vm_info:
                raise Exception("The VM info has not been obtained yet.")

            self.guid = self.instance_id = self.vm_info["InstanceId"]
            self.power_state = self.vm_info["Status"]
            self.guest_os = self.ali_cloud.vm_details[self.vm_name]['OSType'].lower()

            if self.vm_info['PublicIpAddress']['IpAddress']:
                self.ip = self.vm_info['PublicIpAddress']['IpAddress'][0]
            else:
                self.ip = self.vm_info['VpcAttributes']['PrivateIpAddress']['IpAddress'][0]

        except Exception as exp:
            self.log.exception("Exception while getting basic properties of VM")
            raise Exception(exp)

    def snapshots(self):
        """
        get snapshots of the vm.
        return:
                Response object which contains a list of snaphosts
        """

        snapshot = {
            "InstanceId": self.instance_id,
            "RegionId": self.region_id
        }
        snapshots = self.ali_cloud.execute_action("DescribeSnapshots", snapshot)
        return snapshots

    def storage_disks(self):
        """
        get storage disks attatched  to the vm.
        return:
                response object containing the list of the storage disks
        """
        action_params = {
            "InstanceId": self.instance_id,
            "RegionId": self.region_id
        }
        disks = self.ali_cloud.execute_action("DescribeDisks", action_params)
        return disks
