"""
Main file for all the utilities of Virtual Sever

Method:

get_utils_path - get the utils path of VSA Automation

get_testdata_path - path where testdata needs to copied
"""

import os
import base64
from AutomationUtils import logger
from AutomationUtils.pyping import ping
from Cryptodome.Cipher import AES
from Cryptodome import Random

UTILS_PATH = os.path.dirname(os.path.realpath(__file__))
BLOCK_SIZE = 16
key = b'515173A5C402A03398D79B5353B2A080'


def get_testdata_path(machine):
    """
    get the test data path provided base directory

    Args:

        machine         (obj)   - Machine object of controller

    returns:
        _test_data_path (str)   - Test data Path where test data can be generated
        False           (bool)  - if testdata path cannot be retreived

    Exception:
        if failed to create directory

    """
    log = logger.get_log()
    try:
        _vserver_path = os.path.dirname(UTILS_PATH)
        import time
        _testdata_path = os.path.join(_vserver_path, "TestCases", "TestData", str(int(time.time())))
        log.info("checking if directory {} exist".format(_testdata_path))
        if not machine.check_directory_exists(_testdata_path):
            machine.create_directory(_testdata_path)
        if machine.os_info.lower() != 'windows':
            return _testdata_path.replace('/', '//')
        return _testdata_path.replace('/', '\\')
    except Exception:
        log.exception("Error: can't find the VirtualServer utils Path")
        return False


def get_content_indexing_path(machine):
    """
    get the content indexing data path provided base directory

    Args:

        machine         (obj)   - Machine object of controller

    returns:
        _test_data_path (str)   - Content Indexing test data path

        False           (bool)  - if testdata path cannot be retreived

    Exception:
        if failed to create directory

    """
    log = logger.get_log()
    try:
        _vserver_path = os.path.dirname(UTILS_PATH)
        _testdata_path = os.path.join(_vserver_path, "TestCases", "ContentIndexing")

        log.info("checking if directory exist %s" % _testdata_path)

        if not machine.check_directory_exists(_testdata_path):
            raise Exception("The content indexing data path is not available")

        return _testdata_path.replace('/', '\\')

    except Exception:
        log.exception("Error: can't find the VirtualServer utils Path")
        return False


def find_live_browse_db_file(machine, db_path):
    """

    :param machine: Machine object of MA machine used for live browse
    :param db_path: DB path where the live browse db is located
    :return:
        db_name :   (str)   : name of the db used in live browse

    Raise:
        Exception if DB file is not found
    """
    log = logger.get_log()
    try:

        file_name = None
        file_in_path = machine.get_files_in_path(db_path)
        if isinstance(file_in_path, str):
            file_in_path = [os.path.basename(file_in_path)]
        for each_file in file_in_path:
            each_file = os.path.basename(each_file)
            if each_file.strip().endswith(".db"):
                file_name = each_file
                break

        if file_name is None:
            raise Exception("no file found with that extension")

        return os.path.join(db_path, file_name)

    except Exception as err:
        log.exception("An error Occurred in getting live browse db path")
        raise err


def encode_password(message):
    """
    :param value: value of the text needed to be encoded
    :return: encoded value of the password
    """

    if not isinstance(message, bytes):
        message = message.encode()
    IV = Random.new().read(BLOCK_SIZE)
    aes = AES.new(key, AES.MODE_CFB, IV)
    return base64.b64encode(IV + aes.encrypt(message)).decode()


def encode_base64(message):
    """

    Encodes password in base 64 format

    Args:
            message                 (string):   String needs to be encoded

    Returns:
                                    (byte):     base 64 encoded password

    """

    if not isinstance(message, bytes):
        message = message.encode()
    return base64.b64encode(message)


def decode_password(message):
    """
        :param value: value of the text needed to be decoded
        :return: decoded value of the password
        """
    encrypted = base64.b64decode(message)
    IV = encrypted[:BLOCK_SIZE]
    aes = AES.new(key, AES.MODE_CFB, IV)
    cipher = aes.decrypt(encrypted[BLOCK_SIZE:])
    return cipher.decode("utf-8")


def bytesto(bytes, to, bsize=1024):
    """

    Args:
        bytes: bytes to convert
        to: format to be converted into
        bsize: unit of byte size

    Returns: Converted value of the bytes in desired format

    """

    available_units = {'KB': 1, 'MB': 2, 'GB': 3, 'TB': 4}
    result = float(bytes)
    for i in range(available_units[to]):
        result = result / bsize
    return result


def get_os_flavor(guest_os_name):
    """

    Args:
        guest_os_name: Name of the guestOs of the VM

    Returns:
        Flavor to which guestOS belongs to

    """
    os_flavor = None
    if "windows" in guest_os_name.lower():
        os_flavor = "Windows"
    else:
        os_flavor = "Unix"
    return os_flavor


def find(lst, key, value):
    """
    Finds the dict based on a key value given in a list of
    dicts
    Args:
        lst:    list of dicts
        key:    Key that should be checked for
        value:  Value that filters the dict

    Returns:
        Dict when the key with given value is found else returns -1

    """
    for i, dict in enumerate(lst):
        if dict[key] == value:
            return i, dict
    return -1, {}


def validate_ip(ip):
    """
    Validates the Ip given
    Args:
        ip:     (str) - Ip to be validated
    Returns:
        True - if valid IP
        False - if invalid IP
    """
    try:
        response = ping(ip)
        return response.ret_code == 0
    except Exception as err:
        return False


def decorative_log(msg=""):
    """
    Logs to be printed in a different way
    Args:
            msg                     (string):  Log line to be printed


    """
    if msg != "":
        log = logger.get_log()
        log.info("%(boundary)s %(message)s %(boundary)s",
                 {'boundary': "*" * 25,
                  "message": msg})


def get_details_from_config_file(tag1, tag2=None):
    """
    Get details from config file
    Args:
        tag1                  (string):   first tag to look in the config file

        tag2                  (string):   second tag to look in the config file

    Returns:
        value in config file
    """
    from AutomationUtils import config
    config = config.get_config()
    if tag2:
        return eval('config.Virtualization.{}.{}'.format(tag1, tag2))
    else:
        return eval('config.Virtualization.{}.creds'.format(tag1))


def set_inputs(inputs, source_options):
    """
    Sets the input given in the input

    Args:

        inputs                      (dict):     all the inputs in the input json

        source_options              (object):   source object for restores

    """
    log = logger.get_log()
    for each_input_key, each_input_val in inputs.items():
        try:
            getattr(source_options, each_input_key)
            log.info("Setting this attribute {0} in object {1}".format(each_input_key, each_input_val))
            setattr(source_options, each_input_key, each_input_val)
        except:
            # no need to log anything here
            continue


def subclient_initialize(testcase):
    """

    Args:
        testcase                    (obj):  Object of the testcase

    Returns:
        auto_subclient              (obj):  Object of the subclient

    Raises:
            Exception:
                if it fails to create subclient object

    """
    try:
        from . import VirtualServerHelper
        log = logger.get_log()
        log.info("Started executing {} testcase".format(testcase.id))
        decorative_log("Initialize helper objects")
        auto_commcell = VirtualServerHelper.AutoVSACommcell(testcase.commcell, testcase.csdb)
        auto_client = VirtualServerHelper.AutoVSAVSClient(auto_commcell, testcase.client)
        auto_instance = VirtualServerHelper.AutoVSAVSInstance(auto_client,
                                                              testcase.agent, testcase.instance, testcase.tcinputs)
        auto_backupset = VirtualServerHelper.AutoVSABackupset(auto_instance, testcase.backupset)
        auto_subclient = VirtualServerHelper.AutoVSASubclient(auto_backupset, testcase.subclient)
        if not auto_subclient.vm_list:
            raise Exception("No content found in the subclient")
        return auto_subclient
    except Exception as exp:
        logger.get_log().error('Subclient object created failed with error: {} '.format(str(exp)))
        return False


def destination_subclient_initialize(testcase):
    """

    Args:
        testcase                    (obj):  Object of the testcase

    Returns:
        dest_auto_subclient              (obj):  Object of the destination subclient

    Raises:
            Exception:
                if it fails to create destination subclient object

    """
    try:
        from . import VirtualServerHelper
        log = logger.get_log()
        log.info("Started executing {} testcase".format(testcase.id))
        decorative_log("Initialize helper objects")

        vcenter_client = testcase.tcinputs["Destination_Virtualization_client"]
        testcase.client = testcase.commcell.clients.get(vcenter_client)
        testcase.tcinputs["ClientName"] = vcenter_client
        testcase.agent = testcase.client.agents.get('Virtual Server')
        instancekeys = next(iter(testcase.agent.instances.all_instances))
        testcase.tcinputs["InstanceName"] = instancekeys
        testcase.instance = testcase.agent.instances.get(instancekeys)
        backupsetkeys = next(iter(testcase.instance.backupsets.all_backupsets))
        testcase.tcinputs["BackupsetName"] = backupsetkeys
        testcase.backupset = testcase.instance.backupsets.get(backupsetkeys)
        sckeys = next(iter(testcase.backupset.subclients.all_subclients))
        testcase.tcinputs["SubclientName"] = sckeys
        testcase.subclient = testcase.backupset.subclients.get(sckeys)

        dest_auto_commcell = VirtualServerHelper.AutoVSACommcell(testcase.commcell, testcase.csdb)
        dest_auto_client = VirtualServerHelper.AutoVSAVSClient(dest_auto_commcell, testcase.client)
        dest_auto_instance = VirtualServerHelper.AutoVSAVSInstance(dest_auto_client,
                                                                   testcase.agent, testcase.instance, testcase)
        dest_auto_backupset = VirtualServerHelper.AutoVSABackupset(dest_auto_instance, testcase.backupset)
        dest_auto_subclient = VirtualServerHelper.AutoVSASubclient(dest_auto_backupset, testcase.subclient)
        return dest_auto_subclient
    except Exception as exp:
        logger.get_log().error('Destination subclient object created failed with error: {} '.format(str(exp)))
        return False


def discovered_client_initialize(auto_subclient, vm):
    """
        Initialzes objects of discovered client
        Args:
            testcase                    (obj):   Object of the testcase
            vm                   (basestring):   Discovered client name
            auto_subclient              (obj):   Object of the subclient

        Raises:
                Exception:
                    if it fails to create object

        """
    try:
        log = logger.get_log()
        log.info("Initialize helper objects for discovered clients")
        v2client_obj = auto_subclient.auto_commcell.commcell.clients.get(vm)
        vmagent_obj = v2client_obj.agents.get(auto_subclient.vsa_agent.agent_name)
        vminstance_obj = vmagent_obj.instances.get('VMInstance')
        vmbackupset_obj = vminstance_obj.backupsets.get(
            auto_subclient.auto_vsa_backupset.backupset_name)
        vmsub_obj = vmbackupset_obj.subclients.get('default')
        auto_subclient.subclient._subClientEntity = vmsub_obj._subClientEntity

    except Exception as exp:
        raise Exception('Discovered client object created failed with error: {} '.format(str(exp)))
