# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
Class for Subclient
classes defined:
    AutoVSASubclient  - wrapper for VSA Subclient operations

"""

import json
import os
import re
import socket
import time
import pprint
import random
import threading
import requests
from urllib.request import urlopen
from cvpysdk.client import Client
from AutomationUtils.machine import Machine
from VirtualServer.VSAUtils import VirtualServerConstants
from VirtualServer.VSAUtils.VirtualServerConstants import ServiceIds, ServiceOperationEntity
from VirtualServer.VSAUtils import VirtualServerUtils
from VirtualServer.VSAUtils.VsaDiscovery import VsaDiscovery
from cvpysdk.job import Job
from AutomationUtils import logger
from AutomationUtils.idautils import CommonUtils
from cvpysdk.constants import VSAObjects
from Server.Scheduler.schedulerhelper import SchedulerHelper
from . import AutoVSAVSClient, AutoVSAVSInstance

UTILS_PATH = os.path.dirname(os.path.realpath(__file__))


class AutoVSASubclient(object):
    """
    class for performing subclient operations. It act as wrapper for Testcase and SDK
    """

    def __init__(self, backupset_obj, subclient):
        """
        Initialize subclient SDK objects

        Args:
            subclient (obj) - object of subclient class of SDK
        """

        self.auto_vsa_backupset = backupset_obj
        self.log = self.auto_vsa_backupset.log
        self.auto_vsainstance = self.auto_vsa_backupset.auto_vsainstance
        self.auto_vsaclient = self.auto_vsainstance.auto_vsaclient
        self.vsa_agent = self.auto_vsainstance.vsa_agent
        self.auto_commcell = self.auto_vsainstance.auto_commcell
        self.csdb = self.auto_commcell.csdb
        self.hvobj = self.auto_vsainstance.hvobj
        self.subclient = subclient
        self.subclient_name = self.subclient.subclient_name
        self.subclient_id = self.subclient.subclient_id
        self._browse_ma_id = self.subclient.storage_ma_id
        self._browse_ma = self.subclient.storage_ma
        self.quiesce_file_system = True
        self._disk_filter = self.subclient.vm_diskfilter
        self._disk_filter_input = None
        self._vm_record = {}
        self.vm_filter = None
        self._vm_content = None
        self.vm_list = None
        self.backup_folder_name = None
        self.testdata_path = None
        self.disk_restore_dest = None
        self._is_live_browse = False
        self.ma_machine = None
        self.exe_file_path = None
        self._controller_machine = None
        self._is_windows_live_browse = False
        self.restore_obj = None
        self.timestamp = None
        self.backup_option = None
        self.set_content_details()
        self.restore_proxy_client = None
        self.current_job = None
        try:
            self.hvobj.power_on_proxies(self.ips_of_proxies())
        except Exception as exp:
            self.log.info("the following hvobj may not have power_on_proxies function")

    class VmValidation(object):
        def __init__(self, vmobj, vm_restore_options):
            self.vm = vmobj
            self.vm_restore_options = vm_restore_options

        def __eq__(self, other):
            """compares the source vm and restored vm"""

            config_val = (int(self.vm.disk_count) == int(other.vm.disk_count))

            self.log = logger.get_log()
            self.log.info("Source VM:{0}::\nCPU:{1}\nDisk count:{2}\nMemory:{3}".format(
                self.vm.vm_name, self.vm.no_of_cpu, self.vm.disk_count, self.vm.memory
            ))
            self.log.info("Destination VM:{0}::\nCPU:{1}\nDisk count:{2}\nMemory:{3}".format(
                other.vm.vm_name, other.vm.no_of_cpu, other.vm.disk_count, other.vm.memory
            ))
            if config_val:
                if type(self.vm) == type(other.vm):
                    config_val = (float(self.vm.memory) == float(other.vm.memory) and
                                  int(self.vm.no_of_cpu) == int(other.vm.no_of_cpu))
                    if config_val:
                        source = self.vm.VmValidation(self, self.vm_restore_options)
                        dest = self.vm.VmValidation(other, self.vm_restore_options)
                        # self.log.debug("--Details about vm: {}--".format(self.vm.vm_name))
                        # self.log.debug(pprint.pformat(vars(self.vm)))
                        # self.log.debug("--Details about vm: {}--".format(other.vm.vm_name))
                        # self.log.debug(pprint.pformat(vars(other.vm)))
                        config_val = (source == dest)
                else:
                    dest = other.vm.VmConversionValidation(self, self.vm_restore_options)
                    config_val = dest.__eq__(dest)

            return config_val

    class BackupValidation(object):
        def __init__(self, vmobj, backup_option):
            self.vm = vmobj
            self.backup_option = backup_option

        def validate(self):
            validate_obj = self.vm.BackupValidation(self.vm, self.backup_option)
            result = validate_obj.validate()
            return result

    class LiveSyncVmValidation(object):
        def __init__(self, vmobj, schedule=None, replicationjob=None):
            self.vm = vmobj
            self.schedule = schedule
            self.replicationjob = replicationjob

        def __eq__(self, other):
            """ performes hypervisor specific live sync validation"""
            source = self.vm.LiveSyncVmValidation(self, self.schedule, self.replicationjob)
            if isinstance(type(self.vm), type(other.vm)):
                dest = self.vm.LiveSyncVmValidation(other, self.schedule, self.replicationjob)
                config_val = (source == dest)
            else:
                dest = other.vm.LiveSyncVmValidation(other, self.schedule, self.replicationjob)
                config_val = dest.__eq__(dest)
            return config_val

    @property
    def controller_machine(self):
        """
        Returns:
            Controller Machine machine class object
        """
        if not self._controller_machine:
            _controller_client_name = self.auto_commcell.get_client_name_from_hostname(
                socket.gethostbyname_ex(socket.gethostname())[2][0])
            self._controller_machine = Machine(
                _controller_client_name, self.auto_commcell.commcell)
        return self._controller_machine

    @property
    def storage_policy(self):
        """Returns storage policy associated with subclient.Read only attribute"""
        return self.subclient.storage_policy

    @storage_policy.setter
    def storage_policy(self, value):
        """
        Set the Specified Storage Policy

        Args:
            value - storage policy name

        Exception:
            if storage policy does not exist
        """
        self.subclient.storage_policy = value

    @property
    def storage_policy_id(self):
        """Returns storage policy id associated with subclient.Read only attribute"""
        sp_name = self.auto_commcell.commcell.storage_policies.get(
            self.storage_policy)
        return sp_name.storage_policy_id

    @property
    def vm_content(self):
        """
             Returns content  associated with subclient in the form of dict.Read only attribute

             Return:
                 content    (dict)  - with keys
                 {
                'allOrAnyChildren': True,
                'equalsOrNotEquals': True,
                'name': anme of the VM,
                'displayName': display name of the VM,
                'path': source path of the VM,
                'type': 9(VM),1(Host)
                        }

        """

        return self.subclient.content

    @vm_content.setter
    def vm_content(self, content):
        """
        set the specified content as content in subclient

        Args:
            content (str)   -   like [VM]=startswith=test*,test1
                                                        [VM] - represent type
                                                                        like [DNS],[HN]

                                                        startswith  represent equality in GUI
                                                                like endswith,equals

                                                        test* - include all VM starts with test
                                                               adding dyanamic contetn in GUI

                                                        , - to include multiple content

                                                        test1 -  non-dynamic content
        """
        self._vm_content = content
        self.set_content_details()

    @property
    def browse_ma(self):
        """
        Returns the browse MA from which the disk restore is perfomed
        It is read only attribute
        """
        return self._browse_ma, self._browse_ma_id

    def __deepcopy__(self, tempobj):
        """
        over ride deepcopy method to copy every attribute of an objevt to other
        """
        try:
            cls = tempobj.__class__
            result = cls.__new__(cls, tempobj, tempobj.vm_name)
            for k, v in tempobj.__dict__.items():
                setattr(result, k, v)
            return result

        except Exception as err:
            self.log.exception("Failed to deepcopy Exception:" + str(err))
            raise err

    def ips_of_proxies(self):
        """
        collects the proxies and FBR in the setup
        return:
                (dict) which contain hostname as keys and ips as values
        """
        ips = {}
        proxy_list = []
        try:
            fbr_query = "select net_hostname from APP_Client where id in " \
                        "(select attrVal from APP_InstanceProp where componentNameId = " \
                        + self.auto_vsainstance.vsa_instance_id + " and attrName like '%FBR Unix MA%')"
            self.csdb.execute(query=fbr_query)
            fbr_host = self.csdb.fetch_one_row()[0]
            proxy_list.append(fbr_host)
        except Exception as err:
            self.log.exception("Failed to get FBR :" + str(err))
            self.log.info("the setup might not have any FBR")
        if self.subclient.subclient_proxy:
            for proxy in self.subclient.subclient_proxy:
                proxy_list.append(proxy)
        if self.subclient.instance_proxy:
            if self.subclient.instance_proxy not in proxy_list:
                proxy_list.append(self.subclient.instance_proxy)
        if self.auto_vsainstance.vsa_instance.associated_clients:
            for proxy in self.auto_vsainstance.vsa_instance.associated_clients:
                if proxy not in proxy_list:
                    proxy_list.append(proxy)

        import socket
        for proxy in proxy_list:
            d_name = self.auto_commcell.get_hostname_for_client(proxy)
            ips[d_name] = socket.gethostbyname(d_name)
        return ips

    def get_previous_full_backup(self, job_id):
        """
        Get the Previous full backup  for this subclient

        args:
                job_id  (int) - for which previous full has to be fetched

        Exception:
                if failed to get app id


        returns:
            job id  (int) - job id of the previous full backup of given current backup

        """
        try:
            _job_list = []
            _query = "Select id from APP_Application where instance = %s and \
                        backupSet = %s and subclientName = '%s'" % \
                     (self.auto_vsainstance.vsa_instance_id,
                      self.auto_vsa_backupset.backupset_id, self.subclient_name)

            self.csdb.execute(_query)
            _results = self.csdb.fetch_one_row()
            if not _results:
                raise Exception(
                    "An exception occurred in getting previous full backup")

            _AppId = _results[0]

            _query1 = "Select jobId  from JMBkpStats where appId = %s and appType = %s \
                                    and bkpLevel = 1 order by jobId DESC" % \
                      (_AppId, VirtualServerConstants.APP_TYPE)

            self.csdb.execute(_query1)
            _results = self.csdb.fetch_one_row()
            if not _results:
                raise Exception(
                    "An exception occurred in getting previous full backup")

            for each_result in _results:
                tempval = each_result
                if tempval != job_id:
                    _job_list.append(tempval)

            # _job_list.sort(reverse=True)

            return _job_list[0]

        except Exception as err:
            self.log.exception(
                "Failed to get Previous Full Job Exception:" + str(err))
            raise Exception("ERROR - exception while GetPreviousFULLBackup")

    def get_recent_incr_backup(self, job_id):
        """
        Get the Previous incr backup  for this subclient
        args:
                job_id  (int) - for which previous incr has to be fetched
        Exception:
                if failed to get app id
        returns:
            job id  (int) - job id of the previous incr backup of given current backup
        """
        try:
            _job_list = []
            _query = "Select id from APP_Application where instance = %s and \
                        backupSet = %s and subclientName = '%s'" % \
                     (self.auto_vsainstance.vsa_instance_id,
                      self.auto_vsa_backupset.backupset_id, self.subclient_name)

            self.csdb.execute(_query)
            _results = self.csdb.fetch_one_row()
            if not _results:
                raise Exception(
                    "An exception occurred in getting previous incr backup")

            _AppId = _results[0]

            _query1 = "Select jobId  from JMBkpStats where appId = %s and appType = %s \
                                    and bkpLevel = 2 order by jobId DESC" % \
                      (_AppId, VirtualServerConstants.APP_TYPE)

            self.csdb.execute(_query1)
            _results = self.csdb.fetch_one_row()
            if not _results:
                raise Exception(
                    "An exception occurred in getting previous incr backup")

            for each_result in _results:
                tempval = each_result
                if tempval != job_id:
                    _job_list.append(tempval)

            return _job_list[0]

        except Exception as err:
            self.log.exception(
                "Failed to get Previous Incremental Job Exception:" + str(err))
            raise Exception("ERROR - exception while GetPreviousINCREMENTALBackup")

    def check_snapshot_entry_for_job(self, jobid):
        """ Checks if there is an entry for snapshot in db for the Jobid
        args:
            jobid (str):   job id of job to for which check has to performed
            return (bool):   true if db entry exists else false
            raises exception :
                if error occurs
        """
        try:
            _query = f"Select * from SMMetaData where RefId in (Select SMVolumeID from" \
                     f" SMVolume where JobId in ('{jobid}') ) "
            self.csdb.execute(_query)
            result = self.csdb.fetch_one_row()
            self.log.info("DB query result : {0}".format(result))
            if result:
                if result[0] == '':
                    return False
                else:
                    return True
            else:
                raise Exception("An exception occurred while checking snapshot entry"
                                " in db !")
        except Exception as err:
            raise Exception("An exception occurred while checking snapshot entry"
                            " in db :%s" % err)

    def set_content_details(self):
        """
        Update the subclient details in subclient and prepares VM List

        Exception:
                if failed to update subclient
        """
        try:

            if self._vm_content is None:
                vsdiscovery = VsaDiscovery(self.hvobj)
                _content = vsdiscovery.fetch_subclient_content(self.subclient.content)
                _vm_filter = vsdiscovery.fetch_subclient_content(self.subclient.vm_filter)
                self.vm_list = vsdiscovery.merge_rules(_content, _vm_filter, 'not')
            else:
                raise Exception("Failed in setting Subclient Content")
            for each_vm in self.vm_list:
                if not hasattr(self.hvobj.VMs, each_vm):
                    if not self.hvobj.instance_type == 'google cloud platform':
                        self.hvobj.VMs = each_vm
                    else:
                        for data in self.subclient.content:
                            if data['display_name'] == each_vm:
                                self.hvobj.VMs = data['id']
        except Exception as err:
            self.log.exception(
                "Failed to SetContentDetails Exception:" + str(err))
            raise err

    def get_maname_from_policy(self):
        """
        Get the MA Name from policy associated with subclient

        return:
                ma_name     (str)   - ma client name associated with SP

        Exception:
                if failed to get ma name
        """
        try:
            _query = "select DISTINCT AC.name  from MMDataPath MDP, APP_Client AC where \
                        MDP.HostClientId = AC.id AND MDP.CopyId in \
                    (select id from archGroupCopy where archGroupId = %s)" % self.storage_policy_id

            self.csdb.execute(_query)
            _results = self.csdb.fetch_one_row()
            if not _results:
                raise Exception(
                    "An exception occurred in getting ma from policy")

            ma_name = _results[0]
            return ma_name

        except Exception as err:
            self.log.exception(
                "Failed to GetStoragePolicy Exception:" + str(err))
            raise err

    def _process_disk_filter_string(self, controller_string):
        """
        process the filter string to split it into controller , type number

        controller_string: Filter string that is from CS DB
        :return:
            controller_type (str) : it is scsi or ide
            number  (int)          : ilocation of scsi or ide

        """
        try:
            self.log.info("The Controller string is %s" % controller_string)
            _opening_bracket = None
            _possible_brackets = {
                "[": "]",
                "(": ")"
            }
            _brackets_type = list(
                _bracket for _bracket in _possible_brackets.keys() if _bracket in controller_string)
            _total_brackets = len(_brackets_type)
            if _total_brackets == 1:
                _opening_bracket = _brackets_type[0]
                _slot_saperator = "-" if _opening_bracket == "[" else ":"
                if _opening_bracket is not None:
                    controller_string = controller_string.strip()
                    _controller_type = (controller_string.split(_opening_bracket)[0]).strip()
                    _temp = (controller_string.split(_opening_bracket)[1]).strip()
                    _controller_location_string = _temp.split(
                        _possible_brackets[_opening_bracket])[0]
                    _number, _location = _controller_location_string.split(_slot_saperator)
            else:
                _single_slot = [int(slot_value)
                                for slot_value in controller_string.split() if slot_value.isdigit()]
                if len(_single_slot) != 1:
                    raise Exception('Received unformatted slot values')
                _controller_type = controller_string.split(str(_single_slot[0]))[0]
                _number = _single_slot[0]
                _location = _single_slot[0]

            return _controller_type, _number, _location

        except Exception as err:
            self.log.exception(
                "An exception occurred in  process_diks_filter_string %s" % err)
            raise err

    def prepare_disk_filter_list(self):
        """
        Prepares the disk filter list by processing the pattern
        """
        try:

            def process_repository_filters(vm, repository_name):
                return self.hvobj.VMs[vm].get_disks_by_repository(repository_name)

            def process_control_filters(vm, controller_detail):
                if self.hvobj.instance_type in ('vmware', 'xen'):
                    return self.hvobj.VMs[vm].get_disk_in_controller(controller_detail)
                else:
                    controller_type, number, location = self._process_disk_filter_string(
                        controller_detail)
                    return self.hvobj.VMs[vm].get_disk_in_controller(controller_type, number,
                                                                     location)

            def process_disk_path_filters(vm, disk_path):
                return self.hvobj.VMs[vm].get_disk_path_from_pattern(disk_path)

            def process_datastore_uri_filters(vm, datastore_uri):
                return self.hvobj.VMs[vm].get_datastore_uri_by_pattern(datastore_uri)

            def process_disk_by_ostype_filters(vm, _disk_type):
                return self.hvobj.VMs[vm]._get_disks_by_os_type()

            def process_by_disk_tag_filters(vm, tag_name, tag_value):
                return self.hvobj.VMs[vm].get_disks_by_tag(tag_name, tag_value)

            def process_by_disk_label(vm, disk_number):
                return self.hvobj.VMs[vm].get_disk_by_label(disk_number)

            disks_filter_process = {
                "1": process_datastore_uri_filters,
                "2": process_disk_path_filters,
                "3": process_control_filters,
                "4": process_repository_filters,
                "5": process_by_disk_label,
                "6": process_disk_by_ostype_filters,
                "9": process_by_disk_tag_filters
            }

            '''
                Apply Datastore/repository filters first since
                there is good chance of all the disks getting filtered out
            '''
            _disk_filter_sorted = sorted(
                self._disk_filter,
                key=lambda _filter: _filter['filterTypeId'],
                reverse=True)
            # Datastore filter check
            if _disk_filter_sorted[-1]['filterTypeId'] == '1':
                _data_store_filter = _disk_filter_sorted.pop()
                _disk_filter_sorted.insert(0, _data_store_filter)

            for each_vm in self.vm_list:
                self.log.info("Disk Details before filtering are : {0}".format(
                    self.hvobj.VMs[each_vm].disk_dict))
                disk_filter_list = []
                for each_filter in _disk_filter_sorted:
                    if each_filter['filterTypeId'] == '9':
                        disk_path = disks_filter_process[(each_filter['filterTypeId'])](
                            each_vm, each_filter['filter'], each_filter['value'])
                    else:
                        disk_path = disks_filter_process[(each_filter['filterTypeId'])](
                            each_vm, each_filter['filter'])
                    if disk_path is None or disk_path == "" or disk_path == []:
                        self.log.info("the criteria filter %s does not match this VM %s" %
                                      (each_filter['filter'], each_vm))
                    else:
                        if isinstance(disk_path, list):
                            for disk in disk_path:
                                if disk not in disk_filter_list:
                                    disk_filter_list = disk_filter_list + [disk]
                        else:
                            disk_filter_list.append(disk_path)
                    # Do not apply any more filters if all the disks are filtered out
                    if len(disk_filter_list) == len(self.hvobj.VMs[each_vm].disk_dict):
                        break

                disk_filter_list = list(set(disk_filter_list))
                for each_disk in disk_filter_list:
                    del self.hvobj.VMs[each_vm].disk_dict[each_disk]
                self.hvobj.VMs[each_vm].filtered_disks = disk_filter_list

                self.hvobj.VMs[each_vm].disk_count = len(self.hvobj.VMs[each_vm].disk_dict)
                self.log.info("Disk Details after filtering are : {0}".format(
                    self.hvobj.VMs[each_vm].disk_dict))

        except Exception as err:
            self.log.exception("Failed to PrepareDiskFilterListContent Exception:" + str(err))
            raise Exception("ERROR - exception while PrepareDiskFilterListContent")

    def _get_browse_ma(self, fs_restore_options):
        """
        Get the browse Mafor the browse
        Args:
            fs_restore_options: Fs restore options object

        Returns:
            ma_macine - machine object of MA
        """
        try:

            ma_machine = Machine(fs_restore_options._browse_ma_client_name,
                                 self.auto_commcell.commcell)
            if fs_restore_options.is_ma_specified:
                sp_ma_machine = Machine(self._browse_ma, self.auto_commcell.commcell)
                if ma_machine.os_info.lower() != sp_ma_machine.os_info.lower() and\
                        not(fs_restore_options.browse_from_snap and self.hvobj.instance_type == 'hyper-v'):
                    raise Exception(
                        "Browse MA specifies {0} is of difference flavour {1} than SP MA {2} flavour {3}".format(
                            ma_machine, ma_machine.os_info, sp_ma_machine, sp_ma_machine.os_info
                        ))
            else:
                if ma_machine.os_info.lower() == "unix":
                    windows_proxy = None
                    self.log.info("SP MA is unix so setting proxy as browse MA")
                    sub_proxies = self.subclient.subclient_proxy
                    if sub_proxies:
                        self.log.info(
                            "Checking windows in subclientProxy {0}".format(",".join(sub_proxies)))
                        for each_proxy in sub_proxies:
                            temp_ma = Machine(each_proxy, self.auto_commcell.commcell)
                            if temp_ma.os_info.lower() == "windows":
                                ma_machine = temp_ma
                                windows_proxy = each_proxy
                                break
                    else:
                        if not windows_proxy:
                            instance_proxies = self.auto_vsainstance.proxy_list
                            self.log.info("Checking windows in InstanceProxy {0}".format(
                                ",".join(instance_proxies)))
                            for each_proxy in instance_proxies:
                                temp_ma = Machine(each_proxy, self.auto_commcell.commcell)
                                if temp_ma.os_info == "WINDOWS":
                                    ma_machine = temp_ma
                                    windows_proxy = each_proxy
                                    break

                    if not windows_proxy:
                        ma_machine = Machine(self.auto_commcell.commcell.commserv_name,
                                             self.auto_commcell.commcell)
                        fs_restore_options.browse_ma = self.auto_commcell.commcell.commserv_name
                    else:
                        fs_restore_options.browse_ma = windows_proxy

                    # this would slve forcing browse ma for second VM
                    fs_restore_options.is_ma_specified = False

            return ma_machine
        except Exception as err:
            self.log.exception("Failed to _get_browse_ma Exception:" + str(err))
            raise Exception("ERROR - exception while _get_browse_ma")

    def _check_if_windows_live_browse(self, os_name, metadata_collected):
        """
        Decides Live browse validation need to be perfomred or not.

        Args:
                os_name             (str)   - os name of  backups VM

                metdata_collected   (bool)  - True on metadata collected , false on not collected

        return:
                True    - If live browse validation needs to be performed
                False   - if live browse validation need notto be performed

        """
        try:
            if (os_name.lower() == "windows") and (not metadata_collected):
                return True
            else:
                return False

        except BaseException:
            self.log.exception(
                "An exception occurred in checking live browse validation neeeded")
            return False

    def vsa_discovery(self, backup_option, extra_options):
        """
        Creates testdata path and generate testdata and copy the test data to each drive in VM .
        Args:
            backup_option           (object):   object of Backup Options class in options
                                            helper containes all backup options

            extra_options        (dict):     Extra options for testdata creation
        """
        try:

            """
            TODO: Map Disknames with drive letters and stop copying 
                  data to the filtered disks
            """

            self.backup_folder_name = backup_option.backup_type
            self.testdata_path = backup_option.testdata_path

            if not self.testdata_path:
                self.testdata_path = VirtualServerUtils.get_testdata_path(
                    self.controller_machine)
                self.timestamp = os.path.basename(os.path.normpath(self.testdata_path))
                self.auto_vsaclient.timestamp = self.timestamp
            if backup_option.cleanup_testdata_before_backup:
                extra_options['cleanup_testdata_before_backup'] = True
                self.cleanup_testdata(backup_option, extra_options)

            self.log.info("TesdataPath provided is {}".format(self.testdata_path))
            self.log.info("creating test data directory {}".format(self.testdata_path))
            self.log.info("Generating Test data folders")

            zero_size_file = extra_options.get('zero_kb_files', True)
            testdata_size = backup_option.advance_options.get("testdata_size",
                                                              random.randint(40000, 60000))
            attempt = 1
            while attempt < 5:
                try:
                    generate = self.controller_machine.generate_test_data(self.testdata_path, 3, 5,
                                                                          testdata_size, zero_size_file=zero_size_file)
                    break
                except Exception as exp:
                    attempt = attempt + 1
                    time.sleep(60)
            if attempt >= 5:
                generate = self.controller_machine.generate_test_data(self.testdata_path, 3, 5,
                                                                      testdata_size, zero_size_file=zero_size_file)

            if not generate:
                raise Exception(generate)
            for _vm in self.vm_list:
                self.hvobj.VMs[_vm].update_vm_info('All', os_info=True, force_update=True)
                self.log.info("VM selected is {0}".format(_vm))
                '''If all the disks are filtered out
                    avoiding copying test data to all the drives'''
                if len(self.hvobj.VMs[_vm].disk_list) > 0:
                    for _drive in self.hvobj.VMs[_vm].drive_list.values():
                        self.log.info("Copying Testdata to Drive {0}".format(_drive))
                        self.hvobj.timestamp = self.timestamp
                        self.hvobj.copy_test_data_to_each_volume(
                            _vm, _drive, self.backup_folder_name, self.testdata_path)
            special_vm_drive = extra_options.get('special_vm_drive')
            if special_vm_drive:
                testdata_size = backup_option.advance_options.get("testdata_size",
                                                                  random.randint(500000, 700000),
                                                                  hlinks=True,
                                                                  slinks=True)
                special_testdata = self.testdata_path + "special_testdata"
                attempt = 1
                while attempt < 5:
                    try:
                        generate = self.controller_machine.generate_test_data(special_testdata, 4, 6,
                                                                              testdata_size)
                        break
                    except Exception as exp:
                        attempt = attempt + 1
                        time.sleep(60)
                if attempt >= 5:
                    generate = self.controller_machine.generate_test_data(special_testdata, 4, 6,
                                                                          testdata_size)
                if not generate:
                    raise Exception(generate)
                for _vm, _drive in special_vm_drive.items():
                    self.hvobj.copy_test_data_to_each_volume(
                        _vm, _drive, 'special_testdata', special_testdata)
            if backup_option.modify_data:
                self.modify_data()
            if backup_option.delete_data:
                self.delete_data()

        except Exception as err:
            self.log.exception("Exception while doing VSA Discovery :{0}".format(err))
            raise err

    def cleanup_testdata(self, backup_option, extra_options=None):
        """
        Cleans up testdata that is copied from each vm in the subclient

        Args:
            backup_option           (object):   object of Backup Options class in options
                                                helper contains all backup options

            extra_options           (dict):      Extra options for cleanup
        """
        try:
            self.log.info("Testdata cleanup from subclient started")
            if not self.backup_folder_name:
                self.backup_folder_name = backup_option.backup_type
            for _vm in self.vm_list:
                self.hvobj.VMs[_vm].update_vm_info('All', os_info=True, force_update=True)
                self.hvobj.VMs[_vm].get_drive_list(drives="all")
                self.log.info("VM selected is: {}".format(_vm))
                if backup_option.cleanup_testdata_before_backup:
                    for _drive in self.hvobj.VMs[_vm].drive_list.values():
                        for folder_name in [self.backup_folder_name]:
                            _testdata_path = self.hvobj.VMs[_vm].machine.join_path(_drive,
                                                                                   folder_name)
                            self.log.info("Cleaning up {}".format(_testdata_path))
                            if self.hvobj.VMs[_vm].machine.check_directory_exists(_testdata_path):
                                self.hvobj.VMs[_vm].machine.remove_directory(_testdata_path)
                else:
                    for _drive in self.hvobj.VMs[_vm].drive_list.values():
                        _testdata_path = VirtualServerConstants.get_folder_to_be_compared(
                            folder_name=self.backup_folder_name, _driveletter=_drive,
                            timestamp=self.timestamp)
                        self.log.info("Cleaning up {}".format(_testdata_path))
                        if self.hvobj.VMs[_vm].machine.check_directory_exists(_testdata_path):
                            self.hvobj.VMs[_vm].machine.remove_directory(_testdata_path)
                if extra_options:
                    special_vm_drive = extra_options.get('special_vm_drive')
                else:
                    special_vm_drive = None
                if special_vm_drive and _vm in special_vm_drive:
                    _testdata_path = self.hvobj.VMs[_vm].machine.join_path(special_vm_drive[_vm],
                                                                           'special_testdata')
                    if self.hvobj.VMs[_vm].machine.check_directory_exists(_testdata_path):
                        self.hvobj.VMs[_vm].machine.remove_directory(_testdata_path)
            self.log.info("clearing Testdata in controller: {}".format(self.testdata_path))
            self.controller_machine.remove_directory(self.testdata_path)
            if special_vm_drive:
                self.log.info("clearing Testdata in controller: {}".format(self.testdata_path))
                special_testdata = self.testdata_path + "special_testdata"
                self.log.info("clearing Testdata in controller: {}".format(special_testdata))
                self.controller_machine.remove_directory(special_testdata)

            try:
                if self.hvobj.power_off_proxies_flag:
                    if extra_options and 'cleanup_before_backup' in extra_options.keys():
                        if not extra_options['cleanup_before_backup']:
                            self.hvobj.power_off_proxies(self.ips_of_proxies())
                    else:
                        self.hvobj.power_off_proxies(self.ips_of_proxies())
            except Exception as exp:
                self.log.info("the following hvobj may not have power_off_proxies function")

        except Exception as err:
            self.log.exception(
                "Exception while doing cleaning up testdata Discovery: {}".format(err))
            raise err

    def post_restore_clean_up(self, vm_restore_options, source_vm=False, status=False):

        """
            Cleans up VM and its resources after out of place restore

            Args:
                    vm_restore_options              (str):  options that need to be set while
                                                            performing vm restore
                    source_vm                       (bool): whether  source vm has to be powered
                                                            off or not
                    status                          (bool) : whether the tc has passed ot failed
            Raises:
             Exception:
                If unable to clean up VM and its resources
        """
        try:
            for each_vm in self.vm_list:
                if vm_restore_options.dest_client_hypervisor.VMs.get(each_vm, None):
                    if source_vm:
                        self.log.info("Powering off VM {0}".format(each_vm))
                        vm_restore_options.dest_client_hypervisor.VMs[each_vm].power_off()
                    restore_vm_name = "del" + each_vm
                    if status:
                        self.log.info("Cleaning up VM {0}".format(restore_vm_name))
                        vm_restore_options.dest_client_hypervisor.VMs[restore_vm_name].clean_up()
                    else:
                        vm_restore_options.dest_client_hypervisor.VMs[restore_vm_name].power_off()

        except Exception as err:
            self.log.exception(
                "Exception while doing cleaning up VM resources: " + str(err))
            raise err

    def modify_data(self):
        """
        Modifies the data on controller machine and then copies to all VM's drives

        Raise Exception:
                if directory does not exists or fails to modify data
        """
        try:
            if self.controller_machine.check_directory_exists(self.testdata_path):
                self.controller_machine.modify_test_data(self.testdata_path, modify=True)
            for _vm in self.vm_list:
                if len(self.hvobj.VMs[_vm].disk_list) > 0:
                    for _drive in self.hvobj.VMs[_vm].drive_list.values():
                        _dest_path = self.hvobj.VMs[_vm].machine.join_path \
                            (_drive, self.backup_folder_name, "TestData", self.timestamp)
                        if self.hvobj.VMs[_vm].machine.check_directory_exists(_dest_path):
                            self.hvobj.VMs[_vm].machine.remove_directory(_dest_path)
                            self.hvobj.copy_test_data_to_each_volume \
                                (_vm, _drive, self.backup_folder_name, self.testdata_path)
                        else:
                            self.log.error("Failed to modify, directory {} does not exists".
                                           format(_dest_path))
        except Exception as err:
            self.log.exception(
                "Exception while modifying data " + str(err))
            raise err

    def check_jobstatus_to_stop_services(self, name):
        """

        Fetch running job for vms and stop the CVD services on proxy or MA after one VM got completed

        Args:
            name(str)          --  client name to stop services

        Returns:
            job_ids   (dict)  --  Dictionary containing job IDs

        Raise Exception:
                fail to get the running job

        """
        try:
            VirtualServerUtils.decorative_log('checking Database for child job complete status')
            jobvmdict = {}
            for each_vm in self.vm_list:
                query1 = "select jobId from JMBkpJobInfo where applicationId in" \
                   " (select id from APP_Application where subclientName = '"+self.subclient.subclient_name+"' " \
                   "and clientid in (select id from APP_Client where name = '"+each_vm+"')) "
                self.csdb.execute(query1)
                query1output = self.csdb.fetch_all_rows()
                jobvmdict[each_vm] = query1output[0][0]
                for key in sorted(jobvmdict):
                    self.log.info('-sorting dict-')
            VirtualServerUtils.decorative_log('sorted in dictionary format'+str(jobvmdict))
            joblist = jobvmdict.values()
            VirtualServerUtils.decorative_log('getting child job completed status from DB')
            childjobstatus = 0
            while childjobstatus <= 0:
                for eachjob in joblist:
                    query = "select status from JMBkpStats where jobid = "+eachjob+""
                    self.csdb.execute(query)
                    queryoutput = self.csdb.fetch_all_rows()
                    queryoutput = queryoutput[0][0]
                    if queryoutput != None and queryoutput != '':
                        childjobstatus = 1
                        VirtualServerUtils.decorative_log(
                            "successfully got child job completed status from DB-"+str(eachjob))
                        break
                if queryoutput == '1' or queryoutput == '3':
                    self.stop_service(name, 'GxCVD(Instance001)')
            return joblist
        except Exception as err:
            self.log.exception(
                "-----Failed to get individual job IDs of VMs from database-----")
            raise Exception

    def delete_data(self):
        """
        Deletes the first file on controller machine and on all VM's drives

        Raise Exception:
                if directory does not exists or fails to delete data
        """
        try:
            if self.controller_machine.check_directory_exists(self.testdata_path):
                file_list = self.controller_machine.get_files_in_path(self.testdata_path)
                self.controller_machine.delete_file(file_list[0])
            for _vm in self.vm_list:
                if len(self.hvobj.VMs[_vm].disk_list) > 0:
                    for _drive in self.hvobj.VMs[_vm].drive_list.values():
                        _dest_path = self.hvobj.VMs[_vm].machine.join_path \
                            (_drive, self.backup_folder_name, "TestData", self.timestamp)
                        if self.hvobj.VMs[_vm].machine.check_directory_exists(_dest_path):
                            _file_list = self.hvobj.VMs[_vm].machine.get_files_in_path(_dest_path)
                            self.hvobj.VMs[_vm].machine.delete_file(_file_list[0])
                        else:
                            self.log.error("Failed to delete, directory {} does not exists".
                                           format(_dest_path))
        except Exception as err:
            self.log.exception(
                "Exception while deleting data " + str(err))
            raise err

    def get_vm_lastcompleted_job(self):
        """

        Fetch last job ran on the vm
        Returns:
        job IDs    (dict)  --  Dictionary containing  job IDs

        Raise Exception:
                fail to get last job on the vm
        """
        try:
            lastjob = {}
            for each_vm in self.vm_list:
                query = "select TOP 1 PERCENT jobId from JMBkpStats where appid =" \
                       " (select id from APP_Application where subclientName = '"+self.subclient.subclient_name+"' " \
                       "and clientid in (select id from APP_Client where name = '"+each_vm+"')) " \
                       "ORDER by jobid DESC"
                self.csdb.execute(query)
                queryoutput = self.csdb.fetch_all_rows()
                lastjob[each_vm] = queryoutput[0][0]
            for key in sorted(lastjob):
                VirtualServerUtils.decorative_log('sorting dict')
                lastjob = lastjob.values()
                VirtualServerUtils.decorative_log(
                    'JOB and VM details sorted in dictionary format '+str(lastjob))
                return lastjob
        except Exception as err:
            self.log.exception(
                "-----Failed to get child job completed status from DB-----")
            raise Exception

    def backup(self, backup_option, **kwargs):
        """
        Submit VSA backup job

        Args:
                backup_option           (object):   object of Backup Options class in options
                                                    helper containes all backup options

                **kwargs                (dict):   Optional arguments

        Raise Exception:
                if job does not complete successfully

        """
        try:
            if kwargs.get('msg'):
                VirtualServerUtils.decorative_log(kwargs.get('msg'))
            self.backup_option = backup_option
            if not kwargs.get('skip_discovery', False):
                self.vsa_discovery(self.backup_option, kwargs)
            common_utils = CommonUtils(self.auto_commcell.commcell)
            if kwargs.get('template'):
                self.log.info("Converting VMs to template")
                for vm in self.hvobj.VMs:
                    self.hvobj.VMs[vm].convert_vm_to_template()
            if kwargs.get('rdm'):
                self.log.info("calculating the details about rdm disks")
                for vm in self.hvobj.VMs:
                    if not self.hvobj.VMs[vm].rdm_details:
                        self.log.error("Error in getting disk details for the vm")
            self.log.info("Starting Backup Job")
            if 'SYNTHETIC_FULL' in backup_option.backup_type and "BEFORE" in backup_option.incr_level:
                _backup_type = ['INCREMENTAL', 'SYNTHETIC_FULL']
            elif 'SYNTHETIC_FULL' in backup_option.backup_type and "AFTER" in backup_option.incr_level:
                _backup_type = ['SYNTHETIC_FULL', 'INCREMENTAL']
            else:
                _backup_type = [self.backup_option.backup_type]
            if backup_option.set_disk_props:
                if backup_option.disk_props_dict:
                    self.log.info("Setting disk props")
                    # Set provided disk properties for all VMs
                    for vm in self.hvobj.VMs:
                        self.hvobj.VMs[vm].set_disk_props(backup_option.disk_props_dict)
                else:
                    raise Exception("disk_props_dict not provided to set disk props")
            for _bc_type in _backup_type:
                _backup_jobs = self.subclient.backup(_bc_type,
                                                     self.backup_option.run_incr_before_synth,
                                                     self.backup_option.incr_level,
                                                     self.backup_option.collect_metadata,
                                                     self.backup_option.advance_options)
                if not (isinstance(_backup_jobs, list)):
                    _backup_jobs = [_backup_jobs]
                for _backup_job in _backup_jobs:
                    self.backup_job = _backup_job
                self.backup_option.backup_job = self.backup_job
                self.log.info(
                    "Submitted '{0}' backup Job : {1}".format(_bc_type, self.backup_job.job_id))
                if kwargs.get('job_status'):
                    VirtualServerUtils.decorative_log("check job status")
                    _op_id = kwargs.get('op_id')
                    _entity_id = kwargs.get('entity_id')
                    self.current_job = self.backup_job.job_id
                    self.check_job_status_fromDB(_op_id, _entity_id)
                if not _backup_job.wait_for_completion():
                    raise Exception("Failed to run backup with error: {0}"
                                    .format(_backup_job.delay_reason))
                self.log.info("Backup Job {0} completed successfully"
                              "Checking if Job type is Expected for job ID {0}".
                              format(self.backup_job.job_id))
                for _status in \
                        self.backup_option.backup_job.details['jobDetail']['clientStatusInfo'][
                            'vmStatus']:
                    if _status['Status'] == 1:
                        self.log.info('VM:{} failed to backup'.format(_status['vmName']))
                        raise Exception
                self.current_job = self.backup_job.job_id
                if _bc_type == 'SYNTHETIC_FULL':
                    _bc_type = 'Synthetic Full'
                self.auto_commcell.check_backup_job_type_expected(
                    self.backup_job.job_id, _bc_type)

                _bkpcopy_enabled = self.backup_option.advance_options.get("create_backup_copy_immediately")
                if self.backup_option.backup_method == "SNAP" \
                        and _bkpcopy_enabled \
                        and _bc_type != 'Synthetic Full':

                    self.log.info("Backup Type is not Synthfull, looking for backup copy job")
                    retry = 0
                    while retry < 5:
                        try:
                            time.sleep(10)
                            self.backupcopy_job_id = common_utils.get_backup_copy_job_id(
                                self.backup_job.job_id)
                            self.log.info("Backup Copy Job ID : {0}".format(self.backupcopy_job_id))
                            break
                        except Exception:
                            time.sleep(10)
                            retry = retry + 1
                    self.backupcopy_job = Job(self.auto_commcell.commcell, self.backupcopy_job_id)
                    self.log.info("Backup Copy Job {0} :".format(self.backupcopy_job))
                    if kwargs.get('bkpcopy_job_status'):
                        VirtualServerUtils.decorative_log("check backup copy job status")
                        _op_id = kwargs.get('op_id')
                        _entity_id = kwargs.get('entity_id')
                        self.current_job = self.backupcopy_job.job_id
                        self.check_job_status_fromDB(_op_id, _entity_id)
                    if not self.backupcopy_job.wait_for_completion():
                        raise Exception("Failed to run backup copy job with error:{0} "
                                        .format(self.backupcopy_job.delay_reason))
                    for _status in \
                            self.backupcopy_job.details['jobDetail']['clientStatusInfo'][
                                'vmStatus']:
                        if _status['Status'] == 1:
                            self.log.info(
                                'VM:{} failed during backup copy'.format(_status['vmName']))
                            raise Exception
                    # todo Commenting it out now, we will come up with some approach
                    # if self.hvobj.instance_type == "vmware":
                    #     if self.subclient.snapshot_engine_name.lower() == 'netapp' \
                    #             and not backup_option.collect_metadata:
                    #         self.log.info("If its Netapp engine without metadata, Transport mode should be NFS")
                    #         _transport_mode = self.auto_commcell.find_job_transport_mode(self.backupcopy_job_id)
                    #         if _transport_mode == 'nas':
                    #             self.log.info("Transport mode for job {} is nas".format(self.backupcopy_job_id))
                    #         else:
                    #             self.log.error("The Transport mode for job {0} is {1}".
                    #                            format(self.backupcopy_job_id, _transport_mode))
                    #             raise Exception("Transport Mode NAS is expected for Vmware Netapp")
                if hasattr(self.backup_option,
                           "Application_aware") and self.backup_option.Application_aware:
                    self.log.info("It is application Aware backup so checking for other jobs")
                    ida_job_id, workflow_job_id = self.auto_commcell.find_app_aware_jobs(
                        self.backup_job.job_id)
                    ida_job = Job(self.auto_commcell.commcell, ida_job_id)
                    workflow_job = Job(self.auto_commcell.commcell, workflow_job_id)
                    if "Completed" not in (ida_job.status and workflow_job.status):
                        raise Exception(
                            "Ida job {0} or Workflow Job {1} failed , please check the logs".format(
                                ida_job, workflow_job))

        except Exception as err:
            self.log.exception(
                "Exception while submitting Backup job:" + str(err))
            raise err

    def agentless_file_restore(self, fs_restore_options, msg="", discovered_client=None):
        """

        perform Agentless file restore for specific subclinet

        Args:
                fs_restore_options      (object):   options that need to be set while
                                                    performing guest file restore

                msg                     (string):  Log line to be printed

                discovered_client       (string):   Pass the discovered client name if restore has to be performed from
                                                    discovered client.
        Exception:
                        if job fails
                        if validation fails

                """

        try:
            VirtualServerUtils.decorative_log(msg)
            if discovered_client:
                temp_vmlist = [discovered_client]
                VirtualServerUtils.discovered_client_initialize(self, discovered_client)
            else:
                temp_vmlist = self.vm_list
            for _vm in temp_vmlist:
                self.log.info("Guest file restore from {0} VM".format(_vm))
                _path_dict = {}
                flr_options = self.file_level_path(fs_restore_options, _vm)
                fs_restore_options = flr_options['fs_restore_options']
                VirtualServerUtils.decorative_log("Its Agentless restore")
                flr_options['dest_machine'] = fs_restore_options.auto_subclient.hvobj.VMs[
                    _vm].machine
                _drive = next(
                    iter(fs_restore_options.auto_subclient.hvobj.VMs[_vm].drive_list.values()))
                _restore_folder = flr_options['dest_machine'].join_path(_drive, 'agentless')
                if flr_options['dest_machine'].check_directory_exists(_restore_folder):
                    flr_options['dest_machine'].remove_directory(_restore_folder)
                self.fs_restore_dest = flr_options['dest_machine']. \
                    join_path(_restore_folder, self.backup_folder_name, _vm)
                agentless_option = {
                    "vm_user": fs_restore_options.auto_subclient.hvobj.VMs[_vm].user_name,
                    "vm_pass": (VirtualServerUtils.encode_base64(
                        fs_restore_options.auto_subclient.hvobj.VMs[_vm].password)).decode(),
                    "vm_name": fs_restore_options.auto_subclient.hvobj.VMs[_vm].vm_name,
                    "vm_guid": fs_restore_options.auto_subclient.hvobj.VMs[_vm].guid,
                    "vserver": fs_restore_options.auto_subclient.hvobj.server_host_name}
                self.log.info("Restore dest path {}".format(self.fs_restore_dest))
                fs_restore_job = self.subclient. \
                    guest_file_restore(vm_name=_vm,
                                       folder_to_restore=flr_options['_fs_path_to_browse_list'],
                                       destination_client=fs_restore_options.destination_client,
                                       destination_path=self.fs_restore_dest,
                                       copy_precedence=fs_restore_options.copy_precedence,
                                       preserve_level=flr_options['_preserve_level'],
                                       unconditional_overwrite=fs_restore_options.unconditional_overwrite,
                                       browse_ma=flr_options['browse_ma'],
                                       fbr_ma=fs_restore_options.fbr_ma,
                                       agentless=agentless_option)
                self.log.info(" Running restore Job {0} :".format(fs_restore_job))
                if not fs_restore_job.wait_for_completion():
                    raise Exception(
                        "Failed to run file level restore job {0} with error:{1} ".format(
                            fs_restore_job.job_id, fs_restore_job.delay_reason))
                self.log.info("file level restore Job got complete JOb Id:{0}".format(
                    fs_restore_job.job_id))
                if fs_restore_options.validation:
                    self.file_level_validation(flr_options)
                else:
                    self.log.info("Validation is being skipped")
            self.log.info(
                "Ran file level restore from all the VMs and its drives")
        except Exception as err:
            self.log.exception(
                "Exception at restore job please check logs for more details {0}".format(err))
            self.log.error("Restore: FAIL - File level files Restore Failed")
            raise err

    def validate_child_job_type(self, joblist, parentjobtype):
        """

        validate backup job type for child job. Backup job type should be same as parent job type

        Args:
            joblist          (list)   --  job of child vm
            parentjobype     (str)   --   parent job type
        Raise Exception:
                if child job backup type is not same as parent job

        """
        try:
            jobtypevmlist = []
            for eachjob in joblist:
                query = "select bkpLevel from JMBkpStats where jobId ="+eachjob+""
                self.csdb.execute(query)
                queryoutput = self.csdb.fetch_all_rows()
                jobtypevmlist.append(queryoutput)
                if parentjobtype == 'INCREMENTAL' and all(i == [['2']] for i in jobtypevmlist):
                    VirtualServerUtils.decorative_log('Child and parent job type are incremental-'+str(eachjob))
                elif parentjobtype == 'FULL' and all(i == [['1']] for i in jobtypevmlist):
                    VirtualServerUtils.decorative_log('Child and parent job type are full -'+str(eachjob))
                else:
                    self.log.exception("---Child job type is not same as parent job type---")
                    raise Exception
        except Exception as err:
            raise Exception

    def guest_file_restore(self, fs_restore_options, special_vm_drive=None, discovered_client=None,
                           **kwargs):
        """

        perform Guest file restore for specific subclinet

        Args:
                fs_restore_options      (object):   options that need to be set while
                                                    performing guest file restore

                discovered_client       (string):   Pass the discovered client name if restore has to be performed from
                                                    discovered client.

                special_vm_drive        (dict):     Vm and drive where more data needs to be copied

                **kwargs                    : Arbitrary keyword arguments
        Exception:
                        if job fails
                        if validation fails

        """

        try:
            if kwargs.get('msg'):
                VirtualServerUtils.decorative_log(kwargs.get('msg'))
            if discovered_client:
                temp_vmlist = [discovered_client]
                VirtualServerUtils.discovered_client_initialize(self, discovered_client)
            else:
                temp_vmlist = self.vm_list
            for _vm in temp_vmlist:
                if fs_restore_options.browse_from_snap and self.hvobj.instance_type == 'hyper-v' \
                        and self.hvobj.VMs[_vm].guest_os.lower() != 'windows':
                    self.log.info(
                        "File restore from Snap for unix machine is not supported for hyper-V"
                        " skipping restore for VM {0}".format(_vm))
                    continue
                self.log.info("Guest file restore from {0} VM".format(_vm))
                _path_dict = {}
                if fs_restore_options.metadata_collected and \
                        self.hvobj.VMs[_vm].guest_os.lower() != 'windows':
                    drive_list = self.convert_drive_to_volume(_vm)
                else:
                    drive_list = self.hvobj.VMs[_vm].drive_list
                if special_vm_drive and _vm in special_vm_drive:
                    flr_options = self.file_level_path(fs_restore_options, _vm, drive_list,
                                                       special_vm_drive[_vm])
                else:
                    flr_options = self.file_level_path(fs_restore_options, _vm, drive_list)
                fs_restore_options = flr_options['fs_restore_options']
                self.fs_restore_dest = fs_restore_options.client_machine.join_path(
                    fs_restore_options.restore_path,
                    self.backup_folder_name, _vm)
                agentless_option = ""
                dest_client = Machine(fs_restore_options.destination_client,
                                      self.auto_commcell.commcell)
                flr_options['dest_machine'] = dest_client
                self.log.info("Restore dest path {}".format(self.fs_restore_dest))
                if dest_client.check_directory_exists(self.fs_restore_dest):
                    self.log.info(
                        "Directory already exist on Restore proxy, Performing cleanup ")
                    dest_client.remove_directory(self.fs_restore_dest)
                fs_restore_job = self.subclient. \
                    guest_file_restore(vm_name=_vm,
                                       folder_to_restore=flr_options['_fs_path_to_browse_list'],
                                       destination_client=fs_restore_options.destination_client,
                                       destination_path=self.fs_restore_dest,
                                       copy_precedence=fs_restore_options.copy_precedence,
                                       preserve_level=flr_options['_preserve_level'],
                                       unconditional_overwrite=fs_restore_options.unconditional_overwrite,
                                       browse_ma=flr_options['browse_ma'],
                                       fbr_ma=fs_restore_options.fbr_ma,
                                       agentless=agentless_option,
                                       from_date=kwargs.get('from_date', 0),
                                       to_date=kwargs.get('to_date', 0))
                self.log.info(" Running restore Job {0} :".format(fs_restore_job))
                if not fs_restore_job.wait_for_completion():
                    raise Exception(
                        "Failed to run file level restore job {0} with error:{1}".format(
                            fs_restore_job.job_id, fs_restore_job.delay_reason))
                self.log.info("file level restore Job got complete JOb Id:{0}".format(
                    fs_restore_job.job_id))
                if fs_restore_options.validation:
                    if special_vm_drive:
                        self.file_level_validation(flr_options, drive_list, special_vm_drive[_vm])
                    else:
                        self.file_level_validation(flr_options, drive_list)
                else:
                    self.log.info("Validation is being skipped")
                if dest_client.check_directory_exists(self.fs_restore_dest):
                    self.log.info(
                        "Validation Completed, performing post restore cleanup on Restore proxy ")
                    dest_client.remove_directory(self.fs_restore_dest)
            self.log.info(
                "Ran file level restore from all the VMs and its drives")
        except Exception as err:
            self.log.exception(
                "Exception at restore job please check logs for more details {0}".format(err))
            self.log.error("Restore: FAIL - File level files Restore Failed")
            raise err

    def file_level_path(self, fs_restore_options, vm, drive_list=None, special_vm_drive=None):
        """
        Fetch the source location for restore
        Args:
            fs_restore_options              (object):   options that need to be set while
                                                        performing guest file restore

            vm                              (string):   name of the vm whose path needs to be
                                                        fetched

            drive_list                      (dict) :    Dict of drives for the vm

            special_vm_drive                (string):     Drive where more data needs to be copied

        Returns:
            flr_options                      (dict):    Options needed to be used during file level
                                                        restore

        Raises:
            Exception:
                If it fails to fetch the path
        """
        try:
            _path_dict = {}
            if not drive_list:
                drive_list = self.hvobj.VMs[vm].drive_list
            _fs_path_to_browse_list = []
            if self.hvobj.VMs[vm].guest_os.lower() != 'windows':
                _preserve_level = 10
            else:
                _preserve_level = fs_restore_options.preserve_level
            for label, _drive in drive_list.items():
                if fs_restore_options.browse_from_snap \
                        and self.hvobj.VMs[vm].machine.os_info.lower() == "windows":
                    _temp_vm, _temp_vmid = self.subclient._get_vm_ids_and_names_dict_from_browse()
                    _browse_request = self.subclient.guest_files_browse(_temp_vmid[vm],
                                                                        media_agent=fs_restore_options.browse_ma)
                    _path = self.find_snap_guest_file_path(_browse_request[1], label)
                    _path_dict[label] = _path
                    _fs_path_to_browse = [_path, self.backup_folder_name, "TestData",
                                          self.timestamp]
                else:
                    if _drive == "/":
                        _fs_path_to_browse = [self.backup_folder_name, "TestData", self.timestamp]
                    else:
                        _fs_path_to_browse = [label, self.backup_folder_name, "TestData",
                                              self.timestamp]
                _fs_path_to_browse = "\\".join(_fs_path_to_browse)
                _fs_path_to_browse_list.append(_fs_path_to_browse)
                if special_vm_drive and special_vm_drive == _drive:
                    _fs_path_to_browse = [_fs_path_to_browse.split('\\')[0],
                                          "special_testdata", "TestData",
                                          self.timestamp + "special_testdata"]
                    _fs_path_to_browse = "\\".join(_fs_path_to_browse)
                    _fs_path_to_browse_list.append(_fs_path_to_browse)

            if fs_restore_options.is_ma_specified is False:
                browse_ma = ""
            else:
                browse_ma = fs_restore_options.browse_ma

            self._is_windows_live_browse = self._check_if_windows_live_browse(
                self.hvobj.VMs[vm].guest_os,
                fs_restore_options.metadata_collected)

            if not fs_restore_options.metadata_collected:
                if self._is_windows_live_browse:
                    ma_machine = self._get_browse_ma(fs_restore_options)
                    self.ma_machine = ma_machine
                    self.ma_client_name = self.auto_commcell.commcell.clients.get(
                        fs_restore_options._browse_ma_client_name)

            flr_options = {"fs_restore_options": fs_restore_options,
                           "_fs_path_to_browse_list": _fs_path_to_browse_list,
                           "_preserve_level": _preserve_level,
                           "browse_ma": browse_ma,
                           "_path_dict": _path_dict,
                           "vm": vm}
            self.log.info(pprint.pformat(vars(fs_restore_options)))
            self.log.info("flr_options: {}".format(flr_options))
            return flr_options
        except Exception as err:
            self.log.exception(
                "An exception occurred in getting file level source path: {0}".format(err))
            raise err

    def file_level_validation(self, flr_options, drive_list=None, special_vm_drive=None):
        """

        Args:
            flr_options                         (dict):     Options needed to be used during file level
                                                            restore

            drive_list                          (dict) :    Dict of drives for the vm

            special_vm_drive                    (string):   Drive where more data is copied

        Raises:
            Exception:
                If it fails in performing file level validation

        """

        try:
            fs_restore_options = flr_options['fs_restore_options']
            vm = flr_options['vm']
            if not drive_list:
                drive_list = self.hvobj.VMs[vm].drive_list
            if self.hvobj.VMs[
                vm].machine.os_info.lower() != 'windows' and not self.backup_option.collect_metadata:
                try:
                    _restore_dest = flr_options['dest_machine'].get_folders_in_path(
                        self.fs_restore_dest, recurse=False)
                    if flr_options['dest_machine'].os_info.lower() != 'windows':
                        _restore_dest = _restore_dest[1:]
                    if not _restore_dest or len(_restore_dest) != 1:
                        raise Exception
                    else:
                        _restore_dest = _restore_dest[0]
                except Exception as err:
                    self.log.exception(
                        "Error in Destination location. Multiple/No folders in {}. please check and rerun".format(
                            self.fs_restore_dest))
                    raise err
            else:
                _restore_dest = self.fs_restore_dest
            self.log.info("Restore destination: {}".format(_restore_dest))
            for label, _drive in drive_list.items():
                if fs_restore_options.browse_from_snap \
                        and self.hvobj.VMs[vm].machine.os_info.lower() == "windows":
                    label = flr_options['_path_dict'][label]
                self.log.info("Validating {0} of vm: {1}".format(_drive, vm))
                if self.hvobj.VMs[vm].machine.os_info.lower() != "windows":
                    label = label.strip().split("/", 1)[-10:][1].split("/") if label.startswith(
                        '/') else label.strip().split("/")[-10:]
                    label = (flr_options['dest_machine'].join_path(*label)).strip("\\")
                _restore_folder = flr_options['dest_machine'].join_path(_restore_dest,
                                                                        label) if _drive != '/' else _restore_dest
                _restore_folder = flr_options['dest_machine'].join_path(_restore_dest,
                                                                        label) if _drive != '/' else _restore_dest
                _final_restore_path = flr_options['dest_machine'].join_path(_restore_folder, self.backup_folder_name,
                                                                            "TestData",
                                                                            self.timestamp)
                self.log.info('Validation on drive: {}, final restore folder: {}'.format(_drive, _final_restore_path))
                self.fs_testdata_validation(
                    flr_options['dest_machine'],
                    flr_options['dest_machine'].join_path(_restore_folder, self.backup_folder_name,
                                                          "TestData",
                                                          self.timestamp))
                if special_vm_drive and special_vm_drive == _drive:
                    special_path = self.testdata_path + "special_testdata"
                    dest_path = flr_options['dest_machine'].join_path(
                        _restore_folder, "special_testdata",
                        "TestData", self.timestamp + "special_testdata")
                    self.log.info("Verifying testdata for huge size source:{0}, Destination: {1}".
                                  format(special_path, dest_path))
                    self.auto_vsaclient.fs_testdata_validation(
                        flr_options['dest_machine'], special_path, dest_path)
            if not fs_restore_options.metadata_collected and not fs_restore_options.browse_from_snap:
                if self.hvobj.VMs[vm].machine.os_info.lower() == "windows":
                    if self.controller_machine.os_info.lower() != 'windows':
                        self.log.info("Skipping the block level validation for Unix controller. "
                                      "Cross OS executable generation not supported. Support will be added. "
                                      "Sleeping for 12 minutes")
                        time.sleep(720)
                    elif self.ma_machine.os_info.lower() == "windows":
                        attempt = 0
                        while attempt < 3:
                            try:
                                self.block_level_validation(vm,
                                                            fs_restore_options._browse_ma_client_name)
                                break
                            except Exception as exp:
                                attempt = attempt + 1
                                time.sleep(60)
                        if attempt >= 3:
                            self.block_level_validation(vm,
                                                        fs_restore_options._browse_ma_client_name)
                    else:
                        _dest_machine = Machine(fs_restore_options.destination_client,
                                                self.auto_commcell.commcell)
                        if _dest_machine.os_info.lower() == "windows":
                            self.log.info("Browse is done in Proxy. Pseudomount validation "
                                          "will be done on {}".format(
                                fs_restore_options.destination_client))
                            self.block_level_validation(vm,
                                                        fs_restore_options.destination_client)
                        else:
                            self.log.info("Browse must has been done on CS so skipping pseudomount validation"
                                          "please pass the Windows MA in the JSON input if "
                                          "validation is required. sleeping for 12 minutes")
                            time.sleep(720)
                        del _dest_machine
            # vmware live file cleanup validation
            if fs_restore_options.browse_from_snap:
                if self.hvobj.instance_type == "vmware":
                    self.vmware_live_browse_validation(
                        vm, self.backup_job.job_id, self.backupcopy_job_id, fs_restore_options.metadata_collected)
                else:
                    self.log.info("Sleeping for 12 minutes for the snap to unmount")
                    time.sleep(720)
        except Exception as err:
            self.log.exception("An exception occurred in doing file level validation {0}".format(err))
            raise err

    def get_extent_count(self, db_path, db_name):
        """
        Get the extent number of extetnts in live browse DB
        Args:
            db_path                 (str):  db folder path of the livebrowse db

            db_name                 (str):  db file path of the livebrowse db

        Returns:
            extent_count            (int):  Number of extents in DB currently

        Raises:
            Exception:
                if it fails to get the extent count
        """
        try:
            utils_path = VirtualServerUtils.UTILS_PATH
            _exe_file = os.path.join(db_path, "Extentcount.exe")
            if self.ma_machine.check_file_exists(_exe_file):
                self.exe_file_path = _exe_file
            elif self.controller_machine.check_file_exists(os.path.join(utils_path, "Extentcount.exe")):
                self.ma_machine.copy_from_local(os.path.join(utils_path, "Extentcount.exe"), db_path)
                if self.ma_machine.check_file_exists(_exe_file):
                    self.exe_file_path = _exe_file
                else:
                    raise Exception("Failed to copy the extent exe folder")
            elif not self.exe_file_path:
                extent_count_file = os.path.join(utils_path, "Extentcount.py")
                exe_file = self.controller_machine.generate_executable(extent_count_file)
                self.ma_machine.copy_from_local(exe_file, db_path,
                                                raise_exception=True)
                _exe_file = os.path.join(db_path, os.path.split(exe_file)[1:][0])
                if self.ma_machine.check_file_exists(_exe_file):
                    self.exe_file_path = _exe_file
                else:
                    raise Exception("Failed to copy the extent exe folder")
            self.log.info("Sleeping for 2 minutes before trying")
            time.sleep(120)
            cmd = "iex " + "\"" + "& " + "\'" + self.exe_file_path + "\' \'" + db_name + "\'\""
            output = self.ma_machine.execute_command(cmd)
            return int(output.formatted_output.strip())

        except Exception as err:
            self.log.exception("An exception occurred in getting extent count %s", str(err))
            raise err

    def block_level_validation(self, _vm, browse_ma):
        """
        Performs Windows Block level validation

        Args:

            _vm                                     (str):  backup Vm for which guest level
                                                            restore is performed

            browse_ma                               (str):  Ma used for guest file restore
                                                            browse

        Raises:
            Exception:
                When block level validation fails
        """

        try:

            wait_time = 0
            error_count = 0
            pre_extent_count = 0
            do_block_level_validation = False

            _browse_ma_jr_dir = self.auto_commcell.get_job_results_dir(browse_ma)
            browse_ma_obj = Machine(commcell_object=self.auto_commcell.commcell, machine_name=browse_ma)
            reg_value = browse_ma_obj.get_registry_value(value='nPseudoMountPrunerMinFreeSpace',
                                                         win_key=browse_ma_obj.key.replace("%s", "VirtualServer"))
            if reg_value != '':
                do_block_level_validation = True

            min_space_prune_value = self.get_min_space_prune_val(browse_ma_obj, _browse_ma_jr_dir)
            storage_details = browse_ma_obj.get_storage_details()
            available = storage_details[_browse_ma_jr_dir[0]]['available']
            if int(min_space_prune_value) < VirtualServerConstants.pruning_constants.get("min_percentage_limit") or \
                    int(available) < VirtualServerConstants.pruning_constants.get("min_available_limit"):
                do_block_level_validation = True

            if self.hvobj.VMs[_vm].guest_os.lower() == "windows" and do_block_level_validation:
                self.log.info("performing pruning validation on Browse MA {0}".format(browse_ma))
                while wait_time <= 600:
                    db_path = VirtualServerConstants.get_live_browse_db_path(_browse_ma_jr_dir)
                    db_name = VirtualServerUtils.find_live_browse_db_file(self.ma_machine, db_path)
                    try:
                        current_extent_count = self.get_extent_count(db_path, db_name)
                    except:
                        self.log.info("sleeping for 30 secs and retrying for extent_count")
                        time.sleep(30)
                        wait_time = wait_time + 30
                        current_extent_count = self.get_extent_count(db_path, db_name)
                    self.log.info("Current extent count is %s", str(current_extent_count))

                    if current_extent_count > pre_extent_count:
                        self.log.info(
                            "Extents should have reduced but it increased. Will wait for 3 error")
                        error_count = error_count + 1
                        if error_count > 4:
                            raise Exception("pruning is not happening , please check logs")
                    pre_extent_count = current_extent_count
                    wait_time = wait_time + 120
                time.sleep(200)
            else:
                self.log.info("Skipping the block level validation because of less pruning space or MA a linux machine")
                self.log.info("Sleeping for 700 seconds")
                time.sleep(700)

            self.log.info("performing Post un-mount validation")
            # disk Un-mount validation
            # self.disk_count_validation(disk_count_before_restore)
            self.disk_unmount_validation(_vm)
            # Path CleanUp Validation
            live_browse_mount_path = VirtualServerConstants.get_live_browse_mount_path(
                _browse_ma_jr_dir,
                self.hvobj.VMs[_vm].guid,
                self.hvobj.VMs[_vm].guest_os)
            if self.ma_machine.check_directory_exists(live_browse_mount_path):
                files_in_path = self.ma_machine.get_files_in_path(live_browse_mount_path)
                if files_in_path:
                    self.log.info("Not files are cleaned after Live browse, please check the folder %s"
                                  % live_browse_mount_path)
                    raise Exception("Staging Path CleanUp Failed")
                else:
                    self.log.info("Staging Path %s was cleaned successfully", live_browse_mount_path)
            else:
                self.log.info("Staging Path was cleaned successfully along with the mount path directory %s",
                              live_browse_mount_path)

        except Exception as err:
            self.log.exception("Failed in performing Block Level Validation")
            raise err

    def fs_testdata_validation(self, dest_client, dest_location):
        """
        Does Validation of Backed up and data restored from file level

        Args:
                dest_client     (obj)   -   Machine class object of Destination client

                dest_location   (str)   - destination location of file level restore job


        Exception
                if folder comparison fails


        """
        self.auto_vsaclient.fs_testdata_validation(dest_client=dest_client,
                                                   source_location=self.testdata_path,
                                                   dest_location=dest_location)

    def disk_restore(self, disk_restore_options, _skip_validation=False, discovered_client=None, **kwargs):
        """

        perform Disk restore for specific subclinet

        Args:
                disk_restore_options    (object):   represent options that need to be set
                                                    while performing disk  restore

                _skip_validation        (bool): Set to true to skip disk validation

                discovered_client       (string):   Pass the discovered client name if restore has to be performed from
                                                    discovered client.

                **kwargs                         : Arbitrary keyword arguments

        Exception:
                        if job fails
                        if validation fails

        """
        try:
            if kwargs.get('msg'):
                VirtualServerUtils.decorative_log(kwargs.get('msg'))
            if discovered_client:
                temp_vmlist = [discovered_client]
                VirtualServerUtils.discovered_client_initialize(self, discovered_client)
            else:
                temp_vmlist = self.vm_list
            for _vm in temp_vmlist:

                dest_client = Machine(machine_name=disk_restore_options.destination_client,
                                      commcell_object=self.auto_commcell.commcell)
                self.disk_restore_dest = dest_client.join_path(
                    disk_restore_options.restore_path, self.backup_folder_name, _vm)

                # """
                disk_restore_job = self.subclient.disk_restore(
                    _vm, proxy_client=disk_restore_options.destination_client,
                    destination_path=self.disk_restore_dest,
                    copy_precedence=disk_restore_options.copy_precedence,
                    media_agent=disk_restore_options.disk_browse_ma,
                    snap_proxy=disk_restore_options.snap_proxy)

                if not disk_restore_job.wait_for_completion():
                    raise Exception(
                        "Failed to run disk level restore job {0} with error:{1}".format(
                            disk_restore_job.job_id, disk_restore_job.delay_reason))
                self.log.info("Disk restore job completed successfully with job id {0}".format(
                    disk_restore_job.job_id))
                # """
                if self.hvobj.VMs[_vm].guest_os.lower() == "windows":
                    # Commenting out validation for vmware disk level restore for now
                    if not self.hvobj.instance_type == "vmware" and disk_restore_options.validation:
                        _vm_disk_list = self.hvobj.VMs[_vm].disk_list
                        for each_disk in _vm_disk_list:
                            if True in list(map(each_disk.__contains__,
                                                [".avhdx", ".avhd", ".AVHDX", ".AVDH"])):
                                self.log.info("Skipping validation as there are snapsots in VM")
                                _skip_validation = True
                                break

                        if not _skip_validation:
                            self.disk_validation(
                                self.hvobj.VMs[_vm],
                                disk_restore_options._destination_pseudo_client,
                                self.disk_restore_dest,
                                disk_restore_options.client_machine)

                dest_client.remove_directory(self.disk_restore_dest)

        except Exception as err:
            self.log.exception("Exception occurred please check logs")
            raise Exception("Disk Restore Job failed, please check agent logs {0}".format(err))

    def disk_validation(
            self,
            vm_obj,
            destination_client_name,
            disk_restore_destination,
            dest_machine):
        """
        Performs Disk Validation by mounting the restored disk on the Host

        Args:

        _vm                     (str)   - object of VMHelper class

        destination_client_name    (str)   - Pseudo  client name of the destination client

        disk_restore_destination    (str)   - restore path of all the disk

        dest_machine    (obj) - destimation client where disk restores are performed

        Exception:
                        if job fails
                        if validation fails

        """
        try:

            self.log.info("Performed Restore in client %s" %
                          destination_client_name)
            dest_client_hypervisor = self.auto_vsainstance._create_hypervisor_object(
                destination_client_name)

            _list_of_disks = dest_client_hypervisor.get_disk_in_the_path(disk_restore_destination,
                                                                         dest_machine)

            _vm_disk_list = vm_obj.disk_list
            if not _vm_disk_list:
                self.log.info(
                    "Cannot validate the Disk as we cannot find the disk attached to the VM")
                return False

            if not ((_list_of_disks is None) or (_list_of_disks == [])):
                _final_mount_disk_list = []
                for each_disk in _vm_disk_list:
                    each_disk_name = os.path.basename(each_disk).split(".")[0]
                    for disk_path in _list_of_disks:
                        if each_disk_name.lower() in disk_path.lower():
                            _final_mount_disk_list.append(disk_path)

            else:
                self.log.info(
                    "the Disk cannot be validated as we cannot find disk with Hypervisor extension,\
                                                                could be converted disk")
                return False

            if not _final_mount_disk_list:
                _final_mount_disk_list = _list_of_disks

            for _file in _final_mount_disk_list:

                _file = disk_restore_destination + "\\" + _file
                self.log.info("Validation Started For Disk :[%s]" % _file)
                _drive_letter = dest_client_hypervisor.mount_disk(vm_obj, _file, dest_machine)
                if _drive_letter != -1:
                    for each_drive in _drive_letter:
                        dest_folder_path = VirtualServerConstants.get_folder_to_be_compared(
                            self.backup_folder_name, each_drive, self.timestamp)
                        self.log.info("Folder comparison started...")
                        time.sleep(5)
                        self.fs_testdata_validation(
                            dest_machine, dest_folder_path)
                else:
                    self.log.error("ERROR - Error mounting VMDK " + _file)
                    raise Exception("Exception at Mounting Disk ")

                dest_client_hypervisor.un_mount_disk(vm_obj, _file)

        except Exception as err:
            self.log.exception("Exception occurred please check logs")
            dest_client_hypervisor.un_mount_disk(vm_obj, _file)
            raise err

    def attach_disk_restore(self, attach_disk_restore_options, discovered_client=None, **kwargs):
        """

        perform Attach disk restore for specific subclinet

        Args:
                attach_disk_restore_options    (object):   represent options that need to be set
                                                    while performing attach disk  restore

                discovered_client       (string):   Pass the discovered client name if restore has to be performed from
                                                    discovered client.

                **kwargs                         : Arbitrary keyword arguments

        Exception:
                        if job fails
                        if validation fails

        """
        try:
            if kwargs.get('msg'):
                VirtualServerUtils.decorative_log(kwargs.get('msg'))
            if discovered_client:
                temp_vmlist = [discovered_client]
                VirtualServerUtils.discovered_client_initialize(self, discovered_client)
            else:
                temp_vmlist = self.vm_list
            for _vm in temp_vmlist:
                self.hvobj.VMs[_vm].update_vm_info(force_update=True)
                datastore = self.hvobj.VMs[_vm].datastore
                _before_restore = len(self.hvobj.VMs[_vm].disk_list)
                self.log.info("Number of disk in vm {0} before restore is {1}".
                              format(_vm, _before_restore))
                disk_restore_job = self.subclient.attach_disk_restore(
                    _vm, vcenter=attach_disk_restore_options.vcenter,
                    proxy_client=attach_disk_restore_options._dest_client_name,
                    esx=self.hvobj.VMs[_vm].esx_host,
                    datastore=datastore,
                    copy_precedence=attach_disk_restore_options.copy_precedence,
                    media_agent=attach_disk_restore_options.disk_browse_ma,
                    snap_proxy=attach_disk_restore_options.snap_proxy)

                if not disk_restore_job.wait_for_completion():
                    raise Exception(
                        "Failed to run disk level restore job {0} with error:{1}".format(
                            disk_restore_job.job_id, disk_restore_job.delay_reason))
                self.log.info("Attach Disk restore job completed successfully with job id {0}"
                              .format(disk_restore_job.job_id))
                self.log.info("Validating the disks count")
                _after_restore = len(self.hvobj.VMs[_vm].disk_list)
                self.log.info("Number of disk in vm {0} After restore is {1}".
                              format(_vm, _after_restore))
                if _after_restore / _before_restore == 2:
                    self.log.info("Disk restore validation complete")
                else:
                    self.log.error("Disk restore validation failed")
                self.log.info("Deleting the attached disks and updating the disk count")
                if not self.hvobj.VMs[_vm].delete_disks():
                    self.log.exception("Deletion of attached disk failed. clean the disks manually")
                    raise Exception("Failed to delete the disks")

        except Exception as err:
            self.log.exception("Attach Disk Restore Job failed. please check logs")
            raise Exception(
                "Attach Disk Restore Job failed, please check agent logs {0}".format(err))

    def virtual_machine_restore(self, vm_restore_options, discovered_client=None, **kwargs):
        """
        perform Full VM restore for specific subclient

        Args:
                vm_restore_options      (object):   options that need to be set
                                                    while performing vm  restore

                discovered_client       (string):   Pass the discovered client name if restore has to be performed from
                                                    discovered client.

                **kwargs                         : Arbitrary keyword arguments
        Exception:
                        if job fails
                        if validation fails

        """
        try:
            if kwargs.get('msg'):
                VirtualServerUtils.decorative_log(kwargs.get('msg'))
            vm_to_restore = None
            if discovered_client:
                vm_to_restore = discovered_client
                VirtualServerUtils.discovered_client_initialize(self, discovered_client)
            if vm_restore_options.in_place_overwrite:
                def hyperv():
                    vm_restore_job = self.subclient.full_vm_restore_in_place(
                        overwrite=vm_restore_options.unconditional_overwrite,
                        power_on=vm_restore_options.power_on_after_restore,
                        copy_precedence=vm_restore_options.copy_precedence,
                        add_to_failover=vm_restore_options.register_with_failover,
                        snap_proxy=vm_restore_options.snap_proxy)
                    return vm_restore_job

                def vmware():
                    if self.subclient.is_intelli_snap_enabled:
                        if self.subclient.snapshot_engine_name.lower() == 'virtual server agent snap':
                            self.log.info("Inplace full vm restore is not supported for vvol")
                            return None
                    vm_restore_job = self.subclient.full_vm_restore_in_place(
                        vm_to_restore=vm_to_restore,
                        overwrite=vm_restore_options.unconditional_overwrite,
                        power_on=vm_restore_options.power_on_after_restore,
                        copy_precedence=vm_restore_options.copy_precedence,
                        to_time=vm_restore_options._end_time)
                    return vm_restore_job

                def oraclevm():
                    vm_restore_job = self.subclient.full_vm_restore_in_place(
                        overwrite=vm_restore_options.unconditional_overwrite,
                        destination_client=vm_restore_options.proxy_client,
                        power_on=vm_restore_options.power_on_after_restore,
                        copy_precedence=vm_restore_options.copy_precedence)
                    return vm_restore_job

                def azureRM():
                    vm_restore_job = self.subclient.full_vm_restore_in_place(
                        overwrite=vm_restore_options.unconditional_overwrite,
                        power_on=vm_restore_options.power_on_after_restore,
                        copy_precedence=vm_restore_options.copy_precedence)
                    return vm_restore_job

                def fusion_compute():
                    vm_restore_job = self.subclient.full_vm_restore_in_place(
                        overwrite=vm_restore_options.unconditional_overwrite,
                        proxy_client=vm_restore_options.proxy_client,
                        power_on=vm_restore_options.power_on_after_restore,
                        copy_precedence=vm_restore_options.copy_precedence)
                    return vm_restore_job

                def azurestack():
                    vm_restore_job = self.subclient.full_vm_restore_in_place(
                        overwrite=vm_restore_options.unconditional_overwrite,
                        power_on=vm_restore_options.power_on_after_restore,
                        copy_precedence=vm_restore_options.copy_precedence)
                    return vm_restore_job

                def openstack():
                    vm_restore_job = self.subclient.full_vm_restore_in_place(
                        overwrite=vm_restore_options.unconditional_overwrite,
                        power_on=vm_restore_options.power_on_after_restore,
                        proxy_client=vm_restore_options.proxy_client,
                        copy_precedence=vm_restore_options.copy_precedence,
                        datastore=vm_restore_options.restoreObj["Datastore"],
                        esx_host=vm_restore_options.restoreObj["esxHost"],
                        esx_server=vm_restore_options.restoreObj["esxServerName"],
                        datacenter=vm_restore_options.restoreObj["Datacenter"],
                        cluster=vm_restore_options.restoreObj["Cluster"])
                    return vm_restore_job

                def red_hat():
                    vm_restore_job = self.subclient.full_vm_restore_in_place(
                        power_on=vm_restore_options.power_on_after_restore,
                        proxy_client=vm_restore_options.proxy_client,
                        copy_precedence=vm_restore_options.copy_precedence)
                    return vm_restore_job

                def amazon():
                    vm_restore_job = self.subclient.full_vm_restore_in_place(
                        overwrite=vm_restore_options.unconditional_overwrite,
                        power_on=vm_restore_options.power_on_after_restore,
                        copy_precedence=vm_restore_options.copy_precedence)
                    return vm_restore_job

                def oci():
                    vm_restore_job = self.subclient.full_vm_restore_in_place(
                        source_vm_details=vm_restore_options.source_vm_details,
                        vm_to_restore=vm_restore_options.auto_subclient.vm_list[0],
                        power_on=vm_restore_options.power_on_after_restore,
                        proxy_client=vm_restore_options.proxy_client,
                        copy_precedence=vm_restore_options.copy_precedence,
                        indexing_v2=True)
                    return vm_restore_job

                def nutanix():
                    vm_restore_job = self.subclient.full_vm_restore_in_place(
                        overwrite=vm_restore_options.unconditional_overwrite,
                        power_on=vm_restore_options.power_on_after_restore,
                        copy_precedence=vm_restore_options.copy_precedence)
                    return vm_restore_job

                def googlecloud():
                    vm_restore_job = self.subclient.full_vm_restore_in_place(
                        overwrite=vm_restore_options.unconditional_overwrite,
                        power_on=vm_restore_options.power_on_after_restore,
                        copy_precedence=vm_restore_options.copy_precedence)
                    return vm_restore_job

                def xen():
                    vm_restore_job = self.subclient.full_vm_restore_in_place(
                        overwrite=vm_restore_options.unconditional_overwrite,
                        power_on=vm_restore_options.power_on_after_restore,
                        copy_precedence=vm_restore_options.copy_precedence)
                    return vm_restore_job

                hv_dict = {"hyper-v": hyperv, "vmware": vmware, "azure resource manager": azureRM,
                           "openstack": openstack, "oraclevm": oraclevm,
                           "fusioncompute": fusion_compute, "azure stack": azurestack,
                           "red hat virtualization": red_hat, "amazon": amazon,
                           "oracle cloud infrastructure": oci, "nutanix ahv": nutanix,
                           "google cloud platform": googlecloud, "xen": xen}
                vm_restore_job = (
                    hv_dict[vm_restore_options.dest_auto_vsa_instance.vsa_instance_name.lower()])()

            else:

                def hyperv():
                    if self.backup_folder_name:
                        vm_restore_dest = os.path.join(vm_restore_options.destination_path,
                                                       self.backup_folder_name)
                    else:
                        vm_restore_dest = vm_restore_options.destination_path
                    vm_restore_options.destination_path = vm_restore_dest
                    vm_restore_job = self.subclient.full_vm_restore_out_of_place(
                        destination_client=vm_restore_options._destination_pseudo_client,
                        proxy_client=vm_restore_options.proxy_client,
                        destination_path=vm_restore_options.destination_path,
                        overwrite=vm_restore_options.unconditional_overwrite,
                        power_on=vm_restore_options.power_on_after_restore,
                        copy_precedence=vm_restore_options.copy_precedence,
                        add_to_failover=vm_restore_options.register_with_failover,
                        restore_option=vm_restore_options.advanced_restore_options)
                    return vm_restore_job

                def fusion_compute():
                    vm_restore_job = self.subclient.full_vm_restore_out_of_place(
                        destination_client=vm_restore_options._destination_pseudo_client,
                        proxy_client=vm_restore_options.proxy_client,
                        datastore=vm_restore_options.datastore,
                        host=vm_restore_options.host,
                        overwrite=vm_restore_options.unconditional_overwrite,
                        power_on=vm_restore_options.power_on_after_restore,
                        copy_precedence=vm_restore_options.copy_precedence)
                    return vm_restore_job

                def vmware():
                    vm_restore_job = self.subclient.full_vm_restore_out_of_place(
                        vm_to_restore=vm_to_restore,
                        overwrite=vm_restore_options.unconditional_overwrite,
                        power_on=vm_restore_options.power_on_after_restore,
                        proxy_client=vm_restore_options.proxy_client,
                        copy_precedence=vm_restore_options.copy_precedence,
                        disk_option=vm_restore_options.disk_option,
                        vcenter_client=vm_restore_options._dest_client_name,
                        datastore=vm_restore_options._datastore,
                        esx_host=vm_restore_options._host[0],
                        source_ip=vm_restore_options.source_ip,
                        destination_ip=vm_restore_options.destination_ip,
                        network=vm_restore_options._network,
                        to_time=vm_restore_options._end_time,
                        dest_computer_name=vm_restore_options.dest_computer_name,
                        source_subnet=vm_restore_options.source_subnet if hasattr(
                            vm_restore_options, 'source_subnet') else None,
                        source_gateway=vm_restore_options.source_gateway if hasattr(
                            vm_restore_options, 'source_gateway') else None,
                        destination_subnet=vm_restore_options.destination_subnet if hasattr(
                            vm_restore_options, 'destination_subnet') else None,
                        destination_gateway=vm_restore_options.destination_gateway if hasattr(
                            vm_restore_options, 'destination_gateway') else None)
                    return vm_restore_job

                def azure():
                    vm_restore_job = self.subclient.full_vm_restore_out_of_place(
                        overwrite=vm_restore_options.unconditional_overwrite,
                        power_on=vm_restore_options.power_on_after_restore,
                        copy_precedence=vm_restore_options.copy_precedence,
                        resource_group=vm_restore_options.Resource_Group,
                        storage_account=vm_restore_options.Storage_account,
                        datacenter=vm_restore_options.datacenter,
                        disk_type=vm_restore_options.disk_type_dict,
                        restore_option=vm_restore_options.advanced_restore_options)
                    return vm_restore_job

                def azurestack():
                    vm_restore_job = self.subclient.full_vm_restore_out_of_place(
                        overwrite=vm_restore_options.unconditional_overwrite,
                        power_on=vm_restore_options.power_on_after_restore,
                        copy_precedence=vm_restore_options.copy_precedence,
                        resource_group=vm_restore_options.Resource_Group,
                        storage_account=vm_restore_options.Storage_account,
                        restore_option=vm_restore_options.advanced_restore_options)
                    return vm_restore_job

                def oraclevm():
                    vm_restore_job = self.subclient.full_vm_restore_out_of_place(
                        virtualization_client=vm_restore_options._destination_pseudo_client,
                        destination_client=vm_restore_options.proxy_client,
                        repository=vm_restore_options.datastore,
                        server=vm_restore_options.host,
                        overwrite=vm_restore_options.unconditional_overwrite,
                        power_on=vm_restore_options.power_on_after_restore,
                        copy_precedence=vm_restore_options.copy_precedence)
                    return vm_restore_job

                def openstack():
                    vm_restore_job = self.subclient.full_vm_restore_out_of_place(
                        vm_to_restore=vm_restore_options.auto_subclient.vm_list,
                        power_on=vm_restore_options.power_on_after_restore,
                        destination_client=vm_restore_options._destination_pseudo_client,
                        proxy_client=vm_restore_options.proxy_client,
                        copy_precedence=vm_restore_options.copy_precedence,
                        datastore=vm_restore_options.restoreObj["Datastore"],
                        securityGroups=vm_restore_options.restoreObj["securityGroups"],
                        esx_host=vm_restore_options.restoreObj["esxHost"],
                        esx_server=vm_restore_options.restoreObj["esxServerName"],
                        datacenter=vm_restore_options.restoreObj["Datacenter"],
                        cluster=vm_restore_options.restoreObj["Cluster"])

                    return vm_restore_job

                def red_hat():
                    vm_restore_job = self.subclient.full_vm_restore_out_of_place(
                        power_on=vm_restore_options.power_on_after_restore,
                        destination_client=vm_restore_options._destination_pseudo_client,
                        proxy_client=vm_restore_options.proxy_client,
                        cluster=vm_restore_options.cluster,
                        repository=vm_restore_options.repository,
                        copy_precedence=vm_restore_options.copy_precedence)
                    return vm_restore_job

                def amazon():
                    vm_restore_job = self.subclient.full_vm_restore_out_of_place(
                        overwrite=vm_restore_options.unconditional_overwrite,
                        power_on=vm_restore_options.power_on_after_restore,
                        proxy_client=vm_restore_options.proxy_client,
                        copy_precedence=vm_restore_options.copy_precedence)
                    return vm_restore_job

                def oci():
                    vm_restore_job = self.subclient.full_vm_restore_out_of_place(
                        source_vm_details=vm_restore_options.source_vm_details,
                        vm_to_restore=vm_restore_options.auto_subclient.vm_list[0],
                        new_name=vm_restore_options.new_name,
                        destination_client=vm_restore_options._destination_pseudo_client,
                        proxy_client=vm_restore_options.proxy_client,
                        power_on=vm_restore_options.power_on_after_restore,
                        indexing_v2=True)
                    return vm_restore_job

                def nutanix():
                    vm_restore_job = self.subclient.full_vm_restore_out_of_place(
                        overwrite=vm_restore_options.unconditional_overwrite,
                        power_on=vm_restore_options.power_on_after_restore,
                        proxy_client=vm_restore_options.proxy_client,
                        copy_precedence=vm_restore_options.copy_precedence,
                        container=vm_restore_options.container,
                        restore_option=vm_restore_options.advanced_restore_options)
                    return vm_restore_job

                def googlecloud():
                    vm_restore_job = self.subclient.full_vm_restore_out_of_place(
                        overwrite=vm_restore_options.unconditional_overwrite,
                        power_on=vm_restore_options.power_on_after_restore,
                        proxy_client=vm_restore_options.proxy_client,
                        copy_precedence=vm_restore_options.copy_precedence,
                        zone=vm_restore_options.zone,
                        project_id=vm_restore_options.project_id,
                        restore_option=vm_restore_options.advanced_restore_options)
                    return vm_restore_job

                def xen():
                    vm_restore_job = self.subclient.full_vm_restore_out_of_place(
                        destination_client=vm_restore_options._dest_client_name,
                        overwrite=vm_restore_options.unconditional_overwrite,
                        power_on=vm_restore_options.power_on_after_restore,
                        proxy_client=vm_restore_options.proxy_client,
                        copy_precedence=vm_restore_options.copy_precedence,
                        storage=vm_restore_options.storage,
                        xen_server=vm_restore_options.xen_server)
                    return vm_restore_job

                hv_dict = {"hyper-v": hyperv, "fusioncompute": fusion_compute,
                           "openstack": openstack, "vmware": vmware, "oraclevm": oraclevm,
                           "azure resource manager": azure, "azure stack": azurestack,
                           "red hat virtualization": red_hat, "amazon": amazon,
                           "oracle cloud infrastructure": oci, "nutanix ahv": nutanix,
                           "google cloud platform": googlecloud, "xen": xen}
                vm_restore_job = (
                    hv_dict[vm_restore_options.dest_auto_vsa_instance.vsa_instance_name.lower()])()

            if vm_restore_job:
                self.restore_proxy_client = vm_restore_options.proxy_client
                if isinstance(vm_restore_job, tuple):
                    self.log.info("Restore job is :{} ".format(vm_restore_job[0].job_id))
                    if not vm_restore_job[0].wait_for_completion():
                        raise Exception("Failed to run VM restore job {0} with error: {1}".format(
                            vm_restore_job[0].job_id, vm_restore_job[0].delay_reason))
                    vm_restore_options.restore_job = vm_restore_job[0]
                else:
                    self.log.info("Restore job is : " + str(vm_restore_job.job_id))
                    if not vm_restore_job.wait_for_completion():
                        raise Exception("Failed to run VM restore job {0} with error: {1}".format(
                            vm_restore_job.job_id, vm_restore_job.delay_reason))
                    vm_restore_options.restore_job = vm_restore_job
                self.log.info("Job Status : {0}".format(vm_restore_options.restore_job.status))
                if 'one or more error' in vm_restore_job.status:
                    self.log.error("Restore Job Completed with one or more errors")

                if vm_restore_options.validation:
                    self.vm_restore_validator(vm_restore_options, discovered_vm=discovered_client)

        except Exception as err:
            self.log.error("Exception while submitting Restore job:" + str(err))
            raise err

    def verify_cbt_restore(self, vm_restore_job_id, restore_proxy):
        """
        Verify that CBT is used during In-Place VM restore
        Args:
            vm_restore_job_id    (string):     job ID of the full vm restore
            restore_proxy        (string):     proxy to do restore
        Raises:
            Exception:
                if any exception when reading log file
        """
        try:
            machine_ = Machine(machine_name=restore_proxy,
                               commcell_object=self.auto_commcell.commcell)
            client_ = Client(self.auto_commcell.commcell, restore_proxy)
            log_directory = client_.log_directory
            self.log.info("Navigate to the Proxy's Log Files directory to read vsrst log")
            vsrst_log = machine_.join_path(log_directory, "vsrst.log")
            self.log.info("Restore Log: %s", vsrst_log)
            log_line = machine_.read_file(vsrst_log, search_term=vm_restore_job_id)
            self.log.info(log_line)
            list_of_lines = log_line.split("\n")
            self.log.info("Looking for the logline that contains seek optimization")
            seek_percentage = -1
            seek_optimization_line = ""
            for line in list_of_lines:
                if "SeekDecisionVMWare::printStats() - Number of restored items skipped by SEEK optimization".lower() \
                        in line.lower():
                    self.log.info("Found Seek Optimization in line: %s", line)
                    seek_optimization_line = line
                    seekline = re.search("[0-9]*\%", line)
                    self.log.info("Seek Line :" + seekline.group())
                    if seekline:
                        seek_percentage = int(seekline.group().split('%')[0])
                        break
            if seek_percentage != -1 and seek_percentage > 0:
                self.log.info("Seek Optimization Validation  Succeeded with Percentage: %s " % str(
                    seek_percentage))
            else:
                raise Exception("Seek Optimization validation failed: %s" % seek_optimization_line)
        except Exception as exp:
            self.log.exception("CBT was not used during Restore: %s", str(exp))
            raise exp

    def verify_data_pruned(self, vm_name):
        """
        Args:
            vm_name         (str): VM that gets checked if data is pruned
        Raises:
            Exception
                if there's any exception while accessing VM's directory
        """
        try:
            for each_drive in self.source_obj.drive_list:
                dest_location = self.source_obj.machine.join_path(
                    self.source_obj.drive_list[each_drive],
                    "Differential")
                self.log.info("Destination Location: %s" % dest_location)
                if self.hvobj.VMs[vm_name].machine.check_directory_exists(dest_location):
                    raise Exception(
                        "Data copied after backup was not deleted after doing Full VM In-Place Restore")
                else:
                    self.log.info("Data copied after backup was successfully pruned.")
        except Exception as ex:
            raise Exception("Exception while validating if data copied after Backup was pruned.")

    def vm_restore_validator(self, vm_restore_options, type_of_content="VM", discovered_vm=None):
        """

        Args:
            vm_restore_options              (str):  options that need to be set while performing vm restore

            type_of_content                 (string):   type of content of a subclient

            discovered_vm                   (string): Restored discovered vm that has to be validated

        Returns:

        """
        try:
            if discovered_vm:
                self.temp_vm_list = [discovered_vm]
            else:
                self.temp_vm_list = self.vm_list
            for vm in self.temp_vm_list:
                if vm_restore_options.in_place_overwrite:
                    restore_vm_name = vm
                else:
                    restore_vm_name = "del" + vm
                if vm_restore_options.restore_backup_job is not None:
                    self.vm_restore_validation(vm, restore_vm_name, vm_restore_options, 'Basic')
                else:
                    self.vm_restore_validation(vm, restore_vm_name, vm_restore_options)
        except Exception as err:
            self.log.error("Exception while submitting Restore job:" + str(err))
            raise err

    def vm_restore_validation(self, vm, restore_vm, vm_restore_options, prop='Advanced'):
        """
        Param:
             vm:                     -  Source Vm name
             restore_vm:             -  Restored VM Name
             vm_restore_options      -  options of VM restore options class
             prop                    -  only validate basic or advanced properties
        Exception:
            if validation fails

        """
        try:
            self.log.info(
                "-----Validating Source VM : {0} and Restore VM : {1}----".format(str(vm), str(restore_vm)))
            time.sleep(240)
            vm_restore_options.dest_client_hypervisor.update_hosts()
            if vm_restore_options.in_place_overwrite:
                self.log.info("it is Inplace restore")
                self.source_obj = self.__deepcopy__((self.hvobj.VMs[vm]))
            else:
                self.source_obj = self.hvobj.VMs[vm]

            on_premise = VirtualServerConstants.on_premise_hypervisor(
                vm_restore_options.dest_client_hypervisor.instance_type)

            # Adding floating IP for openstack VM
            if vm_restore_options.dest_auto_vsa_instance.vsa_instance_name == 'openstack':
                self.hvobj.VMs[vm].backup_job = self.backup_job
                if not vm_restore_options.in_place:
                    self.hvobj.OpenStackHandler.projectName = vm_restore_options.restoreObj[
                        'Datacenter']
                    self.hvobj.OpenStackHandler.add_floating_ip(restore_vm)
                    time.sleep(30)
            vm_restore_options.dest_client_hypervisor.VMs = restore_vm
            if vm_restore_options.power_on_after_restore:
                attempt = 1
                while attempt < 5:
                    time.sleep(120)
                    self.restore_obj = vm_restore_options.dest_client_hypervisor.VMs[restore_vm]
                    self.restore_obj.update_vm_info('All', os_info=True, force_update=True)
                    if self.restore_obj.ip is not None:
                        self.log.info("Disk Details for restored vm are : {0}".format(
                            self.restore_obj.disk_dict))
                        attempt = 5
                    else:
                        self.log.info("Attempt number %d failed. "
                                      "Waiting for 2 minutes for VM to come up" % attempt)
                        attempt = attempt + 1
                _source = self.VmValidation(self.source_obj, vm_restore_options)
                _dest = self.VmValidation(self.restore_obj, vm_restore_options)
                if _source == _dest:
                    self.log.info("config validation is successful")
                else:
                    self.log.error("error while configuration validation")
                    raise Exception
                if vm_restore_options.in_place_overwrite and on_premise:
                    if not self.restore_obj.guid == self.source_obj.guid:
                        raise Exception(
                            "The GUID id of the in place restored VM does not match the source VM")

                if (self.hvobj.instance_type.lower() == "amazon" and
                        vm_restore_options.dest_auto_vsa_instance.vsa_instance_name.lower() ==
                        "amazon"):
                    if not vm_restore_options.in_place_overwrite:
                        self.source_obj.validate_name_tag = False
                        self.restore_obj.validate_name_tag = False
                    else:
                        self.source_obj.validate_name_tag = True
                        self.restore_obj.validate_name_tag = True
                    if self.source_obj == self.restore_obj:
                        self.log.info("AWS EC2 instance specific validation successful")
                    else:
                        raise Exception("AWS EC2 instance specific validation FAILED")

                if prop == 'Basic':
                    if re.match(VirtualServerConstants.Ip_regex, "%s" % self.restore_obj.ip):
                        raise Exception(
                            "The IP address of the restored VM is not of the proper format"
                            ". Boot validation failed.")

                else:
                    if ((VirtualServerUtils.validate_ip(self.restore_obj.ip)) and (
                            not (self.restore_obj.machine is None))):
                        self.log.info("Testdata validation")
                        if self._disk_filter:
                            drive_list = self.restore_obj.drive_list
                        else:
                            drive_list = self.source_obj.drive_list
                        for each_drive in drive_list:
                            dest_location = self.source_obj.machine.join_path(
                                drive_list[each_drive],
                                self.backup_folder_name, "TestData",
                                self.timestamp)
                            self.fs_testdata_validation(self.restore_obj.machine, dest_location)
                    else:
                        self.log.info(
                            "Skipping test data validation ,"
                            "because the VM might be running a UNIX OS or the provided subclient have filters applied")

            elif ((self.hvobj.VMs[vm].guest_os.lower() == "windows") and (
                    not vm_restore_options.in_place_overwrite) and (on_premise)):
                if self.hvobj.instance_type == "hyper-v":
                    self.restore_obj = vm_restore_options.dest_client_hypervisor.VMs[restore_vm]
                    _source = self.VmValidation(self.source_obj, vm_restore_options)
                    _dest = self.VmValidation(self.restore_obj, vm_restore_options)
                    if _source == _dest:
                        self.log.info("config validation is successful")
                    else:
                        self.log.error("error while configuration validation")
                        raise Exception
                    time.sleep(120)
                    self.restore_obj.power_off()
                    dest_location = os.path.join(vm_restore_options.destination_path, restore_vm,
                                                 "Virtual Hard Disks")
                    self.disk_validation(self.restore_obj,
                                         vm_restore_options._destination_pseudo_client,
                                         dest_location, vm_restore_options.dest_machine)
                    self.hvobj.VMs[vm].delete_vm(restore_vm)
                    vm_restore_options.dest_machine.remove_directory(dest_location)
                else:
                    self.log.info(
                        "Skipping disk validation for powered off restored vm for {0} instance".format(
                            self.hvobj.instance_type))
            else:
                self.log.info("This is Linux VM and the destination Ip is not proper,"
                              " so no Data Validation cannot be performed")
                if self.hvobj.instance_type == "hyper-v":
                    dest_location = os.path.join(vm_restore_options.destination_path, restore_vm,
                                                 "Virtual Hard Disks")
                    self.hvobj.VMs[vm].delete_vm(restore_vm)
                    vm_restore_options.dest_machine.remove_directory(dest_location)

        except Exception as err:
            self.log.exception("Exception occurred in VM restore validation " + str(err))
            raise Exception("Exception in VM restore validation")

    def _get_all_backup_jobs(self):
        """
        Get all the backup jobs for the subclient
        :return:
            job_history     (dict)  -   all the unaged jobs for that subclient
                Ex:
                    {'job_cycle_1': {'bkp_level_full':  {'job_id':['aux_copy_jobid_1','aux_2']},
                                     'bkp_level_incr': {'job_id1':['aux1','aux2'],
                                                        'job_id2':['aux1','aux2']}
                                    },
                     'job_cycle_2': {'bkp_level_synth': {'job_id':['aux1','aux2']}
                                    }
                    }
        """
        job_history = {}

        _query = "select distinct BS.jobID, BS.bkpLevel, BS.fullCycleNum, DS.auxCopyJobId " \
                 "from JMBkpStats as BS join JMJobDataStats as DS ON BS.jobId = DS.jobId " \
                 "where BS.agedTime = 0 and BS.appId={0}".format(self.subclient_id)
        self.csdb.execute(_query)

        _results = self.csdb.fetch_all_rows()

        for result in _results:
            cycle_num = result[2].strip()
            job_id = result[0].strip()
            level = result[1].strip()
            aux_copy = result[3].strip()
            if cycle_num in job_history.keys():
                if level in job_history[cycle_num].keys():
                    if job_id in job_history[cycle_num][level].keys():
                        aux_jobs = job_history[cycle_num][level][job_id]
                        aux_jobs.append(aux_copy)
                        aux_jobs = list(set(aux_jobs))
                        try:
                            aux_jobs.remove('0')
                        except ValueError:
                            pass
                        job_history[cycle_num][level][job_id] = aux_jobs
                    else:
                        job_history[cycle_num][level][job_id] = [aux_copy]
                else:
                    job_history[cycle_num][level] = {job_id: [aux_copy]}
            else:
                job_history[cycle_num] = {level: {}}
                job_history[cycle_num][level] = {job_id: [aux_copy]}

        return job_history

    def reset_subclient_content(self):

        """"
        This is used to set the contents back to the subclient with
        updated GUID of every VM after In place restore in case of Azure

        Raise:
              Exception:
                If unable to set contents in the subclient

        """

        try:
            list = []
            for _vm in self.vm_list:
                self.hvobj.VMs[_vm].update_vm_info('All', True)
                list1 = {'type': VSAObjects.VM, 'display_name': _vm, 'name': self.hvobj.VMs[_vm].guid}
                list.append(list1)

            self.subclient.content = list


        except Exception as err:
            self.log.exception(
                "Exception while setting contents of subclient" + str(err))
            raise err

    def verify_vmfilter_backedup_vms(self):

        """"
        This is used to verify that only non-filtered VMs are backed up

        Raise:
              Exception:
                If filtered VMs are backed up

        """

        try:
            backedup_vms, _vm_ids = self.subclient._get_vm_ids_and_names_dict_from_browse()
            self.log.info(
                "Check if only following VMs {0} got backed up after applying filter ".format(
                    self.vm_list))
            for vm in backedup_vms:
                if vm not in self.vm_list:
                    self.log.info("The VM = {0} should not be backed up".format(vm))
                    raise Exception("VM filter is backing up non-expected VMs")
            if len(self.vm_list) != len(backedup_vms):
                self.log.info(
                    "The VMs {0} should have backed up, but actual backed up VMs are {1}".format(
                        self.vm_list, backedup_vms))
                raise Exception("VM filter is not backing up the expected VMs")
            self.log.info(
                "The VM filter is working properly, it backed up {0}".format(backedup_vms))
        except Exception as err:
            self.log.exception(
                "An Exception occurred while verifying vm filter backed up VMs" % err)
            raise err

    def create_ini_files(self):
        """
        Create a temp folder and files for storing and verifying changeID

        Raises:
             Exception:
                If unable to create temp folder and files

        """
        try:
            _vserver_path = os.path.dirname(VirtualServerUtils.UTILS_PATH)
            path_dir = self.controller_machine.join_path(_vserver_path, "TestCases", "CBT")
            if not self.controller_machine.check_directory_exists(path_dir):
                self.controller_machine.create_directory(path_dir)

            current_date = self.controller_machine.create_current_timestamp_folder(
                path_dir, "date")
            current_time = self.controller_machine.create_current_timestamp_folder(
                current_date, "time")
            self.controller_machine.create_file(
                os.path.join(current_time, "cbtStats.ini"), "$null")
            self.controller_machine.create_file(os.path.join(
                current_time, "usedChangeIDStatus.ini"), "$null")
        except Exception as err:
            self.log.exception(
                "Exception while creating files" + str(err))
            raise err

    def get_changeid_from_metadata(self, backup_type, backupjobid=None):
        """
        Get changeID generated for given backup job

        Args:
                backup_type    (str):    FULL/INCR/DIFF/SYNTH_FULL
                backupjobid    (int):    job ID of the backup job

        Raises:
             Exception:
                If unable to get chnage ID from metadata

        """
        try:
            _vserver_path = os.path.dirname(VirtualServerUtils.UTILS_PATH)
            path_dir = self.controller_machine.join_path(_vserver_path, "TestCases", "CBT")
            curfolder = self.controller_machine.get_latest_timestamp_file_or_folder(path_dir)
            curfolder = self.controller_machine.get_latest_timestamp_file_or_folder(curfolder)
            if self.auto_vsainstance.vsa_instance._instance_name == 'hyper-v':
                for each_vm in self.vm_list:
                    vmguid = self.hvobj.VMs[each_vm].guid
                    pathtoBrowse = "\\" + vmguid
                    fileToWriteTo = open(curfolder + "\\cbtStats.ini", "a")
                    fileToWriteTo.write("\n[" + backup_type + "_" + each_vm + "]\n")
                    fileToWriteTo.close()
                    if backupjobid is None:
                        backupjobid = self.subclient.find_latest_job(include_active=False)
                    response_json = self.get_metadata(backupjobid._job_id, pathtoBrowse)
                    self.write_changeid_tofile(response_json)
            else:
                pathtoBrowse = "\\"
                fileToWriteTo = open(curfolder + "\\cbtStats.ini", "a")
                fileToWriteTo.write("\n[" + backup_type + "]\n")
                fileToWriteTo.close()
                if backupjobid is None:
                    backupjobid = self.subclient.find_latest_job(include_active=False)
                response_json = self.get_metadata(backupjobid._job_id, pathtoBrowse)
                self.write_changeid_tofile(response_json)
        except Exception as err:
            self.log.exception(
                "Exception while getting changeID from Metadata" + str(err))
            raise err

    def get_metadata(self, backupjobid, Pathtobrowse):
        """
        Get metadata for given backup job using browse request

        Args:
                backupjobid    (int):   job ID of the backup job
                Pathtobrowse   (int):   corressponding path for browsing

        Raises:
             Exception:
                If unable to get metdata from browse request

        """
        try:
            options = {}
            options['path'] = Pathtobrowse
            options['_subclient_id'] = self.subclient_id
            _backup_job = Job(self.auto_commcell.commcell, backupjobid)
            from datetime import timezone, datetime
            temp_time = datetime.strptime(_backup_job.start_time, "%Y-%m-%d %H:%M:%S")
            from_time = temp_time.replace(tzinfo=timezone.utc).astimezone(tz=None)
            from_time = datetime.strftime(from_time, "%Y-%m-%d %H:%M:%S")
            temp_time = datetime.strptime(_backup_job._end_time, "%Y-%m-%d %H:%M:%S")
            end_time = temp_time.replace(tzinfo=timezone.utc).astimezone(tz=None)
            end_time = datetime.strftime(end_time, "%Y-%m-%d %H:%M:%S")
            options['from_time'] = from_time
            options['to_time'] = end_time
            options['media_agent'] = self._browse_ma
            options['show_deleted'] = True
            if self.auto_vsainstance.vsa_instance._instance_name == 'hyper-v':
                options['vm_disk_browse'] = True
            paths, response_json = self.auto_vsa_backupset.backupset.browse(options)
            return response_json
        except Exception as err:
            self.log.exception(
                "Exception while getting metadata using browse request" + str(err))
            raise err

    def write_changeid_tofile(self, response_json):
        """
        Find and write changeID generated for given backupjob to cbtStats.ini file

        Args:
                response_json    (str):     Browse response received for given VM

        Raises:
             Exception:
                If unable to find and write changeID

        """
        try:
            # Open the file to write to
            _vserver_path = os.path.dirname(VirtualServerUtils.UTILS_PATH)
            path_dir = self.controller_machine.join_path(_vserver_path, "TestCases", "CBT")
            curfolder = self.controller_machine.get_latest_timestamp_file_or_folder(path_dir)
            curfolder = self.controller_machine.get_latest_timestamp_file_or_folder(curfolder)
            fileToWriteTo = open(curfolder + "\\cbtStats.ini", "a")
            if self.auto_vsainstance.vsa_instance._instance_name == 'azure resource manager':
                for value in response_json.values():
                    for key1, value1 in value.items():
                        if 'name' in key1:
                            path_name = value1
                        if 'advanced_data' in key1:
                            for key2, value2 in value1.items():
                                if 'browseMetaData' in key2:
                                    for key3, value3 in value2.items():
                                        if 'virtualServerMetaData' in key3:
                                            for key4, value4 in value3.items():
                                                if 'changeId' in key4:
                                                    import xml.etree.ElementTree as ET
                                                    tree = ET.ElementTree(ET.fromstring(value4))
                                                    tree_ID = tree.getroot().attrib[
                                                        'recentFullReferencePoint']
                                                    fileToWriteTo.write(
                                                        value['name'] + ":" + tree_ID + "\n")
                full_job_id = self.subclient.find_latest_job(include_active=False)
                job_info = Job(self.auto_commcell.commcell, full_job_id._job_id)
                total_time = int(job_info._summary['jobEndTime']) - int(
                    job_info._summary['jobStartTime'])
                fileToWriteTo.write("[TIME]" + "\n" + "FULL : " + str(total_time))
                fileToWriteTo.close()
            else:
                for key, value in response_json.items():
                    for key1, value1 in value.items():
                        if 'name' in key1:
                            path_name = value1
                        if 'advanced_data' in key1:
                            for key2, value2 in value1.items():
                                if 'browseMetaData' in key2:
                                    for key3, value3 in value2.items():
                                        if 'virtualServerMetaData' in key3:
                                            for key4, value4 in value3.items():
                                                if 'changeId' in key4:
                                                    fileToWriteTo.write(
                                                        path_name + ":" + value4 + "\n")
                fileToWriteTo.close()
        except Exception as err:
            self.log.exception(
                "Exception while finding and writing changeID to file " + str(err))
            raise err

    def parse_diskcbt_stats(self, cbtstat_folder, backup_type):
        """
        Find and copy cbt_stat file from hyperv to controller machine.
        And write changedID used by given backup to usedChangeIDStatus.ini file

        Args:
                cbtstat_folder    (str):     Folder to which CBT stats are stored on HyperV
                backup_type      (str):      FULL/INCR/DIFF/SYNTH_FULL

        Raises:
             Exception:
                If unable to find and write changeID

        """
        try:
            for each_vm in self.vm_list:
                # get CBTstat file and copy it on local system
                vmguid = self.hvobj.VMs[each_vm].guid
                vmcbtstat_folder = os.path.join(cbtstat_folder, str(vmguid).upper())
                for each_proxy in self.auto_vsainstance.proxy_list:
                    proxy_machine = Machine(each_proxy, self.auto_commcell.commcell)
                    if proxy_machine.check_directory_exists(vmcbtstat_folder):
                        break
                _vserver_path = os.path.dirname(VirtualServerUtils.UTILS_PATH)
                cbt_folder = os.path.join(_vserver_path, "TestCases", "CBT")
                cbt_stat = os.path.join(_vserver_path, "TestCases", "CBTStats")
                destvmcbt_stat = os.path.join(cbt_stat, str(vmguid).upper())
                if proxy_machine.is_local_machine:
                    if not proxy_machine.check_directory_exists(destvmcbt_stat):
                        proxy_machine.remove_directory(destvmcbt_stat)
                    proxy_machine.copy_folder(vmcbtstat_folder, destvmcbt_stat)
                else:
                    _dest_base_path = os.path.splitdrive(vmcbtstat_folder)
                    host_name = self.auto_commcell.get_hostname_for_client(each_proxy)
                    remote_vmcbtstat_folder = "\\\\" + host_name + "\\" + _dest_base_path[
                        0].replace(
                        ":", "$") + _dest_base_path[-1]
                    if self.controller_machine.check_directory_exists(destvmcbt_stat):
                        self.controller_machine.remove_directory(destvmcbt_stat)
                    self.controller_machine.copy_from_network_share(remote_vmcbtstat_folder,
                                                                    cbt_stat,
                                                                    self.auto_vsainstance.user_name,
                                                                    self.auto_vsainstance.password)
                proxy_machine.remove_directory(vmcbtstat_folder)
                self.log.info("Copied CBTstat folder at {0}".format(destvmcbt_stat))

                # Find and write changeID used
                curfolder = self.controller_machine.get_latest_timestamp_file_or_folder(cbt_folder)
                curfolder = self.controller_machine.get_latest_timestamp_file_or_folder(curfolder)
                fileToWriteTo = open(curfolder + "\\usedChangeIDStatus.ini", "a")
                fileToWriteTo.write("\n[" + backup_type + "_" + each_vm + "]\n")
                paths1 = [os.path.join(destvmcbt_stat, fn) for fn in
                          next(os.walk(destvmcbt_stat))[2]]
                paths = []
                for element in paths1:
                    if "avhdx.txt" not in element.lower() and (
                            ".vhd" in element.lower() or ".vhdx" in element.lower()):
                        paths.append(element)
                previousChangedIDUsed = {}
                diskName = ""
                for file in paths:
                    statsFile = open(file, "r")
                    lines = statsFile.readlines()
                    for line in lines:
                        if "ChangeId from previous job : " in line:
                            changeIDLine = line
                            subStrChangeID = changeIDLine.index("[")
                            subStrChangeID2 = changeIDLine.index("]")
                            changeID = changeIDLine[subStrChangeID + 1:subStrChangeID2]
                            indexDash = file.rfind("-")
                            diskName = file[indexDash + 1:len(file) - 4]
                            previousChangedIDUsed[diskName] = changeID
                            fileToWriteTo.write(diskName + ":" + changeID + "\n")
                    statsFile.close()
                    os.remove(file)
                fileToWriteTo.close()
        except Exception as err:
            self.log.exception(
                "Exception while parsing and writing used changeID to file " + str(err))
            raise err

    def parse_diskcbt_stats_azure(self, backup_options):
        """
        Find the changedID from the log files on the proxy and copy it to the controller
        And write changedID used by given backup to usedChangeIDStatus.ini file

        Args:
                backup_type      (str):      FULL/INCR/DIFF/SYNTH_FULL

        Raises:
             Exception:
                If unable to find and write changeID

        """
        try:
            log_dirc = self.auto_commcell.log_dir.rstrip('Automation')
            vmlog_file = os.path.join(log_dirc, 'vsbkp.log')
            job_id = self.subclient.find_latest_job(include_active=False)

            for each_proxy in self.auto_vsainstance.proxy_list:
                proxy_machine = Machine(each_proxy, self.auto_commcell.commcell)
                if proxy_machine.check_directory_exists(vmlog_file):
                    break
            _vserver_path = os.path.dirname(VirtualServerUtils.UTILS_PATH)
            cbt_folder = self.controller_machine.join_path(_vserver_path, "TestCases", "CBT")
            cbt_stat = self.controller_machine.join_path(_vserver_path, "TestCases", "CBTStats")
            destvmcbt_stat = self.controller_machine.join_path(cbt_stat, job_id._job_id)
            if proxy_machine.is_local_machine:
                if not proxy_machine.check_directory_exists(destvmcbt_stat):
                    proxy_machine.create_directory(destvmcbt_stat)
                proxy_machine.copy_folder(vmlog_file, destvmcbt_stat)
            else:
                _dest_base_path = os.path.splitdrive(vmlog_file)
                host_name = self.auto_commcell.get_hostname_for_client(each_proxy)
                remote_vmcbtstat_folder = "\\\\" + host_name + "\\" + _dest_base_path[0].replace(
                    ":", "$") + _dest_base_path[-1]
                if not self.controller_machine.check_directory_exists(destvmcbt_stat):
                    self.controller_machine.create_directory(destvmcbt_stat)
                    self.controller_machine.copy_files_from_network_share(
                        destvmcbt_stat,
                        remote_vmcbtstat_folder,
                        self.auto_vsainstance.user_name,
                        self.auto_vsainstance.password)

            self.log.info("Copied log file at {0}".format(destvmcbt_stat))

            # Find and write changeID used
            curfolder = self.controller_machine.get_latest_timestamp_file_or_folder(cbt_folder)
            curfolder = self.controller_machine.get_latest_timestamp_file_or_folder(curfolder)

            if backup_options.backup_type == 'FULL':
                fileToWriteTo = open(curfolder + "\\cbtStats.ini", "a")
                fileToWriteTo.write("\n[" + backup_options.backup_type + "]\n")

                err_occur = []
                pattern = re.compile(str(job_id._job_id) + " " + 'VSBkpWorker::BackupVMDisk()')
                pattern1 = re.compile(str(job_id._job_id) + " " + 'CAzureInfo::InitCBT()')

                with open(vmlog_file, 'rt') as in_file:
                    for linenum, line in enumerate(in_file):
                        if pattern.search(line) is not None or pattern1.search(line):
                            err_occur.append(line.rstrip('\n'))
                    for line in err_occur:
                        if "No change Id supplied" in line:
                            changeIDLine = line
                            subStrChangeID = changeIDLine.index("[")
                            subStrChangeID2 = changeIDLine.index("]")
                            vm_name = changeIDLine[subStrChangeID + 1:subStrChangeID2]
                        if "Saving change Id " in line:
                            changeIDLine = line
                            subStrChangeID = changeIDLine.index("[")
                            subStrChangeID2 = changeIDLine.index("]")
                            changeID = changeIDLine[subStrChangeID + 1:subStrChangeID2]
                            fileToWriteTo.write(vm_name + ":" + changeID + "\n")
                fileToWriteTo.close()

            else:
                fileToWriteTo = open(curfolder + "\\usedChangeIDStatus.ini", "a")
                fileToWriteTo.write("\n[" + backup_options.backup_type + "]\n")

                err_occur = []
                pattern = re.compile(str(job_id._job_id) + " " + 'VSBkpWorker::BackupVMDisk()')

                with open(vmlog_file, 'rt') as in_file:
                    for linenum, line in enumerate(in_file):
                        if pattern.search(line) != None:
                            err_occur.append(line.rstrip('\n'))
                    for line in err_occur:
                        if "Change Id has been supplied" in line:
                            changeIDLine = line
                            subStrChangeID = changeIDLine.index("[")
                            subStrChangeID2 = changeIDLine.index("]")
                            vm_name = changeIDLine[subStrChangeID + 1:subStrChangeID2]
                        if "Saving change Id " in line:
                            changeIDLine = line
                            subStrChangeID = changeIDLine.index("[")
                            subStrChangeID2 = changeIDLine.index("]")
                            changeID = changeIDLine[subStrChangeID + 1:subStrChangeID2]
                            fileToWriteTo.write(vm_name + ":" + changeID + "\n")
                fileToWriteTo.close()

        except Exception as err:
            self.log.exception(
                "Exception while parsing and writing used changeID to file " + str(err))
            raise err

    def verify_changeid_used(self, backup_type):
        """
        Compare and verify if changeID generated by previous backupjob is used by next
        backup job

        Args:
                backup_type      (str):      FULL/INCR/DIFF/SYNTH_FULL

        Returns:
                True            (boolean): if change ID is matched
                False           (boolean): if change id mismatch

        Raises:
             Exception:
                If unable to verify the changeID

        """
        try:
            import configparser
            _vserver_path = os.path.dirname(VirtualServerUtils.UTILS_PATH)
            cbt_folder = self.controller_machine.join_path(_vserver_path, "TestCases", "CBT")
            currentfolder = self.controller_machine.get_latest_timestamp_file_or_folder(cbt_folder)
            currentfolder = self.controller_machine.get_latest_timestamp_file_or_folder(
                currentfolder)
            changeid_generatedfile = self.controller_machine.join_path(
                currentfolder, "cbtStats.ini")
            changeid_usedfile = self.controller_machine.join_path(
                currentfolder, "usedChangeIDStatus.ini")
            Config_Curr = configparser.ConfigParser(strict=False)
            Config_Curr.read(changeid_usedfile)
            Config_Prev = configparser.ConfigParser(strict=False)
            Config_Prev.read(changeid_generatedfile)
            if backup_type == "DIFFERENTIAL":
                cmp_bk_type = "FULL"
            else:
                cmp_bk_type = "INCREMENTAL"
            if self.auto_vsainstance.vsa_instance._instance_name == 'hyper-v':
                for each_vm in self.vm_list:
                    if not Config_Prev.has_section(cmp_bk_type + "_" + each_vm):
                        cmp_bk_type = "FULL"
                    currentDict = Config_Prev[cmp_bk_type + "_" + each_vm]
                    previousDict = Config_Curr[backup_type + "_" + each_vm]
                    for curKey in currentDict:
                        new_key = re.sub(r'^ide\w+\-\w\-', '', curKey)
                        new_key = re.sub(r'^scsi\w+\-\w\-', '', new_key)
                        if currentDict[curKey].strip() != previousDict[new_key].strip():
                            self.log.info(
                                "Used incorrect change IDs for {0} backup for Disk {1}".format(
                                    backup_type,
                                    curKey))
                            return False
                        else:
                            self.log.info(
                                "Used correct change IDs for {0} backup Disk {1}".format(
                                    backup_type, curKey))

                return True

            else:
                if not Config_Prev.has_section(cmp_bk_type):
                    cmp_bk_type = "FULL"
                currentDict = Config_Prev[cmp_bk_type]
                previousDict = Config_Curr[backup_type]
                bkp_time = Config_Prev["TIME"]
                full_backup_time = bkp_time[cmp_bk_type].strip()
                for curKey in currentDict:
                    new_key = re.sub(r'^ide\w+\-\w\-', '', curKey)
                    new_key = re.sub(r'^scsi\w+\-\w\-', '', new_key)
                    if currentDict[curKey].strip() != previousDict[new_key].strip():
                        self.log.info(
                            "Used incorrect change IDs for {0} backup for VM {1}".format(
                                backup_type,
                                curKey))
                        return False, full_backup_time
                    else:
                        self.log.info(
                            "Used correct change IDs for {0} backup VM {1}".format(backup_type,
                                                                                   curKey))

                return True, full_backup_time

        except Exception as err:
            self.log.exception(
                "Exception while verifying changeID used by job " + str(err))
            raise err

    def check_migrate_vm(self):
        """
        Check if more than one proxy/node is available and migrate to best possible node

        Raise Exception:
                If unable to check/migrate the vm

        """
        try:
            if len(self.auto_vsainstance.proxy_list) > 1:
                self.log.info("More than one node available to migrate the VM")
                for each_vm in self.vm_list:
                    self.hvobj.VMs[each_vm].migrate_vm()
                    self.hvobj.VMs[each_vm].recheck_vm_host()
            else:
                self.log.info("No other host is available for migration")
        except Exception as err:
            self.log.exception(
                "An Exception occurred while checking and if possible migrating VM to other node %s" %
                err)
            raise err

    def find_snap_guest_file_path(self, _browse_result, _drive_letter):
        """
        Get the Drive's Serial number
        Args:
            _browse_result                      (dict):     guest file browse for vm from snap at vm level

            _drive_letter                       (string):   drive for which the serial number is evaluated

        Returns:
            _browse_result['snap_display_name'] (string):   Serial number of the _drive_letter

        Raise Exception:
                If unable to verify the changeID
        """
        try:
            if "name" in _browse_result:
                if _browse_result['name'] == _drive_letter:
                    return _browse_result['snap_display_name']
            for v in _browse_result.values():
                if isinstance(v, dict):
                    item = self.find_snap_guest_file_path(v, _drive_letter)
                    if item is not None:
                        return item
        except Exception as err:
            self.log.exception(
                "Exception while getting guest file path for snap " + str(err))
            raise err

    def disk_count_validation(self, disk_count_before_restore):
        """
        Comparing the total number of disks before and after browse
        Args:
            disk_count_before_restore (int) :       Number of disk in the MA before Browse

        Returns:

        """
        try:
            self.log.info("Calculating the disk-count of the MA after restore")
            disk_count_after_restore = self.ma_machine.get_disk_count()
            self.log.info("Performing Live Browse Un-mount Validation")
            if (int(disk_count_before_restore)) >= (int(disk_count_after_restore)):
                self.log.info("Disk Unmounted Successfully")
            else:
                self.log.info(
                    "Failed to unmount disk, Number of Disk before restore:%s and Number of disk after restore:%s" %
                    (disk_count_before_restore, disk_count_after_restore))
                raise Exception("Failed to Un-mount Disk")
        except Exception as err:
            self.log.exception(
                "Exception while disk count validation " + str(err))
            raise err

    def vmware_live_browse_validation(
            self,
            vm,
            snap_backup_job,
            backup_copy_job,
            metadata):
        """
        Validation for live browse from snap
        Args:
            vm                         (str):  Name of the vm which was browsed and restored

            snap_backup_job             (int):  Snap backup job id

            backup_copy_job             (int):  Backup copy job id

            metadata                    (bool): Metadata collected or not

            v2_client                   (bool): Client is v2 or not


        Raises:
            Exception:
                if it fails to do live browse validation
        """
        try:
            if self.auto_commcell.find_job_transport_mode(backup_copy_job) == 'directsan':
                self.log.info("It was proxyless browse. Sleeping for 11 minutes")
                time.sleep(700)
                if not metadata:
                    self.log.info("Disk unmount validation")
                    self.disk_unmount_validation(vm)
            elif self.subclient.snapshot_engine_name == 'Virtual Server Agent Snap':
                self.log.info("VVOL engine. Sleeping for 11 minutes.")
                time.sleep(700)
                self.log.info("More Validation will be added for vvol")
            else:
                self.log.info("Validating Live Browse for Traditional method")
                _flag = False
                if self.auto_commcell.check_v2_indexing(
                        self.subclient._subClientEntity['clientName']):
                    snap_backup_job = self.auto_commcell.get_snap_job_child(vm, snap_backup_job)
                    if self.auto_commcell.get_snap_mount_status(snap_backup_job) != '59':
                        raise Exception("Snap is not mounted")
                vmname = vm + "_" + str(snap_backup_job) + "_GX_BACKUP"
                if not metadata:
                    _list_of_mounted_ds = self.auto_commcell.live_browse_get_ds_info(
                        snap_backup_job)
                    if self.hvobj.check_vms_exist([vmname]):
                        self.log.info(" Sleeping for 11 minutes")
                        time.sleep(700)
                        if not self.hvobj.check_vms_exist([vmname]):
                            self.log.info(" Sleeping for 1 minutes")
                            time.sleep(100)
                            if not self.hvobj.check_ds_exist(_list_of_mounted_ds):
                                _flag = True
                else:
                    if not self.hvobj.check_vms_exist([vmname]):
                        _flag = True
                if _flag:
                    self.log.info("Live browse cleanup Validation successful")
                else:
                    raise Exception("Live Browse Validation failed during cleanup")
        except Exception as err:
            self.log.exception(
                "Exception at Live browse validation %s", str(err))
            raise err

    def validate_rdm_disks(self, copy_precedence, rdm_type=3):
        """
        Validation for RDM disks

        Args:
            copy_precedence                    (int):   Copy precedence to browse

            rdm_type                           (int):    Options passed at the subclient level
                                                            0 -> No options selected
                                                            1 -> No Raw only independent
                                                            2 -> Raw only, no independent
                                                            3 -> Raw and independent

        Raises:
            Exception:
                if it fails to do rdm disks validation
        """
        try:
            self._disk_filter = True
            for _vm in self.vm_list:
                _temp_vm, _temp_vmid = self.subclient._get_vm_ids_and_names_dict_from_browse()
                _browse_request = self.subclient.disk_level_browse(
                    _temp_vmid[_temp_vm[0]],
                    copy_precedence=copy_precedence)
                disk_names = []
                self.hvobj.VMs[_vm].update_vm_info(force_update=True)
                for value in self.hvobj.VMs[_vm].rdm_details.values():
                    if rdm_type == 0:
                        if 'flat' in value[1].lower() and value[2].lower() == 'persistent':
                            disk_names.append(value[0])
                    elif rdm_type == 1:
                        if 'flat' in value[1].lower():
                            disk_names.append(value[0])
                    elif rdm_type == 2:
                        if value[1].lower() in ('flat', 'virtualmode') and value[
                            2].lower() == 'persistent':
                            disk_names.append(value[0])
                    else:
                        disk_names.append(value[0])
                self.log.info("total number of disks qualified: {}".format(len(disk_names)))
                self.log.info(
                    "total no of disks got via browse: {}".format(len(_browse_request[1].items())))
                if len(disk_names) != len(_browse_request[1].items()):
                    self.log.error("RDM disk validation failure")
                    raise Exception("RDM disk count mismatch")

                vm_folder = self.hvobj.VMs[_vm].get_vm_folder
                complete_disk_names = []
                for value1 in _browse_request[1].values():
                    if re.search(r"[\[\]]+", value1['name']):
                        complete_disk_names.append(value1['name'])
                    else:
                        complete_disk_names.append(vm_folder + value1['name'])
                if set(disk_names) == set(complete_disk_names):
                    self.log.info("RDM disks verified successfully for vm {}".format(_vm))
                    self.hvobj.VMs[_vm].disk_count = len(complete_disk_names)
                else:
                    self.log.exception("RDM disks verification Failed for vm {}".format(_vm))
        except Exception as err:
            self.log.exception(
                "Exception at Validating RDM disks {0}".format(err))
            raise err

    def validate_inputs(self, proxy_os="", ma_os="", vm_os="", update_qa=False, vm_check=False):
        """
        validate the OS configuring for the testcase

        Args:
            proxy_os                (str):  expected os of the proxy

            ma_os                   (str):  expected os of the Media agent

            vm_os                   (str):  expected os of the backup vm

            update_qa               (bool): if set to true, inputs will be validated for QA purpose

            vm_check                (bool): if true, check that there is more than one source vm in subclient content

        Raises:
            Exception:
                if it fails to match os details

        """
        try:
            if vm_check:
                if len(self.vm_list) == 1:
                    raise Exception(
                        "There is only one source vm, Correct configuration for automation is to have "
                        "2 source vms atleast")
            if update_qa:
                if proxy_os != "":
                    self.log.info("Validating proxy OS")
                    _proxies = self.subclient.subclient_proxy
                    if _proxies:
                        for _proxy in _proxies:
                            if proxy_os.lower() not in self.auto_commcell.get_client_os_type(
                                    _proxy).lower():
                                raise Exception("OS of Proxy doesn't match")
                    else:
                        if proxy_os.lower() not in self.auto_commcell.get_client_os_type \
                                    (self.subclient.instance_proxy).lower():
                            raise Exception("OS of Proxy doesn't match")
                else:
                    self.log.info("Validating OS of proxy is not required here")
                if ma_os != "":
                    self.log.info("Validating MA OS")
                    if ma_os.lower() not in self.auto_commcell.get_client_os_type(
                            self.subclient.storage_ma).lower():
                        raise Exception("OS of MA doesn't match")
                else:
                    self.log.info("Validating OS of MA is not required here")
                if vm_os != "":
                    self.log.info("validating backup vm OS")
                    for _vm in self.hvobj.VMs:
                        if self.hvobj.VMs[_vm].guest_os.lower() != vm_os:
                            raise Exception("OS of backup vm doesn't match")
                else:
                    self.log.info("Validating OS of backup vm is not required here")
            else:
                self.log.info("Validating inputs is not required here")

        except Exception as err:
            self.log.exception(
                "Exception at Validating OS   %s", str(err))
            raise err

    def assign_browse_ma(self, proxy_os, ma_os, vm_os):
        """

        Assigning the correct browse ma for the subclient automatically

        Args:
            proxy_os                (str):  os of the proxy

            ma_os                   (str):  os of the Media agent

            vm_os                   (str):  os of the backup vm

        Raises:
            Exception:
                if it fails to assign browse MA

                """
        try:
            self.log.info("Assigning the Browse MA")
            if vm_os.lower() == 'windows' and ma_os.lower() == 'linux':
                if proxy_os.lower() == 'linux':
                    self.browse_ma = self.hvobj.commcell.commserv_name
                else:
                    if bool(self.subclient.subclient_proxy):
                        self.browse_ma = self.subclient.subclient_proxy[0]
                    else:
                        self.browse_ma = self.subclient.instance_proxy

        except Exception as err:
            self.log.exception(
                "Exception at assigning browse MA   %s", str(err))
            raise err

    def validate_tags(self):
        """

        Validates Tags and category are intact after restores

        Raises:
            Exception:
                if it fails to validate tags and category

        """
        try:
            for vm in self.vm_list:
                restore_vm = "del" + vm
                self.hvobj.VMs[vm].get_tags_category()
                self.hvobj.VMs[restore_vm].get_tags_category()
                if self.hvobj.VMs[vm].tags_of_vm != self.hvobj.VMs[restore_vm].tags_of_vm:
                    raise Exception("Tags of source and restored vm doest match")
                self.log.info("Source and destination have same tags")
                if self.hvobj.VMs[vm].category_of_vm != self.hvobj.VMs[restore_vm].category_of_vm:
                    raise Exception("Category of source and restored vm doest match")
                self.log.info("Source and destination have same tags category")
                self.log.info("Deleting VM %s", restore_vm)
                self.hvobj.VMs[restore_vm].delete_vm()
        except Exception as err:
            self.log.exception(
                "Exception at Tags validation %s", str(err))
            raise err

    def validate_vcenter(self, version):
        """
        Validates VCenter version is correct for the test

        Args:
            version                     (int):  Version of the VCenter expected

        Raises:
            Exception:
                if it fails to validate VCenter version

        """
        try:
            _proxies = self.subclient.subclient_proxy
            if _proxies:
                _proxy = _proxies[0]
            else:
                _proxy = self.subclient.instance_proxy
            _proxy_machine = Machine(_proxy, self.auto_commcell.commcell)
            if version < 6.5:
                self.log.info("The VCenter version should be either less than 6.5 or registery keys"
                              "should be present")
                if self.hvobj.vcenter_version >= 6.5:
                    if not _proxy_machine.get_registry_value('VirtualServer',
                                                             'bUsePowerCLiForTags') == '1':
                        raise Exception("This testcase requires vcenter 6.0 or the registry key")
            else:
                self.log.info("The VCenter version should be more than 6.5 and no registey keys"
                              "should be present")
                if _proxy_machine.get_registry_value('VirtualServer',
                                                     'bUsePowerCLiForTags') == '1' or \
                        self.hvobj.vcenter_version < 6.5:
                    raise Exception(
                        "This testcase requires VCenter above 6.5 and no the registry key")
            self.log.info("VCenter and registry key verified")
        except Exception as err:
            self.log.exception(
                "Exception at comparing VCenter version {}".format(err))
            raise err

    def validate_content(self, content):
        """
        Validates the content if of correct type

        Args:
            content                 (str):  Type of content required for the testcase

        Raises:
            Exception:
                if it fails to validate subclient content

        """
        try:
            for ctype in self.vm_content:
                if ctype['type'] != content:
                    raise Exception("Expected type of content is: %s", content)
            self.log.info("Content matches the requirement of the testcase")
        except Exception as err:
            self.log.exception(
                "Exception at comparing subclient content %s", str(err))
            raise err

    def disk_unmount_validation(self, vm):
        """
               Checking if the disks are unmounted
        Args:
            vm           (str) :       Name of the vm which needs to be validated

        Raises:
            Exception:
                if it fails to validate disks unmount

               """
        try:
            self.log.info("Calculating the disk-count of the MA after restore")
            _names_of_disks = self.ma_machine.get_mounted_disks()
            if self.hvobj.VMs[vm].guid in _names_of_disks:
                _list_of_disks = _names_of_disks.split('\r\n')
                self.log.info("mounted disks are: {}".format(_list_of_disks))
                self.log.exception("Disks of the vm {} are still mounted".format(vm))
            else:
                self.log.info("Disk unmount validation completed successfully")
        except Exception as err:
            self.log.exception(
                "Exception while disk unmount validation " + str(err))
            raise err

    def configure_live_sync(self, live_sync_options, pattern_dict=None):
        """
        To configure Live Sync on the subclient

        Args:
            live_sync_options   (obj)   -- Object of LiveSyncOptions class in OptionsHelper module

            pattern_dict        (dict)  -- Dictionary to generate the live sync schedule

                Sample:

                    for after_job_completes :
                    {
                        "freq_type": 'after_job_completes',
                        "active_start_date": date_in_%m/%d/%y (str),
                        "active_start_time": time_in_%H/%S (str),
                        "repeat_days": days_to_repeat (int)
                    }

                    for daily:
                    {
                         "freq_type": 'daily',
                         "active_start_time": time_in_%H/%S (str),
                         "repeat_days": days_to_repeat (int)
                    }

                    for weekly:
                    {
                         "freq_type": 'weekly',
                         "active_start_time": time_in_%H/%S (str),
                         "repeat_weeks": weeks_to_repeat (int)
                         "weekdays": list of weekdays ['Monday','Tuesday']
                    }

                    for monthly:
                    {
                         "freq_type": 'monthly',
                         "active_start_time": time_in_%H/%S (str),
                         "repeat_months": weeks_to_repeat (int)
                         "on_day": Day to run schedule (int)
                    }

                    for yearly:
                    {
                         "active_start_time": time_in_%H/%S (str),
                         "on_month": month to run schedule (str) January, Febuary...
                         "on_day": Day to run schedule (int)
                    }

        Returns:
            object - instance of the Schedule class for this Live sync

        In  config.json  file  give  credentials  "cs_machine_uname" , "cs_machine_password"  for  Schedule

        """

        def hyperv():
            return self.subclient.live_sync.configure_live_sync(
                schedule_name=live_sync_options.schedule_name,
                destination_client=live_sync_options.destination_client,
                proxy_client=live_sync_options.proxy_client,
                copy_precedence=live_sync_options.copy_precedence,
                destination_path=live_sync_options.destination_path,
                destination_network=live_sync_options.network,
                power_on=live_sync_options.power_on,
                overwrite=live_sync_options.unconditional_overwrite,
                distribute_vm_workload=live_sync_options.distribute_vm_workload,
                pattern_dict=pattern_dict
            )

        def azure():
            return self.subclient.live_sync.configure_live_sync(
                schedule_name=live_sync_options.schedule_name,
                destination_client=live_sync_options.destination_client,
                proxy_client=live_sync_options.proxy_client,
                copy_precedence=live_sync_options.copy_precedence,
                power_on=live_sync_options.power_on,
                resource_group=live_sync_options.Resource_Group,
                storage_account=live_sync_options.Storage_account,
                networkdisplayname=live_sync_options.network,
                region=live_sync_options.region,
                unconditional_overwrite=live_sync_options.unconditional_overwrite,
                restoreasmanagedvm=live_sync_options.restoreAsManagedVM,
                createpublicip=live_sync_options.createPublicIP,
                instancesize=live_sync_options.instanceSize,
                networkrsg=live_sync_options.networkrsg,
                destsubid=live_sync_options.subscripid,
                pattern_dict=pattern_dict
            )

        def vmware():
            return self.subclient.live_sync.configure_live_sync(
                schedule_name=live_sync_options.schedule_name,
                destination_client=live_sync_options.destination_client,
                proxy_client=live_sync_options.proxy_client,
                copy_precedence=live_sync_options.copy_precedence,
                destination_network=live_sync_options.network_info,
                power_on=live_sync_options.power_on,
                overwrite=live_sync_options.unconditional_overwrite,
                distribute_vm_workload=live_sync_options.distribute_vm_workload,
                datastore=live_sync_options.datastore_info,
                pattern_dict=pattern_dict
            )

        def amazon():
            return self.subclient.live_sync.configure_live_sync(
                schedule_name=live_sync_options.schedule_name,
                destination_client=live_sync_options.destination_client,
                proxy_client=live_sync_options.proxy_client,
                copy_precedence=live_sync_options.copy_precedence,
                power_on=live_sync_options.power_on,
                unconditional_overwrite=live_sync_options.unconditional_overwrite,
                pattern_dict=pattern_dict,
                networkdisplayname=live_sync_options.network,
                region=live_sync_options.region,
                data_center=live_sync_options.data_center,
                network=live_sync_options.network,
                security_groups=live_sync_options.security_groups,
                volume_type=live_sync_options.volume_type
            )

        hv_dict = {"hyper-v": hyperv, "vmware": vmware, "azure resource manager": azure, "amazon": amazon}

        return hv_dict[live_sync_options.auto_subclient.auto_vsainstance.vsa_instance_name]()

    def get_live_sync_destination_subclient(self, schedule_name):
        """

        Get Live Sync Destination Auto Subclient

        Args:
            auto_subclient (AutoVSASubclient): Source AutoVSASubclient for Live Sync

            schedule_name (str):    Schedule name for Live sync

        Returns:
            dest_auto_subclient (AutoVSASubclient): Destination AutoVSASubclient for Live Sync

        """
        self.subclient.live_sync.refresh()
        live_sync_pair = self.subclient.live_sync.get(schedule_name)
        vm_pairs = live_sync_pair.vm_pairs
        vm_pair = live_sync_pair.get(next(iter(vm_pairs)))

        destination_client = self.auto_commcell.commcell.clients.get(vm_pair.destination_client)
        agent = destination_client.agents.get('virtual server')
        instance = agent.instances.get(vm_pair.destination_instance)

        dest_auto_client = AutoVSAVSClient(self.auto_commcell, destination_client)
        dest_auto_vsa_instance = AutoVSAVSInstance(dest_auto_client, agent, instance)

        # Initialize destination VMs if they exist
        destination_vms = [live_sync_pair.get(vm_pair).destination_vm for vm_pair in vm_pairs]

        # Check if VMs exist
        initialize_vms = dest_auto_vsa_instance.hvobj.check_vms_exist(destination_vms)

        if initialize_vms:
            for dest_vm_name in destination_vms:
                dest_auto_vsa_instance.hvobj.VMs = dest_vm_name
        return dest_auto_vsa_instance

    def get_live_sync_destination_vms(self, schedule_name):
        """

        Get Live Sync Destination VMs

        Args:
            schedule_name (str): Schedule name for Live sync

        Returns:
            dest_vm_names (list): Destination VM Names

        """
        self.subclient.live_sync.refresh()
        live_sync_pair = self.subclient.live_sync.get(schedule_name)
        vm_pairs = live_sync_pair.vm_pairs

        dest_vm_names = []
        for vm_pair in vm_pairs:
            dest_vm_names.append(live_sync_pair.get(vm_pair).destination_vm)
        return dest_vm_names

    def get_recent_replication_job(self, schedule_name):
        """

        Returns latest replication job, given schedule name

        Args:
            schedule_name   (str) : Name of schedule

        Returns:
            replication_job (Job) : Job Object for latest replication job

        """

        self.auto_vsaclient.vsa_client.schedules.refresh()
        schedule = self.auto_vsaclient.vsa_client.schedules.get(schedule_name)
        schedule_helper = SchedulerHelper(schedule, self.auto_commcell.commcell)

        # Get latest replication job from schedule helper
        replication_job = schedule_helper.get_jobid_from_taskid()
        return replication_job

    def validate_live_sync(self, live_sync_name, replication_run=True, check_replication_size=True,
                           schedule=None):
        """To validate VSA live sync

        Args:
            live_sync_name  (str)   -- Name of the live sync

            replication_run (bool)  -- Set to True to check if replication job is triggered for the backup, else False
                                        default: True

            check_replication_size (bool) -- Set to False if incremental jo is coverted to full replication

            schedule(object) -- schedule object for replication schedule to be validated

        Raises:
            Exception

                -- If validation fails

        """
        self.subclient.live_sync.refresh()
        live_sync_pair = self.subclient.live_sync.get(live_sync_name)
        vm_pairs = live_sync_pair.vm_pairs
        vm_pair = live_sync_pair.get(next(iter(vm_pairs)))

        destination_client = self.auto_commcell.commcell.clients.get(vm_pair.destination_client)
        dest_auto_client = AutoVSAVSClient(self.auto_commcell, destination_client)

        agent = destination_client.agents.get('virtual server')
        instance = agent.instances.get(vm_pair.destination_instance)

        dest_auto_vsa_instance = AutoVSAVSInstance(dest_auto_client, agent, instance)

        for vm_pair in vm_pairs:
            self.log.info('validating VM pair: "%s"', vm_pair)

            source_vm = self.hvobj.VMs[vm_pair]
            time.sleep(60)
            source_vm.update_vm_info('All', os_info=True, force_update=True)

            dest_vm_name = live_sync_pair.get(vm_pair).destination_vm
            dest_auto_vsa_instance.hvobj.VMs = dest_vm_name
            dest_vm = dest_auto_vsa_instance.hvobj.VMs[dest_vm_name]

            dest_vm.power_on()
            self.log.info('Successfully powered on VM: "%s"', dest_vm_name)

            # Wait for IP to be generated
            wait = 10

            while wait:
                self.log.info('Waiting for 60 seconds for the IP to be generated')
                time.sleep(60)
                try:
                    dest_vm.update_vm_info('All', os_info=True, force_update=True)
                except Exception:
                    pass

                if dest_vm.ip and VirtualServerUtils.validate_ip(dest_vm.ip):
                    break
                wait -= 1
            else:
                self.log.error('Valid IP not generated within 10 minutes')
                raise Exception(f'Valid IP for VM: {dest_vm_name} not generated within 5 minutes')
            self.log.info('IP is generated')

            _source = self.VmValidation(source_vm)
            _dest = self.VmValidation(dest_vm)
            self.log.info(
                "Source validation object {0} and destination validation object is {1}".format(
                    vars(_source), vars(_dest)))

            time.sleep(120)
            vm_pair_obj = live_sync_pair.get(vm_pair)
            self.log.info("Vm pair object is {0}".format(vars(vm_pair_obj)))
            self.log.info("Replicatio job is {0}".format(vm_pair_obj.latest_replication_job))
            if isinstance(vm_pair_obj.latest_replication_job, int):
                replication_job = self.auto_commcell.commcell.job_controller.get(
                    vm_pair_obj.latest_replication_job)
            else:
                replication_job = self.get_recent_replication_job(schedule_name=live_sync_name)

            if replication_run:
                # To validate if replication job completed successfully
                assert str(vm_pair_obj.last_synced_backup_job) == str(self.backup_job.job_id), \
                    f"Replication job failed to sync latest backup job {self.backup_job.job_id}"
                self.log.info('Backup job sync successfull')

                backup_job = self.auto_commcell.commcell.job_controller.get(self.backup_job.job_id)
                if backup_job.backup_level == 'Incremental' and check_replication_size:
                    assert replication_job.size_of_application < (
                            backup_job.size_of_application + 1048576), \
                        "Replication job has replicated more data than expected for Incremental backup"
                    self.log.info('Data replicated for incremental job validation successful')
            else:
                # To validate if replication job never started
                assert str(vm_pair_obj.last_synced_backup_job) != str(self.backup_job.job_id), \
                    f"Replication Job started for Synthetic full, failing case"
                self.log.info('Replication run not started for Synthetic, validation successful')

            # To validate sync status
            assert vm_pair_obj.status == 'IN_SYNC', \
                f'VM pair : "{vm_pair}" \n Status: "{vm_pair_obj.status} \n Validation failed"'
            self.log.info('Sync status validation successful')

            # To validate Configuration
            assert _source == _dest, "error while configuration validation"
            self.log.info("config validation is successful")

            _livesyncsource = self.LiveSyncVmValidation(source_vm, schedule, replication_job)
            _livesyncdest = self.LiveSyncVmValidation(dest_vm, schedule, replication_job)

            self.log.info(
                "Source validation object {0} and destination validation object is {1}".format(
                    vars(_livesyncsource), vars(_livesyncdest)))
            # hypervisor specific validation
            assert _livesyncsource == _livesyncdest, "error while validation  "

            # To validate test data between source and destination
            if replication_run:
                for drive in dest_vm.drive_list.values():
                    dest_path = dest_vm.machine.join_path(drive, self.backup_folder_name,
                                                          "TestData", self.timestamp)
                    self.fs_testdata_validation(dest_vm.machine, dest_path)
                self.log.info('Testdata validation successful')

            dest_vm.power_off()
            self.log.info('Successfully powered off VM: "%s"', dest_vm_name)

            self.log.info('Validation successful for VM pair: "%s"', vm_pair)

    def cleanup_live_sync(self, live_sync_name):
        """To clean up live sync operations

        Args:
             live_sync_name  (str)   -- Name of the live sync

        Raises:
            Exception

                -- If cleanup operation fails

        """
        live_sync_pair = self.subclient.live_sync.get(live_sync_name)
        vm_pairs = live_sync_pair.vm_pairs
        vm_pair = live_sync_pair.get(next(iter(vm_pairs)))

        destination_client = self.auto_commcell.commcell.clients.get(vm_pair.destination_client)

        dest_auto_client = AutoVSAVSClient(self.auto_commcell, destination_client)

        agent = destination_client.agents.get('virtual server')
        instance = agent.instances.get(vm_pair.destination_instance)

        dest_auto_vsa_instance = AutoVSAVSInstance(dest_auto_client, agent, instance)

        for vm_pair in vm_pairs:
            dest_vm_name = live_sync_pair.get(vm_pair).destination_vm
            dest_auto_vsa_instance.hvobj.VMs = dest_vm_name
            dest_vm = dest_auto_vsa_instance.hvobj.VMs[dest_vm_name]

            # To delete the replicated VM
            output = dest_vm.delete_vm(dest_vm_name)
            if output:
                self.log.info('Successfully deleted the replicated VM : "%s"', dest_vm_name)
            else:
                raise Exception(f'Failed to delete the VM {dest_vm_name} please check the logs')

        # To delete the created live sync configuration
        self.subclient._client_object.schedules.delete(live_sync_name)
        self.log.info('Successfully deleted the Live sync configuration schedule %s',
                      live_sync_name)

        self.log.info('Live sync cleanup operation is successful')

    def verify_cbt_backup(self, backup_type, backup_method='Streaming'):
        """
        Verifies CBT validation
        Args:
            backup_type                 (string):   Backup type of the job
            backup_method               (string):   Backup Method of the job

        Raises:
                Exception:
                    if it fails to get cbt details of the backup job

        """
        try:
            if backup_method == 'SNAP':
                backup_job = self.backupcopy_job
            else:
                backup_job = self.backup_job
            for cbt_status in backup_job.details['jobDetail']['clientStatusInfo']['vmStatus']:
                if backup_type in ['FULL']:
                    if cbt_status['CBTStatus'] in ['Enabled', 'Started', 'Not Supported']:
                        self.log.info("CBT is {0} for {1} backup job for vm {2}".
                                      format(cbt_status['CBTStatus'], backup_type,
                                             cbt_status['vmName']))
                        continue
                elif backup_type in ['INCREMENTAL', 'DIFFERENTIAL']:
                    if cbt_status['CBTStatus'] in ['Used', 'Not Supported']:
                        self.log.info(" CBT is Used for {0} backup job for vm {1}".
                                      format(backup_type, cbt_status['vmName']))
                        continue
                self.log.error("CBT is not used/enabled for {0} backup job for vm {1}".
                               format(backup_type, cbt_status['vmName']))
                raise Exception("CBT is not used/enabled")
        except Exception as err:
            self.log.exception(
                "Exception at Validating CBT backup {0}".format(err))
            raise err

    def validate_special_vm(self):
        """
        Validates the configuration of the special vm

        Returns:

            special_vm_drive               (dict):     Dict of VM and drive of big data

        Raises:
                Exception:
                    if it fails to fetch all details of the vm

        """
        try:
            special_vm_drive = {}
            for vm in self.vm_list:
                self.hvobj.VMs[vm].update_vm_info(prop='All', force_update=True)
                if self.hvobj.VMs[vm].guest_os.lower() != 'windows':
                    VirtualServerUtils.decorative_log("Checking the File systems of the Drives.")
                    command = "df -Th"
                    output = self.hvobj.VMs[vm].machine.execute_command(command)
                    tmp = ['/dev/mapper', 'xfs', 'ext2', 'ext3', 'ext4', 'btrfs']
                    if all(x in output.output for x in tmp):
                        self.log.info("{0} File systems found in the VM.".format(str(tmp)))
                    else:
                        self.log.exception("All types of File system is not found in the VM")
                        raise Exception
                    VirtualServerUtils.decorative_log(
                        "Checking if the Disk is mapped as UUID in fstab")
                    command = "cat /etc/fstab"
                    output = self.hvobj.VMs[vm].machine.execute_command(command)
                    if output.output.count("UUID") <= 1:
                        self.log.exception("Data Disk is not mapped with UUID in fstab")
                        raise Exception
                    self.log.info("Data Disk mapped with UUID in fstab")
                    VirtualServerUtils.decorative_log(
                        "Checking If the Drives are of 2GB and 1GB free space.")
                    tmp = self.hvobj.VMs[vm].machine.get_storage_details()
                    max_size = 0
                    selected_drive = ''
                    for drive, size in tmp.items():
                        if drive != 'tmpfs' and drive != 'devtmpfs':
                            if isinstance(size, dict):
                                if size['mountpoint'] != '/boot':
                                    if size['total'] > 2048 and size['available'] > 1024:
                                        if size['available'] > max_size:
                                            max_size = size['available']
                                            selected_drive = size['mountpoint']
                                        continue
                                    else:
                                        self.log.exception("Not All drives are more than 2GB "
                                                           "and/or have free space more than 1 GB")
                                        raise Exception
                        self.log.info("All drives are minimum of 2GB in size and 1 GB free space")

                else:
                    VirtualServerUtils.decorative_log("Checking the File systems of the Drives.")
                    command = "get-ciminstance Win32_logicaldisk | where mediatype -match '12' | select filesystem"
                    output = self.hvobj.VMs[vm].machine.execute_command(command)
                    tmp = ['NTFS', 'FAT', 'FAT32', 'ReFS']
                    if all(x in output.output for x in tmp):
                        self.log.info("{0} File systems found in the VM.".format(str(tmp)))
                    else:
                        self.log.exception("All types of File system is not found in the VM")
                        raise Exception

                    VirtualServerUtils.decorative_log("Looking for GPT Disks")
                    command = "gwmi -query 'Select * from Win32_DiskPartition WHERE Index = 0' |" \
                              "Select-Object DiskIndex, @{Name='GPT';Expression={$_.Type.StartsWith('GPT')}}"
                    if output.output:
                        output = self.hvobj.VMs[vm].machine.execute_command(command)
                        count = output.output.count('True')
                        if count > 0:
                            self.log.info("There are {} GPT disks".format(count))
                        else:
                            self.log.exception("There are no GPT disks")
                            raise Exception
                    else:
                        self.log.exception("Exception in fetching GPT disks")
                        raise Exception

                    VirtualServerUtils.decorative_log("Checking if boot disk is GPT")
                    command = 'gwmi -query "Select * from Win32_DiskPartition WHERE Type like \'%GPT%\'' \
                              ' and Bootable = True" | Select-Object DiskIndex'
                    output = self.hvobj.VMs[vm].machine.execute_command(command)
                    if output.output:
                        self.log.info("Boot Disk is GPT")
                    else:
                        self.log.exception("Boot Disk is not GPT or issue in fetching disks")
                        raise Exception

                    VirtualServerUtils.decorative_log("Looking for Dynamic Disks")
                    command = "gwmi -query 'Select * from Win32_DiskPartition WHERE Index = 0' |" \
                              "Select-Object DiskIndex, @{Name='dynamic';" \
                              "Expression={$_.Type.Contains('Logical Disk Manager')}}"
                    output = self.hvobj.VMs[vm].machine.execute_command(command)
                    count = output.output.count('True')
                    if count > 0:
                        self.log.info("There are {} Dynamic disks".format(count))
                    else:
                        self.log.exception("There are no Dynamic disks")
                        raise Exception

                    VirtualServerUtils.decorative_log(" Verifying for all Types of Dynamic Disks")
                    command = "'list volume' | diskpart"
                    output = self.hvobj.VMs[vm].machine.execute_command(command)
                    tmp = ['RAID-5', 'Mirror', 'Stripe', 'Spanned']
                    if all(x in output.output for x in tmp):
                        self.log.info("{0} Dynamic Disks are Present in the VM".format(str(tmp)))
                    else:
                        self.log.exception("All types of Dynamic Disks are not present in the VM")
                        raise Exception

                    VirtualServerUtils.decorative_log("Checking if the VM has uninitialized disk")
                    command = "get-disk | where partitionstyle -eq 'raw' | where OperationalStatus -eq 'Online'"
                    output = self.hvobj.VMs[vm].machine.execute_command(command)
                    if output.output:
                        self.log.info("VM:{} has uninitialized disk".format(vm))
                    else:
                        self.log.exception("There is no uninitialized disk in the vm:{}".format(vm))
                        raise Exception

                    VirtualServerUtils.decorative_log(
                        "Checking If the Drives are of 2GB and 1GB free space.")
                    tmp = self.hvobj.VMs[vm].machine.get_storage_details()
                    max_size = 0
                    selected_drive = ''
                    for drive, size in tmp.items():
                        if isinstance(size, dict):
                            if size['total'] > 2048 and size['available'] > 1024:
                                if size['available'] > max_size:
                                    max_size = size['available']
                                    selected_drive = drive
                                continue
                            else:
                                self.log.exception("Not All drives are more than 2GB "
                                                   "and/or have free space more than 1 GB")
                                raise Exception
                self.log.info("All drives are minimum of 2GB in size and 1 GB free space")

                VirtualServerUtils.decorative_log("Checking and copying big data.")
                self.log.info("drive should have minimum 20 GB free space")
                if selected_drive != '' and max_size >= 20480:
                    self.log.info("Drive to be used on VM {0} for copying big data: {1}:".
                                  format(vm, selected_drive))
                    if self.hvobj.VMs[vm].guest_os.lower() == 'windows':
                        selected_drive = selected_drive + ":"
                        special_vm_drive[vm] = selected_drive
                        del max_size, selected_drive, tmp, command, output, count
                    else:
                        special_vm_drive[vm] = selected_drive
                        del max_size, selected_drive, tmp, command, output
                else:
                    self.log.exception("No Drive to copy data")
                    raise Exception
            return special_vm_drive
        except Exception as err:
            self.log.exception(
                "Exception at checking special details of the VM: {0}".format(err))
            raise err

    def add_registry_key(self, key, vm=None, folder="VirtualServer", key_val=1, key_type='DWord'):
        """
        Add registry to enable the feature

        Args:

            key    (string):   registry key name

            vm     (string): client machine where registry needs to be set

            folder (string): relative path

            key_val (int)  : value of the key

            key_type (string) : data type definition

        Raises:
            exception:
                if failed to set the registry on client machine

        """

        try:
            if vm:
                _proxy = Machine(vm, self.auto_commcell.commcell)
            else:
                _proxy = Machine(socket.gethostbyname_ex(socket.gethostname())[2][0])

            _proxy.create_registry(folder, key, key_val, key_type)

        except Exception as err:
            self.log.exception(
                "Exception while adding the registry : {0}".format(err))
            raise err

    def delete_registry_key(self, key, vm="None", folder="VirtualServer"):
        """
         Remove registry to disable the feature

         Args:

            key    (string):   registry key name

            vm     (string): client machine where registry needs to be removed

            folder (string): relative path

        Raises:
            exception:
                if failed to delete the registry on client machine
        """

        try:
            if vm:
                _proxy = Machine(vm, self.auto_commcell.commcell)
            else:
                _proxy = Machine(socket.gethostbyname_ex(socket.gethostname())[2][0])

            _proxy.remove_registry(folder, key)

        except Exception as err:
            self.log.exception(
                "Exception while deleting the registry: {0}".format(err))
            raise err

    def get_min_space_prune_val(self, ma_obj, job_result_dir):
        """
        Calculates the min disk space value for pruning to happen

        Args:

            ma_obj    (object):   Browse MA object
            job_result_dir    (string):   Job result directory

        Return:
            min_space_prune_value   (float):    Min disk space value for pruning
        """
        storage_details = ma_obj.get_storage_details()
        min_space_prune_value = 0
        for _drive, _size in storage_details.items():
            if isinstance(_size, dict) and _drive == job_result_dir[0]:
                total = _size['total']
                available = _size['available']
                min_space_prune_value = (available / total) * 100
                min_space_prune_value = min_space_prune_value + (min_space_prune_value * 5) / 100
                if int(min_space_prune_value) == 10:
                    min_space_prune_value = 9
        return min_space_prune_value

    def set_registry_key_for_pruning(self, browse_ma):
        """
        Sets the required registry key from pruning to happen

        Args:

            ma    (string):   Media agent name

        Raises:
            exception:
                if failed to add the registry on media agent

        """
        try:
            ma_obj = Machine(commcell_object=self.auto_commcell.commcell, machine_name=browse_ma)
            ma_client = Client(commcell_object=self.auto_commcell.commcell, client_name=browse_ma)

            min_space_prune_value = self.get_min_space_prune_val(ma_obj, ma_client.job_results_directory)

            self.add_registry_key(key="nPseudoMountPrunerMinFreeSpace", vm=browse_ma,
                                  key_val=int(min_space_prune_value))
            self.add_registry_key(key="nExtentPruneInterval", vm=browse_ma, key_val=60)

            ma_client.restart_services()
            time.sleep(120)
            self.log.info("Sleeping for 120 seconds")
            sleep_count = 0
            while not ma_client.is_ready:
                sleep_count = sleep_count + 1
                time.sleep(30)
                self.log.info("Services are not up.Sleeping for 30 seconds")
                if sleep_count > 3:
                    raise Exception("Services are not up and running")

        except Exception as err:
            self.log.exception(
                "Exception while setting the registry key for pruning: {0}".format(err))
            raise err

    def del_registry_key_for_pruning(self, browse_ma):
        """
        Deletes the required registry key added for pruning to happen

        Args:

            ma    (string):   Media agent name

        Raises:
            exception:
                if failed to delete the registry on media agent

        """
        try:
            ma_client = Client(commcell_object=self.auto_commcell.commcell, client_name=browse_ma)
            self.delete_registry_key(key="nPseudoMountPrunerMinFreeSpace", vm=browse_ma)
            self.delete_registry_key(key="nExtentPruneInterval", vm=browse_ma)
            ma_client.restart_services()
            time.sleep(120)
            self.log.info("Sleeping for 120 seconds")
            sleep_count = 0
            while not ma_client.is_ready:
                sleep_count = sleep_count + 1
                time.sleep(30)
                self.log.info("Services are not up.Sleeping for 30 seconds")
                if sleep_count > 3:
                    raise Exception("Services are not up and running")
        except Exception as err:
            self.log.exception(
                "Exception while deleting the registry key added for pruning: {0}".format(err))
            raise err

    def vm_conversion_validation(self, auto_subclient, vm_restore_options, backup_type):

        vm_names_dict = {}
        for each_vm in auto_subclient.vm_list:
            auto_subclient.hvobj.VMs = each_vm
            vm_names_dict[each_vm] = "del" + each_vm

        self.hvobj.update_hosts()
        self.hvobj.VMs.clear()
        self.hvobj.VMs = list(vm_names_dict.values())
        for source_vm, dest_vm in vm_names_dict.items():
            restore_obj = self.hvobj.VMs[dest_vm]
            source_obj = auto_subclient.hvobj.VMs[source_vm]
            source_obj.update_vm_info('All', os_info=True, force_update=True)
            attempt = 0
            while attempt < 5:
                time.sleep(120)
                try:
                    restore_obj.update_vm_info(prop='All', os_info=True, force_update=True)
                    if restore_obj.ip is None or restore_obj.ip == "":
                        self.log.info("Attempt number %d failed. "
                                      "Waiting for 2 minutes for VM to come up" % attempt)
                        raise Exception
                    else:
                        break

                except Exception as ex:
                    attempt = attempt + 1

            # """
            for each_drive in restore_obj.drive_list:
                dest_testdata_path = restore_obj.machine.join_path(
                    restore_obj.drive_list[each_drive],
                    backup_type, "TestData",
                    auto_subclient.timestamp)

                auto_subclient.fs_testdata_validation(restore_obj.machine,
                                                      dest_testdata_path)
            # """

            _source = self.VmValidation(source_obj, vm_restore_options)
            _dest = self.VmValidation(restore_obj, vm_restore_options)
            if _source == _dest:
                self.log.info("config validation is successful")
            else:
                self.log.error("error while configuration validation")
                raise Exception("Error while configuration validation")

    def check_cbt_backup_time(self, backup_type, backup_method='Streaming'):
        """
        Check if the time taken for CBT incremental backup is less than
        full backup
        Args:
            backup_type                 (string):   Backup type of the job
            backup_method               (string):   Backup Method of the job

        Raises:
                Exception:
                    If it fails to get job details of the backup job
                    :param backup_type:
        """
        try:
            if backup_method == 'SNAP':
                backup_job = self.backupcopy_job
            else:
                backup_job = self.backup_job
            if backup_type == "INCREMENTAL":
                __cur_job_info = self.auto_commcell.get_job_start_end_time(backup_job.job_id)
                incremental_time_diff = int(__cur_job_info[1]) - int(__cur_job_info[0])

            else:
                _recent_incr_bkp_job_id = self.get_recent_incr_backup(backup_job.job_id)
                __recent_job_info = self.auto_commcell.get_job_start_end_time(
                    _recent_incr_bkp_job_id)
                incremental_time_diff = int(__recent_job_info[1]) - int(__recent_job_info[0])

            _prev_full_bkp_job_id = self.get_previous_full_backup(backup_job.job_id)
            __prev_job_info = self.auto_commcell.get_job_start_end_time(_prev_full_bkp_job_id)
            full_time_diff = int(__prev_job_info[1]) - int(__prev_job_info[0])

            if full_time_diff > incremental_time_diff:
                self.log.info(
                    "Validation successful. Time taken to complete CBT incremental backup "
                    "is less than that time taken for full backup")
            else:
                self.log.exception(
                    "Validation failed. Time taken to complete CBT incremental backup"
                    "is more")

        except Exception as err:
            self.log.exception(
                "Exception while fetching job details")
            raise err

    def check_cbt_backup_sizes(self, backup_type, backup_method='Streaming'):
        """
        Verify backup size of CBT incremental backup is less than
        full backup
        Args:
            backup_type                 (string):   Backup type of the job
            backup_method               (string):   Backup Method of the job

        Raises:
                Exception:
                    If it fails to get job details of the backup job
        """
        try:
            if backup_method == 'SNAP':
                backup_job = self.backupcopy_job
            else:
                backup_job = self.backup_job
            if backup_type == "INCREMENTAL":
                __cur_job_info = self.auto_commcell.get_job_backup_size(backup_job.job_id)
                incremental_backup_size = __cur_job_info[0]

            else:
                _recent_incr_bkp_job_id = self.get_recent_incr_backup(backup_job.job_id)
                __recent_job_info = self.auto_commcell.get_job_backup_size(_recent_incr_bkp_job_id)
                incremental_backup_size = __recent_job_info[0]

            _prev_full_bkp_job_id = self.get_previous_full_backup(backup_job.job_id)
            __prev_job_info = self.auto_commcell.get_job_backup_size(_prev_full_bkp_job_id)
            full_backup_size = __prev_job_info[0]

            if int(full_backup_size) > int(incremental_backup_size):
                self.log.info("Validation successful. Backup size of CBT incremental backup "
                              "is less than that of full backup")
            else:
                self.log.exception("Validation failed. Backup size of CBT incremental backup"
                                   "is more")
                raise Exception
        except Exception as err:
            self.log.exception(
                "Exception while fetching job details")
            raise err

    def cbt_check_snapshot(self, backup_type, backup_method='Streaming'):
        """
        Checks if previous full backup job's snapshot is deleted after incremental job

        Args:
            backup_type                 (string):   Backup type of the job
            backup_method               (string):   Backup Method of the job

        Raises:
                Exception:
                    If it fails to get job details of the backup job
        """
        try:
            if backup_method == 'SNAP':
                self.log.info("No CBT Snapshot validation for snap and backup copy")
            else:
                backup_job = self.backup_job
                if backup_type == "INCREMENTAL":
                    _prev_full_bkp_job_id = self.get_previous_full_backup(backup_job.job_id)
                    _prev_full_bkp_job_obj = Job(self.auto_commcell.commcell,
                                                 str(_prev_full_bkp_job_id))
                    for _vm in self.vm_list:
                        snap_exists = \
                        self.hvobj.VMs[_vm].check_disk_snapshots_by_jobid(_prev_full_bkp_job_obj)[0]
                        if snap_exists:
                            self.log.error(
                                "Snapshot for previous full job exists after the Incremental job "
                                "for vm {0}".format(_vm))
                            raise Exception(
                                "Snapshot for previous full job exists after the Incremental job")
                        self.log.info(
                            "CBT Incremental Job validation completed successfully for vm : {0}".format(
                                _vm))
                else:
                    self.log.info("Verifying previous Snapshots only for Streaming Incremental Job")
                    raise Exception
        except Exception as err:
            raise err

    def attach_volume_restore(self, attach_volume_restore_options, msg=""):
        """

        perform Attach Volume to existing instance restore for specific Openstack subclient

        Args:
                attach_volume_restore_options    (object):   represent options that need to be set
                                                    while performing attach volume  restore

                msg                     (string):  Log line to be printed

        Exception:
                        if job fails
                        if validation fails

        """
        try:
            VirtualServerUtils.decorative_log(msg)
            for _vm in self.vm_list:
                self.log.info(
                    "Restoring the volume in Datastore: {}".format(
                        attach_volume_restore_options.datastore))

                def openstack():
                    volume_restore_job = self.subclient.attach_disk_restore(
                        _vm, vcenter=attach_volume_restore_options.vcenter,
                        proxy_client=attach_volume_restore_options._dest_client_name,
                        esx=attach_volume_restore_options.esxHost,
                        datastore=attach_volume_restore_options.datastore,
                        copy_precedence=attach_volume_restore_options.copy_precedence,
                        media_agent=attach_volume_restore_options.disk_browse_ma,
                        snap_proxy=attach_volume_restore_options.snap_proxy,
                        destinationVM=attach_volume_restore_options.dest_vm,
                        destinationVMGUID=attach_volume_restore_options.dest_vm_guid,
                        datacenter=attach_volume_restore_options.datacenter,
                        cluster=attach_volume_restore_options.cluster
                    )
                    return volume_restore_job

                hv_dict = {"openstack": openstack}

                volume_restore_job = (hv_dict[attach_volume_restore_options.dest_auto_vsa_instance.
                    vsa_instance_name.lower()])()

                if not volume_restore_job.wait_for_completion():
                    raise Exception(
                        "Failed to run volume level restore job {0} with error:{1}".format(
                            volume_restore_job.job_id, volume_restore_job.delay_reason))
                self.log.info("Attach Disk restore job completed successfully with job id {0}"
                              .format(volume_restore_job.job_id))
                self.log.info("Performing basic 'Attach Volume to Existing Instance Validation'")
                source_volume_details = self.hvobj.VMs[_vm].disk_list
                destination_volume_details = self.hvobj.OpenStackHandler.get_volume_attachments(
                    attach_volume_restore_options.dest_vm_guid)
                self.hvobj.VMs[_vm].attach_volume_validation(attach_volume_restore_options,
                                                             source_volume_details,
                                                             destination_volume_details)
                self.log.info(
                    "Now..... Detaching and deleting the newly attached volume in destination VM")
                self.hvobj.OpenStackHandler.detach_and_delete_volume(
                    attach_volume_restore_options.dest_vm_guid,
                    attach_volume_restore_options.dest_servers,
                    destination_volume_details)
                self.log.info("Detach and Delete completed successfully")
                self.log.info("Verifying cleanup of attached volumes in the proxy machine")
                self.hvobj.VMs[_vm].backup_job = self.backup_job
                self.hvobj.VMs[_vm].verify_if_snapshots_and_volume_attached()
                self.log.info("All types of Validation completed successfully")
        except Exception as err:
            self.log.exception(
                "Attach Volume to existing instance Restore Process failed. please check logs")
            raise Exception(
                "Attach Volume to existing instance Restore Process failed, please check agent logs {0}".format(
                    err))

    def validate_rfc_files(self, jobid, source_list, delete_rfc=False, skip_parent_validation=False,
                           skip_child_validation=False):
        """
        Function to validate Remote file cache with backup and restore

        Args:
            jobid                (str)  --  Job ID of backup  .

            source_list         (list)  --  list of RFC files to be validated .

            delete_rfc           (bool) --  True: Deletes RFC path
                                            Default :False

            skip_parent_validation (bool) -- Skip Validation of Parent Job

            skip_child_validation  (bool) -- Skip Validation of Child Job(s)

        Raises:
            Exception if file not available in media agent RFC paths
        """
        self.indexpath = self.auto_commcell.get_IndexCache_location(self._browse_ma_id)
        self.log.info("Media Agent (Index Server) Cache Path: %s", self.indexpath)

        media_agent_machine = Machine(machine_name=self._browse_ma,
                                      commcell_object=self.auto_commcell.commcell)
        if not skip_parent_validation:
            self.log.info("Begin: Validation of RFC files in IndexCache for Parent Job %s", jobid)
            rfc_path = media_agent_machine.join_path(self.indexpath,
                                                     'RemoteFileCache', '2',
                                                     self.subclient.subclient_guid, str(jobid))
            self.log.info(
                "Media Agent (Index Server) RFC Path for Parent Job {0}: {1}".format(jobid,
                                                                                     rfc_path))
            compare_result = self.compare_rfc_files(self._browse_ma, rfc_path, source_list)
            if compare_result:
                self.log.info("All files validated in RFC")
            else:
                self.log.info("Failed to validate All files in RFC")
            self.log.info("End: Validation of RFC files in IndexCache for Parent Job %s", jobid)

            if skip_child_validation:
                return

            if delete_rfc:
                self.log.info("Deleting RFC Folder for Parent Job {0}: {1}".format(jobid, rfc_path))
                media_agent_machine.remove_directory(rfc_path)
                self.log.info("RFC Folder %s deleted ", rfc_path)

        if skip_child_validation:
            return

        child_guid_list = self.auto_commcell.get_Child_subclient_GUID_list(jobid)

        for _ChildGUID in child_guid_list:
            self.log.info("Begin: Validation of RFC files in IndexCache for Child Job %s",
                          _ChildGUID[1])
            rfc_path = media_agent_machine.join_path(self.indexpath,
                                                     'RemoteFileCache', '2', _ChildGUID[0],
                                                     str(_ChildGUID[1]))
            self.log.info(
                "Media Agent (Index Server) RFC Path for Child Job {0}: {1}".format(_ChildGUID[1],
                                                                                    rfc_path))
            self.log.info("VM GUID : {0}".format(_ChildGUID[2].lower()))
            rfc_item_list = [_ChildGUID[2].lower() + '_LiveBrowsePrefetchExtents.xml.rfczip']
            self.log.info("RFC Item: %s", rfc_item_list)
            compare_result = self.compare_rfc_files(self._browse_ma, rfc_path, rfc_item_list)
            if compare_result:
                self.log.info("All files validated in RFC")
            else:
                self.log.info("Failed to validate files in RFC")
            self.log.info("End: Validation of RFC files in IndexCache for Child Job %s",
                          _ChildGUID[1])

            self.log.info("Begin: Validate RFC Arch file in DB for Child Job %s", _ChildGUID[1])
            rfc_archfile_cnt = self.auto_commcell.get_rfc_archfile_count(_ChildGUID[1])
            if (int(rfc_archfile_cnt)) >= 1:
                self.log.info("RFC Arch file got created with backup job %s ", str(_ChildGUID[1]))
            else:
                self.log.info("RFC Arch file failed to create with backup job %s ",
                              str(_ChildGUID[1]))
                raise Exception("RFC Arch file failed to create with backup job %s ",
                                str(_ChildGUID[1]))
            self.log.info("End: Validate RFC Arch file in DB for Child Job %s", _ChildGUID[1])

            if delete_rfc:
                self.log.info(
                    "Deleting RFC Folder for Child Job {0}: {1}".format(_ChildGUID[1], rfc_path))
                media_agent_machine.remove_directory(rfc_path)
                self.log.info("RFC Folder %s deleted ", rfc_path)

    def compare_rfc_files(self, ma_machine, rfc_path, source_list):
        """
        Function to Compare Remote file cache from the list to RFC location

        Args:
            ma_machine        (str)  --  IndexServer Machine name

            rfc_path          (str)  --    RFC location

            source_list       (list) --    List of file to compare with

        Return:
            (Bool) as result of comparison

        Raises:
            Exception if file not available in media agent RFC paths
        """
        compare_flag = True
        media_agent_machine = Machine(machine_name=ma_machine,
                                      commcell_object=self.auto_commcell.commcell)
        rfc_list = media_agent_machine.get_files_in_path(rfc_path)
        self.log.info(rfc_list)
        rfc_files = []
        for file in rfc_list:
            rfc_list_fol, rfc_list_file = os.path.split(file)
            rfc_files.append(rfc_list_file.lower())
        self.log.info(rfc_files)

        for rfc_item in source_list:
            if rfc_item.lower() in rfc_files:
                self.log.info("File %s Exists  in RFC Index Cache  ", rfc_item)
            else:
                regexp = [x for x in rfc_files if re.search(rfc_item, x)]
                if len(regexp) != 0:
                    self.log.info("File %s Exists in RFC Index Cache  ", rfc_item)
                else:
                    compare_flag = False
                    self.log.info("File %s is not in RFC Index Cache  ", rfc_item)
                    raise Exception("File %s is not in RFC Index Cache  ", rfc_item)

        return compare_flag

    def verify_if_solr_used(self, child_backup_job, vm_guid):
        """
        Verify that SOLR is used during Guest File Level Browse

        Args:
            child_backup_job    (string):     job ID of the child backup job

            vm_guid             (string):     VM Guid of the VM part of the VM Group

        Raises:
            Exception:
                if Live Browse is used for Browse instead of SOLR
        """
        try:
            machine_ = Machine(machine_name=self.auto_commcell.commserv_name,
                               commcell_object=self.auto_commcell.commcell)
            client_ = Client(self.auto_commcell.commcell, self.auto_commcell.commserv_name)
            install_dir = client_.install_directory
            browse_log = install_dir + "\\Log Files\\Browse.log"
            self.log.info("Browse Log: %s", browse_log)
            log_line = machine_.read_file(browse_log, search_term=child_backup_job)
            self.log.info(log_line)
            if log_line.rfind('idxDbEngineType="2"') > 0 and log_line.rfind(vm_guid) > 0:
                self.log.info("SOLR is used for Browse.")
            else:
                raise Exception("Live Browse is used for Browse.")
        except Exception as exp:
            self.log.exception(
                "Exception occurred while verifying if SOLR was used during Browse. %s", str(exp))
            raise exp

    def validate_file_indexing_archive_files(self, child_backup_job_id):
        """
        Validate archive files after running File Indexing by making sure backup job archive file with 64 or 65600 flag
        is committed in SOLR

        Args:
            child_backup_job_id    (string):     child backup job ID

        Raises:
            Exception:
                if there's any exception while validating archive files
        """
        try:
            archive_file_list = self.auto_commcell.get_backup_job_archive_files(child_backup_job_id)
            self.log.info("Got Archive Files")
            for archive_id in archive_file_list:
                solr_url = self.formulate_solr_query(self.subclient.subclient_name,
                                                     self._browse_ma) + archive_id
                self.log.info("SOLR Query URL: %s", solr_url)
                if "multinode" in solr_url:
                    self.log.info("SOLR Query URL: %s", solr_url)
                    connection = requests.get(solr_url)
                    print(connection.content)
                    if bytes(archive_id, encoding="ascii") in connection.content:
                        self.log.info("Archive File ID %s is committed to SOLR", archive_id)
                        return True
                else:
                    self.log.info("Cloud SOLR Query URL: %s", solr_url)
                    connection = urlopen(solr_url)
                    response = json.load(connection)
                    if int(response["response"]["numFound"]) > 0:
                        self.log.info("Archive File ID %s is committed to SOLR", archive_id)
                        return True
                self.log.error(
                    "Archive ID validation failed. Didn't find archive files in SOLR: %s",
                    archive_file_list)
                return False
        except Exception as exp:
            self.log.exception("There was an exception in validating archive file IDs in SOLR.%s",
                               str(exp))

    def get_in_line_file_indexing_job(self):
        """
        Get the File Indexing job ID that immediately runs after a Backup

        Raises:
            Exception:
                if there's any exception when getting the File Indexing Job ID
        """
        try:
            job_controller = self.auto_commcell.commcell.job_controller
            running_jobs = job_controller.active_jobs(
                client_name=self.auto_vsaclient.vsa_client_name, lookup_time=1,
                job_type="Data Analytics", operation="File Indexing")
            self.log.info(running_jobs)
            subclientId = self.subclient.subclient_id
            self.log.info("Subclient ID: %s" % subclientId)
            for job_id in list(running_jobs.keys()):
                self.log.info("job id: %s" % job_id)
                if str(running_jobs[job_id]['subclient_id']) == str(subclientId) and \
                        running_jobs[job_id]["operation"] == "File Indexing":
                    file_indexing_job_id = job_id
                    return file_indexing_job_id
        except Exception as exp:
            self.log.exception("Exception when getting in-line File Indexing Job ID. %s", str(exp))
            raise exp

    def get_file_indexing_job_details(self, file_indexing_job_id, synthfull=False):
        """
        Get the child Backup job for VMs in the File Indexing job as well as their GUIDs

        Args:
            file_indexing_job_id    (string):     job ID of the File Indexing job

            synthfull (bool) : when true then file indexing job_id will be same as backup job id.

        Returns:
            a Dictionary of VM GUID as key and a list of child backup job & proxy name as the value

        Raises:
            Exception:
                if there's any exception while getting File Indexing job details
        """
        try:
            job_controller = self.auto_commcell.commcell.job_controller
            file_indexing_job_details = job_controller.get(file_indexing_job_id)
            file_indexing_job_details_dict = file_indexing_job_details._get_job_details()
            self.log.info("Here is the File Indexing job details:")
            self.log.info(file_indexing_job_details_dict)
            child_backup_job_info = file_indexing_job_details_dict['jobDetail']['clientStatusInfo'][
                'vmStatus']
            vm_guid_child_job_dict = {}
            for vm in child_backup_job_info:
                child_backup_job_id = vm['jobID']
                proxy_utilized = vm['Agent']
                self.log.info("Got the Child Backup Job ID: %s", child_backup_job_id)
                vm_guid = vm['GUID']
                self.log.info("Got the VM GUID: %s", vm_guid)
                if synthfull:
                    vm_guid_child_job_dict[str(vm_guid)] = [str(file_indexing_job_id),
                                                            proxy_utilized]
                else:
                    vm_guid_child_job_dict[str(vm_guid)] = [str(child_backup_job_id),
                                                            proxy_utilized]
            return vm_guid_child_job_dict
        except Exception as exp:
            self.log.exception("Exception when getting File Indexing Job details. %s", str(exp))
            raise exp

    def post_backup_validation(self, validate_cbt=False, skip_snapshot_validation=False):
        """
        Performs Post Backup Validation

        Args:

        validate_cbt            (bool) : whether to validate cbt or not

        skip_snapshot_validation (bool) : whether to perform snapshot validation or not

        Raises:
            Exception:
            If Validation fails
        """
        try:
            self.log.info("Performing Post Backup Validation")
            backup_type = self.backup_option.backup_type
            if validate_cbt:
                self.verify_cbt_backup(backup_type, self.backup_option.backup_method)
                if backup_type == 'INCREMENTAL':
                    self.log.info("Incremental Backup.. Proceeding with CBT validations")
                    self.check_cbt_backup_time(backup_type, self.backup_option.backup_method)
                    self.check_cbt_backup_sizes(backup_type, self.backup_option.backup_method)
                    if not skip_snapshot_validation:
                        self.cbt_check_snapshot(backup_type, self.backup_option.backup_method)
            if not self.backup_option.backup_method == "SNAP":
                for each_vm in self.vm_list:
                    vm_obj = self.hvobj.VMs[each_vm]
                    backup_validation = self.BackupValidation(vm_obj, self.backup_option)
                    backup_validation.validate()

        except Exception as err:
            self.log.error(str(err))
            raise Exception("Error in post backup validation")

    def validate_disk_filtering(self, copy_precedence=0, **kwargs):
        """
        Validation for Disk Filtering

        Args:
            copy_precedence                    (int):   Copy precedence to browse
            **kwargs                           (dict):   Optional arguments

        Raises:
            Exception:
                if it fails to do disk filter validation
        """
        try:
            snapshot_dict = {}
            for _vm in self.vm_list:
                self.hvobj.VMs[_vm].update_vm_info('All', os_info=True, force_update=True)
                if kwargs.get("validate_snapshot"):
                    snapshot_dict[_vm] = \
                    self.hvobj.VMs[_vm].check_disk_snapshots_by_jobid(self.backup_job, True)[1]
                    self.log.info("Snapshot dict before filtering is {0}".format(str(snapshot_dict[_vm])))
            self.prepare_disk_filter_list()
            for _vm in self.vm_list:
                self.log.info("Browsing for VM {0}".format(_vm))
                browsed_list = []
                flag = 0
                _temp_vm, _temp_vmid = self.subclient._get_vm_ids_and_names_dict_from_browse()
                _browse_request = self.subclient.disk_level_browse(
                    _temp_vmid[_vm],
                    copy_precedence=copy_precedence)
                self.log.info("Disk Browse Response : {0}".format(str(_browse_request[1])))
                self.log.info(
                    "No. of disks after filter in source VM: {} and no of disks in browse response: {}".format(
                        str(len(self.hvobj.VMs[_vm].disk_dict)),
                        str(len(_browse_request[1].items()))))
                if len(self.hvobj.VMs[_vm].disk_dict) != len(_browse_request[1].items()):
                    self.log.error("Disk Filter validation failure. Disk count mismatch")
                    self.log.error(
                        "Disks in Browse are : {0}".format(str(_browse_request[1].values())))
                    raise Exception("Disk count mismatch")
                self.log.info("Comparing the disk names")
                if self.hvobj.instance_type in ('vmware', 'xen'):
                    if self.hvobj.instance_type == 'vmware':
                        vm_folder = self.hvobj.VMs[_vm].get_vm_folder
                        for disk in _browse_request[1].values():
                            if re.search(r"[\[\]]+", disk['name']):
                                browsed_list.append(disk['name'])
                            else:
                                browsed_list.append(vm_folder + disk['name'])
                        self.hvobj.VMs[_vm].disk_validation = False
                    else:
                        for value in _browse_request[1].values():
                            browsed_list.append(value['snap_display_name'].lower())
                    if set(browsed_list).issubset(set(self.hvobj.VMs[_vm].filtered_disks)):
                        self.log.exception("Error in validating disk filter for vm {}".format(_vm))
                        raise Exception("Error in validating disk filter")
                else:
                    for value in _browse_request[1].values():
                        browsed_list.append(value['name'].lower())
                    for disk in self.hvobj.VMs[_vm].filtered_disks:
                        if disk.lower() in browsed_list:
                            flag = -1
                            break
                        if snapshot_dict:
                            self.log.info("Snapshot Validation")
                            if snapshot_dict[_vm][disk]:
                                flag = -1
                                break
                if flag != 0:
                    self.log.error("Filtered disk found in browse response")
                    raise Exception("Error in validating disk filter")
                self.log.info("Disk filter verified successfully")

        except Exception as err:
            self.log.exception(
                "Exception at Validating disk filter {0}".format(err))
            raise err

    def get_final_ma(self):
        """
        Get Media agents names associated with storage policy and which are not index server

        Returns:
            MA (list)  --  List of MA's
            If Indexsever, removes indexsever MA from the list
        Raise Exception:
                if unable to get MA names
        """
        try:
            finalma = self.subclient.get_ma_associated_storagepolicy()
            indexservername = self.get_index_name()
            if indexservername in finalma:
                finalma.remove(indexservername)
                self.log.info("successfully got finalma " + str(finalma))
            return finalma
        except Exception as err:
            self.log.exception(
                "Failed to Get Index name Exception:" + str(err))
            raise err

    # Chirag's code
    def create_azure_vm(self, subnet_resource, subnet_network, subnet_name):
        """
        Creates a new VM in the current resource group on Azure

        Args:
            subnet_resource (basestring): Name of subnet resource group
            subnet_network  (basestring): Name of subnet network
            subnet_name     (basestring): Name of subnet

        Raises :
            Exception on failure
        """
        try:
            for _vm in self.vm_list:
                self.hvobj.VMs[_vm].create_vm(subnet_resource, subnet_network, subnet_name)

        except Exception as err:
            self.log.exception(
                "Exception while creating a new VM")
            raise err

    def validate_network_security_group(self, vm_restore_options):
        """
        Validates if the restored VM is in the same NSG as the original VM

        Args:
                vm_restore_options			  (str):  options that need to be set while
                                                        performing vm restore
        Raises:
         Exception:
            If validation fails
        """
        try:
            for each_vm in self.vm_list:
                nsg = self.hvobj.VMs[each_vm].nsg
                if vm_restore_options.dest_client_hypervisor.VMs.get(each_vm, None):
                    restore_vm_name = "del" + each_vm
                    rnsg = self.hvobj.VMs[restore_vm_name].nsg
                    if nsg != rnsg:
                        self.log.error("NSG validation failed. NSG of restore VM is "
                                       "not same as that of original VM")
                        raise Exception
            self.log.info("NSG validation successful on restored VM.")

        except Exception as err:
            self.log.exception(
                "Exception while validating NSG on restored VM")
            raise err

    def parallel_browse_same_vm(self, vm, browse_ma='', thread_count=4):
        """
        does file level browse for a vm in parallel
        Args:
            vm                          (string):   name of the vm to browse

            browse_ma                   (string):   name of the MA

            thread_count                  (int):      no of thread for the browse

        Raises:
            Exception:
                If fails during multiple browse in parallel for a vm

        """

        try:
            drive_list = self.hvobj.VMs[vm].drive_list

            def browse(name):
                random_drive = random.choice(list(drive_list.keys()))
                _temp_vm, _temp_vmid = self.subclient._get_vm_ids_and_names_dict_from_browse()
                random_drive = _temp_vmid[vm] if random_drive == "/" else \
                    _temp_vmid[vm] + "\\" + random_drive
                self.log.info("Browse Thread {} starting for path {}".format(name, random_drive))
                _browse_request = self.subclient.guest_files_browse(random_drive,
                                                                    media_agent=browse_ma)
                if not _browse_request:
                    self.log.exception("Exception in browse in Browse thread {}"
                                       "for path{}".format(name, random_drive))
                    raise
                self.log.info("Browse Thread {}: {}".format(name, _browse_request))
                self.log.info("Browse Successful for Tread {0} for path {1}".
                              format(name, random_drive))

            threads = []
            for index in range(thread_count):
                VirtualServerUtils.decorative_log("Starting threads {} for vm {}"
                                                  .format(index, vm))
                browse_thread = threading.Thread(target=browse, args=(index,))
                threads.append(browse_thread)
                time.sleep(2)
                browse_thread.start()
            for index, thread in enumerate(threads):
                time.sleep(2)
                thread.join()

        except Exception as exp:
            self.log.exception("failed during parallel_browse_same_vm {}".format(exp))
            raise exp

    def parallel_browse_multiple_vm(self, browse_ma=''):
        """
        does file level browse for multiple vm in parallel
        Args:
            browse_ma                   (string):   name of the MA

        Raises:
            Exception:
                If fails during parallel browse of file level browse of multiple vms

        """

        try:
            vm_path_dict = {}
            for _vm in self.vm_list:
                _temp_vm, _temp_vmid = self.subclient._get_vm_ids_and_names_dict_from_browse()
                drive_list = self.hvobj.VMs[_vm].drive_list
                random_drive = random.choice(list(drive_list.keys()))
                random_drive = _temp_vmid[_vm] if random_drive == "/" else \
                    _temp_vmid[_vm] + "\\" + random_drive
                vm_path_dict[_vm] = random_drive

            def browse(name, vm, path):
                self.log.info(
                    "Browse Thread {} starting for vm {} for path {}".format(name, vm, path))
                _browse_request = self.subclient.guest_files_browse(path, media_agent=browse_ma)
                if not _browse_request:
                    self.log.exception("Exception in browse in browse thread {}"
                                       "for vm {} for path{}".format(name, vm, path))
                    raise
                self.log.info("Thread {}: {}".format(name, _browse_request))
                self.log.info("Browse Successful for Browse thread {} for vm {} for path {}".
                              format(name, vm, path))

            threads = []
            index = 0
            for _vm, _path in vm_path_dict.items():
                VirtualServerUtils.decorative_log("Starting threads {0} for vm {1}"
                                                  .format(index, _vm))
                browse_thread = threading.Thread(target=browse, args=(index, _vm, _path,))
                index += 1
                threads.append(browse_thread)
                time.sleep(2)
                browse_thread.start()
            for index, thread in enumerate(threads):
                time.sleep(2)
                thread.join()

        except Exception as exp:
            self.log.exception("failed during parallel_browse_multiple_vm {}".format(exp))
            raise exp

    def get_index_name(self):
        """
        Get Index server name

        return:
                Index server name     (str)   - Index server name

        Exception:
                if failed to get Index server ID
        """
        try:
            query = "select currentIdxServer from App_IndexDBInfo where backupsetid="\
                        "(select ChildBackupSetId from app_vmbackupset where VMClientId ="\
                        "(select id from app_client where name = '"+self.subclient.content[0]['display_name']+"'))"

            self.csdb.execute(query)
            inderserverid = self.csdb.fetch_all_rows()
            inderserverid = inderserverid.pop(0)
            inderserverid = inderserverid.pop(0)
            query = "select name from app_client where id = "+inderserverid+""
            self.csdb.execute(query)
            inderservername = self.csdb.fetch_all_rows()
            inderservername = inderservername.pop(0)
            inderservername = inderservername.pop(0)
            self.log.info("successfully got indexservername " + str(inderservername))
            return inderservername
        except Exception as err:
            self.log.exception(
                "Failed to Get Index name Exception:" + str(err))
            raise err

    def convert_drive_to_volume(self, vm):
        """
        Converts drive name to volume name for linux with metadata

        Args:
            vm              (string):   Name of teh vm

        Returns:
            _temo_dict      (dict):     New Dict for volume drives for linux vms with metadata
        """
        _temp_dict = {}
        _vol = 1
        for _drive, _label in self.hvobj.VMs[vm].drive_list.items():
            if _label == "/":
                _temp_dict[_drive] = _drive
            else:
                _temp_dict['Volume-{}'.format(_vol)] = 'Volume-{}'.format(_vol)
                _vol += 1
        return _temp_dict

    def get_vms_from_backup_job(self, job_id):
        """
        Fetches VMs from Backup Job

        Args:
            job_id  (int/str)   -   Backup job id to check VMs involved

        Returns:
            List of VMs

        """
        backedup_vms = []
        job = Job(self.auto_commcell.commcell, job_id)
        for vm_detail in job.details['jobDetail']['clientStatusInfo']['vmStatus']:
            backedup_vms.append(vm_detail['vmName'])
        return backedup_vms

    def check_for_snapshot(self, snapshot_name=None, snapshot_time=None):
        """
        Checks for snapshot

        Args:
            snapshot_name  (basestring)   -   Snapshot name to be checked for managed disk
            snapshot_time  (basestring)   -   Snapshot time to be checked for unmamanged disk
        Raises
            Exception:
                If snapshot does not exists
        """
        try:
            for vm in self.vm_list:
                self.log.info("Looking for manually created auto_snapshot for vm {0} ".format(vm))
                if self.hvobj.VMs[vm].managed_disk:
                    snapshot_dict = self.hvobj.VMs[vm].get_disk_snapshots()
                    for disk, details in snapshot_dict.items():
                        flag = 0
                        self.log.info("Looking for manually created auto_snapshot for disk {0} ".format(disk))
                        for each_snapshot in details:
                            if re.match(snapshot_name, each_snapshot[0].get('name'), re.IGNORECASE):
                                flag = 1
                                break
                else:
                    snapshot_dict = self.hvobj.VMs[vm].get_snapshotsonblobs()
                    for disk, details in snapshot_dict.items():
                        flag = 0
                        self.log.info("Looking for manually created snapshot for disk {0} ".format(disk))
                        for each_snapshot in details:
                            if snapshot_time in each_snapshot:
                                flag = 1
                                break
                if flag == 0:
                    raise Exception("Manually created snapshot doesn't exist for VM {0} ".format(vm))

        except Exception as exp:
            raise Exception("Manually created snapshot doesn't exist for VM {0} ".format(vm))

    def service_operation(self, op_id, entity_id, jobidobj = None, **kwargs):
        """
        Call start/stop service function based on user input

        Args:
            op_id  (int)   -   operation id (op_id) user input passed in test case file
                                to start/stop/restart/suspend services

            entity_id (int)   -   entity_id passed in test case file to determine machine
                                    is coordinator/worker proxy or MA
            jobidobj - job id object for job which needs to be suspend/resume

        Raise Exception:
                if failed to perform service operation

        """
        try:
            self.proxy_list = self.subclient.subclient_proxy
            user = kwargs.get('username', None)
            pswd = kwargs.get('password', None)
            if entity_id == ServiceOperationEntity.Co_ordinator.value:
                entity_name = kwargs.get("co_ordinator_node", self.proxy_list[0])
                if op_id == ServiceIds.Stop.value:
                    self.stop_service(entity_name)
                if op_id == ServiceIds.Start.value:
                    self.start_service(entity_name, user, pswd)
                if op_id == ServiceIds.Suspend.value:   #Create job object with current job
                    job_obj = Job(self.auto_commcell.commcell, self.current_job)
                    if jobidobj == None:
                        job_obj.pause(True).pause(True)
                        self.log.info("Job is suspended")
                        time.sleep(300)
                        job_obj.resume(True)
                        self.log.info("Job is resumed")
                    else:
                        jobidobj.pause(True)
                        self.log.info("Job is Suspended")
                        time.sleep(300)
                        jobidobj.resume(True)
                        self.log.info("Resumed")
                if op_id == ServiceIds.Restart.value:
                    proxy_client = Client(self.auto_commcell.commcell, entity_name)
                    proxy_client.restart_services()
                    time.sleep(60)
                    VirtualServerUtils.decorative_log("All Services were restarted successfully on {}".
                                                      format(entity_name))
            if entity_id == ServiceOperationEntity.Worker.value:
                entity_name = kwargs.get("worker_node", self.proxy_list[1])
                if op_id == ServiceIds.Stop.value:
                    self.stop_service(entity_name)
                if op_id == ServiceIds.Start.value:
                    self.start_service(entity_name, user, pswd)
            if entity_id == ServiceOperationEntity.MA.value:
                if op_id == ServiceIds.Stop.value:
                    entity_name = (
                        self.backup_option
                            .backup_job
                            .details['jobDetail']['generalInfo']['mediaAgent']['mediaAgentName']
                            )
                    self.ma_name = entity_name
                    self.stop_service(entity_name)
                if op_id == ServiceIds.Start.value:
                    self.start_service(self.ma_name, user, pswd)
            if entity_id == ServiceOperationEntity.Other.value:
                entity_name = kwargs.get("client_name")
                if op_id == ServiceIds.Stop.value:
                    self.stop_service(entity_name)
                if op_id == ServiceIds.Start.value:
                    self.start_service(entity_name, user, pswd)

        except Exception as exp:
            self.log.error('Failed to perform service operation: ' + str(exp))
            raise Exception

    def check_job_status_fromDB(self, op_id, entity_id):
        """
        Check job status by running DB query until it is completed for some of the VM's

        Args:
            op_id  (int)   -   operation id (op_id) user input passed in test case file
                                to start/stop/restart/suspend services. This will be
                                passed on to service_operation()
            entity_id (int)   -   entity_id passed in test case file to determine machine
                                    is coordinator/worker proxy or MA. This will be
                                    passed on to service_operation()

        Raise Exception:
                if failed to check job status

         """
        try:
            job_status = '4'
            while True:  # check the status of all vm's until one of them is complete
                if [js for js in job_status if js in ['0', '3']]:
                    self.log.info("Some VM's completed")
                    self.service_operation(op_id, entity_id)
                    break
                query = "select status from JMQinetixUpdateStatus " \
                        "where jobId='" + self.current_job + "'"
                self.csdb.execute(query)
                job_status = self.csdb.fetch_all_rows()
                job_status = [item for sublist in job_status for item in sublist]

        except Exception as exp:
            self.log.error('Failed to check job status: ' + str(exp))
            raise Exception

    def start_service(self, name, username, password):
        """
        Creates object of proxy/MA and then starts services on that machine

         Args:
            name (str)  -   name of the machine proxy/MA on which we have to start services
            username (str)  -   username of the machine
            password (str)  -   password of the machine

        Raise Exception:
                if failed to start services

         """
        try:
            self.log.info("Starting services on {}".format(name))
            _machine = Machine(machine_name=name, commcell_object=None,
                               username=username, password=password)
            _instance = _machine.instance
            command_list = [f'start-Service -Name "GxCVD({_instance})"',
                            f'start-Service -Name "GxClMgrS({_instance})"',
                            f'start-Service -Name "GXMMM({_instance})"',
                            f'start-Service -Name "GxBlr({_instance})"']
            for each_command in command_list:
                _machine.execute_command(each_command)
            time.sleep(30)
            VirtualServerUtils.decorative_log("Services started successfully on {}".format(name))

        except Exception as exp:
            self.log.error('Failed to start services: ' + str(exp))
            raise Exception

    def stop_service(self, name, service_name=None):
        """
          Stop service on the machine

           Args:
              name (str)  -   name of the machine proxy/MA on which we have to stop services
              service_name (str) - name of the service to stop service

          Raise Exception:
                  if failed to stop the service

           """
        try:
            _machine = Client(self.auto_commcell.commcell, client_name=name)
            instance = _machine.instance
            if service_name:
                _machine.stop_service(service_name)
                VirtualServerUtils.decorative_log("{0} service stopped successfully on {1}".
                                                  format(service_name, name))
            else:
                _machine.stop_service(f"GxCVD({instance})")
                VirtualServerUtils.decorative_log("CVD service stopped successfully on {}".format(name))

        except Exception as exp:
            self.log.error('Failed to stop service: ' + str(exp))
            raise Exception

    def validate_ip_hostname(self, vm, ip=None, host_name=None):
        """
        Validate hostname and IP of the restored vm
        Args:
            vm                  (str)  -   name of the vm

            ip                  (str)  -   ip to be matched

            host_name           (str)  -   host name to be matched

        Raise Exception:
                if failed to validate IP and/or Host name
        """
        try:
            if ip:
                self.log.info("---___comparing IP of restored vm: {}---___".format(vm))
                self.log.info('IP passed in input: {}'.format(ip))
                self.log.info('IP of the restored vm: {}'.format(self.hvobj.VMs[vm].ip))
                if ip == self.hvobj.VMs[vm].ip:
                    self.log.info('IP given in input and restored vm are same')
                else:
                    raise Exception('IP of the Destination vm doesnt matches')
            if host_name:
                self.log.info("---___comparing Hostname of restored vm: {}---___".format(vm))
                self.log.info('Host name passed in input: {}'.format(host_name))
                _vm_host_name = self.hvobj.VMs[vm].machine.execute_command(
                    'hostname').formatted_output
                self.log.info('Hostname of the restored vm: {}'.format(_vm_host_name))
                if host_name == _vm_host_name:
                    self.log.info('Hostname given in input and restored vm are same')
                else:
                    raise Exception('Hostname of the Destination vm doesnt matches')
        except Exception as exp:
            self.log.error('Failed to Validate IP and/or Hostname ' + str(exp))
            raise Exception
