# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
Does all the Operation for Vmware vm
    VmwareVM:

            _get_vm_host()                  -   Get the host of the VM among the servers list

            _get_vm_info()                  -   Get the particular  information of VM

            _get_disk_list()                -   Gets the disk list opff the VM

            mount_vmdk()                    -   Mount the VMDK and return the drive letter

            un_mount_vmdk()                 -   Unmount the VMDK mounted provided the path

            get_disk_in_controller()       -   get the disk in controller attached

            get_disk_path_from_pattern()   -   get the list of disk from pattern

            power_off()                     -   power off the VM

            power_on()                      -   Power on the VM

            delete_vm()                     -   Delete the VM

            update_vm_info()                -   Updates the VM info

            live_browse_vm_exists()         -   Checks if live vm exists

            live_browse_ds_exists()         -   Checks if live browse Datastrore exists

"""

import time
import os
import re
import collections
from AutomationUtils import machine
from AutomationUtils import logger
from VirtualServer.VSAUtils.VMHelper import HypervisorVM


class VmwareVM(HypervisorVM):
    """
    This is the main file for all  Vmware VM operations
    """

    def __init__(self, hvobj, vm_name):
        """
        Initialization of vmware vm properties

        Args:
            hvobj               (obj):  Hypervisor Object

            vm_name             (str):  Name of the VM
        """
        super(VmwareVM, self).__init__(hvobj, vm_name)
        self.host_machine = machine.Machine()
        self.server_name = hvobj.server_host_name
        self.vm_props_file = "GetVMwareProps.ps1"
        self.prop_dict = {
            "server_name": self.server_name,
            "user": self.host_user_name,
            "pwd": self.host_password,
            "vm_name": self.vm_name,
            "extra_args": "$null"
        }
        self.vm_obj, self.guid, self.ip, self.guest_os, self.host_name, self.power_state, \
        self.no_of_cpu, self.memory, self.esx_host, self.tools, self.template, \
        self.network_name = (None,) * 12
        self.nics, self.datastores = ([],) * 2
        self.nic_count, self.disk_count = (0,) * 2
        self.disk_dict = {}
        self.disk_validation = True
        self._basic_props_initialized = False
        self.update_vm_info()

    class VmValidation(object):
        def __init__(self, vmobj, vm_restore_options):
            self.vm = vmobj
            self.vm_restore_options = vm_restore_options
            self.log = logger.get_log()

        def __eq__(self, other):
            """compares the source vm and restored vm"""

            if self.vm.vm.nic_count == other.vm.vm.nic_count:
                self.log.info("Network count matched")
            else:
                self.log.error("Network count failed")
                return False
            self.vm.vm.update_vm_info(force_update=True)
            other.vm.vm.update_vm_info(force_update=True)
            if self.vm.vm.disk_validation:
                self.log.debug(
                    'Source vm:{} , disk details: {}'.format(self.vm.vm.vm_name, self.vm.vm.get_disk_provision))
                self.log.debug(
                    'Restored vm:{} , disk details: {}'.format(other.vm.vm.vm_name, self.vm.vm.get_disk_provision))
                if self.vm_restore_options.disk_option == 'Original':
                    if collections.Counter(
                            list(self.vm.vm.get_disk_provision.values())) != collections.Counter(
                            list(other.vm.vm.get_disk_provision.values())):
                        source_count = collections.Counter(self.vm.vm.get_disk_provision.values())
                        dest_count = collections.Counter(other.vm.vm.get_disk_provision.values())
                        if source_count['Thin'] != dest_count['Thin'] or ((source_count[
                                                                               'Thick Eager Zero'] +
                                                                           source_count[
                                                                               'Thick Lazy Zero']) != (
                                                                                  dest_count[
                                                                                      'Thick Eager Zero'] +
                                                                                  dest_count[
                                                                                      'Thick Lazy Zero'])):
                            self.log.error("Disk type match failed")
                            return False
                elif self.vm_restore_options.disk_option == 'Thin':
                    if not all(_type == self.vm_restore_options.disk_option for _type in
                               other.vm.vm.get_disk_provision.values()):
                        self.log.error("Disk type match failed")
                        return False
                else:
                    for _type in other.vm.vm.get_disk_provision.values():
                        if _type not in ('Thick Eager Zero', 'Thick Lazy Zero'):
                            self.log.error("Disk type match failed")
                            return False
                self.log.info("Disk type matched")
            return True

    class VmConversionValidation(object):
        def __init__(self, vmobj, vm_restore_options):
            self.vm = vmobj
            self.vm_restore_options = vm_restore_options
            self.log = logger.get_log()

        def __eq__(self, other):
            """compares the restored vm with user inputs"""
            return other.network_name == self.vm_restore_options._network and \
                   other.esx_host == self.vm_restore_options._host and \
                   other.datastore.split(",")[0] == self.vm_restore_options._datastore

    class LiveSyncVmValidation(object):
        def __init__(self, vmobj, schedule, replicationjob=None):
            self.vm = vmobj
            self.schedule = schedule
            self.replicationjob = replicationjob
            self.log = logger.get_log()

        def __eq__(self, other):
            """ validates vm replicated through livesync """
            try:
                if '__GX_BACKUP__' not in other.vm.vm.VMSnapshot:
                    self.log.info('snapshot validation failed')
                    return False
                self.log.info('snapshot validation successful')
                config_val = (int(self.vm.vm.no_of_cpu) == int(other.vm.vm.no_of_cpu) and
                              int(self.vm.vm.disk_count) == int(other.vm.vm.disk_count) and
                              int(self.vm.vm.memory) == int(other.vm.vm.memory))
                if not config_val:
                    return False
                return True
            except Exception as err:
                self.log.exception(
                    "Exception at Validating  {0}".format(err))
                raise err

    @property
    def disk_list(self):
        """
        To fetch the disk in the VM

        Returns:
            disk_list           (list): List of disk in VM
                                        e.g:[vm1.vmdk]

        """
        _temp_disk_list = {}
        for ds in self.vm_obj.layout.disk:
            _dp = ds.diskFile[0]
            _dn = _dp.split("/")[1]
            _temp_disk_list[_dp] = _dn
        if _temp_disk_list:
            _disk_list = _temp_disk_list.keys()
        else:
            _disk_list = []
        return _disk_list

    @property
    def rdm_details(self):
        """
        To fetch the disk in the VM

        Returns:
            _rdm_dict           (dict): Dictionary of disks and its details of the disk

        """
        _rdm_dict = {}
        for dev in self.vm_obj.config.hardware.device:
            if isinstance(dev, self.hvobj.vim.vm.device.VirtualDisk):
                f_name = dev.backing.fileName
                disk_mode = dev.backing.diskMode
                if isinstance(dev.backing,
                              self.hvobj.vim.vm.device.VirtualDisk.RawDiskMappingVer1BackingInfo):
                    compatibility_mode = dev.backing.compatibilityMode
                else:
                    compatibility_mode = 'Flat'
                _rdm_dict[f_name] = [f_name, compatibility_mode, disk_mode]
        self.disk_validation = False
        return _rdm_dict

    @property
    def get_disk_provision(self):
        """
        To get the disk provision type
        Returns:
                disk_provision_list            (dict):     Dictionary of disks and their type

        Raises:
            Exception:
                if failed to get the Provision type of the disks
        """
        try:
            _disk_provision_list = {}
            for dev in self.vm_obj.config.hardware.device:
                if isinstance(dev, self.hvobj.vim.vm.device.VirtualDisk):
                    if isinstance(dev.backing,
                                  self.hvobj.vim.vm.device.VirtualDisk.RawDiskMappingVer1BackingInfo):
                        _provision = 'Ignore_rdm'
                    else:
                        if dev.backing.thinProvisioned:
                            _provision = 'Thin'
                        else:
                            if dev.backing.eagerlyScrub:
                                _provision = 'Thick Eager Zero'
                            else:
                                _provision = 'Thick Lazy Zero'
                    _disk_provision_list[dev.backing.fileName] = _provision
            return _disk_provision_list
        except Exception as err:
            self.log.exception("Exception while collecting detail about rdm %s", str(err))
            raise err

    @property
    def get_vm_folder(self):
        """gets the folder name for the vm"""
        # self.log.info("Parent path: {}".format(self.vm_obj.config.files.logDirectory))
        return self.vm_obj.config.files.logDirectory

    @property
    def datastore(self):
        """Returns the base datastore of the vm"""
        _datastore = self.get_vm_folder
        _datastore = _datastore[_datastore.find("[") + 1:_datastore.find("]")]
        return _datastore

    @property
    def VMSnapshot(self):
        """Return the snapshot on the vm"""
        _temp_snaps = []
        _snaps = None
        if self.vm_obj.snapshot:
            for snaps in self.vm_obj.snapshot.rootSnapshotList:
                _temp_snaps.append(snaps.name)
            _snaps = ','.join(_temp_snaps)
        return _snaps

    @property
    def NicName(self):
        _nics = []
        _nic_name = None
        for nic in self.vm_obj.network:
            _nics.append(nic.name)
        _nic_name = ','.join(_nics)
        return _nic_name

    def update_vm_info(self, prop='Basic', os_info=False, force_update=False,**kwargs):
        """
        Fetches all the properties of the VM

        Args:
            prop                (str):  Basic - Basic properties of VM like HostName,
                                                especially the properties with which
                                                VM can be added as dynamic content

                                        All   - All the possible properties of the VM

            os_info             (bool): To fetch os info or not

            force_update - to refresh all the properties always
                    True : ALways collect  properties
                    False: refresh only if properties are not initialized

            **kwargs                         : Arbitrary keyword arguments

        Raises:
            Exception:
                if failed to update all the properties of the VM

        """
        try:
            if not self._basic_props_initialized or force_update:
                self._get_vm_info()
            if self.power_state == 'poweredOff':
                if self.template:
                    self.convert_template_to_vm()
                else:
                    self.power_on()
                time.sleep(240)
                self._get_vm_info()
            if os_info or prop == 'All':
                self.nics.clear()
                self.datastores.clear()
                self.nic_count = 0
                if 'isolated_network' not in kwargs.keys():
                    self.vm_guest_os = self.guest_os
                    self.get_drive_list()
                self.no_of_cpu = self.vm_obj.summary.config.numCpu
                self.memory = float(self.vm_obj.summary.config.memorySizeMB / 1024)
                self.esx_host = self.vm_obj.runtime.host.name
                for ds in self.vm_obj.datastore:
                    self.datastores.append(ds.name)
                self.disk_count = len(self.vm_obj.layout.disk)
                self.tools = self.vm_obj.guest.guestState
                for nic in self.vm_obj.network:
                    self.nics.append(nic.name)
                    self.nic_count += 1
                self.network_name = self.NicName
                self.get_all_disks_info()
        except Exception as err:
            self.log.exception("Failed to Get  the VM Properties of the VM")
            raise Exception(err)

    def _get_vm_info(self):
        """
        Get the basic or all or specific properties of VM

        Raises:
            Exception:
                if failed to get all the properties of the VM

        """
        try:
            all_vms = self.hvobj.get_content([self.hvobj.vim.VirtualMachine])
            for vm in all_vms:
                if vm.name == self.vm_name:
                    self.vm_obj = vm
                    break
            self.guid = self.vm_obj.config.instanceUuid
            self.ip = self.vm_obj.summary.guest.ipAddress
            if hasattr(self.vm_obj.guest, 'net'):
                ip = [ip.ipAddress for nic in self.vm_obj.guest.net if nic.ipConfig is not None
                      for ip in nic.ipConfig.ipAddress if ip.state == 'preferred']
                if ip:
                    self.ip = ip[0]
            self.power_state = self.vm_obj.runtime.powerState
            if 'win' in self.vm_obj.config.guestFullName.lower():
                self.guest_os = 'Windows'
            else:
                self.guest_os = 'Linux'
            self.template = self.vm_obj.config.template
            self._basic_props_initialized = True

        except Exception as err:
            self.log.exception("Failed to Get Basic Properties of the VM: {}".format(self.vm_name))
            raise Exception(err)

    def power_on(self):
        """
        Power on the VM.

        Raises:
            Exception:
                When power on failed

        """

        try:
            if self.vm_obj.runtime.powerState in ('poweredOn', 'running'):
                self.log.info("VM {} is already powered ON".format(self.vm_name))
                return
            self.log.info("Powering on the vm {}".format(self.vm_name))
            self.hvobj.wait_for_tasks([self.vm_obj.PowerOn()])
            self.power_state = self.vm_obj.runtime.powerState

        except Exception as exp:
            raise Exception("Exception in PowerOn:" + str(exp))

    def power_off(self):
        """
        Power off the VM.

        Raises:
            Exception:
                When power off failed

        """

        try:
            if self.vm_obj.runtime.powerState == 'poweredOff':
                self.log.info("VM {} is already powered off".format(self.vm_name))
                return
            self.log.info("Powering off the vm {}".format(self.vm_name))
            self.vm_obj.ShutdownGuest()
            time.sleep(5)
            self.power_state = self.vm_obj.runtime.powerState

        except Exception as exp:
            raise Exception("Exception in PowerOff:" + str(exp))

    def delete_vm(self):
        """
        Delete the VM.

        Raises:
            Exception:
                When deleting of the vm failed
        """

        try:
            self.log.info("Deleting the vm {}".format(self.vm_name))
            if self.vm_obj.runtime.powerState in ('poweredOn', 'running'):
                self.hvobj.wait_for_tasks([self.vm_obj.PowerOff()])
            self.hvobj.wait_for_tasks([self.vm_obj.Destroy()])

        except Exception as exp:
            self.log.exception("Exception in DeleteVM {0}".format(exp))
            return False

    def convert_vm_to_template(self):
        """
        Convert a vm to template

        Raises:
            Exception:
                if it fails to convert vms to template

        """
        try:
            self.log.info("Converting VM:{} into template".format(self.vm_name))
            if 'poweredOn' in self.vm_obj.runtime.powerState:
                self.power_off()
                time.sleep(60)
            self.vm_obj.MarkAsTemplate()

        except Exception as err:
            self.log.exception("Exception while Converting  vm to template " + str(err))
            raise err

    def convert_template_to_vm(self):
        """
        Convert a template to vm

        Raises:
            Exception:
                if it fails to convert template to vm

        """
        try:
            self._get_vm_info()
            self.log.info("Converting {} into a VM".format(self.vm_name))
            if self.vm_obj.summary.config.template:
                self.vm_obj.MarkAsVirtualMachine(self.vm_obj.runtime.host.parent.resourcePool)
                time.sleep(5)
                self.power_on()
                time.sleep(120)
                self.log.info("Converted {} into a VM Successfully".format(self.vm_name))
            else:
                self.log.Exception("{} is not a template".format(self.vm_name))

        except Exception as err:
            self.log.exception("Exception while Converting  vm to template " + str(err))
            raise err

    def clean_up(self):
        """
        Does the cleanup after the testcase.

        Raises:
            Exception:
                When cleanup failed or unexpected error code is returned

        """

        try:
            self.log.info("Powering off VMs after restore")
            self.power_off()
        except Exception as exp:
            raise Exception("Exception in Cleanup: {0}".format(exp))

    def add_disks(self, disk_size, disk_type='thin'):
        """
        Add disk in the vm
        Args:
            disk_size       (int): Disk size that needs to be added in GB
            disk_type       (string): thin (default)

        Return:
            Status  : True/False
        Raises:
            Exception: if failed to add disk
        """
        try:
            if not isinstance(disk_size, int):
                disk_size = int(disk_size)
            spec = self.hvobj.vim.vm.ConfigSpec()
            unit_number = 0
            for dev in self.vm_obj.config.hardware.device:
                if hasattr(dev.backing, 'fileName'):
                    unit_number = int(dev.unitNumber) + 1
                    if unit_number == 7:
                        unit_number += 1
                    if unit_number >= 16:
                        self.log.error("We don't support this many disks")
                        raise RuntimeError("We don't support this many disks")
                if isinstance(dev, self.hvobj.vim.vm.device.VirtualSCSIController):
                    controller = dev
            dev_changes = []
            new_disk_size = int(disk_size) * 1024 * 1024
            virtual_hdd_spec = self.hvobj.vim.vm.device.VirtualDeviceSpec()
            virtual_hdd_spec.fileOperation = "create"
            virtual_hdd_spec.operation = self.hvobj.vim.vm.device.VirtualDeviceSpec.Operation.add
            virtual_hdd_spec.device = self.hvobj.vim.vm.device.VirtualDisk()
            virtual_hdd_spec.device.backing = self.hvobj.vim.vm.device.VirtualDisk.FlatVer2BackingInfo()
            if disk_type == 'thin':
                virtual_hdd_spec.device.backing.thinProvisioned = True
            virtual_hdd_spec.device.backing.diskMode = 'persistent'
            virtual_hdd_spec.device.unitNumber = unit_number
            virtual_hdd_spec.device.capacityInKB = new_disk_size
            virtual_hdd_spec.device.controllerKey = controller.key
            dev_changes.append(virtual_hdd_spec)
            self.log.info("Adding disk of size %s", disk_size)
            spec.deviceChange = dev_changes
            self.hvobj.wait_for_tasks([self.vm_obj.ReconfigVM_Task(spec=spec)])
            self.log.info("%s GB disk added to vm %s", disk_size, self.vm_obj.config.name)

            return True
        except Exception as err:
            self.log.exception("Exception in adding disk to vm")
            raise err

    def delete_disks(self, disk_names=None):
        """
        Delete the disks in the vm

        Args:
            disk_names              (string):   Disk name which needs to be deleted


        Returns:
            Status                          : True if successful
                                              False if exception
        Raises:
            Exception:
                if failed to delete the disks of the vm
        """
        try:
            if disk_names:
                _disk_to_delete = disk_names
            else:
                _disk_to_delete = '\w*/del_*'
            disk_found = False
            virtual_hdd_spec = self.hvobj.vim.vm.device.VirtualDeviceSpec()
            virtual_hdd_spec.operation = \
                self.hvobj.vim.vm.device.VirtualDeviceSpec.Operation.remove
            virtual_hdd_spec.fileOperation = self.hvobj.vim.vm.device.VirtualDeviceSpec. \
                FileOperation.destroy
            spec = self.hvobj.vim.vm.ConfigSpec()
            for dev in self.vm_obj.config.hardware.device:
                if isinstance(dev, self.hvobj.vim.vm.device.VirtualDisk):
                    if re.search(_disk_to_delete,
                                 dev.backing.fileName) or dev.deviceInfo.label == _disk_to_delete:
                        virtual_hdd_spec.device = dev
                        self.log.info("Deleting Disk {}".format(dev.backing.fileName))
                        spec.deviceChange = [virtual_hdd_spec]
                        self.hvobj.wait_for_tasks([self.vm_obj.ReconfigVM_Task(spec=spec)])
                        self.log.info("Disk {} deleted".
                                      format(virtual_hdd_spec.device.backing.fileName))
                        disk_found = True

            if not disk_found:
                raise RuntimeError('Virtual {} could not '
                                   'be found.'.format(_disk_to_delete))
            return True
        except Exception as err:
            self.log.exception("exception in deleting the disk")
            raise err

    def get_all_disks_info(self):
        """
        Gets the disk details for disk filtering

        Raises:
            Exception:
                When it fails to fetch disk disk details
        """
        try:
            self.disk_dict = {}
            for dev in self.vm_obj.config.hardware.device:
                if isinstance(dev, self.hvobj.vim.vm.device.VirtualDisk):
                    _key = dev.key
                    if _key // 32000:
                        _ctr, _id = int(_key % 32000 / 15), _key % 32000 % 15
                        _virtual_device_node = 'NVME ({}:{})'.format(_ctr, _id)
                    elif _key // 16000:
                        _ctr, _id = int(_key % 16000 / 30), _key % 16000 % 30
                        _virtual_device_node = 'SATA ({}:{})'.format(_ctr, _id)
                    elif _key // 2000:
                        _ctr, _id = int(_key % 2000 / 16), _key % 2000 % 16
                        _virtual_device_node = 'SCSI ({}:{})'.format(_ctr, _id)
                    else:
                        _virtual_device_node = None
                    self.disk_dict[
                        dev.backing.fileName] = _virtual_device_node, dev.backing.datastore.name, \
                                                dev.backing.fileName, dev.deviceInfo.label
        except Exception as err:
            self.log.exception("Failed to fetch disk details")
            raise Exception(err)

    def attach_network_adapter(self):
        """
        Attaches network to the vm

        Raises:
            Exception:
                if failed to attache the network

        """
        try:
            for dev in self.vm_obj.config.hardware.device:
                if isinstance(dev, self.hvobj.vim.vm.device.VirtualVmxnet3) or \
                        isinstance(dev, self.hvobj.vim.vm.device.VirtualE1000e):
                    device = dev
                    device.connectable = self.hvobj.vim.vm.device.VirtualDevice.ConnectInfo(
                        connected=True, startConnected=True, allowGuestControl=True)
                    nicspec = self.hvobj.vim.vm.device.VirtualDeviceSpec(device=device)
                    nicspec.operation = self.hvobj.vim.vm.device.VirtualDeviceSpec.Operation.edit
                    config_spec = self.hvobj.vim.vm.ConfigSpec(deviceChange=[nicspec])
                    self.hvobj.wait_for_tasks([self.vm_obj.ReconfigVM_Task(config_spec)])
                    self.log.info("{} attached sucessfully".format(dev.deviceInfo.label))
        except Exception as err:
            self.log.exception("exception in attaching hte network")
            raise err

    def change_num_cpu(self, num_cpu):
        """
        Changes CPU number
        Args:
            num_cpu                     (int): new cpu number

        """
        spec = self.hvobj.vim.vm.ConfigSpec()
        spec.numCPUs = num_cpu
        if self.power_state != 'poweredOff':
            self.power_off()
        self.hvobj.wait_for_tasks([self.vm_obj.ReconfigVM_Task(spec)])
        self.no_of_cpu = self.vm_obj.summary.config.numCpu

    def change_memory(self, ram_size_gb):
        """
        Changes ram of the vm
        Args:
            ram_size_gb                     (int): new ram of the vm

        Returns:

        """
        spec = self.hvobj.vim.vm.ConfigSpec()
        spec.memoryMB = ram_size_gb * 1024
        if self.power_state != 'poweredOff':
            self.power_off()
        self.hvobj.wait_for_tasks([self.vm_obj.ReconfigVM_Task(spec)])
        self.memory = int(self.vm_obj.summary.config.memorySizeMB / 1024)

    def get_tags_category(self):
        """
        Get's Tag and category of the VM

        Raises:
            Exception:
                if failed to fetch tags and category of the vm

        """
        try:
            self.prop_dict["property"] = 'tagsvalidation'
            _ps_path = os.path.join(
                self.utils_path, self.vm_props_file)
            output = self.host_machine.execute_script(_ps_path, self.prop_dict)
            if output.exception:
                self.log.exception(
                    "Cannot get tag and category for the vm {}".format(output.exception))
            self.tags_of_vm, self.category_of_vm = output.formatted_output.split(";")
            self.tags_of_vm = self.tags_of_vm.split('=')[1].strip()
            self.category_of_vm = self.category_of_vm.split('=')[1].strip()
        except Exception as err:
            self.log.exception("exception in fetching tags and category of the vm")
            raise err

    def backup_validation(self):
        """
        Updates data neded for backup validation
        Returns:

        """
        self.esx_host = self.vm_obj.runtime.host.name
        self.power_state = self.vm_obj.runtime.powerState
        self.tools = self.vm_obj.guest.guestState

    def get_disk_in_controller(self, controller_detail):
        """
        get the disk associated with the virtual device node

        Args:
                controller_detail           (string):  Details about the virtual device node
        Return:
                _disks                 (list): Disk path

        Raises:
            Exception:
                if failed to fetch disk path from the virtual device node

        """
        try:
            _disks = [key for key, value in self.disk_dict.items() if controller_detail in value]
            return _disks

        except Exception as err:
            self.log.exception("Exception in get_disk_in_controller {}".format(err))
            raise Exception(err)

    def get_disks_by_repository(self, datastore_name):
        """
        get the disk associated with the datastore name

        Args:
            datastore_name                  (string) : datastore name

        Returns:
             _disks                     (list): Disk path

        Raises:
            Exception
                when failed to fetch the disk associated with the datastore name
        """
        try:
            _disks = [key for key, value in self.disk_dict.items() if datastore_name in value]
            return _disks

        except Exception as err:
            self.log.exception("Exception in get_disks_by_repository : {}".format(err))
            raise Exception(err)

    def get_disk_path_from_pattern(self, disk_pattern):
        """
        find the disk that matches the disk apttern form disk list

        Args:
                disk_pattern                    (string):   pattern which needs to be matched

        Returns:
             _disks                     (list): Disk path

        Raises:
            Exception
                when failed to fetch the disk associated with the disk pattern
        """
        try:
            rep = {'?': '\w{1}', '!': '^', '*': '.*'}
            rep = dict((re.escape(k), v) for k, v in rep.items())
            pattern = re.compile("|".join(rep.keys()))
            _disk_pattern = pattern.sub(lambda m: rep[re.escape(m.group(0))], disk_pattern)
            _disk_pattern = re.escape(_disk_pattern)
            if _disk_pattern.isalnum():
                _disk_pattern = '^' + _disk_pattern + '$'
            elif _disk_pattern[-1].isalnum():
                _disk_pattern = _disk_pattern + '$'
            _disk_pattern = re.compile(_disk_pattern, re.I)
            _disks = [key for key, value in self.disk_dict.items() if
                      re.findall(_disk_pattern, value[2])]
            return _disks
        except Exception as err:
            self.log.exception("Exception in get_disk_path_from_pattern : {}".format(err))
            raise Exception(err)

    def get_disk_by_label(self, disk_label):
        """
        find the disk that matches the disk label

        Args:
                disk_label                  (string):   Disk label to be matched

        Returns:
             _disk_path                     (list): Disk path

        Raises:
            Exception
                when failed to fetch the disk associated with the disk label
        """
        try:
            _disk_labels = []
            if re.search(r'\[.*?]', disk_label):
                _range_start = int(re.findall(r"\[([^-]+)", disk_label)[0])
                _range_end = int(re.findall(r"\-([^]]+)", disk_label)[0]) + 1
                disk_ranges = range(_range_start, _range_end)
                for _disk in disk_ranges:
                    _disk_labels.append('Hard disk ' + str(_disk))
            else:
                _disk_labels = [disk_label]
            _disks = [key for key, value in self.disk_dict.items() if value[3] in _disk_labels]
            return _disks

        except Exception as err:
            self.log.exception("Exception in get_disk_by_label : {}".format(err))
            raise Exception(err)

    def _get_snapshots_by_name(self, snap_shots, snap_name='Fresh'):
        """

        Args:
            snap_shots                           (list):   snapshot list of the vm

            snap_name                           (string):   Name of the snap to find

        Returns:
            snap_obj                            (list):     Snapshots found after comparing

        Raises:
            Exception:
                When getting the snapshots hierarchy
        """

        snap_obj = []
        for snap_shot in snap_shots:
            if snap_name:
                if snap_shot.name == snap_name:
                    snap_obj.append(snap_shot)
                else:
                    snap_obj = snap_obj + self._get_snapshots_by_name(
                        snap_shot.childSnapshotList, snap_name)
        return snap_obj

    def create_snap(self, snap_name='Fresh', quiesce=False, dump_memory=False):
        """
        Creating a snapshot of the vm

        Args:
            snap_name                           (string):   Name of the snap to create

            quiesce                             (bool):   Quiesce during snapshot creation

            dump_memory                         (bool):   Taking snapshot of vm memory

        Raises:
            Exception:
                When Creating Snapshot of the vm
        """
        try:
            if quiesce and dump_memory:
                raise Exception("Quiesce and snapshot of vm memory can't be enabled together")
            elif self.vm_obj.runtime.powerState not in ('running', 'poweredOn') and (
                    quiesce or dump_memory):
                self.log.warning(
                    "Quiesce or snapshot of vm memory can't be enabled as the vm is powered off."
                    "Taking snapshot without Quiesce or snapshot of vm memory")
                quiesce = False
                dump_memory = False
            elif self.vm_obj.guest.toolsRunningStatus == 'guestToolsNotRunning' and quiesce:
                self.log.warning(
                    "Quiesce can't be enabled as the guest tools is not running"
                    "Taking snapshot without Quiesce")
                quiesce = False
            snap_description = 'Snapshot taken by automation'
            self.log.info(
                "Creating Snapshot {} on vm:{} with quiese:{}, snapshot of vm memory:{}".format(
                    snap_name, self.vm_name, quiesce, dump_memory))
            self.hvobj.wait_for_tasks(
                [self.vm_obj.CreateSnapshot(snap_name, snap_description, dump_memory, quiesce)])
        except Exception as exp:
            self.log.exception(
                "Exception:{} in Creating snapshot for vm: {}".format(exp, self.vm_name))
            return False

    def revert_snap(self, snap_name='Fresh'):
        """
        Reverts a snapshot of the vm

        Args:
            snap_name                           (string):   Name of the snap to revert to

        Raises:
            Exception:
                When reverting of the snapshot
        """
        try:
            if not self.vm_obj.snapshot:
                raise Exception("There is no Snapshot to revert")
            snap_obj = self._get_snapshots_by_name(self.vm_obj.snapshot.rootSnapshotList, snap_name)
            if len(snap_obj) != 1:
                raise Exception("Multiple or no snaps with name:{}. Can't revert".format(snap_name))
            snap_obj = snap_obj[0].snapshot
            self.log.info("Reverting Snapshot named {} of vm:{}".format(snap_name, self.vm_name))
            self.hvobj.wait_for_tasks([snap_obj.RevertToSnapshot_Task()])
        except Exception as exp:
            self.log.exception(
                "Exception:{} in Reverting snapshot for vm: {}".format(exp, self.vm_name))
            return False

    def delete_snap(self, snap_name='Fresh'):
        """
        Deletes a snapshot of the vm

        Args:
            snap_name                           (string):   Name of the snap to delete

        Raises:
            Exception:
                When deleting of the snapshot
        """
        try:
            if not self.vm_obj.snapshot:
                raise Exception("There is no Snapshot to delete")
            snap_obj = self._get_snapshots_by_name(self.vm_obj.snapshot.rootSnapshotList, snap_name)
            if len(snap_obj) != 1:
                raise Exception("Multiple or no snaps with name:{}. Can't delete".format(snap_name))
            snap_obj = snap_obj[0].snapshot
            self.log.info("Deleting Snapshot named {} of vm:{}".format(snap_name, self.vm_name))
            self.hvobj.wait_for_tasks([snap_obj.RemoveSnapshot_Task(True)])
        except Exception as exp:
            self.log.exception(
                "Exception:{} in deleting snapshot for vm: {}".format(exp, self.vm_name))
            return False

    def delete_all_snap(self):
        """
        Deletes all snapshot of the vm
        Raises:
            Exception:
                When deleting all snapshots of the vm
        """
        try:
            self.log.info("Deleting all Snapshots of vm:{}".format(self.vm_name))
            self.hvobj.wait_for_tasks([self.vm_obj.RemoveAllSnapshots()])
        except Exception as exp:
            self.log.exception(
                "Exception:{} in deleting all snapshots for vm: {}".format(exp, self.vm_name))
            return False

    def mount_vmdk(self, disk_path):
        """
        Mounts a VMDK disk to a target VM via HotAdd

        Args:
                disk_path       (str):   [{Datastore}] {VM Name}/{Filename}.vmdk

        Raise Exception:
                if failed to mount the disk to the target vm

        """

        try:
            virtual_hdd_spec = \
                self.hvobj.vim.vm.device.VirtualDeviceSpec()
            spec = self.hvobj.vim.vm.ConfigSpec()

            unit_number = 0
            for dev in self.vm_obj.config.hardware.device:
                if hasattr(dev.backing, 'fileName'):
                    unit_number = int(dev.unitNumber) + 1
                if isinstance(
                        dev,
                        self.hvobj.vim.vm.device.VirtualSCSIController):
                    controller = dev

            virtual_hdd_spec.operation = \
                self.hvobj.vim.vm.device.VirtualDeviceSpec.Operation.add
            virtual_hdd_spec.device = \
                self.hvobj.vim.vm.device.VirtualDisk()
            virtual_hdd_spec.device.backing = \
                self.hvobj.vim.vm.device.VirtualDisk.FlatVer2BackingInfo()
            virtual_hdd_spec.device.backing.diskMode = 'persistent'
            virtual_hdd_spec.device.backing.fileName = disk_path
            virtual_hdd_spec.device.backing.thinProvisioned = True
            virtual_hdd_spec.device.unitNumber = unit_number
            virtual_hdd_spec.device.controllerKey = controller.key

            spec.deviceChange = [virtual_hdd_spec]

            task = self.vm_obj.ReconfigVM_Task(spec=spec)

            answered_id = None

            while task.info.state not in [
                self.hvobj.vim.TaskInfo.State.success,
                self.hvobj.vim.TaskInfo.State.error]:

                if self.vm_obj.runtime.question is not None:

                    if answered_id:
                        continue

                    question_id = self.vm_obj.runtime.question.id
                    choices = self.vm_obj.runtime.question.choice.choiceInfo
                    choice = None
                    for o in choices:
                        if o.summary == "Yes":
                            choice = o.key
                    self.vm_obj.AnswerVM(question_id, choice)
                    answered_id = question_id

        except Exception as err:
            self.log.exception("An error occurred in mounting the VMDK")
            raise Exception(err)

    def unmount_vmdk(self, disk_path):
        """
        Unmounts a VMDK disk from the target VM

        Args:
                disk_path       (str):   [{Datastore}] {VM Name}/{Filename}.vmdk

        Raise Exception:
                if failed to unmount the disk to the target vm

        """
        try:
            hdd_label = self.get_disk_label_from_path(disk_path)

            for dev in self.vm_obj.config.hardware.device:
                if isinstance(dev, self.hvobj.vim.vm.device.VirtualDisk) \
                        and dev.deviceInfo.label == hdd_label:
                    virtual_hdd_device = dev

            virtual_hdd_spec = \
                self.hvobj.vim.vm.device.VirtualDeviceSpec()
            spec = self.hvobj.vim.vm.ConfigSpec()

            virtual_hdd_spec.operation = \
                self.hvobj.vim.vm.device.VirtualDeviceSpec.Operation.remove
            virtual_hdd_spec.device = virtual_hdd_device

            spec.deviceChange = [virtual_hdd_spec]

            self.hvobj.wait_for_tasks(
                [self.vm_obj.ReconfigVM_Task(spec=spec)])

            self.log.info("Successfully unmounted disk")

        except Exception as err:
            self.log.exception("An error occurred in unmounting the VMDK")
            raise Exception(err)

    def get_disk_label_from_path(self, disk_path):
        """
        Gets the disk label of the required disk

        Args:
                disk_path       (str):   [{Datastore}] {VM Name}/{Filename}.vmdk

        Raise Exception:
                if failed to get label of disk

        """

        try:
            self.get_all_disks_info()
            vm_disks = self.disk_dict
            disk_label = vm_disks[disk_path][-1]
            return disk_label

        except Exception as err:
            self.log.exception("An error occurred in fetching disk label")
            raise Exception(err)

    def verify_vmdks_attached(self, disk_paths):
        """
        Checks whether VMDKs in list are attached to vm

        Args:
                disk_path       (str):   [{Datastore}] {VM Name}/{Filename}.vmdk

        Raise Exception:
                if failed to get label of disk

        """
        try:
            self.get_all_disks_info()
            vm_disks = self.disk_dict
            vm_disk_list = list(vm_disks.keys())

            if all(i in vm_disk_list for i in disk_paths):
                return True
            else:
                return False

        except Exception as err:
            self.log.exception("An error while checking if VMDKs attached to VM")
            raise Exception(err)
