# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""Does all the Operation for Vcloud vm"""

import json
import time

import xmltodict
import requests
from VirtualServer.VSAUtils.VMHelper import HypervisorVM


class VcloudVM(HypervisorVM):
    """
        This is the main file for all Vcloud VM operations
    """

    def __init__(self, hvobj, vm_name):
        """
        Initialization of Vcloud VM properties

        Args:

            hvobj           (obj):  Hypervisor Object

            vm_name         (str):  Name of the VM

        """

        super(VcloudVM, self).__init__(hvobj, vm_name)
        self.url = self.hvobj.url
        self.username = self.hvobj.username
        self.password = self.hvobj.password
        self.vm_list = self.hvobj.vm_list
        self.vapp_name = self.hvobj.vapp_name
        self.vm_name = vm_name
        self.href = None
        self.memory = 0
        self.no_of_cpu = 0
        self.disk_count = 0
        self.ip_address = None
        self.network_adapter = None
        self.vcloud_auth_token = None
        self._basic_props_initialized = False
        self.network_name = None
        self.guid = None
        self.subnet_id = None
        self.vm_state = None
        self._vm_info = {}
        self.disk_info = {}
        self.disk_dict = {}
        self.nic = []
        self._disk_list = None
        self._get_vm_info()
        self.update_vm_info()

    @property
    def vm_info(self):
        """
            It is used to fetch VM info. This is read only property
        """
        if self._vm_info[self.vm_name] == {}:
            self._get_vm_info()
        return self._vm_info[self.vm_name]

    @vm_info.setter
    def vm_info(self, value):
        """
             This is to set vmname for VM info
        """
        self._vm_info[self.vm_name] = value

    @property
    def headers(self):
        """
        provide the default headers required

        """
        self.vcloud_auth_token = self.hvobj.check_for_login_validity()
        self._headers = {'Accept': 'application/*+xml;version=27.0',
                         "x-vcloud-authorization": self.vcloud_auth_token}
        return self._headers

    def get_drive_list(self):
        """
        Returns the drive list for the VM
        """
        try:
            super(VcloudVM, self).get_drive_list()
            del self._drives['E']
            return True

        except Exception as err:
            self.log.exception(
                "An Exception Occurred in Getting the Volume Info for the VM ".format(err))
            return False

    def _get_vm_info(self):
        """
        Get all VM information

        Raises:
            Exception:
                if failed to get information about VM
        """
        try:
            self.log.info(
                "VM information :: Getting all information of VM %s" % self.vm_name)
            self.href = self.vm_list[self.vm_name]

            response = requests.get(self.href, headers=self.headers,
                                    auth=(self.user_name, self.password), verify=False)
            data = xmltodict.parse(response.content)
            content = json.dumps(data)
            data = json.loads(content)
            if response.status_code == 200:
                self._vm_info[self.vm_name] = data
                self.log.info(
                    "vm_info %s of VM %s is successfully obtained " % (data, self.vm_name))

            else:
                self.log.info(
                    "There was No VM in Name %s , please check the VM name" % self.vm_name)
                self._vm_info = False
                # No VMs found
                raise Exception("VM  data cannot be collected")

        except Exception as err:
            self.log.exception("Exception in get_vm_info")
            raise Exception(err)

    def get_vm_guid(self):
        """
        gets the GUID of VM

        Raises:
            Exception:
                If the guid cannot be retrieved
        """
        try:
            self.log.info("Getting the guid information for VM %s" % self.vm_name)
            data = self._vm_info
            self.guid = data[self.vm_name]['Vm']['@id']

        except Exception as err:
            self.log.exception("Exception in get_vm_guid")
            raise Exception(err)

    def get_disk_info(self):
        """
        Get disk properties of both OS and data disks of VM

        Raises:
            Exception:
                    if failed to disk information of VM
        """
        try:
            self.log.info(
                "VM information :: Getting all Disk information of VM %s" % self.vm_name)

            response = requests.get(self.href + "/virtualHardwareSection/disks", headers=self.headers,
                                    auth=(self.user_name, self.password), verify=False)
            if response.status_code == 200:
                data = xmltodict.parse(response.content)
                content = json.dumps(data)
                data = json.loads(content)
                self.disk_info = data['RasdItemsList']['Item']
                self.disk_count = len(self.disk_info)

            else:
                self.log.info("unable to get disk infor for %s" % self.vm_name)
                self._vm_info = False
                # No VMs found
                raise Exception("data cannot be collected")

        except Exception as err:
            self.log.exception("Exception in get_disk_info")
            raise Exception(err)

    def get_status_of_vm(self):
        """
        Get the status of VM like started.stopped

        Raises:
            Exception:
                    if failed to get status of VM
        """
        try:
            self.log.info("Get the Status of VM %s" % self.vm_name)
            data = self._vm_info
            if data[self.vm_name]['Vm']['@status'] == '4':
                self.vm_state = "Powered On"
                self.log.info("VM is Powered On")
            elif data[self.vm_name]['Vm']['@status'] == '8':
                self.vm_state = "Powered Off"
                self.log.info("VM is Powered Off")

        except Exception as err:
            self.log.exception("Exception in getStatusofVM")
            raise Exception(err)

    def get_os_type(self):
        """
        Update the OS Type of VM

        Raises:
            Exception:
                    if failed to find OS type of VM
        """

        try:
            self.log.info("Getting the os info for VM %s" % self.vm_name)
            data = self.vm_info
            os_info = data['Vm']['ovf:OperatingSystemSection']['ovf:Description']
            if 'Windows' in os_info:
                guest_os = 'Windows'
            else:
                guest_os = 'Linux'
            self.log.info("os disk detaisl of the VM is %s" % guest_os)
            setattr(self, "guest_os", guest_os)
            self.log.info("OS type is : %s" % self.guest_os)

        except Exception as err:
            self.log.exception("Exception in GetOSType")
            raise Exception(err)

    def get_ip_address(self):
        """
        Get the Ip address and nic of the VM

        Raises:
            Exception:
                    if failed to get IP address of VM
        """
        try:
            self.log.info("Get VM IP details and nic of %s" % self.vm_name)

            response = requests.get(self.href + "/networkConnectionSection",
                                    headers=self.headers, auth=(self.user_name, self.password),
                                    verify=False)
            data = xmltodict.parse(response.content)
            content = json.dumps(data)
            data = json.loads(content)
            network_info = data['NetworkConnectionSection']
            self.ip_address = network_info['NetworkConnection']['IpAddress']
            self.network_name = network_info['NetworkConnection']['@network']
            self.log.info("Ip of vm %s", self.ip_address)
            self.log.info("Network Name of vm %s", self.network_name)
            self.network_adapter = network_info['NetworkConnection']['NetworkAdapterType']
            self.log.info("Network Adapter of vm %s", self.network_adapter)

            setattr(self, "host_name", self.network_adapter)
            setattr(self, "ip", self.ip_address)
        except Exception as err:
            self.log.exception("Exception in get_vm_ip_address")
            raise Exception(err)

    def get_cores(self):
        """
        Get number of CPU of VM

        Raises:
            Exception:
                    if failed to get CPU of VM
        """

        try:
            self.log.info("Get cpu of a VM %s" % self.vm_name)
            response = requests.get(self.href + "/virtualHardwareSection/cpu",
                                    headers=self.headers, auth=(self.user_name, self.password),
                                    verify=False)
            data = xmltodict.parse(response.content)
            content = json.dumps(data)
            data = json.loads(content)
            cpu_info = data['ns5:Item']
            self.no_of_cpu = int(cpu_info['rasd:VirtualQuantity'])
        except Exception as err:
            self.log.exception("Exception in get_cores")
            raise Exception(err)

    def get_memory_info(self):
        """
        Get memory of VM

        Raises:
            Exception:
                    if failed to get CPU, memory information of VM
        """

        try:
            self.log.info("Get Memory info of a VM %s" % self.vm_name)

            response = requests.get(self.href + "/virtualHardwareSection/memory",
                                    headers=self.headers, auth=(self.user_name, self.password),
                                    verify=False)
            data = xmltodict.parse(response.content)
            content = json.dumps(data)
            data = json.loads(content)
            memory_info = data['ns5:Item']
            self.memory = int(memory_info['rasd:VirtualQuantity']) / 1024
        except Exception as err:
            self.log.exception("Exception in get_cores")
            raise Exception(err)

    def update_vm_info(self, prop='Basic', os_info=False, force_update=False):
        """
         fetches all the properties of the VM

         Args:
                 should have code for two possibilties

                 Basic - Basic properties of VM like cores,GUID,disk

                 All   - All the possible properties of the VM

                 Set the property VMGuestOS for creating OS Object

                 fetches some particular specified property

                 force_update - to refresh all the properties always
                    True : ALways collect  properties
                    False: refresh only if properties are not initialized

         Raises:
            Exception:
                 if failed to get all the properties of the VM
         """
        try:
            if self.vm_info:
                if not self._basic_props_initialized or force_update:
                    self.get_vm_guid()
                    self.get_disk_info()
                    self.get_status_of_vm()
                    self._basic_props_initialized = True

                if prop == 'All':
                    self.get_ip_address()
                    self.get_os_type()
                    self.get_cores()
                    self.get_memory_info()
                    self.vm_guest_os = self.guest_os
                    self.get_drive_list()
                elif hasattr(self, prop):
                    return getattr(self, prop, None)

            else:
                self.log.info("VM info was not collected for this VM")
        except Exception as err:
            self.log.exception("Failed to Get  the VM Properties of the VM")
            raise Exception(err)

    def power_off(self):
        """
        power off the VM.

        return:
                True - when power off is successfull

        Exception:
                When power off failed

        """
        try:
            if self.power_state == 'poweredOff':
                self.log.info("VM {} is already powered off".format(self.vm_name))
                return
            self.log.info("Powering off the vm {}".format(self.vm_name))
            response = requests.post(self.href + "/power/action/powerOff", headers=self.headers,
                                       auth=(self.user_name, self.password), verify=False)
            if response.status_code == 202:
                self.log.info("Powered off the VM")
                self.power_state = 'poweredOff'
        except Exception as exp:
            raise Exception("Exception in PowerOff:" + str(exp))

    def delete_vm(self):
        """
        Delete the VM.

        return:
                True - when Delete is successfull

        Exception:
                When Delete failed

        """
        try:
            self.log.info("Deleting the vm {}".format(self.vm_name))
            response = requests.delete(self.href + "?force=true", headers=self.headers,
                                       auth=(self.user_name, self.password), verify=False)
            if response.status_code == 202:
                self.log.info("Deleted the VM")
        except Exception as exp:
            raise Exception("Exception in Deletion:" + str(exp))

    def clean_up(self):
        """
        Clean up the VM resources post restore

        Raises:
             Exception:
                If unable to clean up VM and its resources

        """
        self.log.info("Deleting VMs/Instances after restore")
        self.delete_vm()