# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""Does all the Operation for OCI vm"""

import time
from AutomationUtils import machine
from AutomationUtils import logger
from VirtualServer.VSAUtils.VMHelper import HypervisorVM
from VirtualServer.VSAUtils.VirtualServerUtils import get_details_from_config_file


class OciVM(HypervisorVM):
    """
    This is the main file for all OCI VM operations
    Automation Requirements:
    1.) ICMP traffic needs to be enabled between OCI machine and OCI and testlab machines.
    2.) TCP must be enabled from OCI machines to Testlab machines - For accessing Admin Console
    VMware to OCI conversion
    1.) For conversion we need to have controller in the same subnet as that of restored VM
    2.) To acess Vm in Vcenter for conversion need to enable port no 22 443 and 5985
    """

    def __init__(self, hvobj, vm_name):
        """
        Initialization of OCI vm properties

        Args:
            hvobj               (obj):  Hypervisor Object

            vm_name             (str):  Name of the VM
        """

        super(OciVM, self).__init__(hvobj, vm_name)
        self.oci_config = hvobj.oci_config
        self.server_host_name = hvobj.server_host_name
        self.instance_type = hvobj.instance_type
        self.identity = hvobj.instance_type
        self.user = hvobj.instance_type
        self.root_compartment_id = hvobj.instance_type
        self.vm_name = vm_name
        self._basic_props_initialized = False
        self.power_state = ''
        self.no_of_cpu = ''
        self.disk_count = ''
        self.memory = ''
        self.ip = None
        self.guest_os = None
        self._disk_list = list()
        self._vol_list = list()
        self._disk_vol_dict = dict()
        self.compartment_name = None
        self.subnet_id = None
        self.datastore = None
        self.instance_size = None
        self.vcn_name = None
        self.ipv4 = None
        self.iqn = None
        self.update_vm_info()

    class VmConversionValidation(object):
        def __init__(self, vmobj, vm_restore_options):
            self.vm = vmobj
            self.vm_restore_options = vm_restore_options
            self.log = logger.get_log()

        def __eq__(self, other):

            """
            compares the restored vm properties after restore
            vm_restore_options - contains properties like Availability Domain , Shape, subnet, Compartment

            Return:
                value     (bool)- on successful validation return true
            """
            if self.vm.ipv4 is not None and self.vm.iqn is not None:
                try:
                    self.vm.machine.execute_command(command='Set-Service -Name msiscsi -StartupType Automatic')
                    self.vm.machine.execute_command(command='Start-Service msiscsi')
                    self.vm.machine.execute_command(
                        command='New-IscsiTargetPortal -TargetPortalAddress ' + self.vm.ipv4)
                    sc = 'Connect-IscsiTarget -NodeAddress ' + self.vm.iqn + ' -TargetPortalAddress ' + self.vm.ipv4 + ' -IsPersistent $True -IsMultipathEnabled $True'
                    self.vm.machine.execute_command(command=sc)
                    self.vm.machine.execute_command(command='Set-Disk -Number 1 -IsOffline $False')
                except Exception as exp:
                    self.log.error("Error while attaching secondary Block Volume")
                    raise Exception("Error while adding data disk to restored instance")
            return (self.vm.compartment_name == self.vm_restore_options.compartment_name and
                    self.vm.datastore == self.vm_restore_options.datastore and
                    self.vm.subnet_id == self.vm_restore_options.subnet_id and
                    self.vm.no_of_cpu == self.vm_restore_options.disk_info
                    )

    def vm_power_on(self):
        """
        Power ON the VM by name
        :return:
        """

        self.hvobj.ComputeClient.instance_action(self._get_vm_info()['instance_id'], 'START')

    def vm_power_off(self):
        """
        Power OFF the VM by name
        :return:
        """
        self.hvobj.ComputeClient.instance_action(self._get_vm_info()['instance_id'], 'STOP')

    def vm_power_reboot(self):
        """
        Restart the VM by name
        :return:
        """
        self.hvobj.ComputeClient.instance_action(self._get_vm_info()['instance_id'], 'REBOOT')

    def vm_power_state(self):
        """
        WIP
        :param vm_name:
        :return:
        """
        pass

    def delete_vm(self):
        """
        Terminate the VM.

        return:
                True - when delete is successful

                False - when delete failed
        """
        self.log.info("Delete the VM")
        self.hvobj.ComputeClient.terminate_instance(self._get_vm_info()['instance_id'])

    def clean_up(self):
        """
        Clean up the VM resources post restore

        Raises:
             Exception:
                If unable to clean up VM and its resources

        """
        self.log.info("Deleting VMs/Instances after restore")
        self.delete_vm()

    def _set_credentials(self, os_name):
        """
        Overridden because root login is not possible in out of place restored AWS instance.
        """

        try:
            sections = get_details_from_config_file('oci')
            keys = get_details_from_config_file('oci', 'keys')
            try:
                key_list = keys.split(",")
            except Exception as err:
                key_list = keys
            self.user_name = sections
            if "," in sections:
                if os_name == 'windows':
                    self.user_name = sections.split(",")[2]
                    self.password = sections.split(",")[3]
                else:
                    self.user_name = sections.split(",")[0]
                    self.password = sections.split(",")[1]
                vm_machine = machine.Machine(self.vm_hostname.strip(), username=self.user_name,
                                             password=self.password)
            else:
                self.user_name = sections
                self.password = ''
                vm_machine = machine.Machine(self.vm_hostname.strip(), username=self.user_name,
                                             password=self.password, key_filename=keys,
                                             run_as_sudo=True)
            if vm_machine:
                self.machine = vm_machine
                self.guest_os = self.machine.os_flavour.lower()
                return
        except Exception as err:
            self.log.error(str(err))
            self.log.exception("Could not create Machine object! The following user names are "
                               "incorrect: {0}")
            return

    @property
    def disk_list(self):
        """
        To fetch the disk in the VM

        Returns:
            disk_list           (list): List of volumes in AWS instance

        """
        op = []
        for k, v in self._disk_vol_dict.items():
            op.append(v)
        return op

    def _get_vm_info(self):
        """
        Collect all the VM properties for VM
        :return:
        """
        try:
            self.log.info(
                "Collecting all the VM properties for VM %s" % self.vm_name)
            op = self.hvobj.get_vm_property(self.vm_name)
            return op
        except Exception as err:
            self.log.exception("Failed to Get all the VM Properties of the VM")
            raise Exception(err)

    def update_vm_info(self, prop='Basic', os_info=False, force_update=False):
        """
        Update VM information

        :param prop: TBD
        :param os_info: TBD
        :param force_update: TBD
        :return:
        """
        try:
            if self._basic_props_initialized:
                self.log.info('No new properties to fetch')
                return
            vm_info = self._get_vm_info()
            if vm_info['power_state'].lower() != 'running':
                self.log.info('Turning on the VM: %s' % self.vm_name)
                self.vm_power_on()
                time.sleep(60)
                vm_info = self._get_vm_info()
            for key, value in vm_info['network'].items():
                try:
                    self.ip = value['Private IP']
                    break
                except:
                    pass
            for k, value in vm_info['block_vloume'].items():
                try:
                    if value['instance_id'] == vm_info['instance_id']:
                        self.ipv4 = value['ipv4']
                        self.iqn = value['iqn']
                        break
                except:
                    pass
            self.vm_guest_os = vm_info['OS']
            self.no_of_cpu = vm_info['instanceSize']
            self.memory = vm_info['instanceSize']
            self.disk_count = vm_info['instanceSize']
            if not self.ip:
                self.ip = vm_info['source_ip']
            try:
                self.compartment_name = vm_info["parent_compartment_name"]
                self.subnet_id = vm_info["subnetId"]
                self.datastore = vm_info["Datastore"]
                self.instance_size = vm_info["instanceSize"]
                self.vcn_name = vm_info["networkName"]
            except:
                self.log.info("Compartment Name, SubnetId, Datastore, Instance Size not Found.")
            self._basic_props_initialized = True

        except Exception as err:
            self.log.exception("Failed to Get  the VM Properties of the VM")
            raise Exception(err)

    def get_vm_ip(self):
        """
        gets public IP of a VM by name

        :return: public IP of the machine if any
        """
        return self._get_vm_info()['source_ip']

    def get_vm_ids(self):
        """
        This method can be used to get an instance id by its name
        :return:        (list)   Returns the ID of the VM
        """
        return self._get_vm_info()['instance_id']
